% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/avas.R
\name{avas}
\alias{avas}
\alias{avas.default}
\alias{avas.formula}
\alias{summary.avas}
\alias{print.avas}
\alias{plot.avas}
\title{Additivity and variance stabilization for regression}
\usage{
avas(...)

\method{avas}{default}(
  x,
  y,
  wt = NULL,
  cat = NULL,
  mon = NULL,
  lin = NULL,
  circ = NULL,
  delrsq = 0.01,
  yspan = 0,
  control = NULL,
  ...
)

\method{avas}{formula}(
  formula,
  data = NULL,
  subset = NULL,
  na.action = getOption("na.action"),
  ...
)

\method{summary}{avas}(object, ...)

\method{print}{avas}(x, ..., digits = 4)

\method{plot}{avas}(
  x,
  ...,
  which = 1:(x$p + 1),
  caption = c(list("Response Y AVAS Transformation"), as.list(paste("Carrier",
    rownames(x$x), "AVAS Transformation"))),
  xlab = "Original",
  ylab = "Transformed",
  ask = prod(par("mfcol")) < length(which) && dev.interactive()
)
}
\arguments{
\item{...}{additional arguments which go ignored for avas call. Included for S3 dispatch
consistency. They are utilized when using print as they get passed to cat. 
Also when plotting an ace object they are passed to plot.}

\item{x}{matrix containing the independent variables.}

\item{y}{a vector containing the response variable.}

\item{wt}{an optional vector of weights.}

\item{cat}{an optional integer vector specifying which variables
assume categorical values.  Positive values in \code{cat} refer
to columns of the \code{x} matrix and zero to the response
variable. Variables must be numeric, so a character variable
should first be transformed with as.numeric() and then specified}

\item{mon}{an optional integer vector specifying which variables are
to be transformed by monotone transformations. Positive values
in \code{mon} refer to columns of the \code{x} matrix and zero
to the response variable.}

\item{lin}{an optional integer vector specifying which variables are
to be transformed by linear transformations. Positive values in
\code{lin} refer to columns of the \code{x} matrix and zero to
the response variable.}

\item{circ}{an integer vector specifying which variables assume
circular (periodic) values. Positive values in \code{circ}
refer to columns of the \code{x} matrix and zero to the response
variable.}

\item{delrsq}{numeric(1); Termination threshold for iteration. Stops when
R-squared changes by less than \code{delrsq} in 3 consecutive iterations
(default 0.01).}

\item{yspan}{yspan Optional window size parameter for smoothing the
variance.  Range is \eqn{[0,1]}.  Default is 0 (cross validated
choice). .5 is a reasonable alternative to try.}

\item{control}{named list; control parameters to set. Documented at 
\code{\link{set_control}}.}

\item{formula}{formula; an object of class "\code{\link{formula}}": a
symbolic description of the model to be smoothed.}

\item{data}{an optional data frame, list or environment (or object coercible
by \code{\link{as.data.frame}} to a data frame) containing the variables in
the model. If not found in data, the variables are taken from
\code{environment(formula)}, typically the environment from which
\code{ace} is called.}

\item{subset}{an optional vector specifying a subset of observations to be
used in the fitting process. Only used when a \code{formula}
is specified.}

\item{na.action}{a function which indicates what should happen when the data
contain NAs. The default is set by the \code{na.action} setting of
\code{\link{options}}, and is \code{\link{na.fail}} if that is unset.
The ‘factory-fresh’ default is \code{\link{na.omit}}. Another possible
value is NULL, no action. Value \code{\link{na.exclude}} can be useful.}

\item{object}{an S3 ace object}

\item{digits}{rounding digits for summary/print}

\item{which}{when plotting an ace object which plots to produce.}

\item{caption}{a list of captions for a plot.}

\item{xlab}{the x-axis label when plotting.}

\item{ylab}{the y-axis label when plotting.}

\item{ask}{when plotting should the terminal be asked for input between plots.}
}
\value{
A structure with the following components:
    \item{x}{the input x matrix.}
    \item{y}{the input y vector.}
    \item{tx}{the transformed x values.}
    \item{ty}{the transformed y values.}
    \item{rsq}{the multiple R-squared value for the transformed values.}
    \item{l}{the codes for cat, mon, ...}
    \item{m}{not used in this version of avas}
    \item{yspan}{span used for smoothing the variance}
    \item{iters}{iteration number and rsq for that iteration}
    \item{niters}{number of iterations used}
}
\description{
Estimate transformations of \code{x} and \code{y} such that
  the regression of \code{y} on \code{x} is approximately linear with
  constant variance
}
\examples{

TWOPI <- 8*atan(1)
x <- runif(200,0,TWOPI)
y <- exp(sin(x)+rnorm(200)/2)
a <- avas(x,y)
plot(a) # View response and carrier transformations
plot(a$tx,a$ty) # examine the linearity of the fitted model

# From D. Wang and M. Murphy (2005), Identifying nonlinear relationships
# regression using the ACE algorithm.  Journal of Applied Statistics,
# 32, 243-258, adapted for avas.
X1 <- runif(100)*2-1
X2 <- runif(100)*2-1
X3 <- runif(100)*2-1
X4 <- runif(100)*2-1

# Original equation of Y:
Y <- log(4 + sin(3*X1) + abs(X2) + X3^2 + X4 + .1*rnorm(100))

# Transformed version so that Y, after transformation, is a
# linear function of transforms of the X variables:
# exp(Y) = 4 + sin(3*X1) + abs(X2) + X3^2 + X4

a1 <- avas(cbind(X1,X2,X3,X4),Y)

par(mfrow=c(2,1))

# For each variable, show its transform as a function of
# the original variable and the of the transform that created it,
# showing that the transform is recovered.
plot(X1,a1$tx[,1])
plot(sin(3*X1),a1$tx[,1])

plot(X2,a1$tx[,2])
plot(abs(X2),a1$tx[,2])

plot(X3,a1$tx[,3])
plot(X3^2,a1$tx[,3])

plot(X4,a1$tx[,4])
plot(X4,a1$tx[,4])

plot(Y,a1$ty)
plot(exp(Y),a1$ty)

}
\references{
Rob Tibshirani (1987),
    ``Estimating optimal transformations for regression''. 
    \emph{Journal of the American Statistical Association} \bold{83},
    394ff.
}
