/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

// Test from official test vectors in
// https://github.com/usnistgov/ACVP-Server/tree/v1.1.0.35/gen-val/json-files/ML-KEM-keyGen-FIPS203

#include <vector>

#include "kyber.h"

const uint8_t MlKem768DecapPrivateKey[KYBER768_PRIVATE_KEY_BYTES] = {
    0x1E, 0x4A, 0xC8, 0x7B, 0x1A, 0x69, 0x2A, 0x52, 0x9F, 0xDB, 0xBA, 0xB9,
    0x33, 0x74, 0xC5, 0x7D, 0x11, 0x0B, 0x10, 0xF2, 0xB1, 0xDD, 0xEB, 0xAC,
    0x0D, 0x19, 0x6B, 0x7B, 0xA6, 0x31, 0xB8, 0xE9, 0x29, 0x30, 0x28, 0xA8,
    0xF3, 0x79, 0x88, 0x8C, 0x42, 0x2D, 0xC8, 0xD3, 0x2B, 0xBF, 0x22, 0x60,
    0x10, 0xC2, 0xC1, 0xEC, 0x73, 0x18, 0x90, 0x80, 0x45, 0x6B, 0x05, 0x64,
    0xB2, 0x58, 0xB0, 0xF2, 0x31, 0x31, 0xBC, 0x79, 0xC8, 0xE8, 0xC1, 0x1C,
    0xEF, 0x39, 0x38, 0xB2, 0x43, 0xC5, 0xCE, 0x9C, 0x0E, 0xDD, 0x37, 0xC8,
    0xF9, 0xD2, 0x98, 0x77, 0xDB, 0xBB, 0x61, 0x5B, 0x9B, 0x5A, 0xC3, 0xC9,
    0x48, 0x48, 0x7E, 0x46, 0x71, 0x96, 0xA9, 0x14, 0x3E, 0xFB, 0xC7, 0xCE,
    0xDB, 0x64, 0xB4, 0x5D, 0x4A, 0xCD, 0xA2, 0x66, 0x6C, 0xBC, 0x28, 0x04,
    0xF2, 0xC8, 0x66, 0x2E, 0x12, 0x8F, 0x6A, 0x99, 0x69, 0xEC, 0x15, 0xBC,
    0x0B, 0x93, 0x51, 0xF6, 0xF9, 0x63, 0x46, 0xAA, 0x7A, 0xBC, 0x74, 0x3A,
    0x14, 0xFA, 0x03, 0x0E, 0x37, 0xA2, 0xE7, 0x59, 0x7B, 0xDD, 0xFC, 0x5A,
    0x22, 0xF9, 0xCE, 0xDA, 0xF8, 0x61, 0x48, 0x32, 0x52, 0x72, 0x10, 0xB2,
    0x6F, 0x02, 0x4C, 0x7F, 0x6C, 0x0D, 0xCF, 0x55, 0x1E, 0x97, 0xA4, 0x85,
    0x87, 0x64, 0xC3, 0x21, 0xD1, 0x83, 0x4A, 0xD5, 0x1D, 0x75, 0xBB, 0x24,
    0x6D, 0x27, 0x72, 0x37, 0xB7, 0xBD, 0x41, 0xDC, 0x43, 0x62, 0xD0, 0x63,
    0xF4, 0x29, 0x82, 0x92, 0x27, 0x2D, 0x01, 0x01, 0x17, 0x80, 0xB7, 0x98,
    0x56, 0xB2, 0x96, 0xC4, 0xE9, 0x46, 0x65, 0x8B, 0x79, 0x60, 0x31, 0x97,
    0xC9, 0xB2, 0xA9, 0x9E, 0xC6, 0x6A, 0xCB, 0x06, 0xCE, 0x2F, 0x69, 0xB5,
    0xA5, 0xA6, 0x1E, 0x9B, 0xD0, 0x6A, 0xD4, 0x43, 0xCE, 0xB0, 0xC7, 0x4E,
    0xD6, 0x53, 0x45, 0xA9, 0x03, 0xB6, 0x14, 0xE8, 0x13, 0x68, 0xAA, 0xC2,
    0xB3, 0xD2, 0xA7, 0x9C, 0xA8, 0xCC, 0xAA, 0x1C, 0x3B, 0x88, 0xFB, 0x82,
    0xA3, 0x66, 0x32, 0x86, 0x0B, 0x3F, 0x79, 0x50, 0x83, 0x3F, 0xD0, 0x21,
    0x2E, 0xC9, 0x6E, 0xDE, 0x4A, 0xB6, 0xF5, 0xA0, 0xBD, 0xA3, 0xEC, 0x60,
    0x60, 0xA6, 0x58, 0xF9, 0x45, 0x7F, 0x6C, 0xC8, 0x7C, 0x6B, 0x62, 0x0C,
    0x1A, 0x14, 0x51, 0x98, 0x74, 0x86, 0xE4, 0x96, 0x61, 0x2A, 0x10, 0x1D,
    0x0E, 0x9C, 0x20, 0x57, 0x7C, 0x57, 0x1E, 0xDB, 0x52, 0x82, 0x60, 0x8B,
    0xF4, 0xE1, 0xAC, 0x92, 0x6C, 0x0D, 0xB1, 0xC8, 0x2A, 0x50, 0x4A, 0x79,
    0x9D, 0x89, 0x88, 0x5C, 0xA6, 0x25, 0x2B, 0xD5, 0xB1, 0xC1, 0x83, 0xAF,
    0x70, 0x13, 0x92, 0xA4, 0x07, 0xC0, 0x5B, 0x84, 0x8C, 0x2A, 0x30, 0x16,
    0xC4, 0x06, 0x13, 0xF0, 0x2A, 0x44, 0x9B, 0x3C, 0x79, 0x26, 0xDA, 0x06,
    0x7A, 0x53, 0x31, 0x16, 0x50, 0x68, 0x40, 0x09, 0x75, 0x10, 0x46, 0x0B,
    0xBF, 0xD3, 0x60, 0x73, 0xDC, 0xB0, 0xBF, 0xA0, 0x09, 0xB3, 0x6A, 0x91,
    0x23, 0xEA, 0xA6, 0x8F, 0x83, 0x5F, 0x74, 0xA0, 0x1B, 0x00, 0xD2, 0x09,
    0x78, 0x35, 0x96, 0x4D, 0xF5, 0x21, 0xCE, 0x92, 0x10, 0x78, 0x9C, 0x30,
    0xB7, 0xF0, 0x6E, 0x58, 0x44, 0xB4, 0x44, 0xC5, 0x33, 0x22, 0x39, 0x6E,
    0x47, 0x99, 0xBA, 0xF6, 0xA8, 0x8A, 0xF7, 0x31, 0x58, 0x60, 0xD0, 0x19,
    0x2D, 0x48, 0xC2, 0xC0, 0xDA, 0x6B, 0x5B, 0xA6, 0x43, 0x25, 0x54, 0x3A,
    0xCD, 0xF5, 0x90, 0x0E, 0x8B, 0xC4, 0x77, 0xAB, 0x05, 0x82, 0x00, 0x72,
    0xD4, 0x63, 0xAF, 0xFE, 0xD0, 0x97, 0xE0, 0x62, 0xBD, 0x78, 0xC9, 0x9D,
    0x12, 0xB3, 0x85, 0x13, 0x1A, 0x24, 0x1B, 0x70, 0x88, 0x65, 0xB4, 0x19,
    0x0A, 0xF6, 0x9E, 0xA0, 0xA6, 0x4D, 0xB7, 0x14, 0x48, 0xA6, 0x08, 0x29,
    0x36, 0x9C, 0x75, 0x55, 0x19, 0x8E, 0x43, 0x8C, 0x9A, 0xBC, 0x31, 0x0B,
    0xC7, 0x01, 0x01, 0x91, 0x3B, 0xB1, 0x2F, 0xAA, 0x5B, 0xEE, 0xF9, 0x75,
    0x84, 0x16, 0x17, 0xC8, 0x47, 0xCD, 0x6B, 0x33, 0x6F, 0x87, 0x79, 0x87,
    0x75, 0x38, 0x22, 0x02, 0x0B, 0x92, 0xC4, 0xCC, 0x97, 0x05, 0x5C, 0x9B,
    0x1E, 0x0B, 0x12, 0x8B, 0xF1, 0x1F, 0x50, 0x50, 0x05, 0xB6, 0xAB, 0x0E,
    0x62, 0x77, 0x95, 0xA2, 0x06, 0x09, 0xEF, 0xA9, 0x91, 0xE5, 0x98, 0xB8,
    0x0F, 0x37, 0xB1, 0xC6, 0xA1, 0xC3, 0xA1, 0xE9, 0xAE, 0xE7, 0x02, 0x8F,
    0x77, 0x57, 0x0A, 0xB2, 0x13, 0x91, 0x28, 0xA0, 0x01, 0x08, 0xC5, 0x0E,
    0xB3, 0x05, 0xCD, 0xB8, 0xF9, 0xA6, 0x03, 0xA6, 0xB0, 0x78, 0x41, 0x3F,
    0x6F, 0x9B, 0x14, 0xC6, 0xD8, 0x2B, 0x51, 0x99, 0xCE, 0x59, 0xD8, 0x87,
    0x90, 0x2A, 0x28, 0x1A, 0x02, 0x7B, 0x71, 0x74, 0x95, 0xFE, 0x12, 0x67,
    0x2A, 0x12, 0x7B, 0xBF, 0x9B, 0x25, 0x6C, 0x43, 0x72, 0x0D, 0x7C, 0x16,
    0x0B, 0x28, 0x1C, 0x12, 0x75, 0x7D, 0xA1, 0x35, 0xB1, 0x93, 0x33, 0x52,
    0xBE, 0x4A, 0xB6, 0x7E, 0x40, 0x24, 0x8A, 0xFC, 0x31, 0x8E, 0x23, 0x70,
    0xC3, 0xB8, 0x20, 0x8E, 0x69, 0x5B, 0xDF, 0x33, 0x74, 0x59, 0xB9, 0xAC,
    0xBF, 0xE5, 0xB4, 0x87, 0xF7, 0x6E, 0x9B, 0x4B, 0x40, 0x01, 0xD6, 0xCF,
    0x90, 0xCA, 0x8C, 0x69, 0x9A, 0x17, 0x4D, 0x42, 0x97, 0x2D, 0xC7, 0x33,
    0xF3, 0x33, 0x89, 0xFD, 0xF5, 0x9A, 0x1D, 0xAB, 0xA8, 0x1D, 0x83, 0x49,
    0x55, 0x02, 0x73, 0x34, 0x18, 0x5A, 0xD0, 0x2C, 0x76, 0xCF, 0x29, 0x48,
    0x46, 0xCA, 0x92, 0x94, 0xBA, 0x0E, 0xD6, 0x67, 0x41, 0xDD, 0xEC, 0x79,
    0x1C, 0xAB, 0x34, 0x19, 0x6A, 0xC5, 0x65, 0x7C, 0x5A, 0x78, 0x32, 0x1B,
    0x56, 0xC3, 0x33, 0x06, 0xB5, 0x10, 0x23, 0x97, 0xA5, 0xC0, 0x9C, 0x35,
    0x08, 0xF7, 0x6B, 0x48, 0x28, 0x24, 0x59, 0xF8, 0x1D, 0x0C, 0x72, 0xA4,
    0x3F, 0x73, 0x7B, 0xC2, 0xF1, 0x2F, 0x45, 0x42, 0x26, 0x28, 0xB6, 0x7D,
    0xB5, 0x1A, 0xC1, 0x42, 0x42, 0x76, 0xA6, 0xC0, 0x8C, 0x3F, 0x76, 0x15,
    0x66, 0x5B, 0xBB, 0x8E, 0x92, 0x81, 0x48, 0xA2, 0x70, 0xF9, 0x91, 0xBC,
    0xF3, 0x65, 0xA9, 0x0F, 0x87, 0xC3, 0x06, 0x87, 0xB6, 0x88, 0x09, 0xC9,
    0x1F, 0x23, 0x18, 0x13, 0xB8, 0x66, 0xBE, 0xA8, 0x2E, 0x30, 0x37, 0x4D,
    0x80, 0xAA, 0x0C, 0x02, 0x97, 0x34, 0x37, 0x49, 0x8A, 0x53, 0xB1, 0x4B,
    0xF6, 0xB6, 0xCA, 0x1E, 0xD7, 0x6A, 0xB8, 0xA2, 0x0D, 0x54, 0xA0, 0x83,
    0xF4, 0xA2, 0x6B, 0x7C, 0x03, 0x8D, 0x81, 0x96, 0x76, 0x40, 0xC2, 0x0B,
    0xF4, 0x43, 0x1E, 0x71, 0xDA, 0xCC, 0xE8, 0x57, 0x7B, 0x21, 0x24, 0x0E,
    0x49, 0x4C, 0x31, 0xF2, 0xD8, 0x77, 0xDA, 0xF4, 0x92, 0x4F, 0xD3, 0x9D,
    0x82, 0xD6, 0x16, 0x7F, 0xBC, 0xC1, 0xF9, 0xC5, 0xA2, 0x59, 0xF8, 0x43,
    0xE3, 0x09, 0x87, 0xCC, 0xC4, 0xBC, 0xE7, 0x49, 0x3A, 0x24, 0x04, 0xB5,
    0xE4, 0x43, 0x87, 0xF7, 0x07, 0x42, 0x57, 0x81, 0xB7, 0x43, 0xFB, 0x55,
    0x56, 0x85, 0x58, 0x4E, 0x25, 0x57, 0xCC, 0x03, 0x8B, 0x1A, 0x9B, 0x3F,
    0x40, 0x43, 0x12, 0x1F, 0x54, 0x72, 0xEB, 0x2B, 0x96, 0xE5, 0x94, 0x1F,
    0xEC, 0x01, 0x1C, 0xEE, 0xA5, 0x07, 0x91, 0x63, 0x6C, 0x6A, 0xBC, 0x26,
    0xC1, 0x37, 0x7E, 0xE3, 0xB5, 0x14, 0x6F, 0xC7, 0xC8, 0x5C, 0xB3, 0x35,
    0xB1, 0xE7, 0x95, 0xEE, 0xC2, 0x03, 0x3E, 0xE4, 0x4B, 0x9A, 0xA9, 0x06,
    0x85, 0x24, 0x5E, 0xF7, 0xB4, 0x43, 0x6C, 0x00, 0x0E, 0x66, 0xBC, 0x8B,
    0xCB, 0xF1, 0xCD, 0xB8, 0x03, 0xAC, 0x14, 0x21, 0xB1, 0xFD, 0xB2, 0x66,
    0xD5, 0x29, 0x1C, 0x83, 0x10, 0x37, 0x3A, 0x8A, 0x3C, 0xE9, 0x56, 0x2A,
    0xB1, 0x97, 0x95, 0x38, 0x71, 0xAB, 0x99, 0xF3, 0x82, 0xCC, 0x5A, 0xA9,
    0xC0, 0xF2, 0x73, 0xD1, 0xDC, 0xA5, 0x5D, 0x27, 0x12, 0x85, 0x38, 0x71,
    0xE1, 0xA8, 0x3C, 0xB3, 0xB8, 0x54, 0x50, 0xF7, 0x6D, 0x3F, 0x3C, 0x42,
    0xBA, 0xB5, 0x50, 0x5F, 0x72, 0x12, 0xFD, 0xB6, 0xB8, 0xB7, 0xF6, 0x02,
    0x99, 0x72, 0xA8, 0xF3, 0x75, 0x1E, 0x4C, 0x94, 0xC1, 0x10, 0x8B, 0x02,
    0xD6, 0xAC, 0x79, 0xF8, 0xD9, 0x38, 0xF0, 0x5A, 0x1B, 0x2C, 0x22, 0x9B,
    0x14, 0xB4, 0x2B, 0x31, 0xB0, 0x1A, 0x36, 0x40, 0x17, 0xE5, 0x95, 0x78,
    0xC6, 0xB0, 0x33, 0x83, 0x37, 0x74, 0xCB, 0x9B, 0x57, 0x0F, 0x90, 0x86,
    0xB7, 0x22, 0x90, 0x3B, 0x37, 0x54, 0x46, 0xB4, 0x95, 0xD8, 0xA2, 0x9B,
    0xF8, 0x07, 0x51, 0x87, 0x7A, 0x80, 0xFB, 0x72, 0x4A, 0x02, 0x10, 0xC3,
    0xE1, 0x69, 0x2F, 0x39, 0x7C, 0x2F, 0x1D, 0xDC, 0x2E, 0x6B, 0xA1, 0x7A,
    0xF8, 0x1B, 0x92, 0xAC, 0xFA, 0xBE, 0xF5, 0xF7, 0x57, 0x3C, 0xB4, 0x93,
    0xD1, 0x84, 0x02, 0x7B, 0x71, 0x82, 0x38, 0xC8, 0x9A, 0x35, 0x49, 0xB8,
    0x90, 0x5B, 0x28, 0xA8, 0x33, 0x62, 0x86, 0x7C, 0x08, 0x2D, 0x30, 0x19,
    0xD3, 0xCA, 0x70, 0x70, 0x07, 0x31, 0xCE, 0xB7, 0x3E, 0x84, 0x72, 0xC1,
    0xA3, 0xA0, 0x93, 0x36, 0x1C, 0x5F, 0xEA, 0x6A, 0x7D, 0x40, 0x95, 0x5D,
    0x07, 0xA4, 0x1B, 0x64, 0xE5, 0x00, 0x81, 0xA3, 0x61, 0xB6, 0x04, 0xCC,
    0x51, 0x84, 0x47, 0xC8, 0xE2, 0x57, 0x65, 0xAB, 0x7D, 0x68, 0xB2, 0x43,
    0x27, 0x52, 0x07, 0xAF, 0x8C, 0xA6, 0x56, 0x4A, 0x4C, 0xB1, 0xE9, 0x41,
    0x99, 0xDB, 0xA1, 0x87, 0x8C, 0x59, 0xBE, 0xC8, 0x09, 0xAB, 0x48, 0xB2,
    0xF2, 0x11, 0xBA, 0xDC, 0x6A, 0x19, 0x98, 0xD9, 0xC7, 0x22, 0x7C, 0x13,
    0x03, 0xF4, 0x69, 0xD4, 0x6A, 0x9C, 0x7E, 0x53, 0x03, 0xF9, 0x8A, 0xBA,
    0x67, 0x56, 0x9A, 0xE8, 0x22, 0x7C, 0x16, 0xBA, 0x1F, 0xB3, 0x24, 0x44,
    0x66, 0xA2, 0x5E, 0x7F, 0x82, 0x36, 0x71, 0x81, 0x0C, 0xC2, 0x62, 0x06,
    0xFE, 0xB2, 0x9C, 0x7E, 0x2A, 0x1A, 0x91, 0x95, 0x9E, 0xEB, 0x03, 0xA9,
    0x82, 0x52, 0xA4, 0xF7, 0x41, 0x26, 0x74, 0xEB, 0x9A, 0x4B, 0x27, 0x7E,
    0x1F, 0x25, 0x95, 0xFC, 0xA6, 0x40, 0x33, 0xB4, 0x1B, 0x40, 0x33, 0x08,
    0x12, 0xE9, 0x73, 0x5B, 0x7C, 0x60, 0x75, 0x01, 0xCD, 0x81, 0x83, 0xA2,
    0x2A, 0xFC, 0x33, 0x92, 0x55, 0x37, 0x44, 0xF3, 0x3C, 0x4D, 0x20, 0x25,
    0x26, 0x94, 0x5C, 0x6D, 0x78, 0xA6, 0x0E, 0x20, 0x1A, 0x16, 0x98, 0x7A,
    0x6F, 0xA5, 0x9D, 0x94, 0x46, 0x4B, 0x56, 0x50, 0x65, 0x56, 0x78, 0x48,
    0x24, 0xA0, 0x70, 0x58, 0xF5, 0x73, 0x20, 0xE7, 0x6C, 0x82, 0x5B, 0x93,
    0x47, 0xF2, 0x93, 0x6F, 0x4A, 0x0E, 0x5C, 0xDA, 0xA1, 0x8C, 0xF8, 0x83,
    0x39, 0x45, 0xAE, 0x31, 0x2A, 0x36, 0xB5, 0xF5, 0xA3, 0x81, 0x0A, 0xAC,
    0x82, 0x38, 0x1F, 0xDA, 0xE4, 0xCB, 0x9C, 0x68, 0x31, 0xD8, 0xEB, 0x8A,
    0xBA, 0xB8, 0x50, 0x41, 0x64, 0x43, 0xD7, 0x39, 0x08, 0x6B, 0x1C, 0x32,
    0x6F, 0xC2, 0xA3, 0x97, 0x57, 0x04, 0xE3, 0x96, 0xA5, 0x96, 0x80, 0xC3,
    0xB5, 0xF3, 0x60, 0xF5, 0x48, 0x0D, 0x2B, 0x62, 0x16, 0x9C, 0xD9, 0x4C,
    0xA7, 0x1B, 0x37, 0xBC, 0x58, 0x78, 0xBA, 0x29, 0x85, 0xE0, 0x68, 0xBA,
    0x05, 0x0B, 0x2C, 0xE5, 0x07, 0x26, 0xD4, 0xB4, 0x45, 0x1B, 0x77, 0xAA,
    0xA8, 0x67, 0x6E, 0xAE, 0x09, 0x49, 0x82, 0x21, 0x01, 0x92, 0x19, 0x7B,
    0x1E, 0x92, 0xA2, 0x7F, 0x59, 0x86, 0x8B, 0x78, 0x86, 0x78, 0x87, 0xB9,
    0xA7, 0x0C, 0x32, 0xAF, 0x84, 0x63, 0x0A, 0xA9, 0x08, 0x81, 0x43, 0x79,
    0xE6, 0x51, 0x91, 0x50, 0xBA, 0x16, 0x43, 0x9B, 0x5E, 0x2B, 0x06, 0x03,
    0xD0, 0x6A, 0xA6, 0x67, 0x45, 0x57, 0xF5, 0xB0, 0x98, 0x3E, 0x5C, 0xB6,
    0xA9, 0x75, 0x96, 0x06, 0x9B, 0x01, 0xBB, 0x31, 0x28, 0xC4, 0x16, 0x68,
    0x06, 0x57, 0x20, 0x4F, 0xD0, 0x76, 0x40, 0x39, 0x2E, 0x16, 0xB1, 0x9F,
    0x33, 0x7A, 0x99, 0xA3, 0x04, 0x84, 0x4E, 0x1A, 0xA4, 0x74, 0xE9, 0xC7,
    0x99, 0x06, 0x29, 0x71, 0xF6, 0x72, 0x26, 0x89, 0x60, 0xF5, 0xA8, 0x2F,
    0x95, 0x00, 0x70, 0xBB, 0xE9, 0xC2, 0xA7, 0x19, 0x50, 0xA3, 0x78, 0x5B,
    0xDF, 0x0B, 0x84, 0x40, 0x25, 0x5E, 0xD6, 0x39, 0x28, 0xD2, 0x57, 0x84,
    0x51, 0x68, 0xB1, 0xEC, 0xCC, 0x41, 0x91, 0x32, 0x5A, 0xA7, 0x66, 0x45,
    0x71, 0x9B, 0x28, 0xEB, 0xD8, 0x93, 0x02, 0xDC, 0x67, 0x23, 0xC7, 0x86,
    0xDF, 0x52, 0x17, 0xB2, 0x43, 0x09, 0x9C, 0xA7, 0x82, 0x38, 0xE5, 0x7E,
    0x64, 0x69, 0x2F, 0x20, 0x6B, 0x17, 0x7A, 0xBC, 0x25, 0x96, 0x60, 0x39,
    0x5C, 0xD7, 0x86, 0x0F, 0xB3, 0x5A, 0x16, 0xF6, 0xB2, 0xFE, 0x65, 0x48,
    0xC8, 0x5A, 0xB6, 0x63, 0x30, 0xC5, 0x17, 0xFA, 0x74, 0xCD, 0xF3, 0xCB,
    0x49, 0xD2, 0x6B, 0x11, 0x81, 0x90, 0x1A, 0xF7, 0x75, 0xA1, 0xE1, 0x80,
    0x81, 0x3B, 0x6A, 0x24, 0xC4, 0x56, 0x82, 0x9B, 0x5C, 0x38, 0x10, 0x4E,
    0xCE, 0x43, 0xC7, 0x6A, 0x43, 0x7A, 0x6A, 0x33, 0xB6, 0xFC, 0x6C, 0x5E,
    0x65, 0xC8, 0xA8, 0x94, 0x66, 0xC1, 0x42, 0x54, 0x85, 0xB2, 0x9B, 0x9E,
    0x18, 0x54, 0x36, 0x8A, 0xFC, 0xA3, 0x53, 0xE1, 0x43, 0xD0, 0xA9, 0x0A,
    0x6C, 0x6C, 0x9E, 0x7F, 0xDB, 0x62, 0xA6, 0x06, 0x85, 0x6B, 0x56, 0x14,
    0xF1, 0x2B, 0x64, 0xB7, 0x96, 0x02, 0x0C, 0x35, 0x34, 0xC3, 0x60, 0x5C,
    0xFD, 0xC7, 0x3B, 0x86, 0x71, 0x4F, 0x41, 0x18, 0x50, 0x22, 0x8A, 0x28,
    0xB8, 0xF4, 0xB4, 0x9E, 0x66, 0x34, 0x16, 0xC8, 0x4F, 0x7E, 0x38, 0x1F,
    0x6A, 0xF1, 0x07, 0x13, 0x43, 0xBF, 0x9D, 0x39, 0xB4, 0x54, 0x39, 0x24,
    0x0C, 0xC0, 0x38, 0x97, 0x29, 0x5F, 0xEA, 0x08, 0x0B, 0x14, 0xBB, 0x2D,
    0x81, 0x19, 0xA8, 0x80, 0xE1, 0x64, 0x49, 0x5C, 0x61, 0xBE, 0xBC, 0x71,
    0x39, 0xC1, 0x18, 0x57, 0xC8, 0x5E, 0x17, 0x50, 0x33, 0x8D, 0x63, 0x43,
    0x91, 0x37, 0x06, 0xA5, 0x07, 0xC9, 0x56, 0x64, 0x64, 0xCD, 0x28, 0x37,
    0xCF, 0x91, 0x4D, 0x1A, 0x3C, 0x35, 0xE8, 0x9B, 0x23, 0x5C, 0x6A, 0xB7,
    0xED, 0x07, 0x8B, 0xED, 0x23, 0x47, 0x57, 0xC0, 0x2E, 0xF6, 0x99, 0x3D,
    0x4A, 0x27, 0x3C, 0xB8, 0x15, 0x05, 0x28, 0xDA, 0x4D, 0x76, 0x70, 0x81,
    0x77, 0xE9, 0x42, 0x55, 0x46, 0xC8, 0x3E, 0x14, 0x70, 0x39, 0x76, 0x66,
    0x03, 0xB3, 0x0D, 0xA6, 0x26, 0x8F, 0x45, 0x98, 0xA5, 0x31, 0x94, 0x24,
    0x0A, 0x28, 0x32, 0xA3, 0xD6, 0x75, 0x33, 0xB5, 0x05, 0x6F, 0x9A, 0xAA,
    0xC6, 0x1B, 0x4B, 0x17, 0xB9, 0xA2, 0x69, 0x3A, 0xA0, 0xD5, 0x88, 0x91,
    0xE6, 0xCC, 0x56, 0xCD, 0xD7, 0x72, 0x41, 0x09, 0x00, 0xC4, 0x05, 0xAF,
    0x20, 0xB9, 0x03, 0x79, 0x7C, 0x64, 0x87, 0x69, 0x15, 0xC3, 0x7B, 0x84,
    0x87, 0xA1, 0x44, 0x9C, 0xE9, 0x24, 0xCD, 0x34, 0x5C, 0x29, 0xA3, 0x6E,
    0x08, 0x23, 0x8F, 0x7A, 0x15, 0x7C, 0xC7, 0xE5, 0x16, 0xAB, 0x5B, 0xA7,
    0x3C, 0x80, 0x63, 0xF7, 0x26, 0xBB, 0x5A, 0x0A, 0x03, 0x19, 0xE5, 0x71,
    0x27, 0x43, 0x8C, 0x7F, 0xC6, 0x01, 0xC9, 0x9C, 0xCA, 0xAE, 0x4C, 0x1A,
    0x83, 0x72, 0x6F, 0xDC, 0xB5, 0x04, 0x5E, 0xD1, 0xA8, 0x2A, 0x98, 0x5E,
    0xA9, 0x95, 0x39, 0x6D, 0x77, 0x27, 0x2C, 0x66, 0xCE, 0x49, 0x32, 0x89,
    0xF6, 0x11, 0x09, 0x10, 0xF3, 0x7C, 0x27, 0x41, 0xCE, 0x47, 0x02, 0x6A,
    0x6F, 0x82, 0x61, 0x99, 0x9C, 0x64, 0x82, 0x57, 0x2B, 0x16, 0x93, 0x91,
    0x2E, 0xF1, 0x2E, 0xEB, 0xEA, 0x7A, 0xCF, 0x92, 0x34, 0xFB, 0x40, 0x9F,
    0x2A, 0x60, 0x90, 0xE6, 0xB0, 0xBF, 0xD8, 0x95, 0x46, 0x9D, 0x0B, 0x2A,
    0x92, 0x1B, 0xB7, 0x23, 0xF8, 0x7A, 0x33, 0xEA, 0x54, 0x65, 0xAB, 0x90,
    0xF5, 0x14, 0xB6, 0x76, 0x98, 0xC0, 0x76, 0x8B, 0x6C, 0xA4, 0x98, 0xB0,
    0x22, 0xC5, 0x12, 0xFA, 0x08, 0x75, 0xF0, 0x54, 0xAA, 0x22, 0x65, 0x86,
    0x7E, 0x31, 0xC0, 0xE5, 0x22, 0x65, 0x1E, 0x02, 0x4A, 0x07, 0xD6, 0x0D,
    0xD9, 0xF6, 0x33, 0x16, 0x69, 0x21, 0xF4, 0x12, 0x6B, 0xC2, 0xB6, 0xAA,
    0x01, 0xCC, 0x15, 0xA0, 0x9B, 0x85, 0xBF, 0xF8, 0x21, 0x8C, 0x5A, 0xAE,
    0x95, 0xBC, 0x1F, 0xFB, 0x26, 0xAE, 0x5A, 0x13, 0x76, 0x70, 0xF0, 0x49,
    0x10, 0xCA, 0x9D, 0x72, 0x41, 0xB6, 0x66, 0x0C, 0x39, 0x4C, 0x54, 0x55,
    0x91, 0x77, 0x46, 0xA2, 0x66, 0x82, 0xFB, 0x71, 0xA4, 0x32, 0xEA, 0x95,
    0x30, 0xE8, 0x39, 0xBD, 0xEB, 0x07, 0x43, 0x30, 0x04, 0xF4, 0x5A, 0x0D,
    0xDA, 0xA0, 0xB2, 0x4E, 0x3A, 0x56, 0x6A, 0x54, 0x08, 0x15, 0xF2, 0x81,
    0xE3, 0xFC, 0x25, 0x9A, 0xC6, 0xCB, 0xC0, 0xAC, 0xB8, 0xD6, 0x22, 0x68,
    0xB6, 0x03, 0xBC, 0x67, 0x6A, 0xB4, 0x15, 0xC4, 0x74, 0xBB, 0x94, 0x87,
    0x3E, 0x44, 0x87, 0xAE, 0x31, 0xA4, 0xE3, 0x84, 0x5C, 0x79, 0x90, 0x15,
    0x50, 0x89, 0x0E, 0xE8, 0x78, 0x4E, 0xEF, 0x90, 0x4F, 0xEE, 0x62, 0xBA,
    0x8C, 0x5F, 0x95, 0x2C, 0x68, 0x41, 0x30, 0x52, 0xE0, 0xA7, 0xE3, 0x38,
    0x8B, 0xB8, 0xFF, 0x0A, 0xD6, 0x02, 0xAE, 0x3E, 0xA1, 0x4D, 0x9D, 0xF6,
    0xDD, 0x5E, 0x4C, 0xC6, 0xA3, 0x81, 0xA4, 0x1D, 0xA5, 0xC1, 0x37, 0xEC,
    0xC4, 0x9D, 0xF5, 0x87, 0xE1, 0x78, 0xEA, 0xF4, 0x77, 0x02, 0xEC, 0x62,
    0x37, 0x80, 0x69, 0x1A, 0x32, 0x33, 0xF6, 0x9F, 0x12, 0xBD, 0x9C, 0x9B,
    0x96, 0x37, 0xC5, 0x13, 0x78, 0xAD, 0x71, 0xA8, 0x31, 0x05, 0x52, 0x77,
    0x25, 0x4C, 0xC6, 0x3C, 0x5A, 0xD4, 0xCB, 0x76, 0xB4, 0xAB, 0x82, 0xE5,
    0xFC, 0xA1, 0x35, 0xE8, 0xD2, 0x6A, 0x6B, 0x3A, 0x89, 0xFA, 0x5B, 0x6F};

struct MlKem768Decap {
  KyberParams params;
  uint8_t ciphertext[KYBER768_CIPHERTEXT_BYTES];
  bool expectedResult;
  uint8_t secret[KYBER_SHARED_SECRET_BYTES];
};

std::vector<MlKem768Decap> MlKem768DecapTests = {
    {params_ml_kem768_test_mode,
     {0x74, 0xA2, 0x6C, 0x7D, 0x27, 0x14, 0x6A, 0x22, 0xC7, 0xEA, 0xB4, 0x20,
      0x13, 0x4E, 0x97, 0x37, 0x99, 0xCE, 0xC1, 0xDA, 0x2D, 0xF6, 0x1A, 0xE0,
      0xFA, 0x79, 0x05, 0xA3, 0xA4, 0x74, 0x85, 0xA0, 0x63, 0x07, 0x6B, 0xFA,
      0x22, 0xD6, 0xE4, 0xFE, 0x50, 0x59, 0xDE, 0x0A, 0x32, 0xE3, 0x8F, 0x11,
      0xAB, 0xD6, 0x3F, 0x99, 0x0E, 0x91, 0xBD, 0x0E, 0x3A, 0x5B, 0xC6, 0xE7,
      0x10, 0xDF, 0xE5, 0xDC, 0x0F, 0x6D, 0x4A, 0x18, 0x14, 0x7E, 0xBC, 0x2E,
      0x2D, 0x9B, 0x17, 0x93, 0x74, 0xD8, 0x36, 0x92, 0xC5, 0x3E, 0xFB, 0xD4,
      0x5F, 0x28, 0xA2, 0xA9, 0x28, 0xC2, 0x49, 0x4F, 0x90, 0x35, 0x76, 0xC4,
      0x10, 0xEB, 0x17, 0x73, 0x89, 0x5E, 0xBE, 0xAD, 0xB1, 0x19, 0x96, 0x0E,
      0xEB, 0xDA, 0x9C, 0x3C, 0x71, 0x07, 0x95, 0xA6, 0xD9, 0xB7, 0x81, 0xFC,
      0x58, 0xB3, 0x0D, 0x08, 0x10, 0x7F, 0x4E, 0x20, 0x94, 0x4A, 0x38, 0x2A,
      0xFB, 0x07, 0x9F, 0x31, 0xD2, 0x17, 0x24, 0xF2, 0xC2, 0x6E, 0x6A, 0x53,
      0x41, 0x2F, 0x0A, 0x90, 0x8B, 0xE7, 0x58, 0x6F, 0x2B, 0x3D, 0x6D, 0x7C,
      0x1D, 0xEA, 0x02, 0x70, 0xE9, 0x8A, 0xA2, 0x09, 0x24, 0x4B, 0xD8, 0x8E,
      0xD6, 0x8A, 0xAE, 0x01, 0x43, 0x23, 0x42, 0xBA, 0x5F, 0x49, 0xE0, 0x15,
      0xCB, 0x47, 0x6B, 0x5B, 0x78, 0xD1, 0x5E, 0xA7, 0x7A, 0x35, 0x4C, 0xC9,
      0xE9, 0xFD, 0x07, 0x13, 0x7D, 0x87, 0x60, 0xBE, 0x42, 0xFD, 0x47, 0x46,
      0xC6, 0x2C, 0x02, 0x02, 0x8E, 0x7B, 0x40, 0x5D, 0xDC, 0x95, 0xDF, 0x3D,
      0x02, 0x19, 0x21, 0xCF, 0xED, 0xDB, 0x3D, 0x96, 0x1B, 0x95, 0x7E, 0xCA,
      0x30, 0x2A, 0x26, 0x3D, 0xAB, 0x2D, 0xC1, 0x17, 0xBE, 0xB3, 0xE7, 0x9E,
      0xFA, 0xCF, 0xCF, 0x93, 0x6D, 0xFC, 0x09, 0xFC, 0x0D, 0x19, 0xC3, 0x58,
      0xD7, 0x24, 0xFA, 0x38, 0x1E, 0xA0, 0x6C, 0xA0, 0x67, 0xC3, 0x84, 0xE9,
      0x44, 0x30, 0x2C, 0x39, 0x07, 0xAB, 0x15, 0xA1, 0xDA, 0x4B, 0x41, 0x35,
      0x26, 0x92, 0xAD, 0xD5, 0x9B, 0x06, 0x15, 0x41, 0xF0, 0x7E, 0xFF, 0x25,
      0xEC, 0x42, 0xF4, 0x6E, 0x1A, 0x0E, 0x37, 0x0C, 0xAD, 0x06, 0xFF, 0x3F,
      0xD9, 0x97, 0xD4, 0xD2, 0xC5, 0x64, 0x8A, 0xF7, 0x62, 0x23, 0x1B, 0x38,
      0x2D, 0x05, 0x93, 0x40, 0x19, 0x36, 0xCB, 0xA2, 0x15, 0x51, 0xA2, 0xAE,
      0x30, 0xD8, 0xE8, 0xEF, 0xFC, 0xF4, 0x39, 0x16, 0xB8, 0x31, 0x38, 0xBB,
      0x5E, 0x61, 0x03, 0x64, 0x42, 0x98, 0x79, 0xFA, 0x9C, 0xDD, 0x5B, 0x7D,
      0x3C, 0xF2, 0xFE, 0xAB, 0xAA, 0x1D, 0xC8, 0xD5, 0x0C, 0xE6, 0x94, 0x02,
      0xE2, 0x11, 0x03, 0xE7, 0x95, 0xDF, 0x70, 0x74, 0xD1, 0xFC, 0xF6, 0x5F,
      0x8A, 0x4E, 0x18, 0x98, 0x6D, 0x54, 0x17, 0x78, 0x06, 0x02, 0xC6, 0x3B,
      0xE5, 0xA0, 0x44, 0x86, 0x33, 0x84, 0xBD, 0x3D, 0x8F, 0xFB, 0x68, 0x5E,
      0xAC, 0x56, 0x7E, 0xD8, 0x34, 0x9D, 0xCF, 0x2C, 0xEB, 0x70, 0x2B, 0x73,
      0x75, 0xB1, 0x45, 0x72, 0x99, 0x98, 0x04, 0x9D, 0x13, 0xE2, 0xCD, 0x46,
      0x6C, 0xF2, 0x23, 0x1B, 0x9D, 0x3A, 0x20, 0x01, 0x8E, 0xE9, 0x08, 0xF8,
      0x51, 0x4A, 0x6C, 0x6A, 0x89, 0xDF, 0x72, 0x32, 0xF9, 0x1F, 0xCD, 0x84,
      0xB8, 0x1E, 0xBC, 0x8B, 0xC5, 0x39, 0xE9, 0xA3, 0x7A, 0x43, 0x24, 0x75,
      0x55, 0x64, 0xBE, 0x1B, 0xF4, 0xFA, 0x1F, 0xB4, 0x57, 0x1E, 0x0A, 0xBB,
      0xC9, 0xB5, 0x2F, 0x9D, 0x09, 0x0C, 0x33, 0xBE, 0x59, 0x9D, 0xE6, 0xC8,
      0x53, 0x2C, 0x7C, 0xB7, 0xEC, 0x8B, 0x4E, 0x2D, 0x3C, 0x07, 0x50, 0x52,
      0x80, 0xE9, 0x99, 0x23, 0x86, 0x59, 0x03, 0xFF, 0xD1, 0x8B, 0xC1, 0x3B,
      0x9C, 0x81, 0x64, 0xAA, 0x1E, 0xAE, 0x84, 0xE3, 0x8D, 0x3F, 0x57, 0xFD,
      0xB8, 0x80, 0x17, 0x85, 0xF1, 0x05, 0xA6, 0xA8, 0x57, 0x4B, 0xD2, 0xFE,
      0x9B, 0xF3, 0x05, 0x84, 0x8E, 0x52, 0x53, 0x30, 0xBC, 0x2D, 0x24, 0xF0,
      0x25, 0x7E, 0x47, 0xA4, 0x95, 0x0F, 0x43, 0x3A, 0x92, 0x33, 0xE8, 0xCD,
      0xEB, 0xA8, 0x1D, 0xBA, 0xE7, 0xD8, 0xC1, 0xA0, 0x6D, 0x01, 0xF7, 0x0D,
      0xE6, 0xEF, 0x66, 0x32, 0x07, 0xD8, 0x49, 0x52, 0x82, 0x7B, 0xAB, 0x3D,
      0x45, 0x1C, 0xBE, 0xA0, 0x99, 0x00, 0x07, 0xFB, 0xDB, 0x42, 0x40, 0xFE,
      0x89, 0x9A, 0x70, 0x6F, 0x7C, 0x15, 0x63, 0xE0, 0x5C, 0x70, 0xBE, 0x9D,
      0x57, 0x51, 0x89, 0xEF, 0x83, 0xE0, 0xCF, 0x76, 0x19, 0x5F, 0x66, 0x52,
      0x49, 0x1C, 0xCE, 0x04, 0xF1, 0xCE, 0x20, 0x92, 0x17, 0x0A, 0x92, 0xE0,
      0xDD, 0x73, 0x01, 0x24, 0x6A, 0x4C, 0x44, 0xFC, 0x0B, 0x4E, 0xE6, 0xAA,
      0xA6, 0x3F, 0xC7, 0x02, 0x78, 0x40, 0xAB, 0xD2, 0xEC, 0x25, 0xF6, 0x54,
      0x58, 0x97, 0x38, 0xCD, 0x38, 0xB9, 0xE1, 0x0B, 0x97, 0x5C, 0xFB, 0x6C,
      0x1D, 0x2E, 0xB4, 0xDA, 0x97, 0x73, 0x69, 0x98, 0xF8, 0x4F, 0xDD, 0xDD,
      0x81, 0x0D, 0x72, 0xDA, 0x3C, 0x5A, 0xB1, 0x35, 0x07, 0x42, 0x0D, 0xDB,
      0xFA, 0xA4, 0xF7, 0x75, 0x0C, 0x1F, 0xAE, 0x9C, 0x7D, 0xFB, 0x30, 0xF4,
      0x0A, 0x12, 0xAE, 0xA6, 0x89, 0xFC, 0x78, 0xDA, 0x90, 0x00, 0x20, 0xE3,
      0xAB, 0xB3, 0x2A, 0x36, 0x4D, 0x5C, 0x6B, 0x3C, 0x75, 0x44, 0xA1, 0xB5,
      0x73, 0x4A, 0x41, 0xE9, 0x5C, 0x83, 0x14, 0xB4, 0x48, 0xCD, 0x0B, 0x73,
      0x8D, 0x82, 0x9A, 0xF7, 0x72, 0xA8, 0xF8, 0x1C, 0x51, 0xAD, 0xBA, 0x2D,
      0x85, 0xF3, 0x26, 0xC8, 0xF5, 0xD6, 0x96, 0x1C, 0xF1, 0x2D, 0x44, 0xA9,
      0xBE, 0xDE, 0xA0, 0x0D, 0x1D, 0xF5, 0xB4, 0x8F, 0x42, 0x9B, 0x1C, 0xE0,
      0xC1, 0x5E, 0xA5, 0xF5, 0xBC, 0x10, 0xB0, 0x17, 0x24, 0x7B, 0xA2, 0xC6,
      0xBE, 0x92, 0x2B, 0x05, 0x63, 0xB8, 0xE9, 0x69, 0x86, 0x77, 0xCB, 0x6C,
      0x45, 0xCC, 0xF2, 0x08, 0x1B, 0xF8, 0x42, 0x19, 0xD2, 0x90, 0x4C, 0x11,
      0xFF, 0x92, 0x19, 0x9F, 0x8A, 0xEF, 0xAD, 0x62, 0xD8, 0x60, 0x8E, 0x20,
      0x08, 0x02, 0xC5, 0xA0, 0x72, 0x02, 0xCC, 0x82, 0x0E, 0x9E, 0x52, 0x0E,
      0x31, 0xBF, 0x36, 0xA8, 0x30, 0x02, 0xEC, 0xA4, 0x01, 0x8B, 0x0B, 0x3A,
      0x39, 0x88, 0x01, 0x56, 0x2A, 0xA8, 0x6C, 0x77, 0xAB, 0x0D, 0x50, 0xA8,
      0xFB, 0xC3, 0x76, 0x8B, 0x0A, 0x64, 0x3B, 0x97, 0xE7, 0xF9, 0x07, 0x21,
      0x68, 0xDE, 0x29, 0xB8, 0x17, 0x59, 0x99, 0xC9, 0xAA, 0x48, 0xD3, 0x01,
      0xA3, 0xF0, 0x30, 0x31, 0x72, 0xE9, 0xC7, 0xD4, 0xF1, 0x63, 0x29, 0xD5,
      0xCA, 0x9D, 0x42, 0x39, 0x7C, 0x39, 0x82, 0xE1, 0x0C, 0x9D, 0xA4, 0x2D,
      0xE8, 0x8B, 0xD6, 0xC2, 0xAB, 0x91, 0xC1, 0xE7, 0x1E, 0x77, 0x8E, 0x58,
      0xBB, 0x8F, 0x80, 0x1F, 0x20, 0x7A, 0x88, 0xA9, 0xB4, 0x7F, 0x9C, 0x68,
      0x7A, 0xFB, 0xBA, 0x34, 0xED, 0xA6, 0xD2, 0x89, 0x9E, 0x4F, 0xA0, 0x00,
      0x8A, 0xA2, 0xB5, 0x39, 0x71, 0x17, 0x53, 0xDC, 0x7C, 0x07, 0xF6, 0x14,
      0xE8, 0x14, 0xF6, 0x83, 0xD6, 0xC0, 0x37, 0x56, 0x2A, 0xE1, 0xFB, 0xBE,
      0x6D, 0x7D, 0x5F, 0xA5, 0x4B, 0x7A, 0x6D, 0x94, 0x51, 0xE1, 0x1B, 0x01,
      0xAA, 0xCC, 0xC3, 0xBF, 0x2E, 0xD6, 0x47, 0x42, 0xDD, 0x10, 0x0E, 0x0E,
      0xAB, 0x2D, 0xF6, 0xCC, 0xCF, 0x93, 0x7B, 0x6D, 0x59, 0x81, 0xEC, 0xA0,
      0xE0, 0x1F, 0x32, 0x45, 0xCF, 0x26, 0xA7, 0x2A, 0xD1, 0xAD, 0xF0, 0x66,
      0xC8, 0xF5, 0x43, 0x0D, 0x72, 0xF5, 0x09, 0x96, 0x3A, 0x65, 0x7D, 0x85,
      0xE5, 0x54, 0xC1, 0x4E, 0x26, 0xE8, 0xBE, 0xC5, 0xD5, 0xF3, 0xAB, 0x99,
      0x8C, 0x9B, 0x29, 0xF1, 0x6B, 0x04, 0x74, 0x7D, 0x80, 0x74, 0x9B, 0x30,
      0xE5, 0x1F, 0xD2, 0xA7, 0xF6, 0x90, 0xC2, 0x2F, 0x99, 0x86, 0xAA, 0xF6,
      0x35, 0x8D, 0x6F, 0xAB, 0x8D, 0xED, 0x54, 0x97, 0x1B, 0x32, 0x64, 0x1D,
      0xE2, 0xB2, 0x58, 0x59, 0x0E, 0xEA, 0xA6, 0xBF, 0x1F, 0x32, 0x32, 0x4A,
      0x7C, 0x4C, 0x98, 0x3F, 0x49, 0x46, 0x6D, 0x86},
     true,
     {0x3D, 0x23, 0xB1, 0x0D, 0xF2, 0x32, 0xA1, 0x80, 0x78, 0x6F, 0x61,
      0x26, 0x1E, 0x85, 0x27, 0x82, 0x51, 0x74, 0x65, 0x80, 0xBE, 0xBC,
      0xA6, 0xAC, 0xBA, 0xD6, 0x0A, 0xEF, 0x69, 0x52, 0xBE, 0x69}},

    {params_ml_kem768_test_mode,
     {0x39, 0xEF, 0xB9, 0x00, 0x89, 0xF1, 0xDC, 0x32, 0xA5, 0x43, 0x70, 0xB3,
      0xEE, 0xDF, 0x2B, 0x12, 0x88, 0x0D, 0xC7, 0xD6, 0x57, 0xF0, 0x40, 0x4E,
      0x41, 0xF7, 0xDA, 0xAA, 0x73, 0xE7, 0xF0, 0x6C, 0xB9, 0x0B, 0xBE, 0xEC,
      0x75, 0x44, 0x16, 0x07, 0x68, 0xEC, 0x3B, 0x56, 0x68, 0x1D, 0x05, 0x7A,
      0xE1, 0xDB, 0x58, 0xF0, 0x12, 0x32, 0x86, 0xD3, 0xA8, 0xCD, 0xD0, 0xB4,
      0x14, 0xCF, 0x98, 0x94, 0xFD, 0xA1, 0xCF, 0xF3, 0xA3, 0x7C, 0xF6, 0x7B,
      0x82, 0xC5, 0xC7, 0xAD, 0x34, 0x27, 0xF2, 0xF2, 0xB3, 0x93, 0x97, 0x8B,
      0x94, 0xE5, 0x24, 0xF3, 0x33, 0x34, 0xE4, 0xA9, 0x8A, 0xFF, 0xEA, 0x8D,
      0x75, 0x14, 0xD6, 0xE1, 0x2E, 0x85, 0x08, 0x6E, 0x58, 0xA0, 0xC0, 0x78,
      0xEB, 0xA6, 0x44, 0x35, 0x44, 0x1F, 0x3E, 0x37, 0x02, 0xEA, 0x27, 0xEE,
      0xA9, 0x84, 0xE4, 0x68, 0x93, 0xBB, 0x88, 0x65, 0x72, 0x49, 0x1F, 0x22,
      0xAE, 0x09, 0xF8, 0xD5, 0x07, 0x74, 0xB4, 0xDD, 0xD5, 0xCF, 0x47, 0x8C,
      0xB0, 0xB2, 0xD0, 0x70, 0x43, 0x7E, 0x86, 0x64, 0x5E, 0xF6, 0x2A, 0xA8,
      0x35, 0x99, 0x09, 0x37, 0x32, 0xF8, 0x1A, 0x75, 0xD1, 0xD5, 0xDE, 0x15,
      0xC3, 0x1E, 0xC8, 0x1A, 0xC4, 0xD6, 0x78, 0x52, 0xFD, 0xE0, 0x89, 0xD5,
      0x80, 0xB7, 0x1E, 0x3D, 0xB0, 0x7C, 0x71, 0x39, 0x44, 0x24, 0xE0, 0x93,
      0x6B, 0xF7, 0x4D, 0x0C, 0x94, 0x05, 0xBD, 0x3D, 0xFB, 0x60, 0xB9, 0x20,
      0xE7, 0xEF, 0xA3, 0x8C, 0x72, 0xD5, 0x91, 0x2B, 0xBD, 0x30, 0x1B, 0xD3,
      0xF3, 0x70, 0x9C, 0xBE, 0xEE, 0xB7, 0xBF, 0xD0, 0x76, 0x7B, 0x77, 0xA8,
      0x63, 0x99, 0x13, 0xE8, 0xC2, 0x28, 0xFB, 0xB7, 0xE3, 0xE1, 0x3C, 0x42,
      0x3B, 0xF0, 0x5A, 0xC6, 0x5B, 0x7E, 0x75, 0xF2, 0x9C, 0x90, 0x48, 0xF1,
      0x61, 0xAF, 0x1B, 0x4B, 0x41, 0xC4, 0x95, 0xAD, 0xB5, 0x3F, 0xEC, 0xC5,
      0x7F, 0xED, 0x0D, 0xCF, 0x79, 0x20, 0x50, 0xA2, 0xA5, 0x86, 0xC3, 0x3A,
      0xA4, 0xA7, 0xF6, 0xBC, 0xDA, 0x90, 0x68, 0xEA, 0x29, 0x5F, 0xB6, 0x92,
      0xBD, 0xCA, 0x75, 0x6F, 0xCC, 0x47, 0xCA, 0x0A, 0x8C, 0x84, 0xDB, 0x5D,
      0xCB, 0x6A, 0x61, 0x66, 0x05, 0xF3, 0xD3, 0xA3, 0x4C, 0x4D, 0x23, 0xEC,
      0x14, 0x94, 0x24, 0x92, 0xC0, 0x7E, 0xF1, 0x23, 0xC8, 0xD0, 0x84, 0xDF,
      0x21, 0xF3, 0xB2, 0x14, 0x1D, 0x27, 0x7F, 0xA1, 0x6E, 0x3C, 0xF4, 0xD5,
      0xA3, 0xAB, 0x8D, 0x78, 0xCE, 0x83, 0x70, 0xF4, 0x11, 0xDF, 0x73, 0x76,
      0x47, 0xA2, 0xD6, 0x12, 0x31, 0x20, 0xAE, 0xE1, 0xCC, 0xF7, 0xDE, 0xFC,
      0x35, 0xA5, 0x40, 0x8F, 0xA6, 0x01, 0x3E, 0x94, 0x70, 0x3E, 0x8E, 0x04,
      0xC5, 0x0B, 0xAD, 0xCB, 0xBF, 0x2E, 0x1F, 0xF0, 0xFB, 0x82, 0xDB, 0x4A,
      0xAC, 0x59, 0x5B, 0x9E, 0xAA, 0x9E, 0x37, 0x0C, 0x9C, 0x61, 0x75, 0xCE,
      0xF2, 0x0B, 0x1D, 0x0B, 0x8A, 0x43, 0x09, 0xAB, 0x91, 0x91, 0x84, 0x51,
      0xE6, 0xC8, 0xA6, 0xDF, 0x04, 0xAE, 0x46, 0x8D, 0x44, 0x6F, 0xD9, 0xE8,
      0x3F, 0x92, 0x52, 0xF1, 0x45, 0xA2, 0xB4, 0x4A, 0x19, 0xE7, 0xB2, 0x7D,
      0xA5, 0x60, 0x44, 0x71, 0x7D, 0xB5, 0xA6, 0xED, 0x5F, 0x6E, 0x5C, 0xDD,
      0x90, 0x20, 0x8A, 0xBC, 0x32, 0x42, 0x90, 0x29, 0x2B, 0x1F, 0x2E, 0x84,
      0xFB, 0x69, 0xF5, 0x98, 0x9D, 0x99, 0x21, 0xDC, 0xB4, 0xF0, 0x58, 0xDC,
      0xAF, 0x7B, 0x99, 0xDF, 0x71, 0xB2, 0x6B, 0xD1, 0x09, 0x0E, 0x45, 0x77,
      0x67, 0x95, 0x4B, 0x8A, 0xCC, 0x84, 0xFD, 0xDF, 0xD6, 0x63, 0xD6, 0x40,
      0x27, 0x52, 0x80, 0x77, 0xB3, 0xC9, 0xE3, 0x70, 0x60, 0x09, 0x42, 0xE4,
      0xC1, 0x17, 0x5B, 0x48, 0x7F, 0xBF, 0x25, 0xE2, 0x67, 0x47, 0x4B, 0x52,
      0x38, 0x57, 0x60, 0x10, 0xCC, 0xCE, 0x33, 0x15, 0xCE, 0xDD, 0x56, 0x34,
      0x65, 0x8B, 0x20, 0x28, 0xF3, 0xFB, 0x99, 0x59, 0xD7, 0x7F, 0xA2, 0x37,
      0x56, 0xDB, 0x48, 0x78, 0x69, 0x7C, 0x9B, 0xC4, 0x91, 0xDB, 0xD6, 0x89,
      0x86, 0xB9, 0x07, 0x3D, 0x18, 0x7F, 0x2A, 0x9E, 0x72, 0xC9, 0x43, 0xD9,
      0x4C, 0x97, 0xDA, 0x86, 0x5C, 0xFD, 0x9C, 0x23, 0x50, 0x81, 0x05, 0x63,
      0x7F, 0xED, 0x62, 0xE5, 0x6E, 0x74, 0x55, 0x55, 0x90, 0x9A, 0x49, 0xD2,
      0x3B, 0x86, 0xE6, 0x20, 0xD4, 0x8F, 0xD5, 0x5A, 0x92, 0xCC, 0x22, 0x66,
      0xC3, 0x8B, 0x85, 0x7F, 0x5D, 0xF9, 0xBB, 0x68, 0x3D, 0x60, 0xB0, 0x84,
      0x81, 0x9C, 0xF0, 0x4F, 0x5B, 0xB8, 0xCB, 0xED, 0x05, 0xAC, 0x6F, 0x48,
      0xC5, 0x18, 0xED, 0xB5, 0xB2, 0x22, 0xF5, 0xE6, 0xDC, 0xBB, 0x43, 0x81,
      0x82, 0xA7, 0xBA, 0x3B, 0x22, 0x79, 0xE5, 0x85, 0x68, 0x28, 0xCB, 0xE9,
      0xBD, 0xA6, 0x00, 0x9A, 0x70, 0xD2, 0x0D, 0xA0, 0x82, 0xD2, 0xFF, 0xBD,
      0x09, 0x2E, 0xDA, 0xD4, 0xB2, 0x72, 0xE4, 0x6D, 0x21, 0x5B, 0x8E, 0xCC,
      0x26, 0x22, 0x24, 0x99, 0xF0, 0x24, 0x32, 0x7A, 0x39, 0x1C, 0xEB, 0x00,
      0x77, 0x89, 0x75, 0x7F, 0xF8, 0xFA, 0x82, 0x67, 0x42, 0x9F, 0x05, 0x34,
      0xF3, 0x05, 0xF7, 0x57, 0x09, 0xDC, 0xC4, 0x22, 0x98, 0x03, 0xEA, 0x8E,
      0x61, 0x2F, 0x55, 0x89, 0x0C, 0x5F, 0xDF, 0x82, 0x52, 0x79, 0x4D, 0x5C,
      0x9C, 0x40, 0x58, 0xC2, 0x25, 0x8A, 0x55, 0x99, 0xBA, 0x85, 0x8A, 0x02,
      0xF8, 0x9A, 0x6F, 0xDB, 0x35, 0xC4, 0xF2, 0x36, 0x4A, 0x4C, 0x6B, 0x32,
      0x6A, 0x31, 0xF7, 0xD0, 0x4F, 0x62, 0xC2, 0xFA, 0xFE, 0x51, 0xD2, 0x80,
      0xCD, 0x7A, 0x4C, 0xAB, 0x66, 0x40, 0x4F, 0xDF, 0xD0, 0x33, 0xEA, 0xDD,
      0x07, 0x97, 0x4B, 0xCA, 0xA7, 0xF0, 0xCB, 0x74, 0x01, 0xB9, 0x48, 0x4D,
      0xAF, 0x9F, 0x32, 0x5B, 0x6B, 0xA5, 0x3F, 0xBF, 0x41, 0x21, 0x93, 0x84,
      0xB2, 0x64, 0xF2, 0x4A, 0xA8, 0xD6, 0x52, 0x81, 0x69, 0x32, 0x95, 0xE6,
      0xF7, 0x1F, 0xCA, 0x88, 0x5F, 0x80, 0x80, 0x26, 0x82, 0x9A, 0x3F, 0xC3,
      0x2D, 0xC9, 0x60, 0x3F, 0x0C, 0xED, 0x36, 0xF0, 0xB5, 0x8A, 0x29, 0x6B,
      0x44, 0xAD, 0xDA, 0x3A, 0xAF, 0x10, 0x63, 0x8C, 0x31, 0xF3, 0x54, 0xD1,
      0xA5, 0xAC, 0x34, 0xE7, 0x7D, 0x4D, 0x01, 0x54, 0xC9, 0x54, 0x67, 0x09,
      0xE9, 0x20, 0x25, 0x8F, 0x73, 0xE0, 0x39, 0xFB, 0xC2, 0x23, 0xEE, 0x74,
      0xA2, 0x70, 0x84, 0x01, 0x65, 0xF6, 0x4E, 0x30, 0x51, 0xB1, 0x0B, 0x5E,
      0x63, 0xF9, 0xAC, 0xCF, 0x5D, 0x1E, 0xF4, 0x0E, 0x43, 0xF5, 0x82, 0x3B,
      0x15, 0xF8, 0xC2, 0x5C, 0xAF, 0xCE, 0x69, 0x8A, 0x64, 0xF9, 0xAE, 0x31,
      0x6D, 0x39, 0x05, 0xB8, 0xE5, 0x10, 0xC5, 0x6C, 0xF7, 0x54, 0x4C, 0xA9,
      0x47, 0x19, 0x73, 0x5A, 0x64, 0x0F, 0x2B, 0x8C, 0x3A, 0x2B, 0x82, 0x8A,
      0x04, 0xE0, 0x56, 0x88, 0x63, 0x93, 0x75, 0x95, 0xE5, 0xB9, 0xDA, 0xDA,
      0x33, 0x53, 0x3D, 0x9D, 0x67, 0x6A, 0xA6, 0x57, 0xFE, 0x69, 0x15, 0x2E,
      0x93, 0x15, 0x9A, 0x00, 0xC5, 0x96, 0x2F, 0x4D, 0xFF, 0x9C, 0x90, 0x1A,
      0x9A, 0xB3, 0x2D, 0xB2, 0x8B, 0x93, 0xF4, 0xBA, 0x78, 0x0E, 0x44, 0xA2,
      0xF7, 0x38, 0x78, 0xAA, 0x76, 0xE1, 0x12, 0xE3, 0x49, 0x02, 0x05, 0xAF,
      0x83, 0x00, 0x0E, 0xFD, 0x88, 0x9F, 0xCE, 0xEA, 0x5E, 0x87, 0xAE, 0x9A,
      0xE0, 0x1E, 0xE1, 0xCC, 0xF6, 0xBA, 0x04, 0x61, 0xA8, 0xD8, 0x65, 0x4B,
      0x77, 0x02, 0xC0, 0x9B, 0xB4, 0x1C, 0x4F, 0x61, 0xA0, 0x0D, 0x05, 0xF0,
      0x31, 0xB2, 0x44, 0xED, 0xED, 0x8D, 0x1C, 0xAC, 0x79, 0x16, 0xBE, 0xB9,
      0xAA, 0x67, 0xA3, 0x88, 0x0F, 0x4C, 0x35, 0x16, 0xA8, 0xD8, 0x20, 0x49,
      0x32, 0xEA, 0x00, 0xEF, 0xB3, 0xAA, 0x20, 0x36, 0x9F, 0xB6, 0xBE, 0x40,
      0x48, 0x43, 0xC7, 0x41, 0x1E, 0x88, 0x42, 0x85, 0x68, 0xAB, 0x9A, 0x39,
      0x12, 0x4E, 0xAD, 0x11, 0x52, 0x98, 0xD4, 0x9C, 0x99, 0x86, 0x51, 0xE5,
      0xEF, 0x61, 0x3A, 0x68, 0x19, 0x33, 0x66, 0x83},

     true,
     {0x1D, 0x2D, 0xCA, 0xCE, 0xC1, 0x4C, 0xBB, 0x78, 0xFE, 0x9E, 0x41,
      0x89, 0x37, 0x83, 0x5E, 0xED, 0x08, 0x8C, 0xC0, 0x68, 0x33, 0x00,
      0xC9, 0x65, 0xEF, 0x39, 0x72, 0x08, 0x1F, 0x01, 0xC4, 0xE9}},

    {params_ml_kem768_test_mode,
     {0xA5, 0xC8, 0x1C, 0x76, 0xC2, 0x43, 0x05, 0xE1, 0xCE, 0x5D, 0x81, 0x35,
      0xD4, 0x15, 0x23, 0x68, 0x2E, 0x9E, 0xE6, 0xD7, 0xB4, 0x0A, 0xD4, 0x1D,
      0xF1, 0xF3, 0x7C, 0x9B, 0x17, 0xDC, 0xE7, 0x80, 0x76, 0x01, 0x9A, 0x6B,
      0x0B, 0x7C, 0x95, 0xC9, 0xBE, 0x7A, 0xF2, 0x95, 0x07, 0xB2, 0xD5, 0xA6,
      0x98, 0x7C, 0x8E, 0xE3, 0x25, 0x91, 0x90, 0x85, 0x52, 0x43, 0xE6, 0xE5,
      0x6F, 0x56, 0x20, 0x60, 0x8C, 0x52, 0xD9, 0x6F, 0xAB, 0x10, 0x3A, 0x87,
      0x00, 0xFB, 0xA1, 0xA8, 0x7D, 0xCA, 0x60, 0x78, 0x11, 0x8A, 0x08, 0x71,
      0x76, 0x2C, 0x95, 0x34, 0xC0, 0xC0, 0xC3, 0x97, 0x8C, 0x91, 0xC3, 0xA0,
      0x1F, 0x0F, 0x60, 0x8D, 0xCF, 0x75, 0x78, 0x15, 0x43, 0x8F, 0xE8, 0x95,
      0x7C, 0x8A, 0x85, 0x91, 0x83, 0xB1, 0xB6, 0x72, 0x1A, 0x08, 0x65, 0xBE,
      0xBC, 0x79, 0x9D, 0x4E, 0x5C, 0x0E, 0x7B, 0xD3, 0xEA, 0xE4, 0x85, 0x8E,
      0x6A, 0xB6, 0xA2, 0xE7, 0x65, 0x8E, 0xD8, 0x0D, 0x4E, 0xD1, 0x58, 0xB0,
      0x36, 0xB9, 0x3F, 0xA0, 0x3A, 0xFA, 0x6A, 0xE3, 0x13, 0x6C, 0xF3, 0xD6,
      0x93, 0xC9, 0x11, 0xBC, 0xC7, 0x59, 0x05, 0xE5, 0xB0, 0xCB, 0x28, 0x65,
      0xB9, 0xE9, 0x88, 0x45, 0x22, 0xA7, 0x77, 0x77, 0x61, 0x3E, 0x53, 0x11,
      0x1D, 0x5A, 0x1C, 0x7D, 0x3D, 0xAB, 0x73, 0x4C, 0xEB, 0x03, 0x65, 0x7A,
      0xE0, 0xC8, 0x97, 0x63, 0xE9, 0x94, 0x71, 0x05, 0x47, 0x76, 0xBA, 0xE7,
      0xD5, 0x1B, 0x0E, 0x73, 0xA5, 0xBB, 0x35, 0xAE, 0xC3, 0x0F, 0xF6, 0xBC,
      0x93, 0x68, 0x49, 0x16, 0xFE, 0xF1, 0x16, 0x25, 0x86, 0x45, 0x2F, 0x42,
      0x66, 0x53, 0xE2, 0xCA, 0x84, 0x4D, 0x57, 0x44, 0x30, 0x7F, 0xF9, 0xAE,
      0xB2, 0x87, 0xA6, 0x44, 0x77, 0x83, 0xB2, 0x1A, 0x0E, 0x93, 0x9C, 0x81,
      0x42, 0x1D, 0x63, 0x1F, 0x5D, 0xCB, 0x45, 0x2E, 0x51, 0xED, 0x34, 0xE3,
      0xDA, 0xD1, 0xCF, 0x50, 0x4E, 0x0A, 0x3B, 0x0F, 0x47, 0x11, 0xA8, 0xDC,
      0x64, 0x99, 0xD1, 0x69, 0x1D, 0x10, 0x95, 0x69, 0x33, 0x6C, 0xE1, 0x55,
      0x8A, 0x4C, 0x0A, 0x46, 0x4E, 0x20, 0x87, 0xEA, 0x8F, 0x9E, 0x3B, 0x18,
      0xF7, 0x47, 0xEF, 0x61, 0xF4, 0x57, 0x6A, 0xEB, 0x42, 0xB1, 0x7C, 0xAD,
      0xB7, 0xF0, 0xFD, 0x84, 0xDA, 0x8E, 0x3A, 0x6F, 0x47, 0x1D, 0x95, 0xED,
      0xFA, 0x65, 0xBE, 0x9E, 0x6C, 0x9F, 0x6A, 0xE7, 0x56, 0xA2, 0x2A, 0x4F,
      0x1A, 0x5C, 0x54, 0x3C, 0x26, 0xBA, 0x7B, 0xAD, 0x88, 0xE1, 0x6D, 0x5F,
      0x5B, 0x7E, 0x12, 0xE2, 0xD4, 0xCA, 0x34, 0xB3, 0xA6, 0x4D, 0x17, 0xF8,
      0x7C, 0xCF, 0xC4, 0xFF, 0x8C, 0x5E, 0x4F, 0x53, 0x75, 0x2A, 0x07, 0x7C,
      0x68, 0x72, 0x1E, 0x8C, 0xC8, 0x17, 0xF9, 0xFF, 0x24, 0x87, 0x61, 0x70,
      0xFF, 0x2A, 0xF8, 0x9F, 0xA9, 0x58, 0x55, 0xA5, 0xB1, 0xDE, 0x34, 0x7C,
      0x07, 0xFD, 0xDB, 0xCF, 0xE7, 0x26, 0x4A, 0xA5, 0xED, 0x64, 0x01, 0x49,
      0x15, 0x61, 0xD8, 0x31, 0x53, 0x8F, 0x85, 0x2B, 0x0E, 0xD7, 0xB9, 0xE8,
      0xEB, 0xAF, 0xFC, 0x06, 0x02, 0x84, 0xF2, 0x2D, 0x2B, 0xAE, 0xE5, 0x6F,
      0xA9, 0xF6, 0xD0, 0x14, 0x32, 0xA1, 0x15, 0xA2, 0xD6, 0xA6, 0x4C, 0x38,
      0xAE, 0x0A, 0x50, 0xBA, 0x36, 0x2F, 0xB5, 0x7B, 0x53, 0xE3, 0xE8, 0x55,
      0xB8, 0x3C, 0xE8, 0xC4, 0x22, 0x74, 0x04, 0x55, 0x99, 0xF6, 0x5F, 0xA6,
      0xA8, 0x92, 0x1D, 0x85, 0xF9, 0x4E, 0xD2, 0x30, 0xB5, 0x16, 0x71, 0x2D,
      0xB6, 0xFD, 0x2F, 0xF2, 0x8B, 0x3A, 0x33, 0x71, 0xD9, 0xBE, 0x05, 0x8A,
      0xE7, 0x5C, 0x2F, 0xA5, 0x91, 0xB7, 0xEC, 0x3C, 0x3D, 0xAA, 0x1F, 0x76,
      0x42, 0xBC, 0x26, 0xC3, 0x24, 0xC0, 0x80, 0x90, 0x60, 0x7E, 0x66, 0x62,
      0x15, 0x4D, 0xB3, 0x7C, 0xF7, 0x47, 0x96, 0x7A, 0x1F, 0x9F, 0xC2, 0x90,
      0x89, 0xF5, 0x70, 0xEB, 0xE6, 0x0E, 0xEE, 0xF8, 0x9F, 0xD2, 0x44, 0x81,
      0x02, 0x8C, 0x85, 0xAE, 0xF1, 0xDC, 0x3B, 0x09, 0xF2, 0x2C, 0xD3, 0x69,
      0x1B, 0xBB, 0xB8, 0x21, 0xC7, 0xA8, 0xA0, 0xF3, 0x5A, 0xD1, 0x2B, 0xE1,
      0xDD, 0x19, 0x9B, 0x97, 0x70, 0x48, 0xF3, 0xD4, 0x8C, 0x16, 0xBB, 0x2C,
      0xA9, 0x4C, 0xEC, 0xB8, 0x92, 0x87, 0x70, 0xD5, 0xBB, 0x32, 0x9A, 0x03,
      0x27, 0xE0, 0xB2, 0x86, 0xFA, 0xA1, 0xC6, 0x52, 0x81, 0x03, 0x1A, 0x31,
      0xC8, 0x4F, 0x2E, 0xDC, 0x9C, 0x04, 0xD4, 0x75, 0xED, 0x4E, 0x12, 0x8E,
      0x51, 0xEF, 0xA9, 0x7D, 0x01, 0x48, 0xCB, 0xA6, 0xC9, 0x5F, 0x67, 0x4C,
      0x58, 0x9F, 0x30, 0x1C, 0x26, 0x5B, 0xED, 0x70, 0x8E, 0x9A, 0xD8, 0xDA,
      0x3C, 0x5C, 0xEC, 0xBD, 0xEE, 0xED, 0x35, 0xEF, 0x1E, 0x25, 0x31, 0x32,
      0xBA, 0x89, 0x92, 0x0D, 0x78, 0x6B, 0x88, 0x23, 0x0B, 0x01, 0x3B, 0xCF,
      0x2D, 0xC9, 0x2D, 0x6B, 0x15, 0x7A, 0xFA, 0x8D, 0xA8, 0x59, 0x2C, 0xD0,
      0x74, 0x3D, 0x49, 0x82, 0xBE, 0x60, 0xD7, 0xC2, 0xD5, 0xC4, 0x72, 0xAB,
      0x9F, 0xA7, 0xF4, 0xCC, 0x3D, 0x12, 0xB0, 0xEB, 0xAF, 0x0A, 0xBE, 0x55,
      0x5C, 0x75, 0x80, 0x54, 0x26, 0x84, 0x4D, 0xD9, 0x42, 0x86, 0x43, 0xF8,
      0x44, 0x06, 0xA1, 0xB8, 0xD6, 0xFA, 0xED, 0xFD, 0x8A, 0xE6, 0xE7, 0x3A,
      0x72, 0x77, 0x2A, 0x21, 0x59, 0xAC, 0xAB, 0xD9, 0x72, 0xAE, 0xB6, 0xF7,
      0xDE, 0x09, 0x1A, 0xC5, 0xFD, 0xD7, 0xF4, 0x9A, 0x3D, 0xC6, 0x64, 0x1C,
      0xDF, 0x62, 0x44, 0x6B, 0x4B, 0x04, 0xA3, 0x1F, 0x73, 0xB8, 0x0A, 0x62,
      0xF8, 0x0A, 0x40, 0x4A, 0x8C, 0xB1, 0x8C, 0xE3, 0xE6, 0x54, 0x80, 0xEF,
      0x7B, 0x52, 0xBF, 0x00, 0x91, 0x11, 0x7E, 0x5D, 0x08, 0xEA, 0xE1, 0xB0,
      0xAA, 0xBB, 0x72, 0xE6, 0xDF, 0xFF, 0xF7, 0x6F, 0x6E, 0x44, 0xBB, 0xD7,
      0xEA, 0x57, 0x0D, 0x66, 0x04, 0xBC, 0x2E, 0x74, 0x31, 0x8B, 0xAF, 0xA3,
      0x15, 0xA3, 0x88, 0x61, 0xAA, 0x1B, 0x21, 0xAF, 0xB2, 0xA5, 0x3F, 0x26,
      0x14, 0xF1, 0xD6, 0x40, 0x07, 0x59, 0x84, 0xAE, 0x62, 0xE2, 0xFC, 0xA1,
      0xD1, 0xB4, 0xDB, 0x36, 0x9F, 0x15, 0x70, 0x5C, 0xE7, 0xD4, 0xDF, 0x8A,
      0xE9, 0x82, 0x64, 0x50, 0x10, 0x51, 0xC0, 0xDE, 0xF2, 0x1D, 0x64, 0x5D,
      0x49, 0x62, 0x5A, 0xF0, 0x2C, 0xA4, 0x28, 0xD9, 0xF0, 0xC2, 0xCD, 0x9F,
      0xBA, 0xEE, 0xAB, 0x97, 0xE8, 0xE9, 0x15, 0x16, 0x62, 0xB6, 0x99, 0x2B,
      0x4C, 0x99, 0xAB, 0x1B, 0x92, 0x5D, 0x08, 0x92, 0x03, 0x63, 0x37, 0x3F,
      0x76, 0xD3, 0xFD, 0xF0, 0x82, 0x8C, 0xAA, 0x69, 0xC8, 0xB1, 0xBD, 0xC6,
      0xF5, 0x21, 0xDF, 0x64, 0x1C, 0xF1, 0xC8, 0xA4, 0xE7, 0xEF, 0x0C, 0x23,
      0x28, 0x9A, 0x4E, 0x2C, 0xF1, 0x8A, 0xCE, 0xBB, 0xE4, 0xC1, 0xE6, 0x83,
      0x69, 0xBD, 0x52, 0x35, 0x12, 0x01, 0x42, 0xEC, 0xDD, 0x1A, 0x73, 0x81,
      0x1E, 0x2E, 0x53, 0x3A, 0x64, 0x7D, 0x7A, 0xEE, 0x16, 0xDA, 0xA0, 0x3B,
      0x68, 0x36, 0x39, 0xDC, 0xF1, 0xE1, 0xF1, 0xE7, 0x1C, 0xFA, 0xED, 0x48,
      0xF6, 0x9A, 0xEC, 0x3E, 0x83, 0x17, 0x33, 0xDA, 0x19, 0xCE, 0xBE, 0xC1,
      0xDD, 0xBF, 0x71, 0xCB, 0xAE, 0x08, 0x00, 0xF2, 0xF6, 0xD6, 0x4A, 0x09,
      0x6E, 0xC4, 0x95, 0xD6, 0x2F, 0x43, 0x44, 0xF7, 0xAA, 0x56, 0x21, 0xB3,
      0x22, 0x35, 0x3A, 0x79, 0x5A, 0xA0, 0x99, 0xEA, 0x3A, 0x07, 0x02, 0x72,
      0xD0, 0x53, 0xD4, 0x65, 0x3A, 0x20, 0xCF, 0x21, 0x0E, 0xAA, 0xF1, 0x2C,
      0xAE, 0x60, 0x23, 0xD8, 0xE5, 0x11, 0x8D, 0xF0, 0x4B, 0x38, 0x4A, 0x44,
      0xD1, 0xED, 0xB9, 0x1C, 0x44, 0x98, 0x9E, 0xF7, 0xEE, 0x57, 0xF2, 0xBF,
      0x81, 0xA2, 0x4B, 0xDC, 0x76, 0x80, 0x7D, 0xA9, 0x67, 0xEE, 0x65, 0x25,
      0x41, 0x0C, 0x5C, 0x48, 0x50, 0x67, 0xEF, 0xC3, 0xD3, 0x9A, 0x9A, 0xD4,
      0x2C, 0xC7, 0x53, 0xBA, 0xA5, 0x9A, 0x1F, 0xD2, 0x8A, 0xF3, 0x5C, 0x00,
      0xD1, 0x8A, 0x40, 0x6A, 0x28, 0xFC, 0x79, 0xBA},
     true,
     {0xDC, 0x5B, 0x88, 0x88, 0xBC, 0x1E, 0xBA, 0x5C, 0x19, 0x69, 0xC2,
      0x11, 0x64, 0xEA, 0x43, 0xE2, 0x2E, 0x7A, 0xC0, 0xCD, 0x01, 0x2A,
      0x2F, 0x26, 0xCB, 0x8C, 0x48, 0x7E, 0x69, 0xEF, 0x7C, 0xE4}},

    {params_ml_kem768_test_mode,
     {0x0B, 0xAF, 0x0F, 0x6E, 0x91, 0xEC, 0xAE, 0x31, 0x99, 0xF4, 0x92, 0x16,
      0x31, 0x89, 0x1A, 0x14, 0xC1, 0x3B, 0x41, 0x8B, 0x53, 0x38, 0x49, 0x92,
      0xDA, 0x3A, 0x8D, 0xAD, 0xA7, 0xDE, 0xFF, 0xB9, 0xE1, 0xE5, 0xF5, 0x59,
      0xD2, 0x73, 0x44, 0xB6, 0x0B, 0xE8, 0x1E, 0xCD, 0x01, 0xCA, 0xB1, 0xE3,
      0x16, 0x57, 0x3D, 0x57, 0x1E, 0xD4, 0x6F, 0x59, 0x24, 0x8F, 0x40, 0x23,
      0xDB, 0x02, 0x82, 0x20, 0x7E, 0x73, 0x05, 0x49, 0xCD, 0xB6, 0x0E, 0x79,
      0x3E, 0x4C, 0xD1, 0x7A, 0xC6, 0xF2, 0x80, 0x0E, 0x2D, 0x1F, 0xFB, 0x83,
      0x47, 0x7A, 0x6F, 0xE1, 0xD7, 0x39, 0x92, 0x68, 0x21, 0x23, 0xEA, 0x73,
      0x0C, 0x63, 0x26, 0x9D, 0xB1, 0x30, 0x88, 0xD6, 0xDA, 0x46, 0xD0, 0x86,
      0xCC, 0xEA, 0x21, 0x76, 0x39, 0x8E, 0xAC, 0x66, 0x32, 0x70, 0xB8, 0xB2,
      0xF3, 0x37, 0xA5, 0x5E, 0x19, 0xF4, 0xC5, 0x00, 0xDE, 0x06, 0x6B, 0x54,
      0x41, 0x79, 0x4C, 0x2D, 0x0C, 0xCA, 0xDF, 0xE5, 0xAB, 0xDE, 0x7D, 0x93,
      0xFD, 0x7D, 0x64, 0x68, 0xBC, 0x4F, 0x92, 0x56, 0x33, 0x36, 0x6D, 0x93,
      0x16, 0x78, 0x8B, 0x90, 0xB1, 0x10, 0xA4, 0xD9, 0x94, 0x85, 0xE7, 0xE5,
      0x78, 0x53, 0x7A, 0x26, 0x77, 0x44, 0xFB, 0x26, 0x6A, 0x4F, 0x24, 0x3F,
      0xA0, 0x2E, 0x3A, 0x81, 0xDA, 0x67, 0xED, 0x47, 0x79, 0x23, 0xB3, 0x6B,
      0x37, 0xBE, 0x21, 0xDD, 0xA2, 0x1E, 0xB5, 0x1D, 0xCA, 0x1F, 0x0C, 0xE4,
      0x16, 0x52, 0x14, 0x5F, 0x4C, 0x54, 0x2B, 0x2E, 0x5C, 0x92, 0x26, 0x17,
      0x03, 0x36, 0x08, 0x24, 0x6B, 0xBE, 0x2B, 0x52, 0x50, 0xA3, 0x68, 0x80,
      0x4A, 0xBD, 0xB2, 0xEF, 0x6C, 0x31, 0xC4, 0x91, 0xCE, 0x3D, 0xD8, 0x52,
      0xAE, 0xAB, 0xF6, 0xEE, 0xF1, 0x53, 0x0F, 0x4C, 0x99, 0x28, 0x6B, 0x4B,
      0x59, 0x5D, 0x57, 0xCF, 0x3A, 0x99, 0x58, 0x0B, 0x59, 0xAA, 0xA2, 0xC5,
      0x5E, 0x08, 0x0B, 0x52, 0x30, 0xEA, 0x19, 0xCF, 0x27, 0x01, 0xD2, 0x1A,
      0x37, 0xFE, 0xFD, 0x6F, 0x97, 0x09, 0x65, 0x7A, 0x21, 0xAD, 0xD0, 0x63,
      0xEC, 0xBC, 0x19, 0x7B, 0x5A, 0xD0, 0x68, 0xBE, 0x50, 0x2A, 0x2E, 0x09,
      0x0D, 0x83, 0xF4, 0x15, 0x6B, 0x67, 0x1E, 0x46, 0x61, 0x7B, 0xE6, 0xD6,
      0xA1, 0x7D, 0x04, 0x25, 0xFA, 0xC5, 0x65, 0xC4, 0xA0, 0xE4, 0x89, 0x66,
      0xE9, 0xD9, 0x00, 0xCB, 0x2C, 0x2B, 0x0D, 0x29, 0x6E, 0x0B, 0xAA, 0x9D,
      0x6C, 0x5E, 0x05, 0x14, 0xCD, 0x78, 0x83, 0x40, 0x53, 0x05, 0x8A, 0x97,
      0xD3, 0xDD, 0xF8, 0x15, 0x29, 0x07, 0x98, 0x58, 0x73, 0x74, 0x40, 0x81,
      0x26, 0x70, 0xE8, 0x18, 0xC9, 0x89, 0x16, 0x81, 0xD3, 0x50, 0xEC, 0xEC,
      0x93, 0xDA, 0xE3, 0x89, 0xD5, 0x34, 0xA5, 0xC7, 0x8F, 0x01, 0x81, 0x19,
      0x17, 0x06, 0x1C, 0xAC, 0x00, 0x03, 0xD2, 0xBE, 0xA3, 0x90, 0xEB, 0x63,
      0xFA, 0x0F, 0xE9, 0xBA, 0xBC, 0xD7, 0xFF, 0x30, 0x2D, 0x4B, 0x66, 0x56,
      0x7B, 0x2B, 0xFA, 0x67, 0xB2, 0x0F, 0x96, 0x28, 0x47, 0xD0, 0x10, 0xAA,
      0x41, 0x93, 0xCB, 0xE9, 0xF8, 0xCC, 0x1B, 0x14, 0xF8, 0xB2, 0x37, 0xC2,
      0x26, 0x75, 0xB2, 0x98, 0xA8, 0x37, 0x6D, 0xFB, 0x60, 0x37, 0xBF, 0x7C,
      0xEA, 0x36, 0xBD, 0xEA, 0xD5, 0xB5, 0x05, 0x11, 0x1F, 0x67, 0x73, 0x08,
      0x24, 0xB4, 0x96, 0x48, 0x15, 0xD0, 0x0F, 0x63, 0xEE, 0x98, 0xB9, 0xBE,
      0xA0, 0xF2, 0xF4, 0x7C, 0xC0, 0x07, 0xD5, 0x60, 0x6E, 0xD7, 0xF9, 0x67,
      0xCB, 0x15, 0xCC, 0xD4, 0xAF, 0xBC, 0x99, 0x88, 0x1C, 0xFD, 0x29, 0x7B,
      0xDC, 0x2A, 0x50, 0x9E, 0xD3, 0xCB, 0x32, 0x0D, 0xF5, 0x8D, 0xC4, 0xA5,
      0xBC, 0xD1, 0xCB, 0x10, 0x0B, 0x9D, 0x64, 0x18, 0xCB, 0x8E, 0x0F, 0x40,
      0xDE, 0xF2, 0x93, 0xDA, 0x23, 0x70, 0xCA, 0x72, 0x9B, 0x0F, 0xAB, 0x07,
      0x1F, 0xA6, 0xAE, 0xB0, 0xF3, 0xF5, 0xD1, 0x92, 0x5A, 0xB2, 0xDF, 0x73,
      0x2F, 0x98, 0xDD, 0xBF, 0xF2, 0x3D, 0x54, 0x11, 0xE4, 0x92, 0x1A, 0x1C,
      0x50, 0x6F, 0x2F, 0x93, 0x25, 0x1E, 0x82, 0x2C, 0x4C, 0xF8, 0x39, 0x98,
      0xB0, 0x00, 0xFE, 0x65, 0xED, 0x38, 0x6F, 0x57, 0x45, 0xB1, 0xD4, 0xD9,
      0x1A, 0xD9, 0xF9, 0x8B, 0x45, 0xE7, 0x13, 0xC8, 0xD9, 0x44, 0x40, 0x9E,
      0x9D, 0x35, 0x4F, 0x42, 0xFD, 0xB9, 0x74, 0x9A, 0x51, 0x07, 0xC8, 0x83,
      0x15, 0x62, 0xE6, 0x83, 0x49, 0x8C, 0x55, 0xE1, 0x47, 0x5E, 0x55, 0x2A,
      0xC1, 0x08, 0x58, 0xAB, 0x98, 0x67, 0xBF, 0x80, 0x03, 0xFB, 0x88, 0xB3,
      0xB0, 0x9F, 0x6E, 0x8A, 0xD8, 0xE9, 0x4C, 0xE8, 0x2E, 0x34, 0x2B, 0x17,
      0x80, 0xD6, 0x8E, 0xC8, 0x56, 0x5F, 0xC0, 0x68, 0x4A, 0xB6, 0xC7, 0x98,
      0xBF, 0x09, 0xFA, 0x65, 0xBE, 0x62, 0xC3, 0x7A, 0x08, 0x62, 0xAB, 0xFE,
      0x99, 0xD7, 0xDB, 0xE1, 0x43, 0x1B, 0x4C, 0xFE, 0x00, 0x7B, 0x7E, 0xC7,
      0x93, 0x0B, 0x14, 0xF6, 0xD1, 0x61, 0xBD, 0xCA, 0xAE, 0x22, 0x17, 0xD6,
      0x9D, 0x9F, 0xDB, 0xB4, 0xF8, 0x82, 0xB9, 0xF4, 0x64, 0xF8, 0x64, 0x2A,
      0xCD, 0x9B, 0xA0, 0x18, 0xB9, 0x3A, 0x8E, 0x3A, 0x96, 0x51, 0x94, 0xAC,
      0xCD, 0x96, 0xE6, 0x61, 0xCF, 0x0C, 0xF4, 0xA2, 0x66, 0x20, 0x76, 0xE2,
      0x0E, 0x8B, 0xC3, 0x19, 0x69, 0x3F, 0x19, 0x53, 0xDA, 0xB9, 0x3F, 0xEB,
      0x9B, 0xCA, 0xD6, 0x66, 0x83, 0x2D, 0xF4, 0x2F, 0x25, 0x0F, 0xAD, 0xBC,
      0xFA, 0xF7, 0x42, 0xD6, 0x86, 0x42, 0x02, 0x1B, 0xD6, 0xFF, 0xD9, 0x77,
      0x20, 0xC3, 0xE5, 0xAB, 0x86, 0xD8, 0x2C, 0xE8, 0xB1, 0x4C, 0x02, 0x89,
      0xDB, 0xF5, 0x1B, 0x50, 0xC1, 0x3C, 0xFC, 0xEC, 0x12, 0xA3, 0x92, 0x2D,
      0xCD, 0x2D, 0xE8, 0x47, 0x33, 0x29, 0xAE, 0xB2, 0x35, 0x80, 0xB2, 0x2F,
      0x9C, 0x36, 0xB4, 0xF0, 0x6D, 0x65, 0x79, 0x75, 0x1B, 0xE0, 0x59, 0x31,
      0x20, 0xF8, 0x08, 0xF0, 0xE1, 0x45, 0xD9, 0x4D, 0x1D, 0xDB, 0xBE, 0x1D,
      0x48, 0x9B, 0x74, 0x4C, 0xF6, 0xC3, 0x59, 0x64, 0xC3, 0xDD, 0x96, 0xD9,
      0x5F, 0xB6, 0x93, 0x54, 0x3C, 0x69, 0x76, 0x68, 0x77, 0xDA, 0x80, 0xBD,
      0xE8, 0xAC, 0xDF, 0x62, 0xC3, 0x66, 0xD0, 0xA4, 0xA5, 0x53, 0x18, 0x74,
      0x61, 0xF6, 0x71, 0x37, 0x6F, 0x7E, 0x70, 0xF5, 0x54, 0x96, 0x5D, 0x57,
      0x76, 0x0C, 0xDF, 0x5C, 0x6F, 0x63, 0x66, 0xE3, 0x3B, 0x3B, 0xFB, 0x55,
      0x0C, 0xC1, 0xF9, 0x3D, 0x98, 0xD2, 0x50, 0xF9, 0x0D, 0x7D, 0x36, 0xBC,
      0x01, 0x58, 0x1C, 0x49, 0x41, 0x75, 0x46, 0xBF, 0x6B, 0xBA, 0x9D, 0x10,
      0xD4, 0x1C, 0x0A, 0x00, 0x88, 0x55, 0xF3, 0x21, 0x54, 0x7B, 0xDD, 0x5A,
      0x6C, 0xFA, 0x2A, 0x25, 0x16, 0xF7, 0x14, 0x15, 0xB5, 0xBC, 0x2D, 0x5F,
      0xA1, 0xB9, 0xB7, 0x9F, 0xDC, 0x7F, 0x2B, 0x78, 0xAA, 0x11, 0x33, 0x75,
      0xEC, 0x17, 0x17, 0xF0, 0xF2, 0x73, 0xBD, 0x8C, 0xBE, 0xF5, 0x91, 0x39,
      0x51, 0x8A, 0x4E, 0x8A, 0x67, 0xDB, 0x4D, 0x07, 0x12, 0x57, 0x00, 0x03,
      0x36, 0xBB, 0x07, 0x49, 0x7F, 0x72, 0xFA, 0xAC, 0x2C, 0x1F, 0xC0, 0xF5,
      0x53, 0xB2, 0xEB, 0xA5, 0x34, 0x75, 0xF4, 0x66, 0xA2, 0xB3, 0x6A, 0xFE,
      0x0B, 0x72, 0xB4, 0x34, 0x2E, 0x99, 0x5C, 0x54, 0x4E, 0x6E, 0x14, 0xFF,
      0x7D, 0x32, 0x7F, 0x80, 0xE7, 0xAC, 0x6F, 0x65, 0x19, 0x00, 0x45, 0xF3,
      0x80, 0xB5, 0x97, 0x8F, 0x50, 0xE3, 0x32, 0x72, 0x48, 0x46, 0x26, 0x26,
      0x61, 0x25, 0xA3, 0x9D, 0xA0, 0x8B, 0x46, 0x25, 0x66, 0x24, 0xCE, 0x34,
      0x22, 0x3B, 0xB1, 0x72, 0x99, 0xB8, 0xB8, 0x16, 0x27, 0x53, 0x81, 0x2F,
      0x26, 0x44, 0xC9, 0xA1, 0x3C, 0x51, 0x43, 0x0B, 0x02, 0xAB, 0xD1, 0x88,
      0xDD, 0x1A, 0x45, 0x47, 0xC9, 0x20, 0xBA, 0x27, 0xCD, 0xAF, 0x14, 0x5B,
      0xDE, 0xBC, 0x6F, 0x45, 0xEE, 0xE3, 0xF2, 0xF5, 0x55, 0x53, 0x01, 0x0F,
      0x7B, 0x35, 0xAC, 0x63, 0xA3, 0xC7, 0xC6, 0x1C},
     true,
     {0xDC, 0xBE, 0xB5, 0xE4, 0xE8, 0xB1, 0x4B, 0xD3, 0x03, 0x1D, 0x59,
      0x16, 0xBA, 0x03, 0x25, 0x81, 0x19, 0xA5, 0xDA, 0xCD, 0xAC, 0x85,
      0x0C, 0xB4, 0x83, 0xBD, 0x7A, 0xA8, 0x0B, 0x70, 0x38, 0xD8}},

    {params_ml_kem768_test_mode,
     {0x25, 0x13, 0xDE, 0x1E, 0x55, 0xED, 0x0E, 0x86, 0x26, 0x14, 0x58, 0x7F,
      0xE4, 0x7F, 0x30, 0x8C, 0x90, 0xA1, 0xF4, 0x26, 0x47, 0x0C, 0xA1, 0x29,
      0x3B, 0xDD, 0xF7, 0xB9, 0xDD, 0xD6, 0xC3, 0x68, 0xDC, 0x15, 0x2F, 0x45,
      0xC7, 0x13, 0x54, 0x90, 0x4E, 0xD4, 0x8E, 0x15, 0xA1, 0xCB, 0x44, 0x9B,
      0x4C, 0x45, 0xD0, 0xF2, 0x01, 0xED, 0x5C, 0x7D, 0x3A, 0x04, 0x7A, 0x72,
      0xF0, 0x80, 0x26, 0x5D, 0x66, 0xC4, 0x7D, 0x39, 0x46, 0x90, 0x97, 0xEE,
      0xEA, 0xBB, 0xAA, 0x3B, 0x07, 0xED, 0x1F, 0x1A, 0xEA, 0xB8, 0x0C, 0x7D,
      0x24, 0x55, 0x2F, 0xA8, 0x88, 0x9C, 0x67, 0x4A, 0x5D, 0x48, 0x40, 0x28,
      0x9D, 0xE6, 0xB0, 0xFA, 0x9A, 0x22, 0x2E, 0x69, 0x37, 0x08, 0xD1, 0xF2,
      0x52, 0xDF, 0xE8, 0xB9, 0x93, 0x95, 0x68, 0x83, 0xC0, 0x70, 0x67, 0xC1,
      0xC0, 0x84, 0x4E, 0xF0, 0xBE, 0xB4, 0x9F, 0x63, 0x53, 0x4D, 0x21, 0xD4,
      0x71, 0xD6, 0xB7, 0x27, 0xFF, 0xC5, 0x94, 0x77, 0xF9, 0xE8, 0x9E, 0x5B,
      0xEB, 0x2A, 0xF0, 0xCB, 0xEB, 0x05, 0x2F, 0x00, 0x34, 0x14, 0xDA, 0x40,
      0x70, 0x00, 0x87, 0x53, 0xCF, 0xC0, 0xC6, 0xD0, 0xFA, 0x9D, 0x1C, 0x15,
      0x38, 0x8F, 0xE5, 0x88, 0x6E, 0xAD, 0xD3, 0x47, 0x4F, 0x28, 0xE4, 0x68,
      0x2C, 0x0E, 0x01, 0x78, 0x4A, 0x03, 0x7D, 0xC3, 0x79, 0x93, 0x30, 0xEA,
      0x38, 0x07, 0x67, 0xB0, 0xD0, 0xB6, 0xED, 0xFC, 0x97, 0x30, 0xE0, 0x4D,
      0x10, 0x39, 0x54, 0x8A, 0x6F, 0x83, 0x88, 0x90, 0x98, 0x52, 0x2E, 0xBA,
      0xB6, 0x84, 0xDA, 0x6F, 0xE2, 0x6A, 0x4A, 0x68, 0x91, 0xD8, 0x6D, 0x40,
      0xFC, 0xD9, 0xA2, 0x4F, 0x74, 0x3D, 0x74, 0xB2, 0x3B, 0x15, 0x96, 0x81,
      0x07, 0x27, 0xC8, 0x1B, 0xB3, 0xF9, 0xF3, 0xBA, 0xDF, 0xAE, 0x99, 0x97,
      0x94, 0x9E, 0xE0, 0xE2, 0x49, 0x87, 0xFA, 0x18, 0x2A, 0x00, 0xD7, 0x3D,
      0xCE, 0xAD, 0xF6, 0x67, 0xE9, 0x0E, 0x5A, 0xE7, 0x6A, 0x1F, 0x83, 0xA9,
      0x1F, 0xCE, 0xA7, 0x8C, 0x96, 0x26, 0x9F, 0x0C, 0x95, 0x01, 0xF1, 0xD4,
      0xCE, 0x68, 0x25, 0x06, 0xA7, 0xEA, 0x89, 0x30, 0x2A, 0x14, 0x80, 0xE1,
      0x8C, 0xDC, 0x1F, 0x6D, 0x57, 0xB5, 0x31, 0x2E, 0xAF, 0x80, 0x88, 0x95,
      0xB2, 0x08, 0x97, 0xE9, 0xA7, 0x82, 0xF9, 0x16, 0xCD, 0x75, 0xB4, 0x98,
      0x1D, 0xA1, 0x38, 0x1F, 0x14, 0xEB, 0x1E, 0xC2, 0x48, 0xB2, 0x7F, 0x0E,
      0x69, 0x66, 0xA0, 0xCD, 0x75, 0x41, 0x4A, 0x73, 0x59, 0x28, 0xB2, 0x12,
      0x06, 0x15, 0xD8, 0x8F, 0xA5, 0x7A, 0xF5, 0xC4, 0x0E, 0x61, 0x75, 0x0F,
      0x0A, 0x0F, 0x8E, 0x60, 0x57, 0x47, 0xE7, 0xC3, 0x2D, 0x5A, 0x23, 0xF1,
      0x41, 0x24, 0x96, 0x9C, 0x07, 0x2E, 0x94, 0x9C, 0x84, 0x75, 0xE3, 0x10,
      0x8D, 0x68, 0x9D, 0x2D, 0x20, 0x79, 0x7F, 0xE1, 0x46, 0x18, 0x81, 0x1E,
      0x9A, 0x49, 0x7F, 0xD2, 0x6B, 0x9E, 0x71, 0x35, 0x58, 0x52, 0xD4, 0xB3,
      0x63, 0x40, 0xB6, 0x16, 0x95, 0xE3, 0x74, 0x5F, 0x8D, 0x07, 0x64, 0x4A,
      0xC6, 0xE2, 0xC1, 0x8B, 0x3F, 0xC2, 0x76, 0xD4, 0xD1, 0x9D, 0xB6, 0x9A,
      0x7C, 0xF2, 0x60, 0x86, 0xF1, 0x72, 0xE2, 0xBC, 0xE1, 0x61, 0x8A, 0x74,
      0x0A, 0x0C, 0x73, 0x9F, 0xD5, 0x04, 0xF7, 0x2C, 0x2A, 0x72, 0xAD, 0xB5,
      0x56, 0x4B, 0xC8, 0x5D, 0xAB, 0x4C, 0x9C, 0xE7, 0x90, 0xD7, 0x8D, 0x14,
      0xD3, 0xBD, 0x24, 0x2D, 0xF0, 0x41, 0x06, 0xD9, 0x6C, 0xE7, 0xC3, 0xB3,
      0x92, 0xCC, 0xED, 0x9B, 0x99, 0xDF, 0x35, 0x9F, 0xD5, 0x1F, 0x30, 0x6C,
      0xBC, 0xBD, 0x5B, 0x46, 0xB8, 0x48, 0x7C, 0xD7, 0xB7, 0xED, 0xD3, 0xC5,
      0xC0, 0x29, 0x65, 0xC8, 0x46, 0x30, 0xDA, 0x1B, 0x6B, 0x8B, 0x31, 0x7F,
      0xE5, 0x5F, 0x7C, 0x79, 0xE0, 0x5C, 0xDA, 0xC9, 0xE8, 0x63, 0x02, 0x3D,
      0xAF, 0x47, 0x0E, 0x9C, 0x3F, 0xB8, 0xC0, 0x1F, 0xDF, 0x3A, 0xED, 0xF2,
      0x19, 0x3B, 0xFA, 0x69, 0xA8, 0x06, 0xE2, 0xE7, 0x01, 0x51, 0xAB, 0xCF,
      0x96, 0xD3, 0x1C, 0xF6, 0xA3, 0x17, 0xC0, 0x59, 0xCA, 0x8C, 0x7D, 0x45,
      0x6A, 0x8E, 0x5E, 0xBA, 0xA6, 0xC1, 0x28, 0x3A, 0x31, 0x9F, 0x18, 0x8A,
      0xAA, 0x80, 0xD8, 0x30, 0x1E, 0x32, 0x17, 0x54, 0xE5, 0xFB, 0x4E, 0x0B,
      0x25, 0x59, 0x4B, 0x01, 0xBC, 0x5F, 0x82, 0xFF, 0x25, 0xB0, 0x64, 0xC7,
      0x66, 0x42, 0x4D, 0x65, 0x84, 0x59, 0xEF, 0xD7, 0xA2, 0x0B, 0x65, 0xDB,
      0x18, 0x18, 0x11, 0xE6, 0xD5, 0xA4, 0xBD, 0x15, 0x3F, 0x70, 0x66, 0xBD,
      0x77, 0x57, 0xD2, 0xD4, 0x17, 0xD2, 0x1F, 0x83, 0xD7, 0xC4, 0xCB, 0x6A,
      0x07, 0x03, 0xA4, 0x20, 0x32, 0xF0, 0xFD, 0x19, 0x8D, 0x9D, 0x8B, 0x0F,
      0x91, 0xB3, 0x59, 0xFB, 0xE9, 0x08, 0x43, 0x2C, 0x32, 0x86, 0xE1, 0xEF,
      0x9D, 0x60, 0x17, 0x02, 0x15, 0x7E, 0xFB, 0xAB, 0x68, 0xE0, 0xE7, 0x13,
      0x6B, 0xFC, 0x90, 0xD2, 0x6B, 0xD8, 0xA9, 0xA7, 0x01, 0x8D, 0xE4, 0xC4,
      0xBF, 0x05, 0xCE, 0x46, 0x5F, 0x91, 0x7D, 0x20, 0xA4, 0xF2, 0x21, 0xA4,
      0xEE, 0x78, 0x81, 0x3A, 0x1E, 0x8A, 0x11, 0x7C, 0x84, 0x70, 0x92, 0x97,
      0x01, 0xCC, 0xC2, 0x01, 0xA8, 0x5E, 0x7F, 0x18, 0xB6, 0xBC, 0x96, 0xFE,
      0x80, 0xB1, 0xE0, 0x74, 0x66, 0x15, 0x25, 0xD3, 0xFD, 0x0C, 0xE2, 0x56,
      0x5A, 0xB1, 0x11, 0x55, 0xDA, 0xFE, 0x4D, 0x34, 0x10, 0x32, 0x8D, 0x6D,
      0xBB, 0x4D, 0xD9, 0x9A, 0x84, 0xFE, 0x96, 0x28, 0x3D, 0x32, 0x32, 0x25,
      0x22, 0xB8, 0x8B, 0x3A, 0xA2, 0xA1, 0x1C, 0x03, 0x24, 0xB1, 0xD5, 0x55,
      0x6E, 0xF4, 0x08, 0xD3, 0x7B, 0x0D, 0xF8, 0x02, 0xD1, 0x63, 0xFE, 0x38,
      0xD7, 0xC3, 0x89, 0x16, 0xA2, 0x68, 0x10, 0xBD, 0x17, 0x5D, 0x22, 0x76,
      0x23, 0x53, 0xC3, 0x17, 0x5D, 0xC6, 0x04, 0x0C, 0x89, 0x9E, 0x07, 0xA3,
      0x39, 0xCD, 0x4D, 0xDB, 0xD4, 0xD5, 0x54, 0x9E, 0x02, 0xC0, 0xD6, 0x91,
      0x26, 0x39, 0x36, 0xA9, 0xF6, 0x31, 0x11, 0x41, 0x2B, 0x60, 0xAA, 0x9F,
      0x57, 0x48, 0x63, 0x34, 0xE4, 0x0B, 0x2B, 0xC1, 0xB8, 0xEA, 0xA4, 0x87,
      0xA0, 0x94, 0xE4, 0x5C, 0x3F, 0x77, 0xF7, 0x2E, 0xA7, 0x41, 0xCE, 0x22,
      0x5E, 0xCB, 0xE2, 0xB5, 0xE4, 0xA1, 0xFC, 0x08, 0x00, 0x70, 0xA6, 0x58,
      0xFD, 0xF9, 0xE2, 0xB3, 0x88, 0x72, 0x28, 0x55, 0x26, 0x7B, 0x30, 0xD9,
      0x4B, 0x63, 0xC3, 0xED, 0x35, 0xD4, 0x75, 0xB7, 0xEB, 0x22, 0xE3, 0xD2,
      0x46, 0x2A, 0xBA, 0x9C, 0xF2, 0xA8, 0x6B, 0x73, 0x8E, 0xBB, 0x27, 0x0A,
      0xB2, 0x97, 0x08, 0xA2, 0x61, 0x4A, 0x55, 0x7E, 0x33, 0xA6, 0x20, 0xB5,
      0x07, 0x28, 0x6E, 0x5D, 0x4C, 0xA5, 0x7E, 0x2C, 0xEE, 0xDB, 0x99, 0x65,
      0xFF, 0x1C, 0x3E, 0x17, 0x77, 0xF9, 0x80, 0xCD, 0xFB, 0x14, 0x45, 0xBB,
      0xE0, 0xB6, 0xAC, 0xBA, 0x02, 0x16, 0x98, 0x0F, 0x96, 0x2F, 0xBF, 0xAB,
      0xE2, 0x65, 0xB3, 0xAD, 0xFE, 0x86, 0x41, 0x08, 0x82, 0x87, 0x46, 0x88,
      0x27, 0xAE, 0x60, 0x1B, 0x6A, 0x16, 0x5D, 0xEE, 0xD3, 0x9C, 0x0E, 0x87,
      0x73, 0xBF, 0x20, 0x46, 0xBB, 0xF6, 0x36, 0x34, 0xBD, 0xBC, 0xAF, 0x98,
      0x35, 0x8D, 0x25, 0xFD, 0xE4, 0x75, 0x78, 0x17, 0x33, 0xDD, 0xE8, 0xC6,
      0xD6, 0x38, 0x3D, 0x13, 0xB6, 0xD4, 0x8F, 0xF1, 0xB6, 0x5E, 0x2F, 0xF1,
      0x3A, 0xAA, 0x9C, 0xCC, 0xFC, 0x3C, 0x62, 0x69, 0x35, 0xC5, 0x27, 0x0F,
      0x9E, 0x23, 0xA7, 0x1A, 0x87, 0xCF, 0x2B, 0xD7, 0x93, 0xCB, 0x17, 0x5D,
      0x23, 0xEA, 0x5F, 0xBD, 0x82, 0xC1, 0x8A, 0x18, 0x22, 0x42, 0x8C, 0x32,
      0xDB, 0x9E, 0x31, 0xB9, 0x4B, 0xE3, 0x14, 0x4A, 0xBB, 0x00, 0xF5, 0xAC,
      0xAA, 0xA4, 0x31, 0xC1, 0x73, 0x86, 0x71, 0x9C, 0x3F, 0xF4, 0x7C, 0x38,
      0x72, 0x0B, 0x1A, 0xB0, 0x18, 0x89, 0xDA, 0xD8, 0x77, 0xBA, 0xDC, 0x9F,
      0xC7, 0x16, 0xF6, 0x48, 0xFC, 0x8B, 0x55, 0x1F},
     true,
     {0x2C, 0x37, 0xC4, 0x9E, 0x94, 0xDF, 0x71, 0x5B, 0x3C, 0x09, 0xE6,
      0x3A, 0x39, 0xE0, 0x4D, 0xB8, 0xD2, 0x6B, 0xD2, 0xB9, 0x07, 0x2C,
      0x9B, 0x21, 0x07, 0x6B, 0xDF, 0xC0, 0xB6, 0x08, 0x53, 0x4C}},

    {params_ml_kem768_test_mode,
     {0x8A, 0x43, 0x36, 0xFD, 0xDB, 0x3F, 0x55, 0xD1, 0x6A, 0xDB, 0xBE, 0x54,
      0xC6, 0xEF, 0x0D, 0xB2, 0x7F, 0x20, 0x67, 0x93, 0x93, 0xD8, 0x6E, 0xA4,
      0x59, 0x0C, 0xB6, 0xF5, 0xF0, 0x9B, 0xC4, 0xEB, 0x76, 0x18, 0x1A, 0x13,
      0xC9, 0x82, 0x6F, 0xBD, 0x2A, 0x71, 0x74, 0xBE, 0x8A, 0x11, 0xF1, 0x37,
      0x59, 0xEE, 0x23, 0xDA, 0x15, 0x33, 0x7A, 0x4C, 0x56, 0x12, 0x48, 0x0E,
      0x0A, 0x84, 0x3C, 0xC6, 0xD0, 0x4F, 0x3A, 0x90, 0x2E, 0x14, 0x4E, 0xFD,
      0xC0, 0xAC, 0x11, 0x8B, 0xF8, 0x55, 0x3B, 0x98, 0x4E, 0x75, 0x8E, 0x6D,
      0x7E, 0xD1, 0x37, 0x3B, 0x20, 0xA5, 0x72, 0x62, 0x71, 0xC5, 0xF4, 0xB5,
      0x42, 0xFC, 0xCD, 0x63, 0x79, 0x67, 0x1C, 0xE3, 0x7A, 0x5D, 0x01, 0x28,
      0xF5, 0x55, 0x39, 0xB9, 0xA8, 0x55, 0x17, 0x2C, 0xA2, 0xDA, 0x3B, 0xB6,
      0x82, 0x34, 0x84, 0xA8, 0x7D, 0xC2, 0x33, 0x3F, 0x56, 0xCB, 0xAD, 0xF4,
      0xA6, 0x94, 0xA5, 0xDA, 0xE3, 0x41, 0xA0, 0xE3, 0xFB, 0xB3, 0xD8, 0x52,
      0x92, 0x9F, 0xBA, 0xFB, 0xF4, 0xA5, 0xC1, 0x2C, 0xD3, 0x49, 0x4C, 0xDF,
      0x91, 0x00, 0x10, 0xA0, 0xFA, 0xFB, 0xC0, 0x9B, 0x37, 0x5B, 0xAB, 0xFF,
      0xDE, 0xAC, 0xCD, 0x12, 0xE6, 0xE7, 0xBD, 0x34, 0x7C, 0xBF, 0xBD, 0x0C,
      0x84, 0xCD, 0xAB, 0xB5, 0x00, 0x4C, 0xA1, 0x1D, 0xDC, 0x6D, 0x14, 0xC1,
      0xBD, 0x70, 0x0F, 0xE3, 0xEB, 0x23, 0x71, 0xE3, 0x29, 0x3F, 0x71, 0x85,
      0xE2, 0xA0, 0x65, 0x53, 0x2C, 0x3B, 0x65, 0x29, 0xE6, 0x02, 0x40, 0xE7,
      0xAB, 0x64, 0x56, 0x13, 0x9D, 0x66, 0x74, 0x5F, 0x17, 0xB9, 0x4F, 0xDF,
      0x2C, 0x54, 0xB1, 0x3E, 0xE4, 0xDE, 0xBF, 0x1B, 0x77, 0x09, 0x97, 0x18,
      0x80, 0x4B, 0xAE, 0xAA, 0xAC, 0xD2, 0xBC, 0x60, 0xA1, 0x90, 0x48, 0x7C,
      0xDC, 0x76, 0xAF, 0x2E, 0xEB, 0x90, 0x6E, 0x4C, 0x9F, 0x26, 0x64, 0xA3,
      0x0F, 0xAF, 0xB6, 0x50, 0x13, 0xB8, 0xCA, 0x39, 0x37, 0x93, 0xB6, 0x50,
      0xCA, 0xC4, 0xA9, 0x33, 0x77, 0xA6, 0x51, 0x1D, 0x73, 0x9C, 0x21, 0x36,
      0xCE, 0xC5, 0x9E, 0x1B, 0xD1, 0x45, 0x84, 0x98, 0x9A, 0x59, 0x1E, 0x1F,
      0x3B, 0x7F, 0x6D, 0x72, 0x37, 0xAE, 0xDB, 0x55, 0x68, 0x80, 0x81, 0x0F,
      0xAB, 0xDB, 0x1D, 0x7F, 0x82, 0x50, 0xB6, 0x1A, 0x2D, 0x16, 0xA3, 0x33,
      0x7D, 0xA6, 0x5A, 0xEA, 0x64, 0x4D, 0x7E, 0x22, 0x26, 0xBE, 0x5F, 0x24,
      0xCB, 0xE0, 0x1C, 0x8A, 0x33, 0xA4, 0xCC, 0xA0, 0x6F, 0x6F, 0x64, 0x6A,
      0x3F, 0x54, 0x53, 0xFE, 0x2D, 0x9F, 0xDE, 0xA8, 0xD8, 0x61, 0x3F, 0x49,
      0x1B, 0xCF, 0x2A, 0xEA, 0x95, 0x0D, 0xB1, 0xD9, 0xB4, 0x3C, 0x7C, 0x3F,
      0x86, 0xFA, 0x2F, 0x4A, 0x51, 0xCB, 0x44, 0xEB, 0x97, 0x61, 0x36, 0x3C,
      0x38, 0x72, 0x38, 0x52, 0x92, 0x52, 0x47, 0xD9, 0x2E, 0x37, 0xFC, 0x69,
      0x4D, 0x2C, 0xB0, 0x02, 0x48, 0x02, 0x3D, 0x54, 0x48, 0xCD, 0xE2, 0x86,
      0x71, 0x25, 0x25, 0x0B, 0x17, 0x38, 0x84, 0x40, 0xC1, 0x88, 0xF7, 0xE5,
      0x00, 0xCE, 0xF7, 0x74, 0x7A, 0x10, 0x1E, 0x0B, 0xF2, 0x52, 0x1E, 0x2C,
      0x8A, 0x2D, 0x04, 0xF4, 0x2D, 0x83, 0x4C, 0x02, 0x74, 0xEC, 0xBC, 0x73,
      0xE9, 0x46, 0x12, 0xCC, 0xDB, 0x1C, 0x4B, 0x90, 0x8B, 0xAF, 0x63, 0xC0,
      0x9C, 0x94, 0x5A, 0xD4, 0x64, 0x59, 0x12, 0xA0, 0x66, 0x6E, 0x98, 0x44,
      0xA1, 0x61, 0x4B, 0x7F, 0x34, 0x41, 0x5C, 0x18, 0x42, 0xF9, 0xB1, 0xC7,
      0xDA, 0xF7, 0xEE, 0x44, 0x59, 0xA8, 0x72, 0x4B, 0x70, 0x50, 0xF6, 0xB5,
      0x83, 0x33, 0x41, 0x69, 0x10, 0x19, 0x14, 0x9F, 0x35, 0x1A, 0x7F, 0x11,
      0xAE, 0x24, 0x16, 0xDC, 0xD5, 0xB3, 0x6F, 0x18, 0xB1, 0xA4, 0xB8, 0x2C,
      0xC3, 0xE9, 0x24, 0x11, 0x4C, 0xFC, 0x12, 0x6C, 0xA3, 0x09, 0xE3, 0x19,
      0xD4, 0x97, 0xA5, 0x94, 0xB0, 0xAB, 0x2A, 0xFB, 0x58, 0xC1, 0x9D, 0xEF,
      0x3B, 0xC3, 0xAD, 0x88, 0x5B, 0x29, 0xAE, 0xAC, 0x81, 0xF3, 0x46, 0xA1,
      0x96, 0x83, 0xB8, 0x57, 0x7F, 0x4A, 0x1E, 0x0F, 0x30, 0xBD, 0xC8, 0x5A,
      0x38, 0x14, 0xCD, 0x11, 0x96, 0xE6, 0xB2, 0x9E, 0x55, 0xE5, 0xC0, 0xE4,
      0xE0, 0x28, 0x87, 0x24, 0x77, 0xCB, 0x67, 0x5B, 0x24, 0x08, 0xE1, 0x36,
      0xD1, 0x5E, 0x54, 0xC8, 0x5E, 0x8A, 0x46, 0x84, 0x23, 0xCB, 0x79, 0x5D,
      0x93, 0x48, 0xBF, 0xCC, 0x97, 0x5B, 0x4E, 0xC2, 0x0A, 0x23, 0x99, 0x1E,
      0x6E, 0x9E, 0xF9, 0x1D, 0x67, 0x69, 0x83, 0xAC, 0x26, 0xB6, 0x6C, 0x71,
      0x54, 0x8F, 0xB4, 0x6C, 0x4B, 0xF0, 0x6E, 0x28, 0x0D, 0x7C, 0x55, 0xE7,
      0xB8, 0xDB, 0x90, 0x74, 0x3A, 0x8F, 0x89, 0x3F, 0x95, 0xAE, 0xB4, 0xDE,
      0xD1, 0xDC, 0x65, 0xC5, 0xE0, 0xB6, 0x1F, 0xBA, 0xD9, 0xDA, 0x0D, 0xDA,
      0xC2, 0x74, 0x59, 0x1A, 0xA6, 0xCF, 0x23, 0xC7, 0x9C, 0x09, 0x41, 0x43,
      0x56, 0x58, 0x4F, 0x0B, 0xE0, 0x2C, 0xE9, 0xB5, 0x00, 0xA3, 0xEE, 0x6B,
      0xD4, 0xFA, 0x01, 0x19, 0x78, 0x3F, 0x50, 0xE8, 0x00, 0xED, 0x36, 0xD3,
      0xA4, 0x44, 0x59, 0x34, 0xDC, 0xFD, 0x87, 0xA3, 0x1A, 0xF3, 0xAB, 0xC0,
      0x2C, 0xAC, 0x39, 0xC4, 0xB2, 0x80, 0x68, 0xEE, 0xCC, 0x6D, 0x16, 0xB6,
      0xFA, 0x18, 0x7A, 0x07, 0x3B, 0xA1, 0x43, 0x20, 0x9C, 0x0F, 0x38, 0xAF,
      0xE1, 0x00, 0xBC, 0x70, 0x0D, 0x46, 0x1B, 0x1B, 0x36, 0x4E, 0xD2, 0x98,
      0xAA, 0xFD, 0xFC, 0x71, 0x6F, 0xA6, 0xE3, 0x87, 0x0E, 0x62, 0x58, 0xB6,
      0x66, 0x45, 0x09, 0x1F, 0xCF, 0x94, 0x13, 0xED, 0xF6, 0xBC, 0x79, 0xB7,
      0x51, 0x32, 0xA4, 0x6D, 0x1D, 0xFB, 0xBC, 0xE3, 0xCE, 0x9B, 0x05, 0x58,
      0xEF, 0x00, 0x39, 0x29, 0xCC, 0x6E, 0x3D, 0x57, 0xBC, 0x4F, 0xD3, 0x09,
      0x2E, 0xEA, 0xC4, 0xED, 0x71, 0xB7, 0xB7, 0xFC, 0x70, 0xD0, 0xE6, 0x59,
      0x01, 0xDC, 0x91, 0x96, 0x92, 0x8C, 0x5B, 0x8C, 0xF4, 0xA6, 0x3C, 0x62,
      0x79, 0x77, 0x27, 0xC1, 0x92, 0xCF, 0x1C, 0xE4, 0x31, 0x51, 0x20, 0xA5,
      0x7D, 0x4C, 0x8C, 0xFD, 0x03, 0x14, 0x3A, 0xF8, 0x75, 0x44, 0x32, 0xEE,
      0xBA, 0xDC, 0xAD, 0xBC, 0xD2, 0x6C, 0x2E, 0x3A, 0x14, 0xBB, 0x43, 0xA9,
      0x51, 0xAF, 0xDC, 0x19, 0xEE, 0x67, 0xAA, 0xEC, 0x5D, 0xE0, 0x72, 0x2E,
      0x9D, 0x11, 0xE3, 0x62, 0x7A, 0xD1, 0xB6, 0x24, 0xAD, 0xF0, 0xFB, 0x6F,
      0xD2, 0xA6, 0x73, 0x3B, 0x2B, 0x1B, 0x14, 0x11, 0xDD, 0x14, 0xEE, 0x87,
      0xAD, 0x3B, 0xCB, 0xBC, 0xAD, 0x2E, 0xB4, 0xA3, 0x8E, 0xA0, 0x05, 0x75,
      0xBF, 0xA9, 0x93, 0x32, 0x40, 0x00, 0x83, 0xFC, 0x51, 0x9C, 0x37, 0x33,
      0xF6, 0xED, 0xCC, 0xAA, 0xF7, 0x1D, 0x09, 0xA7, 0x16, 0x4E, 0x18, 0xA9,
      0xE9, 0x58, 0x7A, 0x8D, 0x9B, 0x9A, 0x46, 0x56, 0x3F, 0xD3, 0xF1, 0x4B,
      0xFA, 0x2F, 0x2B, 0x8E, 0xBD, 0x9F, 0xDE, 0xAA, 0xEF, 0x46, 0x6E, 0x59,
      0x1F, 0x50, 0x21, 0x51, 0xE4, 0x3A, 0x7E, 0x11, 0x23, 0x27, 0x3E, 0x5E,
      0x05, 0x74, 0x81, 0x4B, 0x20, 0x25, 0x3A, 0x17, 0x91, 0x7D, 0x7B, 0xDF,
      0x83, 0x70, 0xBC, 0x50, 0x46, 0x1A, 0xC8, 0xD8, 0x61, 0x27, 0xDC, 0x52,
      0x7B, 0x82, 0x90, 0xFE, 0x38, 0x6F, 0x1A, 0xC1, 0xE6, 0xE9, 0xD7, 0xB4,
      0x93, 0xBB, 0x7F, 0xED, 0xEC, 0x9E, 0x5A, 0x82, 0xDC, 0x14, 0x02, 0xDE,
      0xAE, 0x71, 0xB1, 0x8A, 0xB4, 0xB6, 0x58, 0xE4, 0x3F, 0x70, 0x72, 0x59,
      0x03, 0x9E, 0xB9, 0x97, 0x8D, 0x4F, 0xB0, 0xD6, 0x28, 0x39, 0xA0, 0xDD,
      0x8E, 0x3A, 0x11, 0x83, 0xCE, 0x33, 0x0D, 0x57, 0xBC, 0x79, 0x27, 0xF7,
      0xCC, 0xF0, 0x6B, 0xA1, 0x0A, 0x04, 0x78, 0xB7, 0xE2, 0xEC, 0x81, 0x81,
      0x95, 0x17, 0x1A, 0xFF, 0x75, 0xC2, 0x9B, 0x28, 0x3E, 0x75, 0x9F, 0x4D,
      0x2F, 0x5D, 0x55, 0xF0, 0xFF, 0xC3, 0x5E, 0x05, 0x81, 0xD9, 0x8E, 0x58,
      0x21, 0x07, 0xBF, 0x64, 0xA6, 0xD8, 0x06, 0x03},
     true,
     {0x47, 0x03, 0x3B, 0x02, 0xA6, 0xDC, 0x05, 0x6F, 0xFE, 0xB5, 0xFC,
      0x1E, 0x96, 0x20, 0x5C, 0x16, 0x63, 0x74, 0xAB, 0x84, 0xA5, 0xF3,
      0xF7, 0xB0, 0x64, 0x27, 0xBB, 0x00, 0x6E, 0x71, 0xA5, 0xA4}},

    {params_ml_kem768_test_mode,
     {0x60, 0x95, 0xA9, 0x51, 0x75, 0x3A, 0x64, 0x4D, 0xD8, 0x98, 0xD6, 0x91,
      0x38, 0xB4, 0xE5, 0x21, 0xA7, 0x04, 0xDC, 0xFA, 0xAD, 0x44, 0xEB, 0x53,
      0xE2, 0x84, 0xF8, 0x36, 0xA4, 0x69, 0x34, 0x9C, 0x5B, 0x92, 0x79, 0x24,
      0x8A, 0xFC, 0x57, 0xAC, 0x93, 0xFA, 0x34, 0xA6, 0x43, 0xDE, 0x02, 0xB7,
      0x24, 0x61, 0x5C, 0xF5, 0x86, 0x59, 0x27, 0xFE, 0xD6, 0x0A, 0x6B, 0x41,
      0xE4, 0xAB, 0x15, 0xB4, 0xDA, 0x35, 0x99, 0xF1, 0x3D, 0x2C, 0x19, 0x96,
      0xC6, 0xD6, 0x98, 0x94, 0x43, 0xBE, 0x6F, 0xB8, 0x1F, 0x5B, 0xA0, 0x3B,
      0xDD, 0x53, 0x46, 0x2B, 0xE5, 0x81, 0x2A, 0x3E, 0x17, 0x78, 0x76, 0xA1,
      0x02, 0xB0, 0xEB, 0xDF, 0xCB, 0x16, 0xDE, 0x7B, 0x29, 0xB5, 0x12, 0x3A,
      0x79, 0xDD, 0x82, 0xE5, 0xCD, 0x47, 0xAB, 0xA0, 0x27, 0x59, 0xFA, 0xF5,
      0x40, 0x1E, 0x3B, 0xF0, 0x31, 0x44, 0xA9, 0x0A, 0xE9, 0x57, 0xEC, 0x04,
      0xDB, 0x98, 0x64, 0xAD, 0xE1, 0xC5, 0xA7, 0x00, 0xCE, 0xC7, 0x87, 0x2C,
      0xCB, 0x64, 0xFF, 0x93, 0x19, 0x84, 0xDD, 0xC3, 0xFB, 0x8D, 0x49, 0x71,
      0xD7, 0x61, 0xE5, 0x54, 0x41, 0x30, 0x27, 0x8C, 0x75, 0xA1, 0xB0, 0x4E,
      0x64, 0x1E, 0x07, 0x0A, 0x74, 0x77, 0x89, 0xA7, 0x1E, 0x09, 0x40, 0x9C,
      0x15, 0x5C, 0x7D, 0x34, 0x1D, 0x5F, 0x82, 0x8A, 0x57, 0x5E, 0xE7, 0x44,
      0x39, 0x15, 0x59, 0x30, 0xDF, 0x22, 0xFD, 0x77, 0x16, 0x18, 0x5B, 0xDF,
      0x91, 0x74, 0x72, 0x43, 0x2A, 0x30, 0xA6, 0x76, 0x2C, 0x9F, 0xE1, 0xA2,
      0x54, 0x44, 0x2F, 0x75, 0x58, 0x04, 0xD2, 0x95, 0xB1, 0x69, 0x8B, 0x47,
      0xA6, 0x7B, 0xBF, 0xDE, 0x17, 0x82, 0x00, 0xF9, 0xCC, 0x3D, 0x4C, 0x70,
      0x5F, 0x4A, 0xC1, 0xB0, 0x0C, 0x37, 0x2D, 0x46, 0x8E, 0x16, 0xED, 0x3C,
      0xBA, 0xAA, 0x86, 0x2A, 0x25, 0x74, 0xA9, 0x57, 0x4A, 0x72, 0x80, 0x87,
      0x8B, 0xB8, 0x2D, 0xA7, 0xBD, 0x1B, 0x2A, 0x58, 0x94, 0x34, 0x56, 0x83,
      0x8F, 0x2E, 0x6A, 0xA9, 0xF6, 0xEF, 0x18, 0x27, 0xC5, 0xB2, 0x4F, 0xA0,
      0x9D, 0xE0, 0x7E, 0x9B, 0x31, 0x53, 0xB0, 0xF4, 0x4A, 0x4F, 0x2A, 0xEA,
      0x76, 0x10, 0xF9, 0xCC, 0xA9, 0x25, 0x65, 0x74, 0x0E, 0x72, 0x95, 0xBA,
      0x3A, 0xC5, 0x76, 0x4A, 0x20, 0xA4, 0x4D, 0x4E, 0x18, 0x62, 0xE5, 0x5B,
      0x1D, 0xF7, 0x91, 0x3B, 0x27, 0x9F, 0x43, 0x8B, 0x3B, 0x34, 0xE0, 0xC2,
      0x2F, 0xD9, 0x0E, 0x06, 0x49, 0x7F, 0x7D, 0xCF, 0x8D, 0x62, 0x35, 0x24,
      0x47, 0xC2, 0xB8, 0xC5, 0x1C, 0x21, 0x47, 0x96, 0x19, 0x4C, 0xDF, 0x66,
      0xD5, 0x00, 0x12, 0x78, 0xD0, 0xD5, 0x5F, 0x82, 0xFA, 0x31, 0xDA, 0xA7,
      0x2B, 0xA6, 0xCD, 0xA3, 0x4E, 0x60, 0xD6, 0x96, 0xED, 0x79, 0xC7, 0x05,
      0x6B, 0xFE, 0x97, 0x26, 0x5F, 0x3D, 0x1B, 0xC0, 0x77, 0x19, 0xB7, 0x45,
      0xAD, 0xD4, 0xA8, 0x34, 0x04, 0xD9, 0x1A, 0x18, 0x4E, 0x62, 0x9F, 0xC2,
      0x4A, 0xE2, 0x36, 0xCF, 0x6A, 0xFA, 0xE4, 0x62, 0x95, 0xD2, 0x4B, 0x43,
      0x1D, 0x81, 0x9E, 0x36, 0x6F, 0x51, 0xE1, 0xBB, 0x2B, 0x44, 0xB1, 0xFB,
      0x7A, 0x30, 0x60, 0x09, 0x1D, 0xEA, 0x1D, 0x41, 0x62, 0x68, 0xCA, 0x55,
      0x0E, 0xE4, 0xE4, 0x1F, 0xCA, 0x1F, 0x38, 0x7E, 0x94, 0x1D, 0xBE, 0x4E,
      0xBA, 0xE2, 0x22, 0xD3, 0xCF, 0x62, 0x56, 0x32, 0xD1, 0xA6, 0x14, 0x14,
      0x03, 0x8F, 0xD4, 0x37, 0xBF, 0xA2, 0x00, 0x05, 0xEB, 0xC4, 0x04, 0xAD,
      0xCD, 0xE2, 0xDC, 0x10, 0xDB, 0x74, 0x1A, 0x3B, 0x75, 0x34, 0xC4, 0x08,
      0x22, 0x52, 0x0C, 0x47, 0x03, 0xFD, 0xFB, 0x6B, 0x38, 0x0F, 0x7D, 0xB7,
      0x2B, 0x72, 0x5B, 0x33, 0x0D, 0x0C, 0x20, 0xDF, 0x25, 0x6B, 0xBD, 0xDC,
      0x31, 0xE0, 0xEA, 0x20, 0xE6, 0x36, 0xA9, 0xFA, 0xE3, 0x10, 0x18, 0x5A,
      0x50, 0x81, 0x92, 0x3B, 0xAF, 0xE0, 0x41, 0xAC, 0x6F, 0xCD, 0x4E, 0x73,
      0xF5, 0xF7, 0x23, 0x71, 0x42, 0xB7, 0x46, 0x81, 0xF6, 0x37, 0x99, 0x6D,
      0x28, 0xC3, 0xFD, 0xE6, 0x05, 0x22, 0x43, 0x26, 0x9D, 0x19, 0x31, 0x6C,
      0x56, 0x99, 0x37, 0x22, 0xEA, 0xDF, 0x19, 0xA9, 0x85, 0xE5, 0x79, 0xED,
      0x55, 0x9F, 0x97, 0x1E, 0x69, 0xEB, 0x51, 0x25, 0x93, 0x7E, 0xBC, 0x80,
      0xEC, 0xD1, 0x5A, 0x4F, 0x80, 0xD7, 0x06, 0x79, 0x05, 0xA4, 0xD3, 0x9C,
      0x62, 0x20, 0xEF, 0xE4, 0x38, 0x83, 0xCF, 0x22, 0xE9, 0xA3, 0x66, 0xF8,
      0x91, 0x1E, 0x21, 0xD0, 0x49, 0x1B, 0x8F, 0xF6, 0x1F, 0xD0, 0x7B, 0x73,
      0x3E, 0x70, 0x7A, 0x08, 0xDB, 0x40, 0x0E, 0x43, 0x8D, 0xAA, 0x00, 0xD4,
      0x81, 0xC5, 0xAC, 0x62, 0x06, 0x4C, 0xF4, 0x7A, 0xFE, 0x3A, 0xB0, 0x80,
      0x27, 0xB3, 0x89, 0x0E, 0x8C, 0x88, 0x35, 0xCE, 0xAF, 0x81, 0x28, 0xF9,
      0xD8, 0x87, 0xA6, 0xCB, 0x7F, 0xDE, 0x87, 0x9D, 0x96, 0x11, 0xC0, 0x12,
      0x81, 0xA0, 0xF0, 0x2D, 0xE0, 0xE9, 0x69, 0xC9, 0x13, 0x1F, 0x85, 0x12,
      0x13, 0x80, 0x36, 0xEC, 0x19, 0x67, 0xDC, 0xA4, 0x5A, 0xA3, 0x0B, 0xE8,
      0xC5, 0xB1, 0x00, 0x81, 0x13, 0xE1, 0x7A, 0x91, 0xD9, 0xF8, 0xE9, 0x99,
      0x5C, 0x07, 0xC0, 0xB1, 0x3A, 0x45, 0x66, 0x8C, 0x96, 0x35, 0x6F, 0x09,
      0xC3, 0xE0, 0x8F, 0xE4, 0xC7, 0xDF, 0x5F, 0x72, 0x30, 0xE0, 0xC9, 0x3E,
      0xEF, 0x08, 0xE8, 0x95, 0x8B, 0x55, 0xE2, 0x13, 0x71, 0x8C, 0x51, 0x6E,
      0x62, 0x4B, 0x57, 0x76, 0x52, 0x57, 0xD2, 0x16, 0x96, 0xA3, 0x45, 0x8F,
      0xFB, 0xA1, 0x1D, 0xE7, 0x08, 0xC4, 0xEE, 0x9A, 0xF2, 0xED, 0xC5, 0xF3,
      0x74, 0x58, 0xDE, 0xC8, 0xB9, 0x85, 0x07, 0x68, 0x82, 0xD3, 0xF4, 0xDE,
      0xB0, 0x0B, 0xFD, 0x8E, 0x7E, 0xA4, 0xD5, 0x7B, 0xAE, 0xAE, 0xC6, 0xBA,
      0xBC, 0x0E, 0x28, 0xC1, 0x54, 0x19, 0xCC, 0xD7, 0x85, 0xCF, 0x6A, 0xCE,
      0xC9, 0x6D, 0x11, 0x11, 0xCD, 0xD1, 0xDA, 0x9A, 0x15, 0x1F, 0x59, 0xA7,
      0x36, 0x6B, 0x64, 0xA5, 0x3F, 0x04, 0x97, 0xD3, 0xB5, 0xA8, 0xEC, 0xB6,
      0x0D, 0x7C, 0x22, 0x0E, 0x99, 0x12, 0x6C, 0xDE, 0x82, 0x93, 0x8C, 0x7E,
      0x13, 0x1B, 0xD8, 0x41, 0x30, 0x0A, 0xE4, 0x61, 0xA1, 0x81, 0x77, 0x03,
      0xED, 0x5B, 0x05, 0x10, 0xB4, 0x7F, 0x2C, 0x29, 0x80, 0xF1, 0xE1, 0x1C,
      0xFB, 0xEC, 0xB5, 0x24, 0xB2, 0x95, 0xC4, 0x21, 0x87, 0xF1, 0x5B, 0x0C,
      0x9F, 0x6B, 0x0E, 0xB1, 0xE7, 0x0B, 0x3E, 0xC4, 0x3E, 0xD9, 0x55, 0x52,
      0x8B, 0x1E, 0x42, 0xE2, 0xBC, 0xB3, 0x1F, 0x3A, 0x1C, 0xFB, 0x5E, 0x9C,
      0x80, 0x7E, 0x8D, 0x36, 0x6E, 0x92, 0x27, 0xA8, 0x77, 0x84, 0x74, 0x8B,
      0x27, 0x7D, 0x6C, 0x88, 0x5B, 0x13, 0x85, 0xC6, 0xC6, 0x91, 0xB3, 0xDB,
      0xD7, 0x84, 0x1D, 0xD8, 0x97, 0x21, 0xB3, 0xA8, 0xBF, 0x96, 0xEB, 0xA9,
      0x9C, 0x53, 0xD4, 0xBB, 0x3B, 0x41, 0xDB, 0x94, 0x09, 0xB9, 0x92, 0xBC,
      0xC2, 0xD8, 0xFC, 0x53, 0xE7, 0x07, 0x23, 0xCA, 0x1F, 0xDC, 0x13, 0x41,
      0xA3, 0xE6, 0x08, 0xD7, 0xF6, 0x2F, 0x23, 0x22, 0xC6, 0xA9, 0xBA, 0x13,
      0x16, 0x63, 0x96, 0x90, 0xA2, 0x2A, 0xEC, 0xEE, 0x36, 0x4B, 0x4F, 0x13,
      0x94, 0x9A, 0x03, 0x10, 0xFB, 0xA1, 0xA0, 0xE3, 0x5D, 0xDA, 0x5F, 0xF8,
      0x40, 0xDA, 0xBA, 0xC5, 0x50, 0x41, 0xB0, 0x93, 0x1D, 0x9E, 0xBE, 0xC8,
      0x9B, 0x78, 0xDD, 0x93, 0x05, 0x12, 0x34, 0x0B, 0x4B, 0x5D, 0x08, 0x77,
      0xAF, 0x54, 0x6F, 0xF0, 0xF3, 0x42, 0xFB, 0x76, 0xB6, 0x47, 0xD6, 0x04,
      0xEE, 0x2E, 0x20, 0x20, 0x79, 0x24, 0xF3, 0x99, 0x07, 0xD6, 0xE7, 0x2D,
      0xD4, 0xA9, 0xA1, 0xED, 0x0B, 0x6D, 0x73, 0x64, 0xCC, 0xE6, 0x99, 0x81,
      0xF5, 0x6C, 0xBD, 0xED, 0xD5, 0x1C, 0xBA, 0xF6, 0xFD, 0xDB, 0x36, 0xE3,
      0x27, 0xAD, 0x65, 0xD4, 0xFE, 0x28, 0x3D, 0x25, 0x3E, 0x6B, 0xF3, 0xC7,
      0x96, 0x9F, 0xFF, 0x1F, 0x34, 0xDC, 0xC7, 0x42},
     true,
     {0xF0, 0xCF, 0x9C, 0xF0, 0x6A, 0x81, 0xEE, 0x54, 0x5A, 0x33, 0xB3,
      0x10, 0x61, 0x61, 0x17, 0xD6, 0x09, 0x6F, 0xB5, 0x6F, 0x0D, 0x4F,
      0x7E, 0x49, 0xFE, 0x0A, 0x37, 0x55, 0x03, 0x20, 0xD3, 0xC4}},

    {params_ml_kem768_test_mode,
     {0x2A, 0xAC, 0xD2, 0xE6, 0xB8, 0x84, 0xBE, 0x6A, 0x3D, 0xDD, 0x80, 0x15,
      0x5B, 0xDC, 0xA8, 0x0E, 0xBA, 0xF0, 0xE2, 0xBF, 0x71, 0x43, 0x12, 0xBB,
      0xA3, 0x0D, 0x5B, 0x36, 0x7F, 0x2D, 0x95, 0xAC, 0x7B, 0xEC, 0x39, 0x65,
      0xAB, 0x05, 0xAF, 0xA3, 0x70, 0xA4, 0x2A, 0x51, 0x2B, 0x5E, 0xFE, 0x4B,
      0x0D, 0xEF, 0xF3, 0xE1, 0x63, 0xAF, 0x18, 0x6B, 0x72, 0x5B, 0xCA, 0xFD,
      0x2A, 0xFB, 0x2B, 0xD2, 0xA0, 0xDB, 0xAB, 0x74, 0xC2, 0xBF, 0x93, 0x62,
      0xE2, 0x7D, 0x69, 0xB6, 0xB4, 0xB5, 0xAA, 0x65, 0x00, 0xEB, 0xC9, 0x31,
      0x6E, 0xA4, 0x11, 0x27, 0x45, 0xF1, 0xC6, 0xE9, 0x8F, 0x2D, 0xEF, 0x91,
      0x32, 0xC7, 0xC0, 0xBF, 0xFE, 0xAA, 0xFA, 0xF9, 0x94, 0xC8, 0x9B, 0x96,
      0xD3, 0xF4, 0x36, 0xB8, 0x75, 0x17, 0x89, 0x63, 0xFB, 0xC1, 0x8D, 0x2E,
      0x06, 0xEC, 0xAF, 0x38, 0x71, 0x78, 0x7C, 0x1A, 0xE9, 0x3B, 0x32, 0x10,
      0x89, 0x68, 0x37, 0xEC, 0x1D, 0xA8, 0x7F, 0x0F, 0xD8, 0xF1, 0x4A, 0xB7,
      0xC5, 0xCB, 0x25, 0x31, 0xE9, 0x0F, 0x41, 0x5F, 0xEB, 0xDA, 0x37, 0x8E,
      0x54, 0x92, 0xE1, 0xDE, 0xC8, 0x24, 0x3F, 0xE2, 0xE8, 0xA7, 0xBA, 0xA6,
      0xFB, 0x6A, 0x03, 0x4D, 0x9C, 0x52, 0x4E, 0x99, 0xD8, 0x48, 0xA8, 0x04,
      0xF1, 0x50, 0x91, 0x5B, 0xFD, 0x66, 0x06, 0x7C, 0x86, 0x03, 0xB5, 0xDB,
      0x0F, 0xE2, 0x9E, 0x27, 0xD3, 0xF6, 0xCA, 0x62, 0x9E, 0x96, 0xBF, 0x3E,
      0x9C, 0x77, 0xA5, 0x91, 0x97, 0x01, 0xEC, 0x19, 0x64, 0x6C, 0x69, 0xA7,
      0x3D, 0xFA, 0xAB, 0x0A, 0xBA, 0x28, 0xFE, 0x3E, 0x9E, 0xAA, 0xEB, 0x47,
      0x5A, 0x44, 0x1B, 0x9B, 0x0D, 0x62, 0xB2, 0x59, 0xDC, 0x6B, 0x77, 0xDE,
      0xC9, 0x64, 0xAB, 0x57, 0xD5, 0xD7, 0x76, 0x98, 0x8D, 0x54, 0xE6, 0x24,
      0x6C, 0x52, 0x6F, 0x1E, 0x8E, 0xFD, 0xF4, 0x54, 0xE7, 0xF0, 0xDD, 0xAE,
      0xD5, 0x36, 0x3C, 0xE0, 0x2B, 0x27, 0x9C, 0xD3, 0xB5, 0x54, 0xC2, 0x51,
      0x79, 0x3C, 0x3A, 0x61, 0x6C, 0x07, 0xA7, 0xBA, 0xBA, 0x80, 0x62, 0x91,
      0x9A, 0x2B, 0x46, 0xC6, 0x4C, 0x15, 0x2B, 0xC8, 0x87, 0xA2, 0x7E, 0x38,
      0x22, 0x54, 0xEA, 0x6D, 0x50, 0xCC, 0xC0, 0x70, 0x2B, 0x7B, 0xC0, 0x99,
      0x4B, 0xAC, 0x09, 0xB7, 0x89, 0x1F, 0xA6, 0x4A, 0x77, 0x3A, 0xE0, 0xB4,
      0xFB, 0xF8, 0x20, 0x4C, 0x13, 0xA4, 0x95, 0x0F, 0xC2, 0xC4, 0xDF, 0x60,
      0xCE, 0xFE, 0xD7, 0x58, 0x2F, 0xD9, 0xFB, 0xB8, 0xC8, 0x34, 0x42, 0x51,
      0x7B, 0xA0, 0xE3, 0xB6, 0x0D, 0x9A, 0x04, 0xFB, 0xB2, 0x4A, 0xBC, 0xEC,
      0xB3, 0x03, 0xE3, 0xFD, 0xD3, 0x7F, 0x10, 0x37, 0x74, 0x1F, 0xD2, 0x48,
      0x9F, 0x63, 0x21, 0x92, 0xA6, 0xB9, 0xC1, 0x22, 0xA7, 0x34, 0x4C, 0xB7,
      0x81, 0xA0, 0xF6, 0x1D, 0x50, 0x11, 0xEB, 0x02, 0x51, 0xA8, 0x42, 0xAD,
      0x48, 0x38, 0xF9, 0xB8, 0xD5, 0x2E, 0x21, 0xA7, 0x83, 0xF0, 0xD8, 0x39,
      0xE8, 0xBA, 0x22, 0x1C, 0xCD, 0xD6, 0xB9, 0x68, 0xA2, 0xB5, 0xFD, 0x21,
      0xB8, 0x45, 0x8B, 0xF5, 0x3C, 0x9C, 0x80, 0x76, 0xAC, 0x0C, 0x52, 0xC0,
      0xF5, 0x30, 0x97, 0xED, 0x1C, 0x25, 0xC9, 0xF6, 0xF1, 0x24, 0x07, 0x77,
      0x2D, 0x67, 0x43, 0xEB, 0x8E, 0x0C, 0xE8, 0xB1, 0xA9, 0x26, 0xF0, 0xFD,
      0xD0, 0xDB, 0x00, 0x48, 0x2D, 0x95, 0x90, 0x67, 0x5E, 0x56, 0xD4, 0x50,
      0x9C, 0xB5, 0xE5, 0xF3, 0x2F, 0xC3, 0xB4, 0xA2, 0xDA, 0xB2, 0xBA, 0x08,
      0x0F, 0x9A, 0x7C, 0xDD, 0x0B, 0x61, 0x17, 0x42, 0xA8, 0xF8, 0x3C, 0xEE,
      0x1B, 0x09, 0x1E, 0x62, 0x9D, 0x2A, 0x03, 0x71, 0xFD, 0xB5, 0xA6, 0x44,
      0x12, 0xB5, 0xFA, 0x63, 0x71, 0x69, 0x61, 0x52, 0x76, 0x40, 0xD0, 0x28,
      0x85, 0xC4, 0xA0, 0x9B, 0x04, 0xA3, 0xA6, 0xF5, 0xEC, 0x01, 0xA9, 0xE0,
      0xDB, 0xB8, 0xFC, 0x4D, 0xDD, 0x9E, 0x05, 0xBD, 0xB2, 0x40, 0xAC, 0x48,
      0x78, 0xF0, 0xD4, 0x14, 0x61, 0xC4, 0x66, 0x17, 0x77, 0x41, 0x7D, 0x61,
      0x50, 0x42, 0x2F, 0xEA, 0xB6, 0xA3, 0x9F, 0x15, 0x6C, 0xAD, 0xB5, 0xF5,
      0xD3, 0xBE, 0xBE, 0x41, 0x7B, 0xAB, 0xCE, 0xFF, 0x5A, 0xAD, 0x1B, 0x7A,
      0x62, 0x4F, 0xC2, 0x3A, 0xBE, 0x28, 0xB2, 0xAB, 0x2E, 0x82, 0x73, 0xE8,
      0xF4, 0x46, 0x36, 0xA6, 0x0C, 0xDA, 0xD9, 0x23, 0x6D, 0xCB, 0x02, 0xFC,
      0xF8, 0x77, 0x22, 0xC8, 0x99, 0xAA, 0x32, 0x1C, 0x56, 0x4B, 0x25, 0xBC,
      0x33, 0xB4, 0x97, 0x6B, 0xC9, 0x60, 0x3B, 0xB8, 0xB8, 0xAB, 0x18, 0xB5,
      0xB0, 0x46, 0x25, 0x98, 0x1F, 0xB3, 0x8B, 0x2A, 0x42, 0x72, 0x2C, 0xE2,
      0x35, 0x8F, 0xC0, 0xBA, 0x99, 0xEF, 0x4B, 0x12, 0x2C, 0x7B, 0x70, 0xBB,
      0x34, 0x7D, 0x0D, 0x48, 0x2D, 0xA3, 0x06, 0x38, 0xEF, 0x8B, 0x9C, 0x1D,
      0x91, 0x21, 0xD8, 0x3B, 0xCD, 0xBB, 0xEB, 0x2A, 0x60, 0x86, 0x17, 0x05,
      0x4F, 0x4B, 0x3F, 0xDD, 0x33, 0xE9, 0xA0, 0x8F, 0x8D, 0xF9, 0x99, 0xA9,
      0x8E, 0x71, 0x5D, 0xBF, 0x04, 0xF8, 0xEF, 0xAC, 0xF1, 0x23, 0xBB, 0xEB,
      0x37, 0xB9, 0x03, 0x8E, 0x9A, 0xD9, 0x06, 0xE3, 0xC5, 0x70, 0xBB, 0x39,
      0x8C, 0x10, 0xE6, 0xD3, 0x66, 0x47, 0xA2, 0xB0, 0xB2, 0x73, 0x1F, 0xD3,
      0x9F, 0x72, 0x61, 0x71, 0xEF, 0xC7, 0x32, 0x1B, 0xC6, 0x7D, 0x93, 0x6F,
      0x79, 0x89, 0xEA, 0x58, 0x33, 0x6E, 0x54, 0x9A, 0x34, 0xB7, 0x3F, 0x09,
      0x7E, 0x3E, 0xA2, 0xC2, 0x58, 0x87, 0xEC, 0x6A, 0x2E, 0x9F, 0xED, 0x5D,
      0x2C, 0xFF, 0x47, 0x5E, 0x99, 0xF3, 0x92, 0x16, 0x2D, 0x95, 0x9D, 0xE1,
      0xC4, 0xA4, 0xDA, 0xD3, 0xC9, 0x65, 0x42, 0x75, 0x6A, 0xEC, 0x33, 0x67,
      0xF7, 0xB2, 0x51, 0x5F, 0x22, 0x25, 0xBF, 0x7B, 0x70, 0x4B, 0x78, 0x0A,
      0x6D, 0x0B, 0x27, 0x9B, 0x8B, 0x4E, 0xE4, 0x87, 0x9A, 0x9B, 0xBB, 0x2F,
      0x33, 0x03, 0x21, 0x6C, 0xBA, 0xDE, 0xA0, 0x0D, 0x22, 0x9C, 0x03, 0xE3,
      0xE2, 0x84, 0x38, 0x92, 0xFA, 0x8E, 0x5B, 0x0A, 0x60, 0x0D, 0x0E, 0x3E,
      0xBD, 0xD1, 0x4F, 0xA2, 0x29, 0x81, 0x9C, 0xE9, 0xC1, 0x0B, 0x8D, 0x5F,
      0x39, 0x3D, 0xE0, 0x11, 0x9A, 0x5B, 0x50, 0x9B, 0x80, 0xD5, 0x6B, 0x06,
      0x78, 0x34, 0x47, 0xF9, 0x31, 0x17, 0x71, 0x23, 0x82, 0x49, 0x10, 0xC9,
      0xBF, 0xDE, 0x9A, 0x29, 0xFB, 0xA0, 0x25, 0x2E, 0x69, 0xA9, 0x0B, 0x3E,
      0x71, 0x78, 0x32, 0x86, 0x61, 0x15, 0xC0, 0x6E, 0xA7, 0x3B, 0x03, 0x3E,
      0xC3, 0xB0, 0xD4, 0x5D, 0xFD, 0xB6, 0x9A, 0x76, 0xB4, 0x84, 0xDB, 0x0B,
      0xE7, 0xA8, 0x12, 0x15, 0xB3, 0x81, 0x7E, 0x1C, 0x02, 0xF9, 0xA5, 0xDE,
      0xE8, 0x96, 0x7B, 0x14, 0x7D, 0xF9, 0xF6, 0x3C, 0x93, 0xA6, 0xE3, 0x96,
      0xDE, 0x42, 0x51, 0xA5, 0xA7, 0x06, 0xDF, 0xDE, 0x96, 0x70, 0xB8, 0xB2,
      0xF6, 0xC4, 0xC3, 0xE2, 0x50, 0x91, 0x42, 0x25, 0x6F, 0xDD, 0xA9, 0x05,
      0xC1, 0x25, 0xFB, 0xBB, 0x29, 0x4E, 0xB2, 0x9A, 0x3B, 0x4D, 0x9B, 0xE3,
      0xB6, 0x77, 0x62, 0xAF, 0xC0, 0x49, 0xB9, 0x6B, 0x3F, 0x41, 0xB8, 0xC3,
      0x1B, 0xC5, 0xD7, 0xB5, 0x22, 0xDC, 0xD1, 0xAD, 0x12, 0xB2, 0x52, 0x37,
      0x0A, 0x8A, 0x57, 0xE4, 0x2F, 0x6A, 0x9A, 0xC2, 0x6F, 0xF7, 0x84, 0xB3,
      0x74, 0xDA, 0x4B, 0x86, 0xFF, 0xDB, 0x65, 0xCC, 0x75, 0x3C, 0xD0, 0x49,
      0xF1, 0xA2, 0x1C, 0xF8, 0x32, 0x44, 0x7E, 0x1D, 0xF7, 0xBA, 0x7D, 0x0D,
      0x11, 0xE4, 0x03, 0xFC, 0x18, 0xBC, 0x54, 0x55, 0x01, 0xE1, 0x65, 0x68,
      0x59, 0x5A, 0xEB, 0x6B, 0xD7, 0x81, 0x1C, 0x21, 0x4C, 0xF2, 0xFB, 0x1C,
      0xDF, 0xB0, 0x7B, 0xB3, 0x23, 0x21, 0xF5, 0x36, 0xE3, 0x89, 0x6B, 0x6E,
      0xF4, 0xD1, 0x61, 0x30, 0xAD, 0xD7, 0x1B, 0x27, 0x1C, 0xD1, 0x02, 0x7E,
      0x35, 0x53, 0x8D, 0x9E, 0x47, 0x5A, 0x3A, 0x53, 0xDF, 0xEA, 0x43, 0x0C,
      0x15, 0x1D, 0xF7, 0xD5, 0x16, 0xCD, 0x0D, 0x9B},
     true,
     {0x0E, 0xA9, 0x83, 0xFF, 0x9D, 0x76, 0xF0, 0x56, 0xAA, 0x42, 0xBB,
      0x77, 0x2A, 0xA2, 0x7C, 0x8A, 0x16, 0x31, 0x72, 0xF4, 0x3E, 0x6B,
      0xC9, 0xBC, 0x55, 0xB8, 0x30, 0x38, 0xE0, 0x95, 0x79, 0x2B}},

    {params_ml_kem768_test_mode,
     {0x8F, 0xFB, 0xC8, 0x0E, 0x46, 0x62, 0x86, 0x4D, 0x6F, 0x37, 0x3D, 0xC8,
      0x83, 0x7A, 0xA9, 0x1B, 0x3C, 0xC2, 0x6B, 0x68, 0x12, 0x4A, 0xBD, 0x73,
      0xDA, 0xD0, 0x25, 0xA1, 0xD1, 0xC1, 0x88, 0x29, 0xDC, 0xF0, 0x77, 0xD3,
      0x03, 0x57, 0x9E, 0x5F, 0x39, 0xF4, 0xBE, 0x10, 0x1B, 0xB9, 0xE3, 0x55,
      0xDF, 0xB5, 0x32, 0x38, 0x82, 0xEA, 0xCB, 0x3D, 0x18, 0x4E, 0x68, 0x12,
      0xC0, 0x3A, 0x7B, 0xEB, 0xE2, 0x51, 0x66, 0xD5, 0x5F, 0x82, 0x1A, 0x00,
      0xF8, 0x0B, 0x8D, 0x2B, 0xAB, 0x1A, 0x7E, 0xEC, 0x83, 0xD3, 0x84, 0xAF,
      0xDF, 0x30, 0xF6, 0xBB, 0xC9, 0x96, 0x0C, 0x46, 0x62, 0x06, 0x7E, 0xF7,
      0xE2, 0x00, 0xE3, 0x72, 0x68, 0xB9, 0xF5, 0x34, 0x8F, 0xF4, 0x84, 0x64,
      0x27, 0x99, 0x25, 0x8B, 0x45, 0xE5, 0x41, 0x10, 0x1A, 0x21, 0xFD, 0xD6,
      0xFB, 0xFA, 0xA2, 0x37, 0x4A, 0x28, 0xFA, 0xA9, 0x72, 0x04, 0x95, 0x3B,
      0x95, 0xBB, 0xD1, 0xBB, 0x51, 0x97, 0x85, 0x21, 0x0D, 0xA7, 0xC8, 0xA0,
      0x9D, 0x07, 0x1D, 0x8A, 0xFC, 0x9B, 0x29, 0xF2, 0xC3, 0xC2, 0x90, 0x9A,
      0x4C, 0x53, 0x67, 0x14, 0x08, 0xB8, 0x08, 0x3B, 0xCF, 0x5A, 0xE0, 0x3D,
      0x45, 0xC0, 0xCF, 0xBA, 0x39, 0x9F, 0x44, 0xD2, 0x4A, 0x06, 0x32, 0x1B,
      0xB7, 0x4F, 0x68, 0x63, 0xB7, 0xD4, 0xBF, 0x0B, 0xFE, 0x73, 0xC8, 0xAF,
      0x8E, 0xE1, 0xDD, 0xA4, 0x52, 0x12, 0xE3, 0xF9, 0xC8, 0x53, 0xD4, 0xD0,
      0xE1, 0x6F, 0x8E, 0xBD, 0xB8, 0x58, 0x1C, 0x4A, 0xDE, 0xEE, 0x83, 0x3D,
      0x81, 0xA9, 0xE0, 0xA9, 0xE8, 0x58, 0x7E, 0x9C, 0x19, 0xE6, 0x89, 0xE6,
      0xDF, 0x71, 0x55, 0x64, 0xBC, 0xE2, 0x7C, 0xFA, 0x73, 0xBA, 0x16, 0x22,
      0x6A, 0x77, 0xCE, 0x44, 0xDC, 0x49, 0x69, 0x92, 0xF4, 0x1A, 0xB9, 0x18,
      0x64, 0x3C, 0x6D, 0x86, 0xA8, 0xB2, 0x6A, 0xBA, 0x6F, 0x94, 0xF3, 0x50,
      0x2D, 0x22, 0xDD, 0x94, 0xFE, 0x55, 0x48, 0x3F, 0x67, 0xC6, 0x35, 0xB3,
      0x07, 0x74, 0x5D, 0x33, 0xF1, 0x71, 0x33, 0x29, 0x36, 0x39, 0x11, 0x8E,
      0x70, 0xCE, 0x42, 0xC6, 0xDB, 0x73, 0x32, 0xD4, 0x86, 0x2C, 0x73, 0xD5,
      0xB8, 0x44, 0x15, 0x45, 0x4A, 0xD5, 0x1F, 0x89, 0xB5, 0x55, 0x9B, 0x5C,
      0x85, 0xD6, 0xB6, 0xED, 0x47, 0xB6, 0x95, 0x8F, 0x21, 0xFB, 0xC2, 0xAD,
      0xF8, 0xC8, 0xA9, 0xD4, 0x3F, 0xD2, 0xE1, 0xB0, 0xC0, 0x24, 0x18, 0xD2,
      0x27, 0xB8, 0x3F, 0x85, 0xCB, 0xC3, 0xA8, 0x1C, 0x71, 0x9E, 0x86, 0x02,
      0x78, 0x1A, 0xE7, 0x1E, 0x15, 0xE6, 0xD7, 0x14, 0x91, 0x9E, 0x52, 0xFC,
      0xCC, 0xFD, 0x9A, 0x68, 0xB4, 0x75, 0x18, 0x25, 0xBF, 0xBB, 0x53, 0xB7,
      0x94, 0x0B, 0x15, 0xB5, 0x46, 0x15, 0x8D, 0xBB, 0xC6, 0x12, 0xE6, 0x02,
      0xF6, 0x60, 0xB9, 0xE0, 0xFF, 0x43, 0x9E, 0x01, 0x56, 0xC4, 0xC8, 0x79,
      0x23, 0x46, 0x01, 0x4B, 0xA1, 0xB4, 0x83, 0x8C, 0x74, 0x25, 0xAB, 0x34,
      0x74, 0x4D, 0xE5, 0x1D, 0x85, 0x4C, 0xBB, 0xA5, 0x8B, 0x7E, 0x67, 0xE0,
      0x14, 0x12, 0x25, 0x18, 0x03, 0x6C, 0xE1, 0x54, 0x1A, 0x16, 0x75, 0xAF,
      0xEA, 0xE4, 0xF2, 0x9A, 0x53, 0x18, 0x60, 0x2A, 0xBB, 0xD0, 0xA1, 0x54,
      0x0F, 0x33, 0x17, 0x6C, 0x98, 0x4E, 0x30, 0x60, 0x98, 0xDB, 0xD0, 0x8E,
      0x82, 0x2A, 0xBB, 0x55, 0xF9, 0xFF, 0x38, 0xD9, 0xE3, 0x1E, 0xA4, 0x69,
      0x51, 0x50, 0xF2, 0xCB, 0x60, 0xBC, 0x2E, 0xB5, 0xF4, 0x78, 0x0C, 0xBE,
      0xBB, 0x21, 0x0C, 0xF4, 0x86, 0x62, 0xC4, 0x54, 0xC7, 0xA4, 0x23, 0x60,
      0xF3, 0x06, 0xFB, 0x03, 0x61, 0x7C, 0x99, 0x8A, 0xD8, 0xA9, 0x29, 0x7D,
      0x6B, 0x71, 0xA7, 0x12, 0x85, 0xF7, 0xAE, 0x8D, 0xFB, 0x33, 0x6F, 0xA9,
      0x22, 0x54, 0x0C, 0x92, 0xDC, 0x71, 0xF7, 0x77, 0xD3, 0xB4, 0xD1, 0x1D,
      0x87, 0xB8, 0xD0, 0x82, 0xFA, 0x8A, 0x00, 0xDF, 0x64, 0x7C, 0xF7, 0xFE,
      0xB2, 0x74, 0x03, 0xD3, 0xCF, 0x50, 0xD8, 0x29, 0xEE, 0xE3, 0x57, 0x5A,
      0x01, 0xE2, 0xCC, 0xA5, 0x78, 0x49, 0xB1, 0x1B, 0x14, 0xF0, 0x01, 0xBE,
      0x18, 0x0D, 0xD5, 0xFA, 0x13, 0xC0, 0x3B, 0x98, 0xED, 0xEA, 0x63, 0x58,
      0xC5, 0xAB, 0x30, 0xA5, 0x26, 0x02, 0x7C, 0xB4, 0x5E, 0x33, 0xE6, 0x46,
      0xB3, 0x79, 0x88, 0xCC, 0x84, 0xB9, 0x79, 0xCC, 0x5C, 0xFC, 0x3B, 0xFD,
      0xA0, 0x5B, 0xD2, 0xC7, 0xB8, 0xCB, 0x1B, 0x11, 0xAF, 0xEE, 0x00, 0x7E,
      0x20, 0xFC, 0xCF, 0x8D, 0x0F, 0x76, 0x4F, 0x4A, 0x6D, 0x2F, 0x6A, 0x8B,
      0x74, 0x28, 0x18, 0x00, 0xCB, 0xDC, 0xBB, 0xCF, 0x0D, 0xF1, 0xEC, 0x9D,
      0x27, 0xE6, 0xA9, 0x49, 0x68, 0x60, 0x4D, 0x9E, 0xFD, 0x37, 0x92, 0x8B,
      0x68, 0x56, 0xC4, 0x8F, 0x01, 0x08, 0x15, 0x55, 0x95, 0xD0, 0x32, 0x31,
      0xDF, 0xC2, 0x2D, 0xC0, 0xC8, 0xEE, 0x61, 0x40, 0x90, 0xF3, 0x7E, 0x08,
      0x28, 0xB4, 0x8A, 0x4D, 0xD3, 0x71, 0xC6, 0x77, 0xB5, 0xDB, 0xA9, 0x5E,
      0x41, 0x7F, 0x12, 0xC9, 0xA3, 0x96, 0x87, 0x5F, 0xB0, 0x56, 0x23, 0xF7,
      0xA5, 0x44, 0xAE, 0xAE, 0x41, 0xA0, 0xAA, 0x53, 0x6F, 0xB8, 0xD7, 0x67,
      0xBA, 0x2E, 0x14, 0x75, 0x2C, 0x84, 0xE1, 0x47, 0x14, 0x9F, 0x65, 0x5A,
      0xE7, 0xB9, 0x03, 0xCA, 0xA5, 0x91, 0xAE, 0x00, 0x26, 0x7A, 0xDD, 0x3E,
      0xA8, 0x16, 0x61, 0x2A, 0xB0, 0xB9, 0xA5, 0xFB, 0x26, 0x3C, 0x70, 0xC4,
      0x36, 0x70, 0x62, 0xF7, 0x79, 0x42, 0x74, 0xC7, 0x5A, 0xC6, 0x6F, 0x70,
      0x6A, 0xE9, 0x36, 0x99, 0x85, 0x9D, 0x55, 0xB2, 0xE4, 0x96, 0x0E, 0x9D,
      0x53, 0x8F, 0x38, 0xA2, 0xFA, 0xEE, 0x36, 0x6B, 0x80, 0xDC, 0x78, 0xBB,
      0x67, 0x3A, 0x9E, 0x1B, 0x05, 0x7D, 0x71, 0x1F, 0x9D, 0xDB, 0x37, 0x70,
      0x94, 0x7E, 0x6D, 0xD7, 0xBC, 0xFB, 0x42, 0x5B, 0x96, 0x67, 0x05, 0x06,
      0x75, 0x8A, 0xEA, 0x39, 0xA5, 0xEC, 0xB3, 0x3A, 0x1B, 0x76, 0xB8, 0x22,
      0xAF, 0x90, 0x37, 0x87, 0xDA, 0x3B, 0x61, 0xA7, 0xB9, 0x26, 0x3C, 0x0F,
      0xAE, 0x1B, 0x72, 0x9B, 0x1A, 0x2E, 0x16, 0xFE, 0xB5, 0x0C, 0x32, 0xA8,
      0x72, 0x81, 0x81, 0xD4, 0xE8, 0xA9, 0xF8, 0x37, 0x6C, 0x39, 0xF6, 0xAA,
      0xBC, 0x2C, 0x02, 0x23, 0x06, 0xB0, 0x5E, 0x49, 0x4C, 0xF9, 0xB6, 0xAD,
      0xEE, 0xEC, 0x95, 0x88, 0x74, 0x40, 0x50, 0x89, 0x81, 0xD6, 0xA7, 0x47,
      0x07, 0xFC, 0xEF, 0xA2, 0x4B, 0x9F, 0x0D, 0xC3, 0xAA, 0xBC, 0x98, 0x4E,
      0x9C, 0x44, 0x17, 0x4E, 0x6D, 0xFB, 0x51, 0xFC, 0xF4, 0x58, 0x8C, 0x57,
      0xF9, 0x65, 0x9A, 0x8E, 0x7A, 0x6F, 0xAE, 0xAF, 0xBA, 0xE7, 0xAB, 0xE4,
      0x60, 0x04, 0x44, 0x93, 0x6B, 0x37, 0x63, 0x46, 0x3D, 0x4A, 0xE4, 0x11,
      0xDD, 0xC1, 0xC9, 0x85, 0x85, 0xE0, 0xDE, 0x58, 0x86, 0x72, 0x51, 0x07,
      0x9B, 0xE7, 0x20, 0x75, 0x97, 0x32, 0x75, 0x14, 0x18, 0x01, 0xB9, 0x8F,
      0x7B, 0x93, 0x97, 0xC0, 0x96, 0xA5, 0x6B, 0x8C, 0xD8, 0x3C, 0xFB, 0xD3,
      0x74, 0xE1, 0x82, 0xF7, 0xDC, 0xC9, 0xA7, 0xC7, 0x64, 0xDB, 0xBF, 0x4D,
      0x75, 0x76, 0xA1, 0xCC, 0x92, 0x39, 0x84, 0x8E, 0x72, 0x95, 0xD2, 0x9C,
      0xF0, 0x34, 0xA1, 0xA7, 0xAE, 0x33, 0xA3, 0x86, 0xC3, 0xDD, 0xC2, 0x4A,
      0x53, 0x51, 0x68, 0xED, 0x23, 0xD7, 0xAD, 0xE9, 0x43, 0x3B, 0x50, 0xDC,
      0x56, 0x94, 0xC9, 0x69, 0xF4, 0xC5, 0x46, 0xEF, 0x22, 0x93, 0xCD, 0x84,
      0x2F, 0x4B, 0x62, 0xB6, 0xB7, 0x43, 0x5F, 0x59, 0x7C, 0xF5, 0xC1, 0x73,
      0x38, 0x84, 0xE0, 0xA6, 0xAA, 0x47, 0xFA, 0x31, 0x88, 0x7D, 0xED, 0xC6,
      0xC4, 0x02, 0xD8, 0xED, 0x01, 0x3E, 0x49, 0xE5, 0xCA, 0xD7, 0x71, 0x8C,
      0xCE, 0xFE, 0xE0, 0xE6, 0xA0, 0x41, 0x71, 0x5C, 0xC9, 0xAD, 0xD7, 0x99,
      0x65, 0x41, 0x30, 0x49, 0xAB, 0xCE, 0x88, 0x63, 0x6A, 0xA7, 0x54, 0x3E,
      0xE2, 0x60, 0x1F, 0x16, 0x28, 0x38, 0xEF, 0x6B},
     true,
     {0x34, 0x27, 0x65, 0xB7, 0x7A, 0x09, 0xBA, 0x68, 0x63, 0xF2, 0xAD,
      0xA7, 0x82, 0xE3, 0x71, 0x98, 0x03, 0xF7, 0xAB, 0x71, 0x4E, 0xE8,
      0x07, 0xDE, 0x89, 0xA1, 0x61, 0x7B, 0x5C, 0x74, 0xF6, 0x0F}},

    {params_ml_kem768_test_mode,
     {0x17, 0x97, 0x6B, 0xAC, 0x62, 0xF6, 0x6C, 0xEF, 0x2B, 0x6F, 0x94, 0x7C,
      0x12, 0x10, 0x79, 0xB6, 0xF2, 0xE9, 0x35, 0x0C, 0x13, 0x7E, 0x73, 0x8B,
      0xFD, 0x88, 0x4F, 0xF2, 0xBA, 0x6E, 0x21, 0x16, 0x40, 0xA3, 0x0F, 0xBF,
      0x26, 0x95, 0xED, 0xF7, 0x04, 0x6E, 0x1F, 0x52, 0x34, 0xAB, 0x1C, 0x8A,
      0x9B, 0x0E, 0x8A, 0x3F, 0xF8, 0x8E, 0xF1, 0x8C, 0x1E, 0x55, 0x12, 0xD5,
      0xF6, 0x9E, 0x4A, 0x36, 0xCC, 0x93, 0x62, 0xF0, 0x09, 0x20, 0x48, 0x1E,
      0x54, 0x60, 0xB1, 0xFB, 0x0C, 0x2B, 0x9F, 0xF0, 0xCD, 0x0D, 0x95, 0x71,
      0x89, 0x66, 0xAF, 0x7E, 0xC1, 0xF7, 0x6B, 0x8D, 0xA9, 0x3F, 0x6A, 0xB1,
      0x79, 0xA5, 0xDE, 0x70, 0xDE, 0xE3, 0x4C, 0x57, 0x9E, 0x28, 0x4A, 0xE8,
      0x50, 0x4E, 0xD9, 0x6E, 0x1A, 0x85, 0x89, 0x80, 0x76, 0xF6, 0x9A, 0xAF,
      0xEC, 0x13, 0x57, 0x53, 0x3E, 0xBB, 0x63, 0x6F, 0xBA, 0x23, 0x72, 0x20,
      0x4D, 0xAB, 0x87, 0xC4, 0x7A, 0xF2, 0x7D, 0x4D, 0x9E, 0xB1, 0xB4, 0xFF,
      0x42, 0x86, 0xD6, 0xA9, 0xFA, 0x7F, 0xD5, 0x06, 0xC9, 0xFE, 0xBA, 0x59,
      0x6D, 0x20, 0x47, 0xDB, 0x76, 0x5C, 0x1E, 0xBF, 0x1F, 0x79, 0x21, 0x86,
      0x7D, 0x39, 0x44, 0x87, 0xF6, 0xBE, 0x92, 0x6E, 0x6B, 0x03, 0x23, 0x05,
      0x8C, 0xB5, 0x91, 0x19, 0x54, 0x36, 0xEC, 0xC8, 0x05, 0xC8, 0xB8, 0x86,
      0x15, 0xC7, 0xA0, 0x38, 0x33, 0xAA, 0xBF, 0x49, 0x03, 0x37, 0x06, 0x3D,
      0xFE, 0xED, 0x69, 0x8F, 0x7D, 0xA8, 0xDD, 0x58, 0x9A, 0x79, 0x4C, 0x95,
      0x6C, 0x2B, 0xF8, 0xD8, 0xCA, 0x4A, 0xE1, 0x8B, 0x0A, 0x77, 0x67, 0x69,
      0x38, 0x02, 0xBD, 0x6D, 0xD5, 0x3F, 0x54, 0x3E, 0x10, 0x5E, 0xC5, 0x26,
      0xC1, 0xD1, 0xD0, 0x0A, 0xC9, 0xC0, 0xB6, 0x06, 0xBD, 0x9B, 0x3A, 0x1D,
      0x52, 0xCB, 0x8C, 0x56, 0xF8, 0x53, 0x5E, 0xCA, 0xDD, 0x82, 0x39, 0x30,
      0x8F, 0x2F, 0xE7, 0xE1, 0xD7, 0xBF, 0xAC, 0x58, 0x48, 0xB5, 0x47, 0xB4,
      0x57, 0x9A, 0xFC, 0x13, 0xA0, 0xB2, 0xBE, 0xDE, 0xFA, 0x46, 0x32, 0x2F,
      0x92, 0xE2, 0xB7, 0x39, 0x80, 0x69, 0x53, 0x69, 0xC5, 0xF4, 0x8D, 0x37,
      0xF9, 0x34, 0x5F, 0x20, 0xC7, 0x82, 0x0D, 0xB6, 0xDE, 0x09, 0xD5, 0xE8,
      0x31, 0x3B, 0x73, 0xED, 0x70, 0x5B, 0x33, 0x64, 0x6F, 0xB1, 0x4C, 0xC4,
      0xD4, 0x0D, 0x65, 0x29, 0x0A, 0x4C, 0x27, 0x36, 0x0F, 0xBB, 0xD0, 0x80,
      0xE6, 0x1A, 0x16, 0xBB, 0x15, 0xE9, 0x56, 0x0A, 0x09, 0x7E, 0x4A, 0xEC,
      0x16, 0xF8, 0xB8, 0x03, 0x0F, 0xAE, 0x1D, 0x47, 0xE0, 0x24, 0xF1, 0x0C,
      0x33, 0xE6, 0xA1, 0xC5, 0x6A, 0xEB, 0x8E, 0xC2, 0xF6, 0xAD, 0x6E, 0xF4,
      0xB8, 0xFF, 0x04, 0xC6, 0x73, 0x07, 0xB2, 0x3E, 0x47, 0x0F, 0xB3, 0xE5,
      0xBC, 0xB6, 0xF5, 0x33, 0xF9, 0x55, 0xC3, 0x6F, 0xDB, 0x46, 0x51, 0x6A,
      0x07, 0xDF, 0xF2, 0x95, 0x61, 0x30, 0xAD, 0x09, 0x24, 0x15, 0x8C, 0xC2,
      0xA0, 0x83, 0x37, 0x8F, 0xB9, 0xAE, 0x32, 0xDE, 0x89, 0xCF, 0x77, 0x4D,
      0x82, 0xC2, 0xFC, 0x70, 0xDA, 0x48, 0x53, 0x63, 0x72, 0x29, 0x9C, 0x61,
      0x92, 0x7A, 0x5A, 0xE6, 0x7E, 0x55, 0xE7, 0x92, 0xB6, 0x4F, 0xE6, 0x1F,
      0x06, 0xEF, 0xFC, 0x1F, 0x21, 0x6C, 0xC9, 0xD7, 0x39, 0xAD, 0xBF, 0x3B,
      0x21, 0x90, 0xE1, 0xD0, 0x80, 0xE0, 0x0F, 0x16, 0x9F, 0x14, 0x5F, 0xE3,
      0x2A, 0xF7, 0xEC, 0x7C, 0xBA, 0x1D, 0x76, 0xFA, 0x68, 0x39, 0xD5, 0xFD,
      0x20, 0x68, 0xE1, 0xDF, 0xFF, 0x55, 0x77, 0x55, 0xFF, 0x2F, 0x42, 0x71,
      0x20, 0x4A, 0x54, 0x68, 0xC7, 0x9C, 0x7B, 0xB8, 0xD0, 0x0F, 0xAD, 0x63,
      0x93, 0x8F, 0x12, 0xD5, 0x3B, 0x24, 0x3B, 0x3F, 0xF8, 0x66, 0x55, 0x69,
      0x13, 0xEB, 0x57, 0xAD, 0x2A, 0xE0, 0x34, 0xF8, 0xB6, 0x2B, 0x1A, 0x1B,
      0x9D, 0xA2, 0xB1, 0xD4, 0x58, 0x00, 0xB4, 0xCE, 0xF1, 0xE1, 0x94, 0x3A,
      0x0C, 0x92, 0xF0, 0xEF, 0x2E, 0xE9, 0x24, 0xF8, 0x0C, 0xF6, 0x7E, 0xBD,
      0x3D, 0x01, 0x99, 0xD4, 0x5E, 0xD4, 0xDC, 0xC0, 0x01, 0x40, 0x82, 0x9A,
      0x09, 0x92, 0xDB, 0x43, 0x61, 0x6C, 0xC4, 0x68, 0x50, 0x8B, 0x85, 0x2E,
      0xB8, 0x22, 0x06, 0x6A, 0x05, 0xCC, 0x91, 0xD6, 0xBC, 0x2B, 0x47, 0xE5,
      0x62, 0x2B, 0x77, 0x4F, 0x81, 0x28, 0xEC, 0xBB, 0xB9, 0x4C, 0xAD, 0xD1,
      0x55, 0x88, 0xB3, 0x6A, 0x71, 0xE9, 0xFD, 0x97, 0xB0, 0x5D, 0x69, 0xE8,
      0xBA, 0xF0, 0x0D, 0x30, 0xA3, 0xD3, 0xC0, 0x0E, 0x66, 0x3E, 0x00, 0xAF,
      0xC9, 0xF5, 0xE1, 0xBA, 0xC8, 0x53, 0x4E, 0xD5, 0xF6, 0xE5, 0xAB, 0x47,
      0xD7, 0xEF, 0xDF, 0x65, 0x37, 0x75, 0x34, 0x08, 0x29, 0x9A, 0x9E, 0x8D,
      0x5F, 0x5A, 0xE0, 0xFE, 0x36, 0xA9, 0xEC, 0x41, 0xC6, 0xDC, 0x9F, 0x78,
      0xA8, 0x91, 0xBF, 0xA9, 0xC8, 0xE9, 0x0A, 0xA1, 0xA4, 0x57, 0xA0, 0xC0,
      0x1A, 0xF7, 0x0C, 0xBC, 0x9E, 0x55, 0xB6, 0x8A, 0x5D, 0x8C, 0xC5, 0xCD,
      0x3B, 0xD6, 0x88, 0x6A, 0xE1, 0x1F, 0xF5, 0x10, 0xC6, 0xED, 0x0E, 0xB2,
      0xF5, 0xC0, 0x81, 0xB2, 0x59, 0x89, 0x51, 0x8B, 0xA2, 0x17, 0xBC, 0x1C,
      0x15, 0x38, 0x64, 0xE5, 0xBB, 0x31, 0x2E, 0xF0, 0xD4, 0x3D, 0x6D, 0xA4,
      0xA0, 0xFD, 0xE4, 0x4F, 0x11, 0x57, 0xCD, 0x23, 0x8E, 0x8D, 0x70, 0xBE,
      0xB4, 0x20, 0xBD, 0x31, 0x0F, 0x8E, 0x5D, 0xB9, 0xD7, 0x4E, 0xF4, 0xEC,
      0x99, 0x80, 0xCB, 0xA7, 0x43, 0x58, 0xFC, 0x77, 0xC5, 0xD4, 0xFA, 0xE3,
      0x03, 0x6E, 0x17, 0x66, 0x47, 0xD7, 0x8C, 0x73, 0x90, 0x0C, 0x79, 0xBF,
      0xBF, 0x0B, 0xC5, 0x45, 0xAB, 0xF7, 0xCB, 0xB4, 0xDC, 0x7F, 0x60, 0x41,
      0xD4, 0xFA, 0x3B, 0x66, 0xE4, 0xD4, 0x65, 0x5E, 0x24, 0xB1, 0x1D, 0xC3,
      0x0B, 0x00, 0x61, 0xC4, 0x52, 0xA6, 0x05, 0xCE, 0x73, 0x36, 0x2F, 0x2A,
      0x3F, 0x05, 0x23, 0x70, 0xD8, 0x73, 0xFC, 0x68, 0xDF, 0xFC, 0xD3, 0x99,
      0x9F, 0xDE, 0xDB, 0x45, 0xDD, 0x9F, 0x2A, 0x02, 0xB4, 0x69, 0x9B, 0xCF,
      0x1F, 0xC5, 0xF8, 0x88, 0xB0, 0x19, 0xB5, 0x02, 0x84, 0x65, 0xF3, 0x0A,
      0xEF, 0xAD, 0x94, 0x6D, 0x48, 0x12, 0x85, 0xD1, 0x12, 0x2E, 0xA7, 0x8F,
      0x3B, 0xD8, 0xB1, 0x98, 0x25, 0x58, 0xC3, 0x8F, 0xA3, 0xDF, 0x0F, 0x05,
      0x8B, 0x12, 0xEE, 0xBB, 0x11, 0xF4, 0xC7, 0x80, 0x9F, 0x63, 0x34, 0xEA,
      0x1D, 0x7F, 0xE0, 0xB5, 0x29, 0xC0, 0xBC, 0x9C, 0x67, 0x04, 0x46, 0x48,
      0x17, 0x8D, 0x2A, 0xE9, 0x23, 0x2E, 0x4E, 0x88, 0xDD, 0x6D, 0x00, 0x16,
      0xD8, 0xA5, 0x90, 0xB7, 0x70, 0x3F, 0x1A, 0x01, 0x7A, 0x4A, 0x26, 0x71,
      0xBB, 0xB2, 0x4F, 0xA9, 0x7A, 0xDE, 0x1B, 0x61, 0xC4, 0x89, 0xAF, 0xE9,
      0xB3, 0xE6, 0x3C, 0xF4, 0xCC, 0xC4, 0x21, 0x68, 0xC9, 0x88, 0x80, 0x92,
      0x1C, 0x2C, 0x0E, 0xA7, 0xD2, 0x4D, 0xB6, 0xDD, 0x67, 0x6B, 0x77, 0xF7,
      0xB6, 0xC0, 0x52, 0x5C, 0x8D, 0x05, 0x78, 0xC7, 0xF5, 0xA2, 0x0D, 0xBF,
      0x2F, 0x82, 0x87, 0x39, 0x04, 0xD7, 0xCF, 0x25, 0x22, 0xCE, 0x63, 0x60,
      0x39, 0x7B, 0x25, 0x4B, 0x18, 0xC3, 0x05, 0x9A, 0x4B, 0xEA, 0x16, 0x9A,
      0x44, 0xD9, 0xBA, 0x17, 0xCF, 0xDA, 0x18, 0x27, 0xEA, 0xBE, 0xCD, 0x26,
      0x9F, 0xD3, 0x91, 0xCB, 0xC0, 0xD4, 0x9D, 0x71, 0xFA, 0x81, 0xAC, 0x16,
      0xF9, 0xA0, 0xDE, 0xD9, 0xE7, 0x2A, 0x58, 0xD1, 0xBC, 0x22, 0x62, 0x97,
      0x9D, 0x8D, 0x7E, 0x53, 0x1D, 0x1C, 0x46, 0xA8, 0xF1, 0x07, 0xBD, 0xA1,
      0x8A, 0x1D, 0x2C, 0xCD, 0x17, 0x33, 0x41, 0x83, 0xDD, 0x3E, 0x79, 0xD9,
      0x05, 0xAC, 0xA7, 0xDA, 0xD3, 0x48, 0xBC, 0x6D, 0x5C, 0xE1, 0x24, 0xA1,
      0x39, 0x7E, 0xB3, 0xB8, 0x9B, 0xE7, 0x58, 0x07, 0x20, 0xB5, 0xDD, 0x00,
      0xBD, 0x3A, 0x63, 0xDA, 0xD8, 0x13, 0xE0, 0xE9, 0x67, 0xEF, 0xED, 0xF1,
      0x7F, 0x3D, 0x96, 0x0E, 0x70, 0xA4, 0xF8, 0x3F},
     true,
     {0xF1, 0x75, 0xCA, 0x29, 0xD3, 0x67, 0x84, 0xE3, 0xB7, 0xA6, 0xF6,
      0xD8, 0x68, 0x2D, 0xE3, 0x54, 0x81, 0x15, 0xC2, 0x5E, 0xC1, 0x75,
      0x1D, 0xAF, 0x6B, 0x5F, 0xC3, 0x31, 0x8F, 0x69, 0x08, 0x02}},
};
