// Copyright (C) 2017 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#include "cppparsecontext.h"

#include "cppeditortr.h"
#include "cppeditorwidget.h"

#include <QAction>
#include <QDir>
#include <QDebug>

#include <utils/algorithm.h>
#include <utils/qtcassert.h>

namespace CppEditor {
namespace Internal {

void ParseContextModel::update(const ProjectPartInfo &projectPartInfo)
{
    beginResetModel();
    reset(projectPartInfo);
    endResetModel();

    emit updated(areMultipleAvailable());
}

QString ParseContextModel::currentToolTip() const
{
    const QModelIndex index = createIndex(m_currentIndex, 0);
    if (!index.isValid())
        return QString();

    return Tr::tr("<p><b>Active Parse Context</b>:<br/>%1</p>"
                  "<p>Multiple parse contexts (set of defines, include paths, and so on) "
                  "are available for this file.</p>"
                  "<p>Choose a parse context to set it as the preferred one. "
                  "Clear the preference from the context menu.</p>")
                    .arg(data(index, Qt::ToolTipRole).toString());
}

void ParseContextModel::setPreferred(int index)
{
    if (index < 0)
        return;

    emit preferredParseContextChanged(m_projectParts[index]->id());
}

void ParseContextModel::clearPreferred()
{
    emit preferredParseContextChanged(QString());
}

bool ParseContextModel::areMultipleAvailable() const
{
    return m_projectParts.size() >= 2;
}

void ParseContextModel::reset(const ProjectPartInfo &projectPartInfo)
{
    // Sort
    m_hints = projectPartInfo.hints;
    m_projectParts = Utils::sorted(projectPartInfo.projectParts, &ProjectPart::displayName);

    // Determine index for current
    const QString id = projectPartInfo.projectPart->id();
    m_currentIndex = Utils::indexOf(m_projectParts, [id](const ProjectPart::ConstPtr &pp) {
        return pp->id() == id;
    });
    QTC_CHECK(m_currentIndex >= 0);
}

int ParseContextModel::currentIndex() const
{
    return m_currentIndex;
}

bool ParseContextModel::isCurrentPreferred() const
{
    return m_hints & ProjectPartInfo::IsPreferredMatch;
}

QString ParseContextModel::currentId() const
{
    if (m_currentIndex < 0)
        return QString();

    return m_projectParts[m_currentIndex]->id();
}

int ParseContextModel::rowCount(const QModelIndex &parent) const
{
    if (parent.isValid())
        return 0;
    return m_projectParts.size();
}

QVariant ParseContextModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid() || index.row() < 0 || index.row() >= m_projectParts.size())
        return QVariant();

    const int row = index.row();
    if (role == Qt::DisplayRole)
        return m_projectParts[row]->displayName;
    else if (role == Qt::ToolTipRole)
        return m_projectParts[row]->projectFile.nativePath();

    return QVariant();
}

ParseContextWidget::ParseContextWidget(ParseContextModel &parseContextModel, QWidget *parent)
    : QComboBox(parent)
    , m_parseContextModel(parseContextModel)
{
    setSizeAdjustPolicy(QComboBox::AdjustToContents);
    QSizePolicy policy = sizePolicy();
    policy.setHorizontalStretch(1);
    policy.setHorizontalPolicy(QSizePolicy::Maximum);
    setSizePolicy(policy);
    // Set up context menu with a clear action
    setContextMenuPolicy(Qt::ActionsContextMenu);
    m_clearPreferredAction = new QAction(Tr::tr("Clear Preferred Parse Context"), this);
    connect(m_clearPreferredAction, &QAction::triggered, this, [this] {
        m_parseContextModel.clearPreferred();
    });
    addAction(m_clearPreferredAction);

    // Set up sync of this widget and model in both directions
    connect(this, &QComboBox::activated, &m_parseContextModel, &ParseContextModel::setPreferred);
    connect(&m_parseContextModel, &ParseContextModel::updated,
            this, &ParseContextWidget::syncToModel);

    // Set up model
    setModel(&m_parseContextModel);
}

void ParseContextWidget::syncToModel()
{
    const int index = m_parseContextModel.currentIndex();
    if (index < 0)
        return; // E.g. editor was duplicated but no project context was determined yet.

    if (currentIndex() != index)
        setCurrentIndex(index);

    setToolTip(m_parseContextModel.currentToolTip());

    const bool isPreferred = m_parseContextModel.isCurrentPreferred();
    m_clearPreferredAction->setEnabled(isPreferred);
    CppEditorWidget::updateWidgetHighlighting(this, isPreferred);
}

QSize ParseContextWidget::minimumSizeHint() const
{
    // QComboBox always returns the same from sizeHint() and minimumSizeHint().
    // We want sizeHint() to be the preferred and maximum size
    // (horizontalPolicy == Maximum), but want it to be shrinkable, which is not the case
    // if the minimumSizeHint() is the same as sizeHint()
    QSize size = QComboBox::minimumSizeHint();
    size.setWidth(120);
    return size;
}

} // namespace Internal
} // namespace CppEditor
