use std::process::Command;

mod utils;
use utils::Sway;

static SWAYSOME_BIN: &str = env!("CARGO_BIN_EXE_swaysome");

// Poor safeguard that the tests are actually testing the right version, in
// case there is some confusion with a swaysome installed from the system. That
// obviously does not always worked, but has saved me a couple of times already 🙃
#[test]
fn test_binary_version() {
    let output = Command::new(SWAYSOME_BIN)
        .args(["--version"])
        .env_clear()
        .env("SWAYSOCK", "/dev/null")
        .output()
        .expect("Couldn't run swaysome");
    assert_eq!(
        String::from_utf8(output.stdout).unwrap(),
        format!("swaysome {}\n", env!("CARGO_PKG_VERSION"))
    );
}

// This is useful when working on swapping argument parsing libraries.
#[test]
fn test_binary_help() {
    let output = Command::new(SWAYSOME_BIN)
        .args(["-h"])
        .env_clear()
        .env("SWAYSOCK", "/dev/null")
        .output()
        .expect("Couldn't run swaysome");
    assert_eq!(String::from_utf8(output.stdout).unwrap(), "Better multimonitor handling for sway\n\nUsage: swaysome <COMMAND>\n\nCommands:\n  init                         Initialize the workspace groups for all the outputs\n  move                         Move the focused container to another workspace on the same workspace group\n  move-to-group                Move the focused container to the same workspace index on another workspace group\n  focus                        Focus to another workspace on the same workspace group\n  focus-group                  Focus to workspace group\n  focus-all-outputs            Focus to another workspace on all the outputs\n  next-output                  Move the focused container to the next output\n  prev-output                  Move the focused container to the previous output\n  workspace-group-next-output  Move the focused workspace group to the next output\n  workspace-group-prev-output  Move the focused workspace group to the previous output\n  next-group                   Move the focused container to the next group\n  prev-group                   Move the focused container to the previous group\n  rearrange-workspaces         Rearrange already opened workspaces to the correct outputs, useful when plugging new monitors\n  help                         Print this message or the help of the given subcommand(s)\n\nOptions:\n  -h, --help     Print help\n  -V, --version  Print version\n");
}

// We only test the 'init' command, given that the exhaustive command testing
// is done in the library integration tests. Here, we only verify that the
// interaction with `sway` works seamslessly.
#[test]
fn test_binary_interaction_with_sway() {
    let sway = Sway::start();
    let output = Command::new(SWAYSOME_BIN)
        .args(["init", "1"])
        .env_clear()
        .env("SWAYSOCK", sway.sock.clone())
        .output()
        .expect("Couldn't run swaysome");
    assert_eq!(String::from_utf8(output.stderr).unwrap(), "successful connection to socket '/tmp/swaysome_tests/test_binary_interaction_with_sway/swaysock'\nSending command: 'focus output HEADLESS-3' - Command successful\nSending command: 'workspace number 31' - Command successful\nSending command: 'focus output HEADLESS-2' - Command successful\nSending command: 'workspace number 21' - Command successful\nSending command: 'focus output HEADLESS-1' - Command successful\nSending command: 'workspace number 11' - Command successful\n");
}
