// @HEADER
// *****************************************************************************
//                           Intrepid2 Package
//
// Copyright 2007 NTESS and the Intrepid2 contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

/** \file   Intrepid2_CubatureDirectTriSymmetricDef.hpp
    \brief  Definition file for the Intrepid2::CubatureDirectTriSymmetric class.
    \author Created by M. Perego from an initial implementation by John Burkardt 
            (https://people.sc.fsu.edu/~jburkardt/py_src/triangle_symq_rule/triangle_symq_rule.html)
*/

namespace Intrepid2 {

  template <typename DT, typename PT, typename WT>
  CubatureDirectTriSymmetric<DT,PT,WT>::
  CubatureDirectTriSymmetric(const ordinal_type degree)
    : CubatureDirect<DT,PT,WT>(degree, 2) {

    INTREPID2_TEST_FOR_EXCEPTION( degree < 0 ||
                                  degree >= cubatureDataStaticSize, std::out_of_range,
                                  ">>> ERROR (CubatureDirectTriSymmetric): No cubature rule implemented for the desired polynomial degree.");
    
    typedef Kokkos::DynRankView<PT,Kokkos::LayoutRight,Kokkos::HostSpace> pointViewHostType;
    typedef Kokkos::DynRankView<WT,Kokkos::LayoutRight,Kokkos::HostSpace> weightViewHostType;

    this->cubatureData_.numPoints_ = cubatureDataStatic_[this->degree_].numPoints_;
    const Kokkos::pair<ordinal_type,ordinal_type> pointRange(0, this->cubatureData_.numPoints_);
    {
      // src
      const pointViewHostType points_host(const_cast<PT*>( &(cubatureDataStatic_[this->degree_].points_[0][0]) ),
                                          pointRange.second,
                                          Parameters::MaxDimension);

      auto points = Kokkos::create_mirror_view(typename DT::memory_space(), points_host);

      Kokkos::deep_copy(points,points_host);

      // dst
      this->cubatureData_.points_ = Kokkos::DynRankView<PT,DT>("CubatureDirectTriSymmetric::cubatureData_::points_",
                                                                pointRange.second,
                                                                Parameters::MaxDimension);
      // copy
      Kokkos::deep_copy(this->cubatureData_.points_ , points );
    }
    {
      // src
      const weightViewHostType weights(const_cast<PT*>( &(cubatureDataStatic_[this->degree_].weights_[0]) ),
                                       pointRange.second);

      // dst
      this->cubatureData_.weights_ = Kokkos::DynRankView<WT,DT>("CubatureDirectTriSymmetric::cubatureData_::weights_",
                                                                 pointRange.second);
      // copy
      Kokkos::deep_copy(Kokkos::subdynrankview(this->cubatureData_.weights_, Kokkos::ALL()) , Kokkos::subdynrankview(weights, Kokkos::ALL()));
    }
  }
      
  //-------------------------------------------------------------------------------------//
  //                          Definition of cubature templates                           //
  //-------------------------------------------------------------------------------------//
  
  /*
    Cubature templates for lines are defined the reference Triangle:
    
    Triangle      -> (0,0,0),(1,0,0),(0,1,0)
  */
  
  /*
    This static const member contains templates for default triangle rules.
  */
  
  template<typename DT, typename PT, typename WT>
  const typename CubatureDirect<DT,PT,WT>::CubatureDataStatic
  CubatureDirectTriSymmetric<DT,PT,WT>::
  cubatureDataStatic_[cubatureDataStaticSize] = {
    // Cubature templates for the reference triangle {(0,0), (1,0), (0,1)}
    //
    {
      1,
      {{1./3., 1./3., 0.0}},
      {1./2.}
    },
    {
      1,
      {{1./3., 1./3., 0.0}},
      {1./2.}
    },
    {
      3,
      {{1./6., 1./6., 0.0},
       {2./3., 1./6., 0.0},
       {1./6., 2./3., 0.0}},
      {1./6.,
       1./6.,
       1./6.}
    },
    {
      6,
      {{0.4459484909159648, 0.1081030181680703, 0.0},
      {0.4459484909159649, 0.4459484909159649, 0.0},
      {0.1081030181680702, 0.4459484909159651, 0.0},
      {0.0915762135097707, 0.0915762135097709, 0.0},
      {0.8168475729804584, 0.0915762135097707, 0.0},
      {0.0915762135097710, 0.8168475729804585, 0.0}},
      {0.1116907948390057,
      0.1116907948390057,
      0.1116907948390057,
      0.0549758718276609,
      0.0549758718276609,
      0.0549758718276609}
    },
    {
      6,
      {{0.4459484909159648, 0.1081030181680703, 0.0},
      {0.4459484909159649, 0.4459484909159649, 0.0},
      {0.1081030181680702, 0.4459484909159651, 0.0},
      {0.0915762135097707, 0.0915762135097709, 0.0},
      {0.8168475729804584, 0.0915762135097707, 0.0},
      {0.0915762135097710, 0.8168475729804585, 0.0}},
      {0.1116907948390057,
      0.1116907948390057,
      0.1116907948390057,
      0.0549758718276609,
      0.0549758718276609,
      0.0549758718276609}
    },
    {
      7,
      {{0.1012865073234563, 0.1012865073234565, 0.0},
      {0.7974269853530872, 0.1012865073234562, 0.0},
      {0.1012865073234565, 0.7974269853530874, 0.0},
      {0.4701420641051151, 0.0597158717897699, 0.0},
      {0.4701420641051151, 0.4701420641051151, 0.0},
      {0.0597158717897698, 0.4701420641051153, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0}},
      {0.0629695902724136,
      0.0629695902724136,
      0.0629695902724136,
      0.0661970763942531,
      0.0661970763942531,
      0.0661970763942531,
      0.1125000000000000}
    },
    {
      12,
      {{0.2194299825497829, 0.2194299825497831, 0.0},
      {0.5611400349004340, 0.2194299825497830, 0.0},
      {0.2194299825497830, 0.5611400349004342, 0.0},
      {0.4801379641122150, 0.0397240717755700, 0.0},
      {0.4801379641122151, 0.4801379641122150, 0.0},
      {0.0397240717755699, 0.4801379641122153, 0.0},
      {0.1416190159239681, 0.0193717243612408, 0.0},
      {0.8390092597147910, 0.1416190159239681, 0.0},
      {0.0193717243612410, 0.8390092597147911, 0.0},
      {0.8390092597147911, 0.0193717243612408, 0.0},
      {0.1416190159239681, 0.8390092597147912, 0.0},
      {0.0193717243612406, 0.1416190159239684, 0.0}},
      {0.0856665620764905,
      0.0856665620764905,
      0.0856665620764905,
      0.0403655447965155,
      0.0403655447965155,
      0.0403655447965155,
      0.0203172798968304,
      0.0203172798968304,
      0.0203172798968304,
      0.0203172798968304,
      0.0203172798968304,
      0.0203172798968304}
    },
    {
      15,
      {{0.4731956536892510, 0.0536086926214979, 0.0},
      {0.4731956536892510, 0.4731956536892511, 0.0},
      {0.0536086926214978, 0.4731956536892513, 0.0},
      {0.0577976400545063, 0.0577976400545066, 0.0},
      {0.8844047198909870, 0.0577976400545063, 0.0},
      {0.0577976400545066, 0.8844047198909872, 0.0},
      {0.2593390118657857, 0.0469712061300856, 0.0},
      {0.6936897820041288, 0.2593390118657857, 0.0},
      {0.0469712061300856, 0.6936897820041289, 0.0},
      {0.6936897820041288, 0.0469712061300856, 0.0},
      {0.2593390118657857, 0.6936897820041289, 0.0},
      {0.0469712061300854, 0.2593390118657859, 0.0},
      {0.2416636063972473, 0.2416636063972475, 0.0},
      {0.5166727872055051, 0.2416636063972474, 0.0},
      {0.2416636063972475, 0.5166727872055052, 0.0}},
      {0.0265904166483803,
      0.0265904166483803,
      0.0265904166483803,
      0.0204590851970285,
      0.0204590851970285,
      0.0204590851970285,
      0.0278772702703456,
      0.0278772702703456,
      0.0278772702703456,
      0.0278772702703456,
      0.0278772702703456,
      0.0278772702703456,
      0.0638626242805669,
      0.0638626242805669,
      0.0638626242805669}
    },
    {
      16,
      {{0.1705693077517601, 0.1705693077517604, 0.0},
      {0.6588613844964795, 0.1705693077517602, 0.0},
      {0.1705693077517604, 0.6588613844964796, 0.0},
      {0.4592925882927231, 0.0814148234145537, 0.0},
      {0.4592925882927231, 0.4592925882927232, 0.0},
      {0.0814148234145537, 0.4592925882927233, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.0505472283170309, 0.0505472283170312, 0.0},
      {0.8989055433659379, 0.0505472283170309, 0.0},
      {0.0505472283170313, 0.8989055433659381, 0.0},
      {0.2631128296346381, 0.0083947774099577, 0.0},
      {0.7284923929554042, 0.2631128296346381, 0.0},
      {0.0083947774099577, 0.7284923929554045, 0.0},
      {0.7284923929554042, 0.0083947774099577, 0.0},
      {0.2631128296346381, 0.7284923929554044, 0.0},
      {0.0083947774099575, 0.2631128296346383, 0.0}},
      {0.0516086852673591,
      0.0516086852673591,
      0.0516086852673591,
      0.0475458171336423,
      0.0475458171336423,
      0.0475458171336423,
      0.0721578038388936,
      0.0162292488115991,
      0.0162292488115991,
      0.0162292488115991,
      0.0136151570872175,
      0.0136151570872175,
      0.0136151570872175,
      0.0136151570872175,
      0.0136151570872175,
      0.0136151570872175}
    },
    {
      19,
      {{0.4896825191987376, 0.0206349616025248, 0.0},
      {0.4896825191987376, 0.4896825191987377, 0.0},
      {0.0206349616025247, 0.4896825191987378, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.1882035356190326, 0.1882035356190329, 0.0},
      {0.6235929287619344, 0.1882035356190327, 0.0},
      {0.1882035356190329, 0.6235929287619346, 0.0},
      {0.2219629891607657, 0.0368384120547363, 0.0},
      {0.7411985987844980, 0.2219629891607657, 0.0},
      {0.0368384120547364, 0.7411985987844981, 0.0},
      {0.7411985987844980, 0.0368384120547363, 0.0},
      {0.2219629891607658, 0.7411985987844981, 0.0},
      {0.0368384120547361, 0.2219629891607660, 0.0},
      {0.4370895914929366, 0.1258208170141268, 0.0},
      {0.4370895914929366, 0.4370895914929367, 0.0},
      {0.1258208170141267, 0.4370895914929368, 0.0},
      {0.0447295133944526, 0.0447295133944529, 0.0},
      {0.9105409732110945, 0.0447295133944525, 0.0},
      {0.0447295133944530, 0.9105409732110946, 0.0}},
      {0.0156673501135695,
      0.0156673501135695,
      0.0156673501135695,
      0.0485678981413995,
      0.0398238694636052,
      0.0398238694636052,
      0.0398238694636052,
      0.0216417696886447,
      0.0216417696886447,
      0.0216417696886447,
      0.0216417696886447,
      0.0216417696886447,
      0.0216417696886447,
      0.0389137705023872,
      0.0389137705023872,
      0.0389137705023872,
      0.0127888378293490,
      0.0127888378293490,
      0.0127888378293490}
    },
    {
      25,
      {{0.4951734598011704, 0.0096530803976591, 0.0},
      {0.4951734598011704, 0.4951734598011706, 0.0},
      {0.0096530803976590, 0.4951734598011707, 0.0},
      {0.0191394152428411, 0.0191394152428414, 0.0},
      {0.9617211695143175, 0.0191394152428411, 0.0},
      {0.0191394152428416, 0.9617211695143175, 0.0},
      {0.1337347551008691, 0.0347236204823275, 0.0},
      {0.8315416244168033, 0.1337347551008691, 0.0},
      {0.0347236204823275, 0.8315416244168037, 0.0},
      {0.8315416244168033, 0.0347236204823275, 0.0},
      {0.1337347551008692, 0.8315416244168035, 0.0},
      {0.0347236204823273, 0.1337347551008694, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.3266931362813369, 0.0375827273411917, 0.0},
      {0.6357241363774715, 0.3266931362813369, 0.0},
      {0.0375827273411917, 0.6357241363774715, 0.0},
      {0.6357241363774714, 0.0375827273411917, 0.0},
      {0.3266931362813369, 0.6357241363774715, 0.0},
      {0.0375827273411916, 0.3266931362813372, 0.0},
      {0.1844850126852464, 0.1844850126852466, 0.0},
      {0.6310299746295069, 0.1844850126852465, 0.0},
      {0.1844850126852466, 0.6310299746295072, 0.0},
      {0.4282348209437188, 0.1435303581125623, 0.0},
      {0.4282348209437188, 0.4282348209437189, 0.0},
      {0.1435303581125623, 0.4282348209437190, 0.0}},
      {0.0048962952492092,
      0.0048962952492092,
      0.0048962952492092,
      0.0031926796150594,
      0.0031926796150594,
      0.0031926796150594,
      0.0144811407316282,
      0.0144811407316282,
      0.0144811407316282,
      0.0144811407316282,
      0.0144811407316282,
      0.0144811407316282,
      0.0418074371869869,
      0.0193695245430094,
      0.0193695245430094,
      0.0193695245430094,
      0.0193695245430094,
      0.0193695245430094,
      0.0193695245430094,
      0.0393168848731886,
      0.0393168848731886,
      0.0393168848731886,
      0.0376236639842720,
      0.0376236639842720,
      0.0376236639842720}
    },
    {
      28,
      {{0.0308468956355879, 0.0308468956355882, 0.0},
      {0.9383062087288239, 0.0308468956355879, 0.0},
      {0.0308468956355883, 0.9383062087288241, 0.0},
      {0.4987801651784607, 0.0024396696430785, 0.0},
      {0.4987801651784607, 0.4987801651784608, 0.0},
      {0.0024396696430784, 0.4987801651784610, 0.0},
      {0.1593036198376935, 0.0143666625695556, 0.0},
      {0.8263297175927509, 0.1593036198376935, 0.0},
      {0.0143666625695557, 0.8263297175927511, 0.0},
      {0.8263297175927509, 0.0143666625695556, 0.0},
      {0.1593036198376935, 0.8263297175927511, 0.0},
      {0.0143666625695554, 0.1593036198376938, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.1132078272866939, 0.1132078272866941, 0.0},
      {0.7735843454266120, 0.1132078272866939, 0.0},
      {0.1132078272866941, 0.7735843454266123, 0.0},
      {0.4366550163931761, 0.1266899672136478, 0.0},
      {0.4366550163931761, 0.4366550163931761, 0.0},
      {0.1266899672136477, 0.4366550163931763, 0.0},
      {0.2144834586192693, 0.2144834586192694, 0.0},
      {0.5710330827614613, 0.2144834586192693, 0.0},
      {0.2144834586192694, 0.5710330827614615, 0.0},
      {0.3106312163134631, 0.0476640669721508, 0.0},
      {0.6417047167143861, 0.3106312163134631, 0.0},
      {0.0476640669721508, 0.6417047167143863, 0.0},
      {0.6417047167143860, 0.0476640669721508, 0.0},
      {0.3106312163134631, 0.6417047167143861, 0.0},
      {0.0476640669721507, 0.3106312163134634, 0.0}},
      {0.0061246484753540,
      0.0061246484753540,
      0.0061246484753540,
      0.0062327459369407,
      0.0062327459369407,
      0.0062327459369407,
      0.0072788116689046,
      0.0072788116689046,
      0.0072788116689046,
      0.0072788116689046,
      0.0072788116689046,
      0.0072788116689046,
      0.0407225673546757,
      0.0200646211906541,
      0.0200646211906541,
      0.0200646211906541,
      0.0315474360799494,
      0.0315474360799494,
      0.0315474360799494,
      0.0339225538718475,
      0.0339225538718475,
      0.0339225538718475,
      0.0203214243279433,
      0.0203214243279433,
      0.0203214243279433,
      0.0203214243279433,
      0.0203214243279433,
      0.0203214243279433}
    },
    {
      33,
      {{0.2714625070149260, 0.2714625070149262, 0.0},
      {0.4570749859701477, 0.2714625070149261, 0.0},
      {0.2714625070149261, 0.4570749859701479, 0.0},
      {0.1092578276593542, 0.1092578276593545, 0.0},
      {0.7814843446812912, 0.1092578276593542, 0.0},
      {0.1092578276593545, 0.7814843446812915, 0.0},
      {0.4401116486585931, 0.1197767026828138, 0.0},
      {0.4401116486585931, 0.4401116486585931, 0.0},
      {0.1197767026828138, 0.4401116486585933, 0.0},
      {0.2554542286385173, 0.1162960196779266, 0.0},
      {0.6282497516835561, 0.2554542286385174, 0.0},
      {0.1162960196779266, 0.6282497516835562, 0.0},
      {0.6282497516835561, 0.1162960196779266, 0.0},
      {0.2554542286385174, 0.6282497516835562, 0.0},
      {0.1162960196779265, 0.2554542286385176, 0.0},
      {0.1272797172335894, 0.0213824902561706, 0.0},
      {0.8513377925102400, 0.1272797172335893, 0.0},
      {0.0213824902561707, 0.8513377925102402, 0.0},
      {0.8513377925102400, 0.0213824902561706, 0.0},
      {0.1272797172335894, 0.8513377925102402, 0.0},
      {0.0213824902561704, 0.1272797172335897, 0.0},
      {0.2916556797383409, 0.0230341563552672, 0.0},
      {0.6853101639063919, 0.2916556797383410, 0.0},
      {0.0230341563552672, 0.6853101639063921, 0.0},
      {0.6853101639063919, 0.0230341563552672, 0.0},
      {0.2916556797383410, 0.6853101639063920, 0.0},
      {0.0230341563552670, 0.2916556797383412, 0.0},
      {0.4882037509455415, 0.0235924981089169, 0.0},
      {0.4882037509455415, 0.4882037509455416, 0.0},
      {0.0235924981089169, 0.4882037509455417, 0.0},
      {0.0246463634363355, 0.0246463634363358, 0.0},
      {0.9507072731273287, 0.0246463634363355, 0.0},
      {0.0246463634363359, 0.9507072731273288, 0.0}},
      {0.0312706065979513,
      0.0312706065979513,
      0.0312706065979513,
      0.0142430260344387,
      0.0142430260344387,
      0.0142430260344387,
      0.0249591674640305,
      0.0249591674640305,
      0.0249591674640305,
      0.0216136818297071,
      0.0216136818297071,
      0.0216136818297071,
      0.0216136818297071,
      0.0216136818297071,
      0.0216136818297071,
      0.0075418387882557,
      0.0075418387882557,
      0.0075418387882557,
      0.0075418387882557,
      0.0075418387882557,
      0.0075418387882557,
      0.0108917925193038,
      0.0108917925193038,
      0.0108917925193038,
      0.0108917925193038,
      0.0108917925193038,
      0.0108917925193038,
      0.0121334190407260,
      0.0121334190407260,
      0.0121334190407260,
      0.0039658212549868,
      0.0039658212549868,
      0.0039658212549868}
    },
    {
      37,
      {{0.4961358947410461, 0.0077282105179079, 0.0},
      {0.4961358947410460, 0.4961358947410461, 0.0},
      {0.0077282105179078, 0.4961358947410462, 0.0},
      {0.4696086896534919, 0.0607826206930162, 0.0},
      {0.4696086896534918, 0.4696086896534920, 0.0},
      {0.0607826206930162, 0.4696086896534921, 0.0},
      {0.2311102849490822, 0.2311102849490824, 0.0},
      {0.5377794301018355, 0.2311102849490823, 0.0},
      {0.2311102849490823, 0.5377794301018356, 0.0},
      {0.2920786885766364, 0.0189880043837590, 0.0},
      {0.6889333070396045, 0.2920786885766364, 0.0},
      {0.0189880043837591, 0.6889333070396048, 0.0},
      {0.6889333070396045, 0.0189880043837590, 0.0},
      {0.2920786885766364, 0.6889333070396046, 0.0},
      {0.0189880043837589, 0.2920786885766367, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.2667452533103511, 0.0977360310660166, 0.0},
      {0.6355187156236324, 0.2667452533103512, 0.0},
      {0.0977360310660165, 0.6355187156236326, 0.0},
      {0.6355187156236323, 0.0977360310660166, 0.0},
      {0.2667452533103513, 0.6355187156236323, 0.0},
      {0.0977360310660164, 0.2667452533103514, 0.0},
      {0.4144775702790545, 0.1710448594418909, 0.0},
      {0.4144775702790545, 0.4144775702790546, 0.0},
      {0.1710448594418908, 0.4144775702790547, 0.0},
      {0.1135599125721331, 0.1135599125721333, 0.0},
      {0.7728801748557336, 0.1135599125721331, 0.0},
      {0.1135599125721334, 0.7728801748557337, 0.0},
      {0.1267997757838373, 0.0219663442065293, 0.0},
      {0.8512338800096334, 0.1267997757838373, 0.0},
      {0.0219663442065294, 0.8512338800096335, 0.0},
      {0.8512338800096334, 0.0219663442065293, 0.0},
      {0.1267997757838374, 0.8512338800096335, 0.0},
      {0.0219663442065290, 0.1267997757838377, 0.0},
      {0.0248959314912162, 0.0248959314912165, 0.0},
      {0.9502081370175671, 0.0248959314912162, 0.0},
      {0.0248959314912167, 0.9502081370175673, 0.0}},
      {0.0049707381805363,
      0.0049707381805363,
      0.0049707381805363,
      0.0163906208018615,
      0.0163906208018615,
      0.0163906208018615,
      0.0230312047963891,
      0.0230312047963891,
      0.0230312047963891,
      0.0090627493231004,
      0.0090627493231004,
      0.0090627493231004,
      0.0090627493231004,
      0.0090627493231004,
      0.0090627493231004,
      0.0258113233321454,
      0.0186059802286308,
      0.0186059802286308,
      0.0186059802286308,
      0.0186059802286308,
      0.0186059802286308,
      0.0186059802286308,
      0.0234735477710776,
      0.0234735477710776,
      0.0234735477710776,
      0.0154515489878799,
      0.0154515489878799,
      0.0154515489878799,
      0.0076965363418911,
      0.0076965363418911,
      0.0076965363418911,
      0.0076965363418911,
      0.0076965363418911,
      0.0076965363418911,
      0.0040146998976292,
      0.0040146998976292,
      0.0040146998976292}
    },
    {
      42,
      {{0.4176447193404539, 0.1647105613190922, 0.0},
      {0.4176447193404539, 0.4176447193404541, 0.0},
      {0.1647105613190921, 0.4176447193404541, 0.0},
      {0.2983728821362577, 0.0146469500556545, 0.0},
      {0.6869801678080878, 0.2983728821362578, 0.0},
      {0.0146469500556545, 0.6869801678080880, 0.0},
      {0.6869801678080878, 0.0146469500556545, 0.0},
      {0.2983728821362577, 0.6869801678080880, 0.0},
      {0.0146469500556543, 0.2983728821362580, 0.0},
      {0.0617998830908725, 0.0617998830908728, 0.0},
      {0.8764002338182547, 0.0617998830908724, 0.0},
      {0.0617998830908729, 0.8764002338182548, 0.0},
      {0.3368614597963450, 0.0929162493569719, 0.0},
      {0.5702222908466831, 0.3368614597963451, 0.0},
      {0.0929162493569718, 0.5702222908466833, 0.0},
      {0.5702222908466831, 0.0929162493569719, 0.0},
      {0.3368614597963450, 0.5702222908466833, 0.0},
      {0.0929162493569718, 0.3368614597963452, 0.0},
      {0.2734775283088386, 0.2734775283088388, 0.0},
      {0.4530449433823227, 0.2734775283088387, 0.0},
      {0.2734775283088388, 0.4530449433823228, 0.0},
      {0.1772055324125434, 0.1772055324125436, 0.0},
      {0.6455889351749130, 0.1772055324125434, 0.0},
      {0.1772055324125437, 0.6455889351749131, 0.0},
      {0.0193909612487009, 0.0193909612487012, 0.0},
      {0.9612180775025979, 0.0193909612487009, 0.0},
      {0.0193909612487014, 0.9612180775025979, 0.0},
      {0.4889639103621786, 0.0220721792756428, 0.0},
      {0.4889639103621787, 0.4889639103621786, 0.0},
      {0.0220721792756427, 0.4889639103621788, 0.0},
      {0.1722666878213555, 0.0571247574036480, 0.0},
      {0.7706085547749963, 0.1722666878213556, 0.0},
      {0.0571247574036480, 0.7706085547749967, 0.0},
      {0.7706085547749965, 0.0571247574036480, 0.0},
      {0.1722666878213556, 0.7706085547749966, 0.0},
      {0.0571247574036478, 0.1722666878213558, 0.0},
      {0.1189744976969568, 0.0012683309328721, 0.0},
      {0.8797571713701711, 0.1189744976969568, 0.0},
      {0.0012683309328722, 0.8797571713701712, 0.0},
      {0.8797571713701712, 0.0012683309328721, 0.0},
      {0.1189744976969569, 0.8797571713701712, 0.0},
      {0.0012683309328719, 0.1189744976969572, 0.0}},
      {0.0163941767720627,
      0.0163941767720627,
      0.0163941767720627,
      0.0072181540567669,
      0.0072181540567669,
      0.0072181540567669,
      0.0072181540567669,
      0.0072181540567669,
      0.0072181540567669,
      0.0072168498348884,
      0.0072168498348884,
      0.0072168498348884,
      0.0192857553935303,
      0.0192857553935303,
      0.0192857553935303,
      0.0192857553935303,
      0.0192857553935303,
      0.0192857553935303,
      0.0258870522536458,
      0.0258870522536458,
      0.0258870522536458,
      0.0210812943684965,
      0.0210812943684965,
      0.0210812943684965,
      0.0024617018012001,
      0.0024617018012001,
      0.0024617018012001,
      0.0109417906847144,
      0.0109417906847144,
      0.0109417906847144,
      0.0123328766062819,
      0.0123328766062819,
      0.0123328766062819,
      0.0123328766062819,
      0.0123328766062819,
      0.0123328766062819,
      0.0025051144192503,
      0.0025051144192503,
      0.0025051144192503,
      0.0025051144192503,
      0.0025051144192503,
      0.0025051144192503}
    },
    {
      49,
      {{0.1299782299330778, 0.1299782299330780, 0.0},
      {0.7400435401338442, 0.1299782299330778, 0.0},
      {0.1299782299330779, 0.7400435401338445, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.4600769492970597, 0.0798461014058806, 0.0},
      {0.4600769492970597, 0.4600769492970598, 0.0},
      {0.0798461014058805, 0.4600769492970599, 0.0},
      {0.1823217834071913, 0.0845942214821918, 0.0},
      {0.7330839951106168, 0.1823217834071913, 0.0},
      {0.0845942214821919, 0.7330839951106169, 0.0},
      {0.7330839951106169, 0.0845942214821918, 0.0},
      {0.1823217834071914, 0.7330839951106171, 0.0},
      {0.0845942214821916, 0.1823217834071916, 0.0},
      {0.1502003840652387, 0.0160270897863455, 0.0},
      {0.8337725261484157, 0.1502003840652387, 0.0},
      {0.0160270897863456, 0.8337725261484159, 0.0},
      {0.8337725261484158, 0.0160270897863455, 0.0},
      {0.1502003840652388, 0.8337725261484159, 0.0},
      {0.0160270897863453, 0.1502003840652391, 0.0},
      {0.3231113151637127, 0.0976504424302424, 0.0},
      {0.5792382424060449, 0.3231113151637127, 0.0},
      {0.0976504424302423, 0.5792382424060452, 0.0},
      {0.5792382424060449, 0.0976504424302424, 0.0},
      {0.3231113151637127, 0.5792382424060450, 0.0},
      {0.0976504424302423, 0.3231113151637129, 0.0},
      {0.4916858166302972, 0.0166283667394056, 0.0},
      {0.4916858166302972, 0.4916858166302973, 0.0},
      {0.0166283667394055, 0.4916858166302975, 0.0},
      {0.2215323407951419, 0.2215323407951421, 0.0},
      {0.5569353184097160, 0.2215323407951420, 0.0},
      {0.2215323407951421, 0.5569353184097161, 0.0},
      {0.3969337374090606, 0.2061325251818789, 0.0},
      {0.3969337374090606, 0.3969337374090606, 0.0},
      {0.2061325251818788, 0.3969337374090607, 0.0},
      {0.3079476814836729, 0.0184542519046332, 0.0},
      {0.6735980666116939, 0.3079476814836729, 0.0},
      {0.0184542519046332, 0.6735980666116942, 0.0},
      {0.6735980666116940, 0.0184542519046332, 0.0},
      {0.3079476814836729, 0.6735980666116941, 0.0},
      {0.0184542519046331, 0.3079476814836731, 0.0},
      {0.0563419176961000, 0.0563419176961003, 0.0},
      {0.8873161646077997, 0.0563419176960999, 0.0},
      {0.0563419176961004, 0.8873161646077998, 0.0},
      {0.0380352293011093, 0.0011135352740137, 0.0},
      {0.9608512354248769, 0.0380352293011093, 0.0},
      {0.0011135352740139, 0.9608512354248772, 0.0},
      {0.9608512354248770, 0.0011135352740137, 0.0},
      {0.0380352293011093, 0.9608512354248772, 0.0},
      {0.0011135352740135, 0.0380352293011096, 0.0}},
      {0.0036987520335230,
      0.0036987520335230,
      0.0036987520335230,
      0.0148652098740357,
      0.0107970439682192,
      0.0107970439682192,
      0.0107970439682192,
      0.0121150043915628,
      0.0121150043915628,
      0.0121150043915628,
      0.0121150043915628,
      0.0121150043915628,
      0.0121150043915628,
      0.0056142521494390,
      0.0056142521494390,
      0.0056142521494390,
      0.0056142521494390,
      0.0056142521494390,
      0.0056142521494390,
      0.0155376102352555,
      0.0155376102352555,
      0.0155376102352555,
      0.0155376102352555,
      0.0155376102352555,
      0.0155376102352555,
      0.0079161381750109,
      0.0079161381750109,
      0.0079161381750109,
      0.0231436430525990,
      0.0231436430525990,
      0.0231436430525990,
      0.0231680206956036,
      0.0231680206956036,
      0.0231680206956036,
      0.0082183810464139,
      0.0082183810464139,
      0.0082183810464139,
      0.0082183810464139,
      0.0082183810464139,
      0.0082183810464139,
      0.0075422371237985,
      0.0075422371237985,
      0.0075422371237985,
      0.0012376330072789,
      0.0012376330072789,
      0.0012376330072789,
      0.0012376330072789,
      0.0012376330072789,
      0.0012376330072789}
    },
    {
      55,
      {{0.4137694858270852, 0.0096649544036603, 0.0},
      {0.5765655597692547, 0.4137694858270852, 0.0},
      {0.0096649544036603, 0.5765655597692547, 0.0},
      {0.5765655597692546, 0.0096649544036603, 0.0},
      {0.4137694858270852, 0.5765655597692547, 0.0},
      {0.0096649544036602, 0.4137694858270854, 0.0},
      {0.3041794482294797, 0.0303059433551864, 0.0},
      {0.6655146084153339, 0.3041794482294797, 0.0},
      {0.0303059433551864, 0.6655146084153342, 0.0},
      {0.6655146084153339, 0.0303059433551864, 0.0},
      {0.3041794482294798, 0.6655146084153339, 0.0},
      {0.0303059433551863, 0.3041794482294800, 0.0},
      {0.0666744722402382, 0.0666744722402385, 0.0},
      {0.8666510555195234, 0.0666744722402381, 0.0},
      {0.0666744722402385, 0.8666510555195236, 0.0},
      {0.0896090890227058, 0.0108129727761038, 0.0},
      {0.8995779382011904, 0.0896090890227058, 0.0},
      {0.0108129727761038, 0.8995779382011906, 0.0},
      {0.8995779382011904, 0.0108129727761038, 0.0},
      {0.0896090890227059, 0.8995779382011906, 0.0},
      {0.0108129727761035, 0.0896090890227062, 0.0},
      {0.2966153724003829, 0.1066531605361485, 0.0},
      {0.5967314670634685, 0.2966153724003830, 0.0},
      {0.1066531605361485, 0.5967314670634687, 0.0},
      {0.5967314670634686, 0.1066531605361485, 0.0},
      {0.2966153724003830, 0.5967314670634686, 0.0},
      {0.1066531605361483, 0.2966153724003832, 0.0},
      {0.2413216807013783, 0.2413216807013785, 0.0},
      {0.5173566385972432, 0.2413216807013784, 0.0},
      {0.2413216807013784, 0.5173566385972433, 0.0},
      {0.4127980959552237, 0.1744038080895527, 0.0},
      {0.4127980959552237, 0.4127980959552238, 0.0},
      {0.1744038080895526, 0.4127980959552238, 0.0},
      {0.1697633551502897, 0.0513543153440131, 0.0},
      {0.7788823295056971, 0.1697633551502898, 0.0},
      {0.0513543153440132, 0.7788823295056972, 0.0},
      {0.7788823295056972, 0.0513543153440131, 0.0},
      {0.1697633551502898, 0.7788823295056971, 0.0},
      {0.0513543153440130, 0.1697633551502900, 0.0},
      {0.1500637365870350, 0.1500637365870353, 0.0},
      {0.6998725268259297, 0.1500637365870350, 0.0},
      {0.1500637365870353, 0.6998725268259298, 0.0},
      {0.2140487799258473, 0.0036969427073556, 0.0},
      {0.7822542773667971, 0.2140487799258473, 0.0},
      {0.0036969427073556, 0.7822542773667974, 0.0},
      {0.7822542773667971, 0.0036969427073556, 0.0},
      {0.2140487799258473, 0.7822542773667972, 0.0},
      {0.0036969427073554, 0.2140487799258476, 0.0},
      {0.4695480309966850, 0.0609039380066301, 0.0},
      {0.4695480309966850, 0.4695480309966850, 0.0},
      {0.0609039380066300, 0.4695480309966852, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.0170416294057183, 0.0170416294057186, 0.0},
      {0.9659167411885631, 0.0170416294057182, 0.0},
      {0.0170416294057187, 0.9659167411885633, 0.0}},
      {0.0040911052766111,
      0.0040911052766111,
      0.0040911052766111,
      0.0040911052766111,
      0.0040911052766111,
      0.0040911052766111,
      0.0069918035623268,
      0.0069918035623268,
      0.0069918035623268,
      0.0069918035623268,
      0.0069918035623268,
      0.0069918035623268,
      0.0062127127977805,
      0.0062127127977805,
      0.0062127127977805,
      0.0028759349852486,
      0.0028759349852486,
      0.0028759349852486,
      0.0028759349852486,
      0.0028759349852486,
      0.0028759349852486,
      0.0158230308409916,
      0.0158230308409916,
      0.0158230308409916,
      0.0158230308409916,
      0.0158230308409916,
      0.0158230308409916,
      0.0205920205348963,
      0.0205920205348963,
      0.0205920205348963,
      0.0204926098934077,
      0.0204926098934077,
      0.0204926098934077,
      0.0088265405235517,
      0.0088265405235517,
      0.0088265405235517,
      0.0088265405235517,
      0.0088265405235517,
      0.0088265405235517,
      0.0143917483513744,
      0.0143917483513744,
      0.0143917483513744,
      0.0023073453198645,
      0.0023073453198645,
      0.0023073453198645,
      0.0023073453198645,
      0.0023073453198645,
      0.0023073453198645,
      0.0135468347338552,
      0.0135468347338552,
      0.0135468347338552,
      0.0231139551570957,
      0.0018945676191321,
      0.0018945676191321,
      0.0018945676191321}
    },
    {
      60,
      {{0.4171034443615992, 0.1657931112768017, 0.0},
      {0.4171034443615992, 0.4171034443615993, 0.0},
      {0.1657931112768016, 0.4171034443615994, 0.0},
      {0.0725054707990024, 0.0115751759031807, 0.0},
      {0.9159193532978169, 0.0725054707990024, 0.0},
      {0.0115751759031809, 0.9159193532978170, 0.0},
      {0.9159193532978169, 0.0115751759031807, 0.0},
      {0.0725054707990024, 0.9159193532978170, 0.0},
      {0.0115751759031804, 0.0725054707990027, 0.0},
      {0.1803581162663705, 0.1803581162663707, 0.0},
      {0.6392837674672587, 0.1803581162663706, 0.0},
      {0.1803581162663708, 0.6392837674672588, 0.0},
      {0.4154754592952291, 0.0132296727600870, 0.0},
      {0.5712948679446841, 0.4154754592952291, 0.0},
      {0.0132296727600869, 0.5712948679446843, 0.0},
      {0.5712948679446841, 0.0132296727600870, 0.0},
      {0.4154754592952290, 0.5712948679446841, 0.0},
      {0.0132296727600868, 0.4154754592952293, 0.0},
      {0.2717918700553548, 0.0131358708340028, 0.0},
      {0.7150722591106424, 0.2717918700553549, 0.0},
      {0.0131358708340029, 0.7150722591106425, 0.0},
      {0.7150722591106424, 0.0131358708340028, 0.0},
      {0.2717918700553550, 0.7150722591106424, 0.0},
      {0.0131358708340026, 0.2717918700553552, 0.0},
      {0.2992189424769703, 0.1575054779268700, 0.0},
      {0.5432755795961597, 0.2992189424769703, 0.0},
      {0.1575054779268699, 0.5432755795961599, 0.0},
      {0.5432755795961597, 0.1575054779268700, 0.0},
      {0.2992189424769702, 0.5432755795961599, 0.0},
      {0.1575054779268698, 0.2992189424769705, 0.0},
      {0.2857065024365866, 0.2857065024365867, 0.0},
      {0.4285869951268267, 0.2857065024365867, 0.0},
      {0.2857065024365867, 0.4285869951268269, 0.0},
      {0.3062815917461865, 0.0673493778673613, 0.0},
      {0.6263690303864522, 0.3062815917461865, 0.0},
      {0.0673493778673613, 0.6263690303864524, 0.0},
      {0.6263690303864522, 0.0673493778673613, 0.0},
      {0.3062815917461865, 0.6263690303864524, 0.0},
      {0.0673493778673612, 0.3062815917461867, 0.0},
      {0.1687225134952594, 0.0780423405682825, 0.0},
      {0.7532351459364581, 0.1687225134952594, 0.0},
      {0.0780423405682825, 0.7532351459364584, 0.0},
      {0.7532351459364581, 0.0780423405682825, 0.0},
      {0.1687225134952594, 0.7532351459364582, 0.0},
      {0.0780423405682823, 0.1687225134952597, 0.0},
      {0.0666540634795968, 0.0666540634795971, 0.0},
      {0.8666918730408060, 0.0666540634795968, 0.0},
      {0.0666540634795971, 0.8666918730408062, 0.0},
      {0.1591922874727927, 0.0160176423621193, 0.0},
      {0.8247900701650880, 0.1591922874727927, 0.0},
      {0.0160176423621194, 0.8247900701650882, 0.0},
      {0.8247900701650880, 0.0160176423621193, 0.0},
      {0.1591922874727927, 0.8247900701650882, 0.0},
      {0.0160176423621192, 0.1591922874727930, 0.0},
      {0.0147554916607538, 0.0147554916607541, 0.0},
      {0.9704890166784920, 0.0147554916607538, 0.0},
      {0.0147554916607542, 0.9704890166784922, 0.0},
      {0.4655978716188903, 0.0688042567622194, 0.0},
      {0.4655978716188902, 0.4655978716188904, 0.0},
      {0.0688042567622194, 0.4655978716188905, 0.0}},
      {0.0136554632640510,
      0.0136554632640510,
      0.0136554632640510,
      0.0022921742008679,
      0.0022921742008679,
      0.0022921742008679,
      0.0022921742008679,
      0.0022921742008679,
      0.0022921742008679,
      0.0131563152940090,
      0.0131563152940090,
      0.0131563152940090,
      0.0051992199779198,
      0.0051992199779198,
      0.0051992199779198,
      0.0051992199779198,
      0.0051992199779198,
      0.0051992199779198,
      0.0043461072505006,
      0.0043461072505006,
      0.0043461072505006,
      0.0043461072505006,
      0.0043461072505006,
      0.0043461072505006,
      0.0130858129676685,
      0.0130858129676685,
      0.0130858129676685,
      0.0130858129676685,
      0.0130858129676685,
      0.0130858129676685,
      0.0188581185763976,
      0.0188581185763976,
      0.0188581185763976,
      0.0112438862733456,
      0.0112438862733456,
      0.0112438862733456,
      0.0112438862733456,
      0.0112438862733456,
      0.0112438862733456,
      0.0102789491602273,
      0.0102789491602273,
      0.0102789491602273,
      0.0102789491602273,
      0.0102789491602273,
      0.0102789491602273,
      0.0062295004011527,
      0.0062295004011527,
      0.0062295004011527,
      0.0039891501029648,
      0.0039891501029648,
      0.0039891501029648,
      0.0039891501029648,
      0.0039891501029648,
      0.0039891501029648,
      0.0013869437888188,
      0.0013869437888188,
      0.0013869437888188,
      0.0125097254752487,
      0.0125097254752487,
      0.0125097254752487}
    },
    {
      67,
      {{0.3850440344131636, 0.0904270403543407, 0.0},
      {0.5245289252324958, 0.3850440344131637, 0.0},
      {0.0904270403543406, 0.5245289252324959, 0.0},
      {0.5245289252324957, 0.0904270403543407, 0.0},
      {0.3850440344131637, 0.5245289252324958, 0.0},
      {0.0904270403543405, 0.3850440344131639, 0.0},
      {0.4749182113240457, 0.0501635773519086, 0.0},
      {0.4749182113240458, 0.4749182113240458, 0.0},
      {0.0501635773519085, 0.4749182113240459, 0.0},
      {0.1516385069726047, 0.1516385069726050, 0.0},
      {0.6967229860547902, 0.1516385069726048, 0.0},
      {0.1516385069726050, 0.6967229860547904, 0.0},
      {0.0472761418326517, 0.0124989324834955, 0.0},
      {0.9402249256838527, 0.0472761418326518, 0.0},
      {0.0124989324834957, 0.9402249256838529, 0.0},
      {0.9402249256838527, 0.0124989324834955, 0.0},
      {0.0472761418326518, 0.9402249256838529, 0.0},
      {0.0124989324834952, 0.0472761418326521, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.3020619577128708, 0.0540117353390243, 0.0},
      {0.6439263069481050, 0.3020619577128708, 0.0},
      {0.0540117353390243, 0.6439263069481052, 0.0},
      {0.6439263069481049, 0.0540117353390243, 0.0},
      {0.3020619577128708, 0.6439263069481049, 0.0},
      {0.0540117353390241, 0.3020619577128710, 0.0},
      {0.2565061597742415, 0.0105050188192420, 0.0},
      {0.7329888214065164, 0.2565061597742415, 0.0},
      {0.0105050188192420, 0.7329888214065168, 0.0},
      {0.7329888214065166, 0.0105050188192420, 0.0},
      {0.2565061597742415, 0.7329888214065168, 0.0},
      {0.0105050188192418, 0.2565061597742418, 0.0},
      {0.4110671018759194, 0.1778657962481611, 0.0},
      {0.4110671018759194, 0.4110671018759195, 0.0},
      {0.1778657962481610, 0.4110671018759197, 0.0},
      {0.1784791255658876, 0.0661224580284034, 0.0},
      {0.7553984164057089, 0.1784791255658876, 0.0},
      {0.0661224580284036, 0.7553984164057090, 0.0},
      {0.7553984164057089, 0.0661224580284034, 0.0},
      {0.1784791255658877, 0.7553984164057090, 0.0},
      {0.0661224580284032, 0.1784791255658879, 0.0},
      {0.2656146099053742, 0.2656146099053743, 0.0},
      {0.4687707801892516, 0.2656146099053742, 0.0},
      {0.2656146099053743, 0.4687707801892517, 0.0},
      {0.0037589443410682, 0.0037589443410685, 0.0},
      {0.9924821113178632, 0.0037589443410682, 0.0},
      {0.0037589443410687, 0.9924821113178633, 0.0},
      {0.2685733063960138, 0.1490669101257739, 0.0},
      {0.5823597834782123, 0.2685733063960138, 0.0},
      {0.1490669101257739, 0.5823597834782125, 0.0},
      {0.5823597834782123, 0.1490669101257739, 0.0},
      {0.2685733063960138, 0.5823597834782125, 0.0},
      {0.1490669101257737, 0.2685733063960140, 0.0},
      {0.4110656686746184, 0.0116918246746672, 0.0},
      {0.5772425066507145, 0.4110656686746184, 0.0},
      {0.0116918246746671, 0.5772425066507147, 0.0},
      {0.5772425066507145, 0.0116918246746672, 0.0},
      {0.4110656686746185, 0.5772425066507145, 0.0},
      {0.0116918246746671, 0.4110656686746186, 0.0},
      {0.1327788302713893, 0.0143315247789420, 0.0},
      {0.8528896449496687, 0.1327788302713893, 0.0},
      {0.0143315247789422, 0.8528896449496687, 0.0},
      {0.8528896449496688, 0.0143315247789420, 0.0},
      {0.1327788302713893, 0.8528896449496690, 0.0},
      {0.0143315247789417, 0.1327788302713896, 0.0},
      {0.0724387055673328, 0.0724387055673330, 0.0},
      {0.8551225888653341, 0.0724387055673327, 0.0},
      {0.0724387055673332, 0.8551225888653343, 0.0}},
      {0.0076641290972766,
      0.0076641290972766,
      0.0076641290972766,
      0.0076641290972766,
      0.0076641290972766,
      0.0076641290972766,
      0.0065535137458694,
      0.0065535137458694,
      0.0065535137458694,
      0.0101591694227292,
      0.0101591694227292,
      0.0101591694227292,
      0.0021087583873722,
      0.0021087583873722,
      0.0021087583873722,
      0.0021087583873722,
      0.0021087583873722,
      0.0021087583873722,
      0.0153742606195580,
      0.0081829542069933,
      0.0081829542069933,
      0.0081829542069933,
      0.0081829542069933,
      0.0081829542069933,
      0.0081829542069933,
      0.0038649176400031,
      0.0038649176400031,
      0.0038649176400031,
      0.0038649176400031,
      0.0038649176400031,
      0.0038649176400031,
      0.0167359970299240,
      0.0167359970299240,
      0.0167359970299240,
      0.0084558269587401,
      0.0084558269587401,
      0.0084558269587401,
      0.0084558269587401,
      0.0084558269587401,
      0.0084558269587401,
      0.0155581983010030,
      0.0155581983010030,
      0.0155581983010030,
      0.0002660028084739,
      0.0002660028084739,
      0.0002660028084739,
      0.0137964432442898,
      0.0137964432442898,
      0.0137964432442898,
      0.0137964432442898,
      0.0137964432442898,
      0.0137964432442898,
      0.0047930622371807,
      0.0047930622371807,
      0.0047930622371807,
      0.0047930622371807,
      0.0047930622371807,
      0.0047930622371807,
      0.0038208524863598,
      0.0038208524863598,
      0.0038208524863598,
      0.0038208524863598,
      0.0038208524863598,
      0.0038208524863598,
      0.0068951433023835,
      0.0068951433023835,
      0.0068951433023835}
    },
    {
      73,
      {{0.1424222825711269, 0.0050051423523504, 0.0},
      {0.8525725750765226, 0.1424222825711269, 0.0},
      {0.0050051423523504, 0.8525725750765228, 0.0},
      {0.8525725750765226, 0.0050051423523504, 0.0},
      {0.1424222825711269, 0.8525725750765228, 0.0},
      {0.0050051423523502, 0.1424222825711273, 0.0},
      {0.0525262798541034, 0.0525262798541037, 0.0},
      {0.8949474402917929, 0.0525262798541034, 0.0},
      {0.0525262798541039, 0.8949474402917930, 0.0},
      {0.0600838999627024, 0.0097770614386769, 0.0},
      {0.9301390385986208, 0.0600838999627023, 0.0},
      {0.0097770614386771, 0.9301390385986208, 0.0},
      {0.9301390385986208, 0.0097770614386769, 0.0},
      {0.0600838999627025, 0.9301390385986208, 0.0},
      {0.0097770614386767, 0.0600838999627027, 0.0},
      {0.1307006699605345, 0.0391424494346089, 0.0},
      {0.8301568806048565, 0.1307006699605346, 0.0},
      {0.0391424494346090, 0.8301568806048567, 0.0},
      {0.8301568806048566, 0.0391424494346089, 0.0},
      {0.1307006699605346, 0.8301568806048567, 0.0},
      {0.0391424494346087, 0.1307006699605348, 0.0},
      {0.3113183832239869, 0.1293128097679790, 0.0},
      {0.5593688070080342, 0.3113183832239869, 0.0},
      {0.1293128097679790, 0.5593688070080343, 0.0},
      {0.5593688070080342, 0.1293128097679790, 0.0},
      {0.3113183832239869, 0.5593688070080343, 0.0},
      {0.1293128097679789, 0.3113183832239870, 0.0},
      {0.1114480557169985, 0.1114480557169988, 0.0},
      {0.7771038885660025, 0.1114480557169985, 0.0},
      {0.1114480557169988, 0.7771038885660028, 0.0},
      {0.0116390273279224, 0.0116390273279228, 0.0},
      {0.9767219453441548, 0.0116390273279224, 0.0},
      {0.0116390273279229, 0.9767219453441548, 0.0},
      {0.2551621331531247, 0.2551621331531249, 0.0},
      {0.4896757336937503, 0.2551621331531249, 0.0},
      {0.2551621331531249, 0.4896757336937504, 0.0},
      {0.2214339418891134, 0.0745611893043552, 0.0},
      {0.7040048688065315, 0.2214339418891134, 0.0},
      {0.0745611893043552, 0.7040048688065317, 0.0},
      {0.7040048688065313, 0.0745611893043552, 0.0},
      {0.2214339418891134, 0.7040048688065317, 0.0},
      {0.0745611893043550, 0.2214339418891137, 0.0},
      {0.4039697179663860, 0.1920605640672279, 0.0},
      {0.4039697179663860, 0.4039697179663861, 0.0},
      {0.1920605640672278, 0.4039697179663862, 0.0},
      {0.3540259269997119, 0.0408883144649781, 0.0},
      {0.6050857585353100, 0.3540259269997119, 0.0},
      {0.0408883144649781, 0.6050857585353102, 0.0},
      {0.6050857585353100, 0.0408883144649781, 0.0},
      {0.3540259269997119, 0.6050857585353101, 0.0},
      {0.0408883144649780, 0.3540259269997121, 0.0},
      {0.1781710060796274, 0.1781710060796276, 0.0},
      {0.6436579878407451, 0.1781710060796274, 0.0},
      {0.1781710060796276, 0.6436579878407452, 0.0},
      {0.4591943889568276, 0.0816112220863448, 0.0},
      {0.4591943889568276, 0.4591943889568277, 0.0},
      {0.0816112220863447, 0.4591943889568278, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.4925124498658742, 0.0149751002682516, 0.0},
      {0.4925124498658742, 0.4925124498658743, 0.0},
      {0.0149751002682516, 0.4925124498658744, 0.0},
      {0.2418941040068926, 0.0149236389074385, 0.0},
      {0.7431822570856689, 0.2418941040068926, 0.0},
      {0.0149236389074385, 0.7431822570856692, 0.0},
      {0.7431822570856689, 0.0149236389074385, 0.0},
      {0.2418941040068927, 0.7431822570856690, 0.0},
      {0.0149236389074383, 0.2418941040068929, 0.0},
      {0.3646204143387100, 0.0020691038491024, 0.0},
      {0.6333104818121875, 0.3646204143387101, 0.0},
      {0.0020691038491024, 0.6333104818121877, 0.0},
      {0.6333104818121876, 0.0020691038491024, 0.0},
      {0.3646204143387101, 0.6333104818121877, 0.0},
      {0.0020691038491022, 0.3646204143387103, 0.0}},
      {0.0014628462439401,
      0.0014628462439401,
      0.0014628462439401,
      0.0014628462439401,
      0.0014628462439401,
      0.0014628462439401,
      0.0035546968113975,
      0.0035546968113975,
      0.0035546968113975,
      0.0016636944202970,
      0.0016636944202970,
      0.0016636944202970,
      0.0016636944202970,
      0.0016636944202970,
      0.0016636944202970,
      0.0048477595408121,
      0.0048477595408121,
      0.0048477595408121,
      0.0048477595408121,
      0.0048477595408121,
      0.0048477595408121,
      0.0131731323537227,
      0.0131731323537227,
      0.0131731323537227,
      0.0131731323537227,
      0.0131731323537227,
      0.0131731323537227,
      0.0076174782585024,
      0.0076174782585024,
      0.0076174782585024,
      0.0008825962091543,
      0.0008825962091543,
      0.0008825962091543,
      0.0158764272937650,
      0.0158764272937650,
      0.0158764272937650,
      0.0090540372952153,
      0.0090540372952153,
      0.0090540372952153,
      0.0090540372952153,
      0.0090540372952153,
      0.0090540372952153,
      0.0157686793226198,
      0.0157686793226198,
      0.0157686793226198,
      0.0080511047304697,
      0.0080511047304697,
      0.0080511047304697,
      0.0080511047304697,
      0.0080511047304697,
      0.0080511047304697,
      0.0123259905267924,
      0.0123259905267924,
      0.0123259905267924,
      0.0114917854885616,
      0.0114917854885616,
      0.0114917854885616,
      0.0172345804254527,
      0.0051609410912095,
      0.0051609410912095,
      0.0051609410912095,
      0.0042279624195467,
      0.0042279624195467,
      0.0042279624195467,
      0.0042279624195467,
      0.0042279624195467,
      0.0042279624195467,
      0.0016410687574199,
      0.0016410687574199,
      0.0016410687574199,
      0.0016410687574199,
      0.0016410687574199,
      0.0016410687574199}
    },
    {
      79,
      {{0.1862949977445409, 0.1862949977445411, 0.0},
      {0.6274100045109180, 0.1862949977445409, 0.0},
      {0.1862949977445411, 0.6274100045109182, 0.0},
      {0.0373108805988846, 0.0373108805988849, 0.0},
      {0.9253782388022305, 0.0373108805988846, 0.0},
      {0.0373108805988850, 0.9253782388022306, 0.0},
      {0.4762456115404990, 0.0475087769190020, 0.0},
      {0.4762456115404990, 0.4762456115404990, 0.0},
      {0.0475087769190019, 0.4762456115404993, 0.0},
      {0.0640905856084340, 0.0048549376076238, 0.0},
      {0.9310544767839422, 0.0640905856084340, 0.0},
      {0.0048549376076240, 0.9310544767839423, 0.0},
      {0.9310544767839422, 0.0048549376076238, 0.0},
      {0.0640905856084341, 0.9310544767839423, 0.0},
      {0.0048549376076235, 0.0640905856084343, 0.0},
      {0.2156070573900944, 0.1062272047202701, 0.0},
      {0.6781657378896355, 0.2156070573900944, 0.0},
      {0.1062272047202701, 0.6781657378896357, 0.0},
      {0.6781657378896355, 0.1062272047202701, 0.0},
      {0.2156070573900943, 0.6781657378896357, 0.0},
      {0.1062272047202699, 0.2156070573900946, 0.0},
      {0.4455510569559248, 0.1088978860881504, 0.0},
      {0.4455510569559248, 0.4455510569559249, 0.0},
      {0.1088978860881503, 0.4455510569559250, 0.0},
      {0.1591337076570672, 0.0075707805046966, 0.0},
      {0.8332955118382362, 0.1591337076570672, 0.0},
      {0.0075707805046966, 0.8332955118382365, 0.0},
      {0.8332955118382362, 0.0075707805046966, 0.0},
      {0.1591337076570672, 0.8332955118382364, 0.0},
      {0.0075707805046964, 0.1591337076570676, 0.0},
      {0.3178601238357719, 0.1398080719918000, 0.0},
      {0.5423318041724280, 0.3178601238357721, 0.0},
      {0.1398080719918000, 0.5423318041724282, 0.0},
      {0.5423318041724280, 0.1398080719918000, 0.0},
      {0.3178601238357719, 0.5423318041724282, 0.0},
      {0.1398080719917999, 0.3178601238357722, 0.0},
      {0.2545792676733390, 0.2545792676733392, 0.0},
      {0.4908414646533217, 0.2545792676733392, 0.0},
      {0.2545792676733392, 0.4908414646533218, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.1985181322287882, 0.0465603649076644, 0.0},
      {0.7549215028635475, 0.1985181322287882, 0.0},
      {0.0465603649076644, 0.7549215028635476, 0.0},
      {0.7549215028635474, 0.0465603649076644, 0.0},
      {0.1985181322287882, 0.7549215028635476, 0.0},
      {0.0465603649076642, 0.1985181322287885, 0.0},
      {0.3934253478170998, 0.2131493043658003, 0.0},
      {0.3934253478170998, 0.3934253478171000, 0.0},
      {0.2131493043658002, 0.3934253478171000, 0.0},
      {0.0999522962881386, 0.0383636847753747, 0.0},
      {0.8616840189364867, 0.0999522962881386, 0.0},
      {0.0383636847753748, 0.8616840189364868, 0.0},
      {0.8616840189364866, 0.0383636847753747, 0.0},
      {0.0999522962881387, 0.8616840189364868, 0.0},
      {0.0383636847753744, 0.0999522962881389, 0.0},
      {0.4200237588162241, 0.0098315482928026, 0.0},
      {0.5701446928909734, 0.4200237588162241, 0.0},
      {0.0098315482928025, 0.5701446928909736, 0.0},
      {0.5701446928909734, 0.0098315482928026, 0.0},
      {0.4200237588162241, 0.5701446928909735, 0.0},
      {0.0098315482928025, 0.4200237588162243, 0.0},
      {0.3331348173095874, 0.0549874791429869, 0.0},
      {0.6118777035474257, 0.3331348173095875, 0.0},
      {0.0549874791429869, 0.6118777035474259, 0.0},
      {0.6118777035474257, 0.0549874791429869, 0.0},
      {0.3331348173095874, 0.6118777035474258, 0.0},
      {0.0549874791429867, 0.3331348173095877, 0.0},
      {0.2805814114236652, 0.0107372128560111, 0.0},
      {0.7086813757203236, 0.2805814114236653, 0.0},
      {0.0107372128560111, 0.7086813757203239, 0.0},
      {0.7086813757203237, 0.0107372128560111, 0.0},
      {0.2805814114236653, 0.7086813757203237, 0.0},
      {0.0107372128560110, 0.2805814114236655, 0.0},
      {0.0109761410283976, 0.0109761410283979, 0.0},
      {0.9780477179432044, 0.0109761410283976, 0.0},
      {0.0109761410283981, 0.9780477179432047, 0.0},
      {0.1093835967117145, 0.1093835967117148, 0.0},
      {0.7812328065765708, 0.1093835967117145, 0.0},
      {0.1093835967117148, 0.7812328065765709, 0.0}},
      {0.0091734629742529,
      0.0091734629742529,
      0.0091734629742529,
      0.0021612754106656,
      0.0021612754106656,
      0.0021612754106656,
      0.0071018253034085,
      0.0071018253034085,
      0.0071018253034085,
      0.0011298696021258,
      0.0011298696021258,
      0.0011298696021258,
      0.0011298696021258,
      0.0011298696021258,
      0.0011298696021258,
      0.0077226078220993,
      0.0077226078220993,
      0.0077226078220993,
      0.0077226078220993,
      0.0077226078220993,
      0.0077226078220993,
      0.0094523999332324,
      0.0094523999332324,
      0.0094523999332324,
      0.0022028974185585,
      0.0022028974185585,
      0.0022028974185585,
      0.0022028974185585,
      0.0022028974185585,
      0.0022028974185585,
      0.0116917457318277,
      0.0116917457318277,
      0.0116917457318277,
      0.0116917457318277,
      0.0116917457318277,
      0.0116917457318277,
      0.0140832013075203,
      0.0140832013075203,
      0.0140832013075203,
      0.0139101107014531,
      0.0059863985789547,
      0.0059863985789547,
      0.0059863985789547,
      0.0059863985789547,
      0.0059863985789547,
      0.0059863985789547,
      0.0137880506290704,
      0.0137880506290704,
      0.0137880506290704,
      0.0041457115276139,
      0.0041457115276139,
      0.0041457115276139,
      0.0041457115276139,
      0.0041457115276139,
      0.0041457115276139,
      0.0036956815002553,
      0.0036956815002553,
      0.0036956815002553,
      0.0036956815002553,
      0.0036956815002553,
      0.0036956815002553,
      0.0086672255672194,
      0.0086672255672194,
      0.0086672255672194,
      0.0086672255672194,
      0.0086672255672194,
      0.0086672255672194,
      0.0035782002384577,
      0.0035782002384577,
      0.0035782002384577,
      0.0035782002384577,
      0.0035782002384577,
      0.0035782002384577,
      0.0007988407910666,
      0.0007988407910666,
      0.0007988407910666,
      0.0078302307760746,
      0.0078302307760746,
      0.0078302307760746}
    },
    {
      87,
      {{0.2989362353149825, 0.2989362353149826, 0.0},
      {0.4021275293700348, 0.2989362353149826, 0.0},
      {0.2989362353149825, 0.4021275293700349, 0.0},
      {0.2891894960785947, 0.2052955593351616, 0.0},
      {0.5055149445862437, 0.2891894960785948, 0.0},
      {0.2052955593351616, 0.5055149445862438, 0.0},
      {0.5055149445862437, 0.2052955593351616, 0.0},
      {0.2891894960785947, 0.5055149445862438, 0.0},
      {0.2052955593351615, 0.2891894960785949, 0.0},
      {0.4970078754686856, 0.0059842490626289, 0.0},
      {0.4970078754686856, 0.4970078754686856, 0.0},
      {0.0059842490626288, 0.4970078754686857, 0.0},
      {0.2378733825979940, 0.0069318090314681, 0.0},
      {0.7551948083705379, 0.2378733825979940, 0.0},
      {0.0069318090314683, 0.7551948083705379, 0.0},
      {0.7551948083705379, 0.0069318090314681, 0.0},
      {0.2378733825979941, 0.7551948083705379, 0.0},
      {0.0069318090314680, 0.2378733825979943, 0.0},
      {0.4036175865463851, 0.1927648269072298, 0.0},
      {0.4036175865463851, 0.4036175865463852, 0.0},
      {0.1927648269072297, 0.4036175865463852, 0.0},
      {0.3188653107948282, 0.1237794004054928, 0.0},
      {0.5573552887996790, 0.3188653107948283, 0.0},
      {0.1237794004054928, 0.5573552887996791, 0.0},
      {0.5573552887996790, 0.1237794004054928, 0.0},
      {0.3188653107948283, 0.5573552887996791, 0.0},
      {0.1237794004054926, 0.3188653107948285, 0.0},
      {0.2318736253704010, 0.0389913626232203, 0.0},
      {0.7291350120063786, 0.2318736253704010, 0.0},
      {0.0389913626232204, 0.7291350120063789, 0.0},
      {0.7291350120063788, 0.0389913626232203, 0.0},
      {0.2318736253704010, 0.7291350120063789, 0.0},
      {0.0389913626232202, 0.2318736253704013, 0.0},
      {0.1331671229413703, 0.0095362475297106, 0.0},
      {0.8572966295289191, 0.1331671229413703, 0.0},
      {0.0095362475297107, 0.8572966295289193, 0.0},
      {0.8572966295289192, 0.0095362475297106, 0.0},
      {0.1331671229413703, 0.8572966295289193, 0.0},
      {0.0095362475297104, 0.1331671229413706, 0.0},
      {0.3468079798099110, 0.0530521917012168, 0.0},
      {0.6001398284888722, 0.3468079798099111, 0.0},
      {0.0530521917012168, 0.6001398284888723, 0.0},
      {0.6001398284888722, 0.0530521917012168, 0.0},
      {0.3468079798099111, 0.6001398284888722, 0.0},
      {0.0530521917012167, 0.3468079798099113, 0.0},
      {0.1189885776227193, 0.1189885776227196, 0.0},
      {0.7620228447545612, 0.1189885776227193, 0.0},
      {0.1189885776227196, 0.7620228447545613, 0.0},
      {0.1902887180912784, 0.1902887180912786, 0.0},
      {0.6194225638174430, 0.1902887180912785, 0.0},
      {0.1902887180912786, 0.6194225638174431, 0.0},
      {0.2165996231899825, 0.1004580200741145, 0.0},
      {0.6829423567359030, 0.2165996231899825, 0.0},
      {0.1004580200741146, 0.6829423567359031, 0.0},
      {0.6829423567359030, 0.1004580200741145, 0.0},
      {0.2165996231899825, 0.6829423567359032, 0.0},
      {0.1004580200741143, 0.2165996231899828, 0.0},
      {0.4815978686532166, 0.0368042626935668, 0.0},
      {0.4815978686532166, 0.4815978686532166, 0.0},
      {0.0368042626935668, 0.4815978686532168, 0.0},
      {0.4498127917753624, 0.1003744164492753, 0.0},
      {0.4498127917753624, 0.4498127917753625, 0.0},
      {0.1003744164492752, 0.4498127917753625, 0.0},
      {0.1288298079620515, 0.0494510655685406, 0.0},
      {0.8217191264694078, 0.1288298079620515, 0.0},
      {0.0494510655685407, 0.8217191264694079, 0.0},
      {0.8217191264694079, 0.0494510655685406, 0.0},
      {0.1288298079620517, 0.8217191264694079, 0.0},
      {0.0494510655685403, 0.1288298079620518, 0.0},
      {0.0536275755461449, 0.0536275755461451, 0.0},
      {0.8927448489077100, 0.0536275755461448, 0.0},
      {0.0536275755461453, 0.8927448489077101, 0.0},
      {0.3609534080189222, 0.0102546358729245, 0.0},
      {0.6287919561081534, 0.3609534080189222, 0.0},
      {0.0102546358729245, 0.6287919561081535, 0.0},
      {0.6287919561081533, 0.0102546358729245, 0.0},
      {0.3609534080189222, 0.6287919561081534, 0.0},
      {0.0102546358729244, 0.3609534080189224, 0.0},
      {0.0107424564328283, 0.0107424564328286, 0.0},
      {0.9785150871343431, 0.0107424564328283, 0.0},
      {0.0107424564328287, 0.9785150871343433, 0.0},
      {0.0557195650723719, 0.0103019036434239, 0.0},
      {0.9339785312842042, 0.0557195650723719, 0.0},
      {0.0103019036434240, 0.9339785312842044, 0.0},
      {0.9339785312842040, 0.0103019036434239, 0.0},
      {0.0557195650723720, 0.9339785312842044, 0.0},
      {0.0103019036434236, 0.0557195650723723, 0.0}},
      {0.0107255609645661,
      0.0107255609645661,
      0.0107255609645661,
      0.0087477080778816,
      0.0087477080778816,
      0.0087477080778816,
      0.0087477080778816,
      0.0087477080778816,
      0.0087477080778816,
      0.0022189148485329,
      0.0022189148485329,
      0.0022189148485329,
      0.0021030601440749,
      0.0021030601440749,
      0.0021030601440749,
      0.0021030601440749,
      0.0021030601440749,
      0.0021030601440749,
      0.0115003523266419,
      0.0115003523266419,
      0.0115003523266419,
      0.0092237424239664,
      0.0092237424239664,
      0.0092237424239664,
      0.0092237424239664,
      0.0092237424239664,
      0.0092237424239664,
      0.0052349520926624,
      0.0052349520926624,
      0.0052349520926624,
      0.0052349520926624,
      0.0052349520926624,
      0.0052349520926624,
      0.0022404065609508,
      0.0022404065609508,
      0.0022404065609508,
      0.0022404065609508,
      0.0022404065609508,
      0.0022404065609508,
      0.0072501529594855,
      0.0072501529594855,
      0.0072501529594855,
      0.0072501529594855,
      0.0072501529594855,
      0.0072501529594855,
      0.0068280162261151,
      0.0068280162261151,
      0.0068280162261151,
      0.0097276209303754,
      0.0097276209303754,
      0.0097276209303754,
      0.0079520183527140,
      0.0079520183527140,
      0.0079520183527140,
      0.0079520183527140,
      0.0079520183527140,
      0.0079520183527140,
      0.0061072050816922,
      0.0061072050816922,
      0.0061072050816922,
      0.0098072376139120,
      0.0098072376139120,
      0.0098072376139120,
      0.0049059859112752,
      0.0049059859112752,
      0.0049059859112752,
      0.0049059859112752,
      0.0049059859112752,
      0.0049059859112752,
      0.0035760425506419,
      0.0035760425506419,
      0.0035760425506419,
      0.0034199424289672,
      0.0034199424289672,
      0.0034199424289672,
      0.0034199424289672,
      0.0034199424289672,
      0.0034199424289672,
      0.0007543496361893,
      0.0007543496361893,
      0.0007543496361893,
      0.0016327142920220,
      0.0016327142920220,
      0.0016327142920220,
      0.0016327142920220,
      0.0016327142920220,
      0.0016327142920220}
    },
    {
      96,
      {{0.3851845246273021, 0.2296309507453958, 0.0},
      {0.3851845246273021, 0.3851845246273022, 0.0},
      {0.2296309507453957, 0.3851845246273022, 0.0},
      {0.4577694113676721, 0.0844611772646559, 0.0},
      {0.4577694113676721, 0.4577694113676721, 0.0},
      {0.0844611772646559, 0.4577694113676722, 0.0},
      {0.0698421694674436, 0.0078762822215824, 0.0},
      {0.9222815483109740, 0.0698421694674436, 0.0},
      {0.0078762822215824, 0.9222815483109743, 0.0},
      {0.9222815483109740, 0.0078762822215824, 0.0},
      {0.0698421694674436, 0.9222815483109743, 0.0},
      {0.0078762822215821, 0.0698421694674439, 0.0},
      {0.2945582590299500, 0.2945582590299502, 0.0},
      {0.4108834819400997, 0.2945582590299502, 0.0},
      {0.2945582590299501, 0.4108834819400998, 0.0},
      {0.1885105236302838, 0.1885105236302840, 0.0},
      {0.6229789527394320, 0.1885105236302839, 0.0},
      {0.1885105236302841, 0.6229789527394323, 0.0},
      {0.4219818887935349, 0.1560362224129302, 0.0},
      {0.4219818887935349, 0.4219818887935350, 0.0},
      {0.1560362224129301, 0.4219818887935351, 0.0},
      {0.0903988311664077, 0.0447522843483359, 0.0},
      {0.8648488844852564, 0.0903988311664078, 0.0},
      {0.0447522843483360, 0.8648488844852564, 0.0},
      {0.8648488844852564, 0.0447522843483359, 0.0},
      {0.0903988311664079, 0.8648488844852564, 0.0},
      {0.0447522843483357, 0.0903988311664081, 0.0},
      {0.4113417640205587, 0.0382752347008638, 0.0},
      {0.5503830012785774, 0.4113417640205588, 0.0},
      {0.0382752347008638, 0.5503830012785776, 0.0},
      {0.5503830012785774, 0.0382752347008638, 0.0},
      {0.4113417640205588, 0.5503830012785775, 0.0},
      {0.0382752347008637, 0.4113417640205590, 0.0},
      {0.3321061050074464, 0.1027470759869314, 0.0},
      {0.5651468190056221, 0.3321061050074464, 0.0},
      {0.1027470759869314, 0.5651468190056224, 0.0},
      {0.5651468190056222, 0.1027470759869314, 0.0},
      {0.3321061050074463, 0.5651468190056224, 0.0},
      {0.1027470759869313, 0.3321061050074466, 0.0},
      {0.3625762804324673, 0.0074002412347108, 0.0},
      {0.6300234783328220, 0.3625762804324673, 0.0},
      {0.0074002412347108, 0.6300234783328221, 0.0},
      {0.6300234783328220, 0.0074002412347108, 0.0},
      {0.3625762804324673, 0.6300234783328221, 0.0},
      {0.0074002412347107, 0.3625762804324675, 0.0},
      {0.2900668241166688, 0.1910812979667202, 0.0},
      {0.5188518779166110, 0.2900668241166688, 0.0},
      {0.1910812979667201, 0.5188518779166111, 0.0},
      {0.5188518779166110, 0.1910812979667202, 0.0},
      {0.2900668241166688, 0.5188518779166111, 0.0},
      {0.1910812979667200, 0.2900668241166690, 0.0},
      {0.4961611784097086, 0.0076776431805828, 0.0},
      {0.4961611784097086, 0.4961611784097087, 0.0},
      {0.0076776431805827, 0.4961611784097089, 0.0},
      {0.2879318028241718, 0.0439916453934559, 0.0},
      {0.6680765517823724, 0.2879318028241719, 0.0},
      {0.0439916453934559, 0.6680765517823725, 0.0},
      {0.6680765517823722, 0.0439916453934559, 0.0},
      {0.2879318028241718, 0.6680765517823725, 0.0},
      {0.0439916453934557, 0.2879318028241721, 0.0},
      {0.2167869333649411, 0.1086899418626720, 0.0},
      {0.6745231247723867, 0.2167869333649412, 0.0},
      {0.1086899418626721, 0.6745231247723870, 0.0},
      {0.6745231247723867, 0.1086899418626720, 0.0},
      {0.2167869333649412, 0.6745231247723870, 0.0},
      {0.1086899418626718, 0.2167869333649414, 0.0},
      {0.1458737198735252, 0.0091447113749641, 0.0},
      {0.8449815687515108, 0.1458737198735252, 0.0},
      {0.0091447113749641, 0.8449815687515109, 0.0},
      {0.8449815687515108, 0.0091447113749641, 0.0},
      {0.1458737198735252, 0.8449815687515109, 0.0},
      {0.0091447113749638, 0.1458737198735255, 0.0},
      {0.1762974348245001, 0.0482549241146414, 0.0},
      {0.7754476410608585, 0.1762974348245001, 0.0},
      {0.0482549241146414, 0.7754476410608587, 0.0},
      {0.7754476410608585, 0.0482549241146414, 0.0},
      {0.1762974348245001, 0.7754476410608587, 0.0},
      {0.0482549241146412, 0.1762974348245004, 0.0},
      {0.2439906460394931, 0.0091639092481852, 0.0},
      {0.7468454447123216, 0.2439906460394931, 0.0},
      {0.0091639092481853, 0.7468454447123218, 0.0},
      {0.7468454447123217, 0.0091639092481852, 0.0},
      {0.2439906460394931, 0.7468454447123218, 0.0},
      {0.0091639092481851, 0.2439906460394934, 0.0},
      {0.0291084706708074, 0.0291084706708077, 0.0},
      {0.9417830586583850, 0.0291084706708073, 0.0},
      {0.0291084706708079, 0.9417830586583850, 0.0},
      {0.1154315382192049, 0.1154315382192051, 0.0},
      {0.7691369235615900, 0.1154315382192049, 0.0},
      {0.1154315382192052, 0.7691369235615901, 0.0},
      {0.0179343210529390, 0.0017984649889484, 0.0},
      {0.9802672139581126, 0.0179343210529390, 0.0},
      {0.0017984649889486, 0.9802672139581127, 0.0},
      {0.9802672139581127, 0.0017984649889484, 0.0},
      {0.0179343210529391, 0.9802672139581129, 0.0},
      {0.0017984649889481, 0.0179343210529393, 0.0}},
      {0.0067465419418053,
      0.0067465419418053,
      0.0067465419418053,
      0.0069306997621171,
      0.0069306997621171,
      0.0069306997621171,
      0.0012977192371156,
      0.0012977192371156,
      0.0012977192371156,
      0.0012977192371156,
      0.0012977192371156,
      0.0012977192371156,
      0.0105378819787261,
      0.0105378819787261,
      0.0105378819787261,
      0.0080106495625745,
      0.0080106495625745,
      0.0080106495625745,
      0.0094265462769207,
      0.0094265462769207,
      0.0094265462769207,
      0.0037587889088942,
      0.0037587889088942,
      0.0037587889088942,
      0.0037587889088942,
      0.0037587889088942,
      0.0037587889088942,
      0.0055986567359814,
      0.0055986567359814,
      0.0055986567359814,
      0.0055986567359814,
      0.0055986567359814,
      0.0055986567359814,
      0.0088595467447551,
      0.0088595467447551,
      0.0088595467447551,
      0.0088595467447551,
      0.0088595467447551,
      0.0088595467447551,
      0.0024521301987785,
      0.0024521301987785,
      0.0024521301987785,
      0.0024521301987785,
      0.0024521301987785,
      0.0024521301987785,
      0.0108532097777545,
      0.0108532097777545,
      0.0108532097777545,
      0.0108532097777545,
      0.0108532097777545,
      0.0108532097777545,
      0.0026446698329922,
      0.0026446698329922,
      0.0026446698329922,
      0.0058311114336715,
      0.0058311114336715,
      0.0058311114336715,
      0.0058311114336715,
      0.0058311114336715,
      0.0058311114336715,
      0.0078550813112852,
      0.0078550813112852,
      0.0078550813112852,
      0.0078550813112852,
      0.0078550813112852,
      0.0078550813112852,
      0.0020533435357878,
      0.0020533435357878,
      0.0020533435357878,
      0.0020533435357878,
      0.0020533435357878,
      0.0020533435357878,
      0.0052817924838735,
      0.0052817924838735,
      0.0052817924838735,
      0.0052817924838735,
      0.0052817924838735,
      0.0052817924838735,
      0.0025270384487923,
      0.0025270384487923,
      0.0025270384487923,
      0.0025270384487923,
      0.0025270384487923,
      0.0025270384487923,
      0.0017845545829282,
      0.0017845545829282,
      0.0017845545829282,
      0.0072078565640523,
      0.0072078565640523,
      0.0072078565640523,
      0.0003202142655857,
      0.0003202142655857,
      0.0003202142655857,
      0.0003202142655857,
      0.0003202142655857,
      0.0003202142655857}
    },
    {
      103,
      {{0.1595037989247572, 0.0238702536543536, 0.0},
      {0.8166259474208891, 0.1595037989247572, 0.0},
      {0.0238702536543537, 0.8166259474208892, 0.0},
      {0.8166259474208892, 0.0238702536543536, 0.0},
      {0.1595037989247573, 0.8166259474208892, 0.0},
      {0.0238702536543534, 0.1595037989247576, 0.0},
      {0.1141013603223645, 0.0051898217608445, 0.0},
      {0.8807088179167908, 0.1141013603223645, 0.0},
      {0.0051898217608446, 0.8807088179167911, 0.0},
      {0.8807088179167909, 0.0051898217608445, 0.0},
      {0.1141013603223646, 0.8807088179167911, 0.0},
      {0.0051898217608443, 0.1141013603223648, 0.0},
      {0.0390072687570320, 0.0390072687570323, 0.0},
      {0.9219854624859357, 0.0390072687570320, 0.0},
      {0.0390072687570324, 0.9219854624859359, 0.0},
      {0.0955398781717349, 0.0327410291887064, 0.0},
      {0.8717190926395586, 0.0955398781717349, 0.0},
      {0.0327410291887065, 0.8717190926395588, 0.0},
      {0.8717190926395587, 0.0327410291887064, 0.0},
      {0.0955398781717349, 0.8717190926395588, 0.0},
      {0.0327410291887062, 0.0955398781717352, 0.0},
      {0.3111622680517019, 0.0024475998559664, 0.0},
      {0.6863901320923316, 0.3111622680517020, 0.0},
      {0.0024475998559664, 0.6863901320923319, 0.0},
      {0.6863901320923317, 0.0024475998559664, 0.0},
      {0.3111622680517019, 0.6863901320923319, 0.0},
      {0.0024475998559662, 0.3111622680517022, 0.0},
      {0.2056172320580521, 0.0087252895853085, 0.0},
      {0.7856574783566395, 0.2056172320580521, 0.0},
      {0.0087252895853085, 0.7856574783566396, 0.0},
      {0.7856574783566393, 0.0087252895853085, 0.0},
      {0.2056172320580521, 0.7856574783566396, 0.0},
      {0.0087252895853083, 0.2056172320580524, 0.0},
      {0.0472616294497253, 0.0071625399102445, 0.0},
      {0.9455758306400303, 0.0472616294497253, 0.0},
      {0.0071625399102447, 0.9455758306400303, 0.0},
      {0.9455758306400303, 0.0071625399102445, 0.0},
      {0.0472616294497255, 0.9455758306400303, 0.0},
      {0.0071625399102442, 0.0472616294497257, 0.0},
      {0.3585095935696251, 0.0685269541872130, 0.0},
      {0.5729634522431619, 0.3585095935696251, 0.0},
      {0.0685269541872131, 0.5729634522431620, 0.0},
      {0.5729634522431619, 0.0685269541872130, 0.0},
      {0.3585095935696250, 0.5729634522431620, 0.0},
      {0.0685269541872129, 0.3585095935696253, 0.0},
      {0.4803288773373085, 0.0393422453253830, 0.0},
      {0.4803288773373085, 0.4803288773373086, 0.0},
      {0.0393422453253830, 0.4803288773373087, 0.0},
      {0.2404827720350127, 0.1017283293272843, 0.0},
      {0.6577888986377031, 0.2404827720350127, 0.0},
      {0.1017283293272843, 0.6577888986377033, 0.0},
      {0.6577888986377032, 0.1017283293272843, 0.0},
      {0.2404827720350126, 0.6577888986377033, 0.0},
      {0.1017283293272841, 0.2404827720350129, 0.0},
      {0.0868410482076331, 0.0868410482076333, 0.0},
      {0.8263179035847337, 0.0868410482076330, 0.0},
      {0.0868410482076334, 0.8263179035847338, 0.0},
      {0.3943235060115415, 0.2113529879769170, 0.0},
      {0.3943235060115415, 0.3943235060115416, 0.0},
      {0.2113529879769169, 0.3943235060115416, 0.0},
      {0.2662513178772472, 0.2662513178772474, 0.0},
      {0.4674973642455053, 0.2662513178772473, 0.0},
      {0.2662513178772474, 0.4674973642455055, 0.0},
      {0.1729323031292240, 0.0583515752375155, 0.0},
      {0.7687161216332605, 0.1729323031292239, 0.0},
      {0.0583515752375156, 0.7687161216332608, 0.0},
      {0.7687161216332605, 0.0583515752375155, 0.0},
      {0.1729323031292240, 0.7687161216332608, 0.0},
      {0.0583515752375153, 0.1729323031292242, 0.0},
      {0.3163043076538381, 0.1548301554055163, 0.0},
      {0.5288655369406456, 0.3163043076538382, 0.0},
      {0.1548301554055163, 0.5288655369406458, 0.0},
      {0.5288655369406456, 0.1548301554055163, 0.0},
      {0.3163043076538382, 0.5288655369406458, 0.0},
      {0.1548301554055161, 0.3163043076538383, 0.0},
      {0.1371293873116475, 0.1371293873116478, 0.0},
      {0.7257412253767047, 0.1371293873116475, 0.0},
      {0.1371293873116478, 0.7257412253767048, 0.0},
      {0.4989594312095863, 0.0020811375808274, 0.0},
      {0.4989594312095863, 0.4989594312095864, 0.0},
      {0.0020811375808273, 0.4989594312095865, 0.0},
      {0.3977585768030076, 0.0147589697299452, 0.0},
      {0.5874824534670471, 0.3977585768030077, 0.0},
      {0.0147589697299451, 0.5874824534670474, 0.0},
      {0.5874824534670471, 0.0147589697299452, 0.0},
      {0.3977585768030076, 0.5874824534670473, 0.0},
      {0.0147589697299450, 0.3977585768030079, 0.0},
      {0.4446924421277275, 0.1106151157445450, 0.0},
      {0.4446924421277275, 0.4446924421277276, 0.0},
      {0.1106151157445449, 0.4446924421277277, 0.0},
      {0.1987498063965361, 0.1987498063965364, 0.0},
      {0.6025003872069274, 0.1987498063965362, 0.0},
      {0.1987498063965363, 0.6025003872069277, 0.0},
      {0.2787941698141023, 0.0329937081925328, 0.0},
      {0.6882121219933649, 0.2787941698141023, 0.0},
      {0.0329937081925328, 0.6882121219933651, 0.0},
      {0.6882121219933649, 0.0329937081925328, 0.0},
      {0.2787941698141023, 0.6882121219933650, 0.0},
      {0.0329937081925327, 0.2787941698141025, 0.0},
      {0.0090164402055982, 0.0090164402055985, 0.0},
      {0.9819671195888033, 0.0090164402055982, 0.0},
      {0.0090164402055986, 0.9819671195888034, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0}},
      {0.0012640830276911,
      0.0012640830276911,
      0.0012640830276911,
      0.0012640830276911,
      0.0012640830276911,
      0.0012640830276911,
      0.0011125098648622,
      0.0011125098648622,
      0.0011125098648622,
      0.0011125098648622,
      0.0011125098648622,
      0.0011125098648622,
      0.0019578701295165,
      0.0019578701295165,
      0.0019578701295165,
      0.0026640152155974,
      0.0026640152155974,
      0.0026640152155974,
      0.0026640152155974,
      0.0026640152155974,
      0.0026640152155974,
      0.0011405518381279,
      0.0011405518381279,
      0.0011405518381279,
      0.0011405518381279,
      0.0011405518381279,
      0.0011405518381279,
      0.0020573751722081,
      0.0020573751722081,
      0.0020573751722081,
      0.0020573751722081,
      0.0020573751722081,
      0.0020573751722081,
      0.0009762956639454,
      0.0009762956639454,
      0.0009762956639454,
      0.0009762956639454,
      0.0009762956639454,
      0.0009762956639454,
      0.0074905566965996,
      0.0074905566965996,
      0.0074905566965996,
      0.0074905566965996,
      0.0074905566965996,
      0.0074905566965996,
      0.0056989446339004,
      0.0056989446339004,
      0.0056989446339004,
      0.0080606208185086,
      0.0080606208185086,
      0.0080606208185086,
      0.0080606208185086,
      0.0080606208185086,
      0.0080606208185086,
      0.0044799585127568,
      0.0044799585127568,
      0.0044799585127568,
      0.0118373042315640,
      0.0118373042315640,
      0.0118373042315640,
      0.0119039314437499,
      0.0119039314437499,
      0.0119039314437499,
      0.0052351282465650,
      0.0052351282465650,
      0.0052351282465650,
      0.0052351282465650,
      0.0052351282465650,
      0.0052351282465650,
      0.0104221979294844,
      0.0104221979294844,
      0.0104221979294844,
      0.0104221979294844,
      0.0104221979294844,
      0.0104221979294844,
      0.0072797246963709,
      0.0072797246963709,
      0.0072797246963709,
      0.0012037723020907,
      0.0012037723020907,
      0.0012037723020907,
      0.0035488894172609,
      0.0035488894172609,
      0.0035488894172609,
      0.0035488894172609,
      0.0035488894172609,
      0.0035488894172609,
      0.0094759753346694,
      0.0094759753346694,
      0.0094759753346694,
      0.0099676389400525,
      0.0099676389400525,
      0.0099676389400525,
      0.0050877873283535,
      0.0050877873283535,
      0.0050877873283535,
      0.0050877873283535,
      0.0050877873283535,
      0.0050877873283535,
      0.0005326806164147,
      0.0005326806164147,
      0.0005326806164147,
      0.0126265301615181}
    },
    {
      112,
      {{0.3333333333333333, 0.3333333333333334, 0.0},
      {0.4188909749106027, 0.1622180501787945, 0.0},
      {0.4188909749106027, 0.4188909749106028, 0.0},
      {0.1622180501787944, 0.4188909749106029, 0.0},
      {0.1623606337169263, 0.1623606337169265, 0.0},
      {0.6752787325661472, 0.1623606337169263, 0.0},
      {0.1623606337169265, 0.6752787325661475, 0.0},
      {0.2414797600735940, 0.1703672824624437, 0.0},
      {0.5881529574639622, 0.2414797600735941, 0.0},
      {0.1703672824624437, 0.5881529574639625, 0.0},
      {0.5881529574639622, 0.1703672824624437, 0.0},
      {0.2414797600735940, 0.5881529574639625, 0.0},
      {0.1703672824624436, 0.2414797600735942, 0.0},
      {0.3289758089242263, 0.1697597958607361, 0.0},
      {0.5012643952150375, 0.3289758089242264, 0.0},
      {0.1697597958607360, 0.5012643952150377, 0.0},
      {0.5012643952150376, 0.1697597958607361, 0.0},
      {0.3289758089242264, 0.5012643952150377, 0.0},
      {0.1697597958607359, 0.3289758089242266, 0.0},
      {0.0409856290011169, 0.0409856290011173, 0.0},
      {0.9180287419977657, 0.0409856290011169, 0.0},
      {0.0409856290011174, 0.9180287419977659, 0.0},
      {0.0067312708878883, 0.0067312708878885, 0.0},
      {0.9865374582242232, 0.0067312708878882, 0.0},
      {0.0067312708878887, 0.9865374582242232, 0.0},
      {0.0931674097798811, 0.0383182258210194, 0.0},
      {0.8685143643990995, 0.0931674097798812, 0.0},
      {0.0383182258210195, 0.8685143643990996, 0.0},
      {0.8685143643990995, 0.0383182258210194, 0.0},
      {0.0931674097798812, 0.8685143643990996, 0.0},
      {0.0383182258210191, 0.0931674097798815, 0.0},
      {0.3945202798001943, 0.0926564815207575, 0.0},
      {0.5128232386790481, 0.3945202798001944, 0.0},
      {0.0926564815207574, 0.5128232386790483, 0.0},
      {0.5128232386790481, 0.0926564815207575, 0.0},
      {0.3945202798001944, 0.5128232386790482, 0.0},
      {0.0926564815207574, 0.3945202798001946, 0.0},
      {0.1626774163944774, 0.0411887142484754, 0.0},
      {0.7961338693570472, 0.1626774163944774, 0.0},
      {0.0411887142484756, 0.7961338693570472, 0.0},
      {0.7961338693570472, 0.0411887142484754, 0.0},
      {0.1626774163944775, 0.7961338693570472, 0.0},
      {0.0411887142484752, 0.1626774163944777, 0.0},
      {0.2535890142188795, 0.0395709049701581, 0.0},
      {0.7068400808109625, 0.2535890142188795, 0.0},
      {0.0395709049701581, 0.7068400808109626, 0.0},
      {0.7068400808109625, 0.0395709049701581, 0.0},
      {0.2535890142188795, 0.7068400808109626, 0.0},
      {0.0395709049701579, 0.2535890142188797, 0.0},
      {0.3622522413177913, 0.0385927001748961, 0.0},
      {0.5991550585073127, 0.3622522413177914, 0.0},
      {0.0385927001748961, 0.5991550585073128, 0.0},
      {0.5991550585073125, 0.0385927001748961, 0.0},
      {0.3622522413177913, 0.5991550585073127, 0.0},
      {0.0385927001748960, 0.3622522413177915, 0.0},
      {0.2816225777061608, 0.0945349617365990, 0.0},
      {0.6238424605572400, 0.2816225777061609, 0.0},
      {0.0945349617365990, 0.6238424605572404, 0.0},
      {0.6238424605572401, 0.0945349617365990, 0.0},
      {0.2816225777061608, 0.6238424605572404, 0.0},
      {0.0945349617365988, 0.2816225777061611, 0.0},
      {0.3832726649926592, 0.0073879946322942, 0.0},
      {0.6093393403750466, 0.3832726649926593, 0.0},
      {0.0073879946322942, 0.6093393403750468, 0.0},
      {0.6093393403750464, 0.0073879946322942, 0.0},
      {0.3832726649926592, 0.6093393403750467, 0.0},
      {0.0073879946322941, 0.3832726649926595, 0.0},
      {0.2737503525162605, 0.0075460031623128, 0.0},
      {0.7187036443214266, 0.2737503525162605, 0.0},
      {0.0075460031623129, 0.7187036443214270, 0.0},
      {0.7187036443214266, 0.0075460031623128, 0.0},
      {0.2737503525162606, 0.7187036443214267, 0.0},
      {0.0075460031623126, 0.2737503525162608, 0.0},
      {0.4962552776757351, 0.0074894446485298, 0.0},
      {0.4962552776757351, 0.4962552776757352, 0.0},
      {0.0074894446485297, 0.4962552776757354, 0.0},
      {0.0941213427973660, 0.0072345584577821, 0.0},
      {0.8986440987448518, 0.0941213427973660, 0.0},
      {0.0072345584577822, 0.8986440987448521, 0.0},
      {0.8986440987448518, 0.0072345584577821, 0.0},
      {0.0941213427973661, 0.8986440987448520, 0.0},
      {0.0072345584577819, 0.0941213427973664, 0.0},
      {0.2642313154382725, 0.2642313154382727, 0.0},
      {0.4715373691234548, 0.2642313154382726, 0.0},
      {0.2642313154382727, 0.4715373691234549, 0.0},
      {0.1803961518867657, 0.0955662695273653, 0.0},
      {0.7240375785858689, 0.1803961518867657, 0.0},
      {0.0955662695273654, 0.7240375785858691, 0.0},
      {0.7240375785858689, 0.0955662695273653, 0.0},
      {0.1803961518867657, 0.7240375785858693, 0.0},
      {0.0955662695273650, 0.1803961518867660, 0.0},
      {0.4806125617925032, 0.0387748764149935, 0.0},
      {0.4806125617925033, 0.4806125617925033, 0.0},
      {0.0387748764149934, 0.4806125617925034, 0.0},
      {0.1747373462828057, 0.0079879218808480, 0.0},
      {0.8172747318363464, 0.1747373462828057, 0.0},
      {0.0079879218808480, 0.8172747318363466, 0.0},
      {0.8172747318363462, 0.0079879218808480, 0.0},
      {0.1747373462828057, 0.8172747318363466, 0.0},
      {0.0079879218808478, 0.1747373462828060, 0.0},
      {0.0963284955992151, 0.0963284955992153, 0.0},
      {0.8073430088015696, 0.0963284955992151, 0.0},
      {0.0963284955992154, 0.8073430088015697, 0.0},
      {0.0372914720512912, 0.0080749108702088, 0.0},
      {0.9546336170785000, 0.0372914720512912, 0.0},
      {0.0080749108702090, 0.9546336170785000, 0.0},
      {0.9546336170785000, 0.0080749108702088, 0.0},
      {0.0372914720512914, 0.9546336170785000, 0.0},
      {0.0080749108702085, 0.0372914720512916, 0.0},
      {0.3753529267020863, 0.2492941465958275, 0.0},
      {0.3753529267020863, 0.3753529267020864, 0.0},
      {0.2492941465958274, 0.3753529267020864, 0.0}},
      {0.0062728449228002,
      0.0065552663509426,
      0.0065552663509426,
      0.0065552663509426,
      0.0051895080282001,
      0.0051895080282001,
      0.0051895080282001,
      0.0070725229032424,
      0.0070725229032424,
      0.0070725229032424,
      0.0070725229032424,
      0.0070725229032424,
      0.0070725229032424,
      0.0076372213006623,
      0.0076372213006623,
      0.0076372213006623,
      0.0076372213006623,
      0.0076372213006623,
      0.0076372213006623,
      0.0019168498654646,
      0.0019168498654646,
      0.0019168498654646,
      0.0003086272527483,
      0.0003086272527483,
      0.0003086272527483,
      0.0026831357270839,
      0.0026831357270839,
      0.0026831357270839,
      0.0026831357270839,
      0.0026831357270839,
      0.0026831357270839,
      0.0075159271748706,
      0.0075159271748706,
      0.0075159271748706,
      0.0075159271748706,
      0.0075159271748706,
      0.0075159271748706,
      0.0036020670873987,
      0.0036020670873987,
      0.0036020670873987,
      0.0036020670873987,
      0.0036020670873987,
      0.0036020670873987,
      0.0044524384640818,
      0.0044524384640818,
      0.0044524384640818,
      0.0044524384640818,
      0.0044524384640818,
      0.0044524384640818,
      0.0049736259378412,
      0.0049736259378412,
      0.0049736259378412,
      0.0049736259378412,
      0.0049736259378412,
      0.0049736259378412,
      0.0071761757890788,
      0.0071761757890788,
      0.0071761757890788,
      0.0071761757890788,
      0.0071761757890788,
      0.0071761757890788,
      0.0021210746334019,
      0.0021210746334019,
      0.0021210746334019,
      0.0021210746334019,
      0.0021210746334019,
      0.0021210746334019,
      0.0020406375385583,
      0.0020406375385583,
      0.0020406375385583,
      0.0020406375385583,
      0.0020406375385583,
      0.0020406375385583,
      0.0021716233610854,
      0.0021716233610854,
      0.0021716233610854,
      0.0012946061911990,
      0.0012946061911990,
      0.0012946061911990,
      0.0012946061911990,
      0.0012946061911990,
      0.0012946061911990,
      0.0102600043357549,
      0.0102600043357549,
      0.0102600043357549,
      0.0059217810712716,
      0.0059217810712716,
      0.0059217810712716,
      0.0059217810712716,
      0.0059217810712716,
      0.0059217810712716,
      0.0051762473854263,
      0.0051762473854263,
      0.0051762473854263,
      0.0018536133821232,
      0.0018536133821232,
      0.0018536133821232,
      0.0018536133821232,
      0.0018536133821232,
      0.0018536133821232,
      0.0050136965336945,
      0.0050136965336945,
      0.0050136965336945,
      0.0008984737927233,
      0.0008984737927233,
      0.0008984737927233,
      0.0008984737927233,
      0.0008984737927233,
      0.0008984737927233,
      0.0094972932586763,
      0.0094972932586763,
      0.0094972932586763}
    },
    {
      120,
      {{0.3876420304045634, 0.2247159391908732, 0.0},
      {0.3876420304045634, 0.3876420304045635, 0.0},
      {0.2247159391908731, 0.3876420304045636, 0.0},
      {0.2110045080614966, 0.2110045080614968, 0.0},
      {0.5779909838770066, 0.2110045080614966, 0.0},
      {0.2110045080614967, 0.5779909838770069, 0.0},
      {0.4404169274793433, 0.0018188666342744, 0.0},
      {0.5577642058863823, 0.4404169274793434, 0.0},
      {0.0018188666342743, 0.5577642058863825, 0.0},
      {0.5577642058863823, 0.0018188666342744, 0.0},
      {0.4404169274793434, 0.5577642058863823, 0.0},
      {0.0018188666342743, 0.4404169274793436, 0.0},
      {0.2994923158045084, 0.2994923158045086, 0.0},
      {0.4010153683909830, 0.2994923158045085, 0.0},
      {0.2994923158045084, 0.4010153683909831, 0.0},
      {0.1590079061973279, 0.0369601415796715, 0.0},
      {0.8040319522230005, 0.1590079061973279, 0.0},
      {0.0369601415796716, 0.8040319522230007, 0.0},
      {0.8040319522230007, 0.0369601415796715, 0.0},
      {0.1590079061973279, 0.8040319522230007, 0.0},
      {0.0369601415796713, 0.1590079061973282, 0.0},
      {0.0372229259924406, 0.0372229259924409, 0.0},
      {0.9255541480151184, 0.0372229259924406, 0.0},
      {0.0372229259924410, 0.9255541480151186, 0.0},
      {0.1773537967572529, 0.0788580680056353, 0.0},
      {0.7437881352371117, 0.1773537967572529, 0.0},
      {0.0788580680056354, 0.7437881352371120, 0.0},
      {0.7437881352371118, 0.0788580680056353, 0.0},
      {0.1773537967572530, 0.7437881352371120, 0.0},
      {0.0788580680056351, 0.1773537967572532, 0.0},
      {0.1451092435745002, 0.1451092435745005, 0.0},
      {0.7097815128509992, 0.1451092435745003, 0.0},
      {0.1451092435745006, 0.7097815128509993, 0.0},
      {0.2700667358209594, 0.0688475294314979, 0.0},
      {0.6610857347475426, 0.2700667358209594, 0.0},
      {0.0688475294314979, 0.6610857347475428, 0.0},
      {0.6610857347475427, 0.0688475294314979, 0.0},
      {0.2700667358209594, 0.6610857347475428, 0.0},
      {0.0688475294314977, 0.2700667358209596, 0.0},
      {0.3413910330211499, 0.1159998076409602, 0.0},
      {0.5426091593378899, 0.3413910330211499, 0.0},
      {0.1159998076409602, 0.5426091593378901, 0.0},
      {0.5426091593378899, 0.1159998076409602, 0.0},
      {0.3413910330211499, 0.5426091593378900, 0.0},
      {0.1159998076409601, 0.3413910330211500, 0.0},
      {0.3739379797195844, 0.0483174342873770, 0.0},
      {0.5777445859930386, 0.3739379797195844, 0.0},
      {0.0483174342873769, 0.5777445859930388, 0.0},
      {0.5777445859930387, 0.0483174342873770, 0.0},
      {0.3739379797195845, 0.5777445859930387, 0.0},
      {0.0483174342873768, 0.3739379797195846, 0.0},
      {0.0991330633416822, 0.0071283145012574, 0.0},
      {0.8937386221570603, 0.0991330633416822, 0.0},
      {0.0071283145012576, 0.8937386221570605, 0.0},
      {0.8937386221570603, 0.0071283145012574, 0.0},
      {0.0991330633416823, 0.8937386221570605, 0.0},
      {0.0071283145012572, 0.0991330633416825, 0.0},
      {0.2995064186296745, 0.2036929105842510, 0.0},
      {0.4968006707860745, 0.2995064186296746, 0.0},
      {0.2036929105842510, 0.4968006707860746, 0.0},
      {0.4968006707860745, 0.2036929105842510, 0.0},
      {0.2995064186296745, 0.4968006707860746, 0.0},
      {0.2036929105842509, 0.2995064186296747, 0.0},
      {0.4247593045405747, 0.1504813909188506, 0.0},
      {0.4247593045405746, 0.4247593045405748, 0.0},
      {0.1504813909188505, 0.4247593045405749, 0.0},
      {0.1786298486036162, 0.0072361617479482, 0.0},
      {0.8141339896484356, 0.1786298486036162, 0.0},
      {0.0072361617479482, 0.8141339896484358, 0.0},
      {0.8141339896484356, 0.0072361617479482, 0.0},
      {0.1786298486036164, 0.8141339896484358, 0.0},
      {0.0072361617479479, 0.1786298486036166, 0.0},
      {0.3620688018959720, 0.0129138832500325, 0.0},
      {0.6250173148539955, 0.3620688018959721, 0.0},
      {0.0129138832500325, 0.6250173148539957, 0.0},
      {0.6250173148539954, 0.0129138832500325, 0.0},
      {0.3620688018959721, 0.6250173148539956, 0.0},
      {0.0129138832500325, 0.3620688018959722, 0.0},
      {0.0887929154893665, 0.0376879497842591, 0.0},
      {0.8735191347263743, 0.0887929154893666, 0.0},
      {0.0376879497842592, 0.8735191347263744, 0.0},
      {0.8735191347263743, 0.0376879497842591, 0.0},
      {0.0887929154893666, 0.8735191347263744, 0.0},
      {0.0376879497842588, 0.0887929154893669, 0.0},
      {0.2336228101417152, 0.1370066940870710, 0.0},
      {0.6293704957712137, 0.2336228101417153, 0.0},
      {0.1370066940870710, 0.6293704957712140, 0.0},
      {0.6293704957712138, 0.1370066940870710, 0.0},
      {0.2336228101417152, 0.6293704957712140, 0.0},
      {0.1370066940870708, 0.2336228101417155, 0.0},
      {0.4622087087487061, 0.0755825825025878, 0.0},
      {0.4622087087487061, 0.4622087087487062, 0.0},
      {0.0755825825025877, 0.4622087087487063, 0.0},
      {0.2565954097090198, 0.0245400602475244, 0.0},
      {0.7188645300434557, 0.2565954097090198, 0.0},
      {0.0245400602475245, 0.7188645300434559, 0.0},
      {0.7188645300434559, 0.0245400602475244, 0.0},
      {0.2565954097090199, 0.7188645300434559, 0.0},
      {0.0245400602475242, 0.2565954097090201, 0.0},
      {0.0929497017007698, 0.0929497017007700, 0.0},
      {0.8141005965984600, 0.0929497017007698, 0.0},
      {0.0929497017007701, 0.8141005965984602, 0.0},
      {0.0410688191117846, 0.0071888282616930, 0.0},
      {0.9517423526265223, 0.0410688191117846, 0.0},
      {0.0071888282616933, 0.9517423526265224, 0.0},
      {0.9517423526265224, 0.0071888282616930, 0.0},
      {0.0410688191117848, 0.9517423526265224, 0.0},
      {0.0071888282616928, 0.0410688191117850, 0.0},
      {0.0078353442826036, 0.0078353442826039, 0.0},
      {0.9843293114347924, 0.0078353442826036, 0.0},
      {0.0078353442826041, 0.9843293114347926, 0.0},
      {0.4890393696603955, 0.0219212606792091, 0.0},
      {0.4890393696603955, 0.4890393696603955, 0.0},
      {0.0219212606792090, 0.4890393696603957, 0.0},
      {0.2794161886492607, 0.0008914643174981, 0.0},
      {0.7196923470332411, 0.2794161886492607, 0.0},
      {0.0008914643174981, 0.7196923470332415, 0.0},
      {0.7196923470332411, 0.0008914643174981, 0.0},
      {0.2794161886492608, 0.7196923470332413, 0.0},
      {0.0008914643174980, 0.2794161886492610, 0.0}},
      {0.0068449257741361,
      0.0068449257741361,
      0.0068449257741361,
      0.0057936316180053,
      0.0057936316180053,
      0.0057936316180053,
      0.0008374089159674,
      0.0008374089159674,
      0.0008374089159674,
      0.0008374089159674,
      0.0008374089159674,
      0.0008374089159674,
      0.0090088203508508,
      0.0090088203508508,
      0.0090088203508508,
      0.0031557390123797,
      0.0031557390123797,
      0.0031557390123797,
      0.0031557390123797,
      0.0031557390123797,
      0.0031557390123797,
      0.0016986488609524,
      0.0016986488609524,
      0.0016986488609524,
      0.0047575107837279,
      0.0047575107837279,
      0.0047575107837279,
      0.0047575107837279,
      0.0047575107837279,
      0.0047575107837279,
      0.0057457629312824,
      0.0057457629312824,
      0.0057457629312824,
      0.0054421968062185,
      0.0054421968062185,
      0.0054421968062185,
      0.0054421968062185,
      0.0054421968062185,
      0.0054421968062185,
      0.0079201761439492,
      0.0079201761439492,
      0.0079201761439492,
      0.0079201761439492,
      0.0079201761439492,
      0.0079201761439492,
      0.0053200853477544,
      0.0053200853477544,
      0.0053200853477544,
      0.0053200853477544,
      0.0053200853477544,
      0.0053200853477544,
      0.0012726358126745,
      0.0012726358126745,
      0.0012726358126745,
      0.0012726358126745,
      0.0012726358126745,
      0.0012726358126745,
      0.0089569104461381,
      0.0089569104461381,
      0.0089569104461381,
      0.0089569104461381,
      0.0089569104461381,
      0.0089569104461381,
      0.0079556550687292,
      0.0079556550687292,
      0.0079556550687292,
      0.0016318698410246,
      0.0016318698410246,
      0.0016318698410246,
      0.0016318698410246,
      0.0016318698410246,
      0.0016318698410246,
      0.0027273191839872,
      0.0027273191839872,
      0.0027273191839872,
      0.0027273191839872,
      0.0027273191839872,
      0.0027273191839872,
      0.0026362809607147,
      0.0026362809607147,
      0.0026362809607147,
      0.0026362809607147,
      0.0026362809607147,
      0.0026362809607147,
      0.0068700412960113,
      0.0068700412960113,
      0.0068700412960113,
      0.0068700412960113,
      0.0068700412960113,
      0.0068700412960113,
      0.0068271375937640,
      0.0068271375937640,
      0.0068271375937640,
      0.0036571704539664,
      0.0036571704539664,
      0.0036571704539664,
      0.0036571704539664,
      0.0036571704539664,
      0.0036571704539664,
      0.0045914106299100,
      0.0045914106299100,
      0.0045914106299100,
      0.0008464918170637,
      0.0008464918170637,
      0.0008464918170637,
      0.0008464918170637,
      0.0008464918170637,
      0.0008464918170637,
      0.0004032551441623,
      0.0004032551441623,
      0.0004032551441623,
      0.0042220429732605,
      0.0042220429732605,
      0.0042220429732605,
      0.0007558510392294,
      0.0007558510392294,
      0.0007558510392294,
      0.0007558510392294,
      0.0007558510392294,
      0.0007558510392294}
    },
    {
      130,
      {{0.0800716549403165, 0.0047946609754367, 0.0},
      {0.9151336840842468, 0.0800716549403165, 0.0},
      {0.0047946609754368, 0.9151336840842469, 0.0},
      {0.9151336840842468, 0.0047946609754367, 0.0},
      {0.0800716549403166, 0.9151336840842469, 0.0},
      {0.0047946609754365, 0.0800716549403169, 0.0},
      {0.0316436115715308, 0.0291551962068358, 0.0},
      {0.9392011922216335, 0.0316436115715307, 0.0},
      {0.0291551962068360, 0.9392011922216336, 0.0},
      {0.9392011922216333, 0.0291551962068358, 0.0},
      {0.0316436115715308, 0.9392011922216336, 0.0},
      {0.0291551962068356, 0.0316436115715311, 0.0},
      {0.0667371225764661, 0.0667371225764664, 0.0},
      {0.8665257548470675, 0.0667371225764660, 0.0},
      {0.0667371225764665, 0.8665257548470677, 0.0},
      {0.0753800475153987, 0.0262093640224987, 0.0},
      {0.8984105884621026, 0.0753800475153986, 0.0},
      {0.0262093640224987, 0.8984105884621029, 0.0},
      {0.8984105884621026, 0.0262093640224987, 0.0},
      {0.0753800475153986, 0.8984105884621030, 0.0},
      {0.0262093640224984, 0.0753800475153990, 0.0},
      {0.0331000343360323, 0.0056981179168752, 0.0},
      {0.9612018477470925, 0.0331000343360323, 0.0},
      {0.0056981179168754, 0.9612018477470927, 0.0},
      {0.9612018477470925, 0.0056981179168752, 0.0},
      {0.0331000343360324, 0.9612018477470927, 0.0},
      {0.0056981179168749, 0.0331000343360326, 0.0},
      {0.0063401164920767, 0.0063401164920770, 0.0},
      {0.9873197670158463, 0.0063401164920766, 0.0},
      {0.0063401164920771, 0.9873197670158466, 0.0},
      {0.1324861896145673, 0.0417247227421209, 0.0},
      {0.8257890876433117, 0.1324861896145673, 0.0},
      {0.0417247227421210, 0.8257890876433118, 0.0},
      {0.8257890876433117, 0.0417247227421209, 0.0},
      {0.1324861896145674, 0.8257890876433118, 0.0},
      {0.0417247227421207, 0.1324861896145676, 0.0},
      {0.1086871329144021, 0.1000456591065276, 0.0},
      {0.7912672079790702, 0.1086871329144021, 0.0},
      {0.1000456591065276, 0.7912672079790705, 0.0},
      {0.7912672079790704, 0.1000456591065276, 0.0},
      {0.1086871329144022, 0.7912672079790705, 0.0},
      {0.1000456591065273, 0.1086871329144024, 0.0},
      {0.4937530328963848, 0.0124939342072305, 0.0},
      {0.4937530328963847, 0.4937530328963848, 0.0},
      {0.0124939342072304, 0.4937530328963849, 0.0},
      {0.2502723132905265, 0.1206144022052490, 0.0},
      {0.6291132845042245, 0.2502723132905265, 0.0},
      {0.1206144022052490, 0.6291132845042247, 0.0},
      {0.6291132845042245, 0.1206144022052490, 0.0},
      {0.2502723132905265, 0.6291132845042245, 0.0},
      {0.1206144022052489, 0.2502723132905267, 0.0},
      {0.3890220620427617, 0.0295379425169078, 0.0},
      {0.5814399954403304, 0.3890220620427618, 0.0},
      {0.0295379425169078, 0.5814399954403305, 0.0},
      {0.5814399954403304, 0.0295379425169078, 0.0},
      {0.3890220620427618, 0.5814399954403304, 0.0},
      {0.0295379425169077, 0.3890220620427621, 0.0},
      {0.3585092964276616, 0.0873784651638445, 0.0},
      {0.5541122384084940, 0.3585092964276616, 0.0},
      {0.0873784651638445, 0.5541122384084942, 0.0},
      {0.5541122384084940, 0.0873784651638445, 0.0},
      {0.3585092964276616, 0.5541122384084941, 0.0},
      {0.0873784651638444, 0.3585092964276618, 0.0},
      {0.3887874971075940, 0.2224250057848120, 0.0},
      {0.3887874971075940, 0.3887874971075941, 0.0},
      {0.2224250057848119, 0.3887874971075941, 0.0},
      {0.1868691794762216, 0.0763119015129594, 0.0},
      {0.7368189190108190, 0.1868691794762216, 0.0},
      {0.0763119015129595, 0.7368189190108192, 0.0},
      {0.7368189190108190, 0.0763119015129594, 0.0},
      {0.1868691794762216, 0.7368189190108192, 0.0},
      {0.0763119015129592, 0.1868691794762219, 0.0},
      {0.2731471009290787, 0.2731471009290788, 0.0},
      {0.4537057981418425, 0.2731471009290788, 0.0},
      {0.2731471009290788, 0.4537057981418426, 0.0},
      {0.4147059095903063, 0.0020575309653709, 0.0},
      {0.5832365594443230, 0.4147059095903063, 0.0},
      {0.0020575309653708, 0.5832365594443231, 0.0},
      {0.5832365594443230, 0.0020575309653709, 0.0},
      {0.4147059095903063, 0.5832365594443230, 0.0},
      {0.0020575309653707, 0.4147059095903066, 0.0},
      {0.3194153053834387, 0.1704787284972490, 0.0},
      {0.5101059661193124, 0.3194153053834388, 0.0},
      {0.1704787284972489, 0.5101059661193126, 0.0},
      {0.5101059661193124, 0.1704787284972490, 0.0},
      {0.3194153053834387, 0.5101059661193125, 0.0},
      {0.1704787284972488, 0.3194153053834389, 0.0},
      {0.1437376261997640, 0.0079996080914843, 0.0},
      {0.8482627657087517, 0.1437376261997640, 0.0},
      {0.0079996080914844, 0.8482627657087518, 0.0},
      {0.8482627657087517, 0.0079996080914843, 0.0},
      {0.1437376261997642, 0.8482627657087518, 0.0},
      {0.0079996080914840, 0.1437376261997644, 0.0},
      {0.4718285633211660, 0.0563428733576680, 0.0},
      {0.4718285633211661, 0.4718285633211661, 0.0},
      {0.0563428733576679, 0.4718285633211662, 0.0},
      {0.1542014303645442, 0.1542014303645444, 0.0},
      {0.6915971392709114, 0.1542014303645442, 0.0},
      {0.1542014303645444, 0.6915971392709116, 0.0},
      {0.2837881388594704, 0.0511658736851378, 0.0},
      {0.6650459874553918, 0.2837881388594705, 0.0},
      {0.0511658736851378, 0.6650459874553920, 0.0},
      {0.6650459874553918, 0.0511658736851378, 0.0},
      {0.2837881388594705, 0.6650459874553919, 0.0},
      {0.0511658736851376, 0.2837881388594707, 0.0},
      {0.2120431633022055, 0.2120431633022057, 0.0},
      {0.5759136733955887, 0.2120431633022056, 0.0},
      {0.2120431633022057, 0.5759136733955889, 0.0},
      {0.4359854193843832, 0.1280291612312336, 0.0},
      {0.4359854193843832, 0.4359854193843832, 0.0},
      {0.1280291612312335, 0.4359854193843834, 0.0},
      {0.2165466664734771, 0.0227845992508957, 0.0},
      {0.7606687342756272, 0.2165466664734771, 0.0},
      {0.0227845992508957, 0.7606687342756275, 0.0},
      {0.7606687342756272, 0.0227845992508957, 0.0},
      {0.2165466664734772, 0.7606687342756273, 0.0},
      {0.0227845992508955, 0.2165466664734774, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.3128985030748800, 0.0094732979122136, 0.0},
      {0.6776281990129065, 0.3128985030748800, 0.0},
      {0.0094732979122136, 0.6776281990129066, 0.0},
      {0.6776281990129064, 0.0094732979122136, 0.0},
      {0.3128985030748800, 0.6776281990129066, 0.0},
      {0.0094732979122134, 0.3128985030748803, 0.0},
      {0.2264347974077175, 0.0004640077321757, 0.0},
      {0.7731011948601069, 0.2264347974077175, 0.0},
      {0.0004640077321757, 0.7731011948601070, 0.0},
      {0.7731011948601069, 0.0004640077321757, 0.0},
      {0.2264347974077175, 0.7731011948601070, 0.0},
      {0.0004640077321755, 0.2264347974077178, 0.0}},
      {0.0006992632240802,
      0.0006992632240802,
      0.0006992632240802,
      0.0006992632240802,
      0.0006992632240802,
      0.0006992632240802,
      0.0006027823868584,
      0.0006027823868584,
      0.0006027823868584,
      0.0006027823868584,
      0.0006027823868584,
      0.0006027823868584,
      0.0024569126514830,
      0.0024569126514830,
      0.0024569126514830,
      0.0016527723564839,
      0.0016527723564839,
      0.0016527723564839,
      0.0016527723564839,
      0.0016527723564839,
      0.0016527723564839,
      0.0005428536714984,
      0.0005428536714984,
      0.0005428536714984,
      0.0005428536714984,
      0.0005428536714984,
      0.0005428536714984,
      0.0002634765583409,
      0.0002634765583409,
      0.0002634765583409,
      0.0032017989498564,
      0.0032017989498564,
      0.0032017989498564,
      0.0032017989498564,
      0.0032017989498564,
      0.0032017989498564,
      0.0023071055381892,
      0.0023071055381892,
      0.0023071055381892,
      0.0023071055381892,
      0.0023071055381892,
      0.0023071055381892,
      0.0026510795909336,
      0.0026510795909336,
      0.0026510795909336,
      0.0071897366137994,
      0.0071897366137994,
      0.0071897366137994,
      0.0071897366137994,
      0.0071897366137994,
      0.0071897366137994,
      0.0041298836085434,
      0.0041298836085434,
      0.0041298836085434,
      0.0041298836085434,
      0.0041298836085434,
      0.0041298836085434,
      0.0068639791080428,
      0.0068639791080428,
      0.0068639791080428,
      0.0068639791080428,
      0.0068639791080428,
      0.0068639791080428,
      0.0097340339185915,
      0.0097340339185915,
      0.0097340339185915,
      0.0051988227640871,
      0.0051988227640871,
      0.0051988227640871,
      0.0051988227640871,
      0.0051988227640871,
      0.0051988227640871,
      0.0097678234616237,
      0.0097678234616237,
      0.0097678234616237,
      0.0009285737354991,
      0.0009285737354991,
      0.0009285737354991,
      0.0009285737354991,
      0.0009285737354991,
      0.0009285737354991,
      0.0087995835903476,
      0.0087995835903476,
      0.0087995835903476,
      0.0087995835903476,
      0.0087995835903476,
      0.0087995835903476,
      0.0014833808313283,
      0.0014833808313283,
      0.0014833808313283,
      0.0014833808313283,
      0.0014833808313283,
      0.0014833808313283,
      0.0057642518173285,
      0.0057642518173285,
      0.0057642518173285,
      0.0066276297242726,
      0.0066276297242726,
      0.0066276297242726,
      0.0050535622160444,
      0.0050535622160444,
      0.0050535622160444,
      0.0050535622160444,
      0.0050535622160444,
      0.0050535622160444,
      0.0084721725392641,
      0.0084721725392641,
      0.0084721725392641,
      0.0082062003012940,
      0.0082062003012940,
      0.0082062003012940,
      0.0031346689230403,
      0.0031346689230403,
      0.0031346689230403,
      0.0031346689230403,
      0.0031346689230403,
      0.0031346689230403,
      0.0102433312946116,
      0.0022957791936993,
      0.0022957791936993,
      0.0022957791936993,
      0.0022957791936993,
      0.0022957791936993,
      0.0022957791936993,
      0.0005697744579341,
      0.0005697744579341,
      0.0005697744579341,
      0.0005697744579341,
      0.0005697744579341,
      0.0005697744579341}
    },
    {
      141,
      {{0.3807140211811871, 0.2385719576376257, 0.0},
      {0.3807140211811871, 0.3807140211811872, 0.0},
      {0.2385719576376256, 0.3807140211811873, 0.0},
      {0.4466678037038646, 0.1066643925922709, 0.0},
      {0.4466678037038646, 0.4466678037038647, 0.0},
      {0.1066643925922708, 0.4466678037038648, 0.0},
      {0.4161413788054121, 0.1677172423891757, 0.0},
      {0.4161413788054121, 0.4161413788054122, 0.0},
      {0.1677172423891756, 0.4161413788054124, 0.0},
      {0.0803046477884383, 0.0803046477884386, 0.0},
      {0.8393907044231230, 0.0803046477884383, 0.0},
      {0.0803046477884388, 0.8393907044231231, 0.0},
      {0.2870421965934966, 0.0307306047272729, 0.0},
      {0.6822271986792305, 0.2870421965934966, 0.0},
      {0.0307306047272730, 0.6822271986792307, 0.0},
      {0.6822271986792305, 0.0307306047272729, 0.0},
      {0.2870421965934966, 0.6822271986792307, 0.0},
      {0.0307306047272727, 0.2870421965934968, 0.0},
      {0.3450878417155684, 0.1291526400634497, 0.0},
      {0.5257595182209819, 0.3450878417155684, 0.0},
      {0.1291526400634497, 0.5257595182209820, 0.0},
      {0.5257595182209819, 0.1291526400634497, 0.0},
      {0.3450878417155684, 0.5257595182209820, 0.0},
      {0.1291526400634496, 0.3450878417155686, 0.0},
      {0.2334004066698710, 0.2334004066698712, 0.0},
      {0.5331991866602577, 0.2334004066698711, 0.0},
      {0.2334004066698712, 0.5331991866602579, 0.0},
      {0.3759301570486618, 0.0280334860952500, 0.0},
      {0.5960363568560882, 0.3759301570486618, 0.0},
      {0.0280334860952500, 0.5960363568560885, 0.0},
      {0.5960363568560882, 0.0280334860952500, 0.0},
      {0.3759301570486618, 0.5960363568560882, 0.0},
      {0.0280334860952499, 0.3759301570486621, 0.0},
      {0.3011654651665091, 0.3011654651665093, 0.0},
      {0.3976690696669817, 0.3011654651665092, 0.0},
      {0.3011654651665091, 0.3976690696669818, 0.0},
      {0.1747799663549000, 0.1747799663549002, 0.0},
      {0.6504400672901998, 0.1747799663549000, 0.0},
      {0.1747799663549002, 0.6504400672902000, 0.0},
      {0.3169455889331320, 0.2091309211376688, 0.0},
      {0.4739234899291993, 0.3169455889331320, 0.0},
      {0.2091309211376687, 0.4739234899291995, 0.0},
      {0.4739234899291992, 0.2091309211376688, 0.0},
      {0.3169455889331320, 0.4739234899291995, 0.0},
      {0.2091309211376686, 0.3169455889331322, 0.0},
      {0.4072283930427198, 0.0660389128497387, 0.0},
      {0.5267326941075414, 0.4072283930427200, 0.0},
      {0.0660389128497386, 0.5267326941075416, 0.0},
      {0.5267326941075414, 0.0660389128497387, 0.0},
      {0.4072283930427199, 0.5267326941075415, 0.0},
      {0.0660389128497386, 0.4072283930427201, 0.0},
      {0.2135535984578239, 0.0410305768191818, 0.0},
      {0.7454158247229943, 0.2135535984578240, 0.0},
      {0.0410305768191818, 0.7454158247229945, 0.0},
      {0.7454158247229943, 0.0410305768191818, 0.0},
      {0.2135535984578240, 0.7454158247229945, 0.0},
      {0.0410305768191816, 0.2135535984578242, 0.0},
      {0.3288528780688926, 0.0052996403717990, 0.0},
      {0.6658474815593083, 0.3288528780688927, 0.0},
      {0.0052996403717991, 0.6658474815593085, 0.0},
      {0.6658474815593083, 0.0052996403717990, 0.0},
      {0.3288528780688927, 0.6658474815593083, 0.0},
      {0.0052996403717989, 0.3288528780688930, 0.0},
      {0.4855650541851628, 0.0288698916296745, 0.0},
      {0.4855650541851628, 0.4855650541851628, 0.0},
      {0.0288698916296744, 0.4855650541851630, 0.0},
      {0.1392953061421487, 0.0630739954149509, 0.0},
      {0.7976306984429004, 0.1392953061421487, 0.0},
      {0.0630739954149511, 0.7976306984429004, 0.0},
      {0.7976306984429004, 0.0630739954149509, 0.0},
      {0.1392953061421489, 0.7976306984429004, 0.0},
      {0.0630739954149507, 0.1392953061421490, 0.0},
      {0.0325715201801815, 0.0325715201801818, 0.0},
      {0.9348569596396367, 0.0325715201801814, 0.0},
      {0.0325715201801819, 0.9348569596396370, 0.0},
      {0.2552462546969780, 0.1489628509382402, 0.0},
      {0.5957908943647818, 0.2552462546969780, 0.0},
      {0.1489628509382402, 0.5957908943647819, 0.0},
      {0.5957908943647818, 0.1489628509382402, 0.0},
      {0.2552462546969780, 0.5957908943647819, 0.0},
      {0.1489628509382400, 0.2552462546969783, 0.0},
      {0.2083760156003741, 0.0946970824331307, 0.0},
      {0.6969269019664952, 0.2083760156003741, 0.0},
      {0.0946970824331307, 0.6969269019664954, 0.0},
      {0.6969269019664952, 0.0946970824331307, 0.0},
      {0.2083760156003741, 0.6969269019664954, 0.0},
      {0.0946970824331305, 0.2083760156003744, 0.0},
      {0.4400105519462155, 0.0055807170152601, 0.0},
      {0.5544087310385245, 0.4400105519462155, 0.0},
      {0.0055807170152600, 0.5544087310385247, 0.0},
      {0.5544087310385244, 0.0055807170152601, 0.0},
      {0.4400105519462155, 0.5544087310385245, 0.0},
      {0.0055807170152600, 0.4400105519462157, 0.0},
      {0.1275709019046775, 0.1275709019046777, 0.0},
      {0.7448581961906449, 0.1275709019046775, 0.0},
      {0.1275709019046778, 0.7448581961906449, 0.0},
      {0.3022209412278211, 0.0750769024331962, 0.0},
      {0.6227021563389826, 0.3022209412278212, 0.0},
      {0.0750769024331963, 0.6227021563389828, 0.0},
      {0.6227021563389826, 0.0750769024331962, 0.0},
      {0.3022209412278211, 0.6227021563389828, 0.0},
      {0.0750769024331961, 0.3022209412278213, 0.0},
      {0.0819468025835337, 0.0069825293244590, 0.0},
      {0.9110706680920073, 0.0819468025835337, 0.0},
      {0.0069825293244591, 0.9110706680920074, 0.0},
      {0.9110706680920073, 0.0069825293244590, 0.0},
      {0.0819468025835338, 0.9110706680920074, 0.0},
      {0.0069825293244588, 0.0819468025835340, 0.0},
      {0.0343649699121420, 0.0060935694037648, 0.0},
      {0.9595414606840931, 0.0343649699121420, 0.0},
      {0.0060935694037650, 0.9595414606840932, 0.0},
      {0.9595414606840931, 0.0060935694037648, 0.0},
      {0.0343649699121421, 0.9595414606840934, 0.0},
      {0.0060935694037646, 0.0343649699121423, 0.0},
      {0.0801120738471011, 0.0350344225276974, 0.0},
      {0.8848535036252014, 0.0801120738471011, 0.0},
      {0.0350344225276975, 0.8848535036252017, 0.0},
      {0.8848535036252014, 0.0350344225276974, 0.0},
      {0.0801120738471012, 0.8848535036252017, 0.0},
      {0.0350344225276972, 0.0801120738471014, 0.0},
      {0.0066392191809586, 0.0066392191809589, 0.0},
      {0.9867215616380824, 0.0066392191809586, 0.0},
      {0.0066392191809590, 0.9867215616380827, 0.0},
      {0.1472134318989225, 0.0193520013180390, 0.0},
      {0.8334345667830385, 0.1472134318989225, 0.0},
      {0.0193520013180390, 0.8334345667830388, 0.0},
      {0.8334345667830385, 0.0193520013180390, 0.0},
      {0.1472134318989225, 0.8334345667830388, 0.0},
      {0.0193520013180388, 0.1472134318989227, 0.0},
      {0.2297196532578432, 0.0073324725490405, 0.0},
      {0.7629478741931164, 0.2297196532578432, 0.0},
      {0.0073324725490405, 0.7629478741931165, 0.0},
      {0.7629478741931163, 0.0073324725490405, 0.0},
      {0.2297196532578433, 0.7629478741931164, 0.0},
      {0.0073324725490403, 0.2297196532578435, 0.0},
      {0.1476555211198698, 0.0004903284434630, 0.0},
      {0.8518541504366672, 0.1476555211198698, 0.0},
      {0.0004903284434630, 0.8518541504366673, 0.0},
      {0.8518541504366672, 0.0004903284434630, 0.0},
      {0.1476555211198698, 0.8518541504366673, 0.0},
      {0.0004903284434627, 0.1476555211198702, 0.0}},
      {0.0047800424837300,
      0.0047800424837300,
      0.0047800424837300,
      0.0047050799047271,
      0.0047050799047271,
      0.0047050799047271,
      0.0060251135120752,
      0.0060251135120752,
      0.0060251135120752,
      0.0026063109364009,
      0.0026063109364009,
      0.0026063109364009,
      0.0027658974168834,
      0.0027658974168834,
      0.0027658974168834,
      0.0027658974168834,
      0.0027658974168834,
      0.0027658974168834,
      0.0062787181020182,
      0.0062787181020182,
      0.0062787181020182,
      0.0062787181020182,
      0.0062787181020182,
      0.0062787181020182,
      0.0067356576990247,
      0.0067356576990247,
      0.0067356576990247,
      0.0031975763497272,
      0.0031975763497272,
      0.0031975763497272,
      0.0031975763497272,
      0.0031975763497272,
      0.0031975763497272,
      0.0078739828906813,
      0.0078739828906813,
      0.0078739828906813,
      0.0056412212723492,
      0.0056412212723492,
      0.0056412212723492,
      0.0068576966152754,
      0.0068576966152754,
      0.0068576966152754,
      0.0068576966152754,
      0.0068576966152754,
      0.0068576966152754,
      0.0049311350594948,
      0.0049311350594948,
      0.0049311350594948,
      0.0049311350594948,
      0.0049311350594948,
      0.0049311350594948,
      0.0032276864524648,
      0.0032276864524648,
      0.0032276864524648,
      0.0032276864524648,
      0.0032276864524648,
      0.0032276864524648,
      0.0014639131808995,
      0.0014639131808995,
      0.0014639131808995,
      0.0014639131808995,
      0.0014639131808995,
      0.0014639131808995,
      0.0035586187064373,
      0.0035586187064373,
      0.0035586187064373,
      0.0035653176552435,
      0.0035653176552435,
      0.0035653176552435,
      0.0035653176552435,
      0.0035653176552435,
      0.0035653176552435,
      0.0013886697644771,
      0.0013886697644771,
      0.0013886697644771,
      0.0061738315654307,
      0.0061738315654307,
      0.0061738315654307,
      0.0061738315654307,
      0.0061738315654307,
      0.0061738315654307,
      0.0053468502948082,
      0.0053468502948082,
      0.0053468502948082,
      0.0053468502948082,
      0.0053468502948082,
      0.0053468502948082,
      0.0016212337988197,
      0.0016212337988197,
      0.0016212337988197,
      0.0016212337988197,
      0.0016212337988197,
      0.0016212337988197,
      0.0048716224614089,
      0.0048716224614089,
      0.0048716224614089,
      0.0054653055464567,
      0.0054653055464567,
      0.0054653055464567,
      0.0054653055464567,
      0.0054653055464567,
      0.0054653055464567,
      0.0010075615636448,
      0.0010075615636448,
      0.0010075615636448,
      0.0010075615636448,
      0.0010075615636448,
      0.0010075615636448,
      0.0005983868042366,
      0.0005983868042366,
      0.0005983868042366,
      0.0005983868042366,
      0.0005983868042366,
      0.0005983868042366,
      0.0021635790176803,
      0.0021635790176803,
      0.0021635790176803,
      0.0021635790176803,
      0.0021635790176803,
      0.0021635790176803,
      0.0002877212028353,
      0.0002877212028353,
      0.0002877212028353,
      0.0023111935558906,
      0.0023111935558906,
      0.0023111935558906,
      0.0023111935558906,
      0.0023111935558906,
      0.0023111935558906,
      0.0016971268694035,
      0.0016971268694035,
      0.0016971268694035,
      0.0016971268694035,
      0.0016971268694035,
      0.0016971268694035,
      0.0004233030678819,
      0.0004233030678819,
      0.0004233030678819,
      0.0004233030678819,
      0.0004233030678819,
      0.0004233030678819}
    },
    {
      150,
      {{0.3039829225164841, 0.3039829225164842, 0.0},
      {0.3920341549670316, 0.3039829225164842, 0.0},
      {0.3039829225164841, 0.3920341549670319, 0.0},
      {0.0455054005583464, 0.0215243853694561, 0.0},
      {0.9329702140721974, 0.0455054005583464, 0.0},
      {0.0215243853694563, 0.9329702140721976, 0.0},
      {0.9329702140721974, 0.0215243853694561, 0.0},
      {0.0455054005583465, 0.9329702140721976, 0.0},
      {0.0215243853694558, 0.0455054005583467, 0.0},
      {0.2133944547670873, 0.0490696693575595, 0.0},
      {0.7375358758753532, 0.2133944547670873, 0.0},
      {0.0490696693575595, 0.7375358758753534, 0.0},
      {0.7375358758753532, 0.0490696693575595, 0.0},
      {0.2133944547670873, 0.7375358758753533, 0.0},
      {0.0490696693575593, 0.2133944547670876, 0.0},
      {0.0048041261966579, 0.0048041261966582, 0.0},
      {0.9903917476066839, 0.0048041261966579, 0.0},
      {0.0048041261966584, 0.9903917476066839, 0.0},
      {0.4582799042404119, 0.0834401915191761, 0.0},
      {0.4582799042404120, 0.4582799042404120, 0.0},
      {0.0834401915191761, 0.4582799042404122, 0.0},
      {0.2421025119193196, 0.1776584502963703, 0.0},
      {0.5802390377843101, 0.2421025119193198, 0.0},
      {0.1776584502963703, 0.5802390377843101, 0.0},
      {0.5802390377843101, 0.1776584502963703, 0.0},
      {0.2421025119193198, 0.5802390377843101, 0.0},
      {0.1776584502963702, 0.2421025119193199, 0.0},
      {0.3862679735700421, 0.2274640528599159, 0.0},
      {0.3862679735700420, 0.3862679735700422, 0.0},
      {0.2274640528599158, 0.3862679735700422, 0.0},
      {0.3271907320191700, 0.1898123562927369, 0.0},
      {0.4829969116880932, 0.3271907320191700, 0.0},
      {0.1898123562927368, 0.4829969116880933, 0.0},
      {0.4829969116880932, 0.1898123562927369, 0.0},
      {0.3271907320191700, 0.4829969116880933, 0.0},
      {0.1898123562927368, 0.3271907320191702, 0.0},
      {0.1419981669331742, 0.0044583820232893, 0.0},
      {0.8535434510435363, 0.1419981669331742, 0.0},
      {0.0044583820232894, 0.8535434510435367, 0.0},
      {0.8535434510435365, 0.0044583820232893, 0.0},
      {0.1419981669331742, 0.8535434510435367, 0.0},
      {0.0044583820232890, 0.1419981669331745, 0.0},
      {0.1753963931914617, 0.0876779764843520, 0.0},
      {0.7369256303241862, 0.1753963931914617, 0.0},
      {0.0876779764843521, 0.7369256303241863, 0.0},
      {0.7369256303241862, 0.0876779764843520, 0.0},
      {0.1753963931914618, 0.7369256303241863, 0.0},
      {0.0876779764843519, 0.1753963931914620, 0.0},
      {0.3921396133344145, 0.0631803276344107, 0.0},
      {0.5446800590311748, 0.3921396133344145, 0.0},
      {0.0631803276344107, 0.5446800590311750, 0.0},
      {0.5446800590311748, 0.0631803276344107, 0.0},
      {0.3921396133344145, 0.5446800590311749, 0.0},
      {0.0631803276344106, 0.3921396133344148, 0.0},
      {0.2582640721504621, 0.2582640721504623, 0.0},
      {0.4834718556990756, 0.2582640721504622, 0.0},
      {0.2582640721504622, 0.4834718556990758, 0.0},
      {0.3341456150359213, 0.0041494641339237, 0.0},
      {0.6617049208301551, 0.3341456150359213, 0.0},
      {0.0041494641339236, 0.6617049208301553, 0.0},
      {0.6617049208301550, 0.0041494641339237, 0.0},
      {0.3341456150359213, 0.6617049208301551, 0.0},
      {0.0041494641339235, 0.3341456150359216, 0.0},
      {0.1741461960511821, 0.0227948049259162, 0.0},
      {0.8030589990229017, 0.1741461960511821, 0.0},
      {0.0227948049259163, 0.8030589990229018, 0.0},
      {0.8030589990229016, 0.0227948049259162, 0.0},
      {0.1741461960511822, 0.8030589990229018, 0.0},
      {0.0227948049259160, 0.1741461960511825, 0.0},
      {0.2758390080718242, 0.0227008443717970, 0.0},
      {0.7014601475563789, 0.2758390080718242, 0.0},
      {0.0227008443717971, 0.7014601475563790, 0.0},
      {0.7014601475563788, 0.0227008443717970, 0.0},
      {0.2758390080718242, 0.7014601475563790, 0.0},
      {0.0227008443717968, 0.2758390080718244, 0.0},
      {0.0262226671646523, 0.0061496485426640, 0.0},
      {0.9676276842926838, 0.0262226671646522, 0.0},
      {0.0061496485426642, 0.9676276842926838, 0.0},
      {0.9676276842926838, 0.0061496485426640, 0.0},
      {0.0262226671646523, 0.9676276842926840, 0.0},
      {0.0061496485426637, 0.0262226671646526, 0.0},
      {0.2430472023659261, 0.1120336293422710, 0.0},
      {0.6449191682918028, 0.2430472023659262, 0.0},
      {0.1120336293422710, 0.6449191682918030, 0.0},
      {0.6449191682918028, 0.1120336293422710, 0.0},
      {0.2430472023659263, 0.6449191682918030, 0.0},
      {0.1120336293422708, 0.2430472023659264, 0.0},
      {0.2299929840579072, 0.0047814897729871, 0.0},
      {0.7652255261691058, 0.2299929840579072, 0.0},
      {0.0047814897729871, 0.7652255261691059, 0.0},
      {0.7652255261691057, 0.0047814897729871, 0.0},
      {0.2299929840579072, 0.7652255261691059, 0.0},
      {0.0047814897729869, 0.2299929840579075, 0.0},
      {0.1058958441786275, 0.1058958441786278, 0.0},
      {0.7882083116427446, 0.1058958441786275, 0.0},
      {0.1058958441786279, 0.7882083116427449, 0.0},
      {0.4295522021188993, 0.1408955957622014, 0.0},
      {0.4295522021188993, 0.4295522021188994, 0.0},
      {0.1408955957622013, 0.4295522021188995, 0.0},
      {0.2956082808724017, 0.0624487421796329, 0.0},
      {0.6419429769479655, 0.2956082808724017, 0.0},
      {0.0624487421796328, 0.6419429769479658, 0.0},
      {0.6419429769479654, 0.0624487421796329, 0.0},
      {0.2956082808724017, 0.6419429769479655, 0.0},
      {0.0624487421796327, 0.2956082808724020, 0.0},
      {0.1213934507540912, 0.0502111859134281, 0.0},
      {0.8283953633324806, 0.1213934507540912, 0.0},
      {0.0502111859134282, 0.8283953633324808, 0.0},
      {0.8283953633324806, 0.0502111859134281, 0.0},
      {0.1213934507540914, 0.8283953633324808, 0.0},
      {0.0502111859134279, 0.1213934507540915, 0.0},
      {0.3738238003102096, 0.0257279987428787, 0.0},
      {0.6004482009469115, 0.3738238003102097, 0.0},
      {0.0257279987428787, 0.6004482009469118, 0.0},
      {0.6004482009469116, 0.0257279987428787, 0.0},
      {0.3738238003102097, 0.6004482009469116, 0.0},
      {0.0257279987428786, 0.3738238003102100, 0.0},
      {0.4427834065202436, 0.0056465659934662, 0.0},
      {0.5515700274862902, 0.4427834065202436, 0.0},
      {0.0056465659934661, 0.5515700274862906, 0.0},
      {0.5515700274862904, 0.0056465659934662, 0.0},
      {0.4427834065202436, 0.5515700274862904, 0.0},
      {0.0056465659934661, 0.4427834065202438, 0.0},
      {0.3377986632005823, 0.1180890697150951, 0.0},
      {0.5441122670843226, 0.3377986632005824, 0.0},
      {0.1180890697150950, 0.5441122670843228, 0.0},
      {0.5441122670843226, 0.1180890697150951, 0.0},
      {0.3377986632005823, 0.5441122670843228, 0.0},
      {0.1180890697150949, 0.3377986632005825, 0.0},
      {0.0922291891952822, 0.0182422910122947, 0.0},
      {0.8895285197924230, 0.0922291891952822, 0.0},
      {0.0182422910122948, 0.8895285197924232, 0.0},
      {0.8895285197924231, 0.0182422910122947, 0.0},
      {0.0922291891952822, 0.8895285197924232, 0.0},
      {0.0182422910122945, 0.0922291891952825, 0.0},
      {0.4848411325625894, 0.0303177348748211, 0.0},
      {0.4848411325625895, 0.4848411325625895, 0.0},
      {0.0303177348748211, 0.4848411325625896, 0.0},
      {0.1586376888630596, 0.1586376888630598, 0.0},
      {0.6827246222738805, 0.1586376888630597, 0.0},
      {0.1586376888630599, 0.6827246222738806, 0.0},
      {0.0608391923927586, 0.0608391923927589, 0.0},
      {0.8783216152144825, 0.0608391923927586, 0.0},
      {0.0608391923927590, 0.8783216152144826, 0.0},
      {0.0702907404781327, 0.0012002556014872, 0.0},
      {0.9285090039203800, 0.0702907404781327, 0.0},
      {0.0012002556014873, 0.9285090039203803, 0.0},
      {0.9285090039203802, 0.0012002556014872, 0.0},
      {0.0702907404781328, 0.9285090039203803, 0.0},
      {0.0012002556014869, 0.0702907404781331, 0.0}},
      {0.0071812331503231,
      0.0071812331503231,
      0.0071812331503231,
      0.0010587697788405,
      0.0010587697788405,
      0.0010587697788405,
      0.0010587697788405,
      0.0010587697788405,
      0.0010587697788405,
      0.0029478361172571,
      0.0029478361172571,
      0.0029478361172571,
      0.0029478361172571,
      0.0029478361172571,
      0.0029478361172571,
      0.0001555676043408,
      0.0001555676043408,
      0.0001555676043408,
      0.0044258525054466,
      0.0044258525054466,
      0.0044258525054466,
      0.0062786281083384,
      0.0062786281083384,
      0.0062786281083384,
      0.0062786281083384,
      0.0062786281083384,
      0.0062786281083384,
      0.0071052931952241,
      0.0071052931952241,
      0.0071052931952241,
      0.0065530570620497,
      0.0065530570620497,
      0.0065530570620497,
      0.0065530570620497,
      0.0065530570620497,
      0.0065530570620497,
      0.0008894477096278,
      0.0008894477096278,
      0.0008894477096278,
      0.0008894477096278,
      0.0008894477096278,
      0.0008894477096278,
      0.0040059646433475,
      0.0040059646433475,
      0.0040059646433475,
      0.0040059646433475,
      0.0040059646433475,
      0.0040059646433475,
      0.0042323478671383,
      0.0042323478671383,
      0.0042323478671383,
      0.0042323478671383,
      0.0042323478671383,
      0.0042323478671383,
      0.0065463742945445,
      0.0065463742945445,
      0.0065463742945445,
      0.0011883821338439,
      0.0011883821338439,
      0.0011883821338439,
      0.0011883821338439,
      0.0011883821338439,
      0.0011883821338439,
      0.0021384711675090,
      0.0021384711675090,
      0.0021384711675090,
      0.0021384711675090,
      0.0021384711675090,
      0.0021384711675090,
      0.0025656386910187,
      0.0025656386910187,
      0.0025656386910187,
      0.0025656386910187,
      0.0025656386910187,
      0.0025656386910187,
      0.0004742702129447,
      0.0004742702129447,
      0.0004742702129447,
      0.0004742702129447,
      0.0004742702129447,
      0.0004742702129447,
      0.0051229932808521,
      0.0051229932808521,
      0.0051229932808521,
      0.0051229932808521,
      0.0051229932808521,
      0.0051229932808521,
      0.0011622726822320,
      0.0011622726822320,
      0.0011622726822320,
      0.0011622726822320,
      0.0011622726822320,
      0.0011622726822320,
      0.0039049716855432,
      0.0039049716855432,
      0.0039049716855432,
      0.0067737978016627,
      0.0067737978016627,
      0.0067737978016627,
      0.0044625580701383,
      0.0044625580701383,
      0.0044625580701383,
      0.0044625580701383,
      0.0044625580701383,
      0.0044625580701383,
      0.0029589418861770,
      0.0029589418861770,
      0.0029589418861770,
      0.0029589418861770,
      0.0029589418861770,
      0.0029589418861770,
      0.0031302673214843,
      0.0031302673214843,
      0.0031302673214843,
      0.0031302673214843,
      0.0031302673214843,
      0.0031302673214843,
      0.0015700888217440,
      0.0015700888217440,
      0.0015700888217440,
      0.0015700888217440,
      0.0015700888217440,
      0.0015700888217440,
      0.0063935692114779,
      0.0063935692114779,
      0.0063935692114779,
      0.0063935692114779,
      0.0063935692114779,
      0.0063935692114779,
      0.0016024580953965,
      0.0016024580953965,
      0.0016024580953965,
      0.0016024580953965,
      0.0016024580953965,
      0.0016024580953965,
      0.0038410552892975,
      0.0038410552892975,
      0.0038410552892975,
      0.0059713348201241,
      0.0059713348201241,
      0.0059713348201241,
      0.0025641260023390,
      0.0025641260023390,
      0.0025641260023390,
      0.0003625672974930,
      0.0003625672974930,
      0.0003625672974930,
      0.0003625672974930,
      0.0003625672974930,
      0.0003625672974930}
    },
    {
      159,
      {{0.0589421088402292, 0.0027287432479211, 0.0},
      {0.9383291479118497, 0.0589421088402292, 0.0},
      {0.0027287432479212, 0.9383291479118498, 0.0},
      {0.9383291479118497, 0.0027287432479211, 0.0},
      {0.0589421088402293, 0.9383291479118498, 0.0},
      {0.0027287432479208, 0.0589421088402295, 0.0},
      {0.3497880100093318, 0.1571776998671935, 0.0},
      {0.4930342901234747, 0.3497880100093319, 0.0},
      {0.1571776998671934, 0.4930342901234749, 0.0},
      {0.4930342901234748, 0.1571776998671935, 0.0},
      {0.3497880100093319, 0.4930342901234748, 0.0},
      {0.1571776998671934, 0.3497880100093320, 0.0},
      {0.3230018235435502, 0.0021009666448276, 0.0},
      {0.6748972098116223, 0.3230018235435502, 0.0},
      {0.0021009666448276, 0.6748972098116225, 0.0},
      {0.6748972098116223, 0.0021009666448276, 0.0},
      {0.3230018235435501, 0.6748972098116225, 0.0},
      {0.0021009666448274, 0.3230018235435504, 0.0},
      {0.4989148246376862, 0.0021703507246276, 0.0},
      {0.4989148246376862, 0.4989148246376863, 0.0},
      {0.0021703507246276, 0.4989148246376864, 0.0},
      {0.1581458542495161, 0.0681658088137464, 0.0},
      {0.7736883369367373, 0.1581458542495161, 0.0},
      {0.0681658088137465, 0.7736883369367377, 0.0},
      {0.7736883369367374, 0.0681658088137464, 0.0},
      {0.1581458542495162, 0.7736883369367376, 0.0},
      {0.0681658088137463, 0.1581458542495164, 0.0},
      {0.0295494682613534, 0.0108309586036094, 0.0},
      {0.9596195731350373, 0.0295494682613534, 0.0},
      {0.0108309586036096, 0.9596195731350372, 0.0},
      {0.9596195731350373, 0.0108309586036094, 0.0},
      {0.0295494682613534, 0.9596195731350375, 0.0},
      {0.0108309586036091, 0.0295494682613537, 0.0},
      {0.2918191734265371, 0.2189323419801726, 0.0},
      {0.4892484845932904, 0.2918191734265372, 0.0},
      {0.2189323419801725, 0.4892484845932905, 0.0},
      {0.4892484845932904, 0.2189323419801726, 0.0},
      {0.2918191734265371, 0.4892484845932905, 0.0},
      {0.2189323419801724, 0.2918191734265372, 0.0},
      {0.0755221785129958, 0.0212868962407333, 0.0},
      {0.9031909252462709, 0.0755221785129958, 0.0},
      {0.0212868962407334, 0.9031909252462711, 0.0},
      {0.9031909252462710, 0.0212868962407333, 0.0},
      {0.0755221785129958, 0.9031909252462711, 0.0},
      {0.0212868962407330, 0.0755221785129961, 0.0},
      {0.1171166695088989, 0.0408472165761025, 0.0},
      {0.8420361139149987, 0.1171166695088989, 0.0},
      {0.0408472165761026, 0.8420361139149987, 0.0},
      {0.8420361139149987, 0.0408472165761025, 0.0},
      {0.1171166695088989, 0.8420361139149989, 0.0},
      {0.0408472165761022, 0.1171166695088992, 0.0},
      {0.0111662181081692, 0.0016034964960438, 0.0},
      {0.9872302853957871, 0.0111662181081691, 0.0},
      {0.0016034964960439, 0.9872302853957873, 0.0},
      {0.9872302853957869, 0.0016034964960438, 0.0},
      {0.0111662181081692, 0.9872302853957873, 0.0},
      {0.0016034964960435, 0.0111662181081695, 0.0},
      {0.4343804267617306, 0.1312391464765388, 0.0},
      {0.4343804267617307, 0.4343804267617307, 0.0},
      {0.1312391464765387, 0.4343804267617308, 0.0},
      {0.0410973356271179, 0.0410973356271182, 0.0},
      {0.9178053287457638, 0.0410973356271179, 0.0},
      {0.0410973356271184, 0.9178053287457639, 0.0},
      {0.2080456492790871, 0.1015459852268340, 0.0},
      {0.6904083654940788, 0.2080456492790872, 0.0},
      {0.1015459852268340, 0.6904083654940790, 0.0},
      {0.6904083654940788, 0.1015459852268340, 0.0},
      {0.2080456492790871, 0.6904083654940790, 0.0},
      {0.1015459852268338, 0.2080456492790874, 0.0},
      {0.3922101149804348, 0.0415270612688227, 0.0},
      {0.5662628237507425, 0.3922101149804349, 0.0},
      {0.0415270612688227, 0.5662628237507427, 0.0},
      {0.5662628237507424, 0.0415270612688227, 0.0},
      {0.3922101149804348, 0.5662628237507425, 0.0},
      {0.0415270612688226, 0.3922101149804351, 0.0},
      {0.3597112755099758, 0.0938490411451324, 0.0},
      {0.5464396833448917, 0.3597112755099759, 0.0},
      {0.0938490411451324, 0.5464396833448918, 0.0},
      {0.5464396833448917, 0.0938490411451324, 0.0},
      {0.3597112755099759, 0.5464396833448918, 0.0},
      {0.0938490411451323, 0.3597112755099761, 0.0},
      {0.2458746994828754, 0.0086860298043841, 0.0},
      {0.7454392707127403, 0.2458746994828755, 0.0},
      {0.0086860298043842, 0.7454392707127405, 0.0},
      {0.7454392707127405, 0.0086860298043841, 0.0},
      {0.2458746994828755, 0.7454392707127405, 0.0},
      {0.0086860298043840, 0.2458746994828757, 0.0},
      {0.1670027381749231, 0.0175891240440456, 0.0},
      {0.8154081377810313, 0.1670027381749231, 0.0},
      {0.0175891240440457, 0.8154081377810315, 0.0},
      {0.8154081377810313, 0.0175891240440456, 0.0},
      {0.1670027381749231, 0.8154081377810315, 0.0},
      {0.0175891240440454, 0.1670027381749234, 0.0},
      {0.2084053051324008, 0.2084053051324010, 0.0},
      {0.5831893897351982, 0.2084053051324008, 0.0},
      {0.2084053051324009, 0.5831893897351984, 0.0},
      {0.1150085986319464, 0.0055235245122126, 0.0},
      {0.8794678768558409, 0.1150085986319464, 0.0},
      {0.0055235245122127, 0.8794678768558412, 0.0},
      {0.8794678768558410, 0.0055235245122126, 0.0},
      {0.1150085986319465, 0.8794678768558412, 0.0},
      {0.0055235245122124, 0.1150085986319467, 0.0},
      {0.1607458844319635, 0.1607458844319638, 0.0},
      {0.6785082311360726, 0.1607458844319636, 0.0},
      {0.1607458844319638, 0.6785082311360728, 0.0},
      {0.3153953981173191, 0.0238589269426556, 0.0},
      {0.6607456749400253, 0.3153953981173191, 0.0},
      {0.0238589269426557, 0.6607456749400253, 0.0},
      {0.6607456749400252, 0.0238589269426556, 0.0},
      {0.3153953981173192, 0.6607456749400253, 0.0},
      {0.0238589269426554, 0.3153953981173194, 0.0},
      {0.2232226502248206, 0.0402953345447718, 0.0},
      {0.7364820152304076, 0.2232226502248206, 0.0},
      {0.0402953345447719, 0.7364820152304077, 0.0},
      {0.7364820152304076, 0.0402953345447718, 0.0},
      {0.2232226502248207, 0.7364820152304077, 0.0},
      {0.0402953345447716, 0.2232226502248209, 0.0},
      {0.2883958599187324, 0.0678784043114471, 0.0},
      {0.6437257357698205, 0.2883958599187324, 0.0},
      {0.0678784043114471, 0.6437257357698207, 0.0},
      {0.6437257357698205, 0.0678784043114471, 0.0},
      {0.2883958599187324, 0.6437257357698207, 0.0},
      {0.0678784043114470, 0.2883958599187327, 0.0},
      {0.4884016029326027, 0.0231967941347945, 0.0},
      {0.4884016029326027, 0.4884016029326028, 0.0},
      {0.0231967941347945, 0.4884016029326029, 0.0},
      {0.2673663502727756, 0.1395356071810827, 0.0},
      {0.5930980425461417, 0.2673663502727756, 0.0},
      {0.1395356071810827, 0.5930980425461420, 0.0},
      {0.5930980425461417, 0.1395356071810827, 0.0},
      {0.2673663502727755, 0.5930980425461420, 0.0},
      {0.1395356071810825, 0.2673663502727758, 0.0},
      {0.3023864112151284, 0.3023864112151286, 0.0},
      {0.3952271775697430, 0.3023864112151285, 0.0},
      {0.3023864112151284, 0.3952271775697432, 0.0},
      {0.4057653952988916, 0.0080665857041666, 0.0},
      {0.5861680189969418, 0.4057653952988916, 0.0},
      {0.0080665857041666, 0.5861680189969420, 0.0},
      {0.5861680189969418, 0.0080665857041666, 0.0},
      {0.4057653952988916, 0.5861680189969419, 0.0},
      {0.0080665857041665, 0.4057653952988918, 0.0},
      {0.1144268129944254, 0.1144268129944257, 0.0},
      {0.7711463740111488, 0.1144268129944254, 0.0},
      {0.1144268129944257, 0.7711463740111490, 0.0},
      {0.4647624310807389, 0.0704751378385222, 0.0},
      {0.4647624310807389, 0.4647624310807390, 0.0},
      {0.0704751378385221, 0.4647624310807391, 0.0},
      {0.0737218813900997, 0.0737218813901000, 0.0},
      {0.8525562372198003, 0.0737218813900997, 0.0},
      {0.0737218813901001, 0.8525562372198006, 0.0},
      {0.3906191787832637, 0.2187616424334726, 0.0},
      {0.3906191787832638, 0.3906191787832639, 0.0},
      {0.2187616424334725, 0.3906191787832639, 0.0},
      {0.1863207276753595, 0.0001234468122874, 0.0},
      {0.8135558255123531, 0.1863207276753596, 0.0},
      {0.0001234468122875, 0.8135558255123532, 0.0},
      {0.8135558255123529, 0.0001234468122874, 0.0},
      {0.1863207276753596, 0.8135558255123531, 0.0},
      {0.0001234468122872, 0.1863207276753599, 0.0}},
      {0.0003846264857381,
      0.0003846264857381,
      0.0003846264857381,
      0.0003846264857381,
      0.0003846264857381,
      0.0003846264857381,
      0.0052261073484612,
      0.0052261073484612,
      0.0052261073484612,
      0.0052261073484612,
      0.0052261073484612,
      0.0052261073484612,
      0.0006764119746262,
      0.0006764119746262,
      0.0006764119746262,
      0.0006764119746262,
      0.0006764119746262,
      0.0006764119746262,
      0.0007582310515785,
      0.0007582310515785,
      0.0007582310515785,
      0.0032195683537669,
      0.0032195683537669,
      0.0032195683537669,
      0.0032195683537669,
      0.0032195683537669,
      0.0032195683537669,
      0.0006360972185858,
      0.0006360972185858,
      0.0006360972185858,
      0.0006360972185858,
      0.0006360972185858,
      0.0006360972185858,
      0.0067726232860038,
      0.0067726232860038,
      0.0067726232860038,
      0.0067726232860038,
      0.0067726232860038,
      0.0067726232860038,
      0.0013807653747952,
      0.0013807653747952,
      0.0013807653747952,
      0.0013807653747952,
      0.0013807653747952,
      0.0013807653747952,
      0.0022563434227437,
      0.0022563434227437,
      0.0022563434227437,
      0.0022563434227437,
      0.0022563434227437,
      0.0022563434227437,
      0.0001347461159294,
      0.0001347461159294,
      0.0001347461159294,
      0.0001347461159294,
      0.0001347461159294,
      0.0001347461159294,
      0.0055855501475840,
      0.0055855501475840,
      0.0055855501475840,
      0.0014302457530784,
      0.0014302457530784,
      0.0014302457530784,
      0.0046931653381418,
      0.0046931653381418,
      0.0046931653381418,
      0.0046931653381418,
      0.0046931653381418,
      0.0046931653381418,
      0.0039114592291533,
      0.0039114592291533,
      0.0039114592291533,
      0.0039114592291533,
      0.0039114592291533,
      0.0039114592291533,
      0.0053151981815983,
      0.0053151981815983,
      0.0053151981815983,
      0.0053151981815983,
      0.0053151981815983,
      0.0053151981815983,
      0.0015256194116726,
      0.0015256194116726,
      0.0015256194116726,
      0.0015256194116726,
      0.0015256194116726,
      0.0015256194116726,
      0.0019126833358650,
      0.0019126833358650,
      0.0019126833358650,
      0.0019126833358650,
      0.0019126833358650,
      0.0019126833358650,
      0.0062696017213116,
      0.0062696017213116,
      0.0062696017213116,
      0.0008708476156508,
      0.0008708476156508,
      0.0008708476156508,
      0.0008708476156508,
      0.0008708476156508,
      0.0008708476156508,
      0.0052857089291138,
      0.0052857089291138,
      0.0052857089291138,
      0.0028345767408327,
      0.0028345767408327,
      0.0028345767408327,
      0.0028345767408327,
      0.0028345767408327,
      0.0028345767408327,
      0.0032909988819261,
      0.0032909988819261,
      0.0032909988819261,
      0.0032909988819261,
      0.0032909988819261,
      0.0032909988819261,
      0.0045894620824638,
      0.0045894620824638,
      0.0045894620824638,
      0.0045894620824638,
      0.0045894620824638,
      0.0045894620824638,
      0.0030672005823595,
      0.0030672005823595,
      0.0030672005823595,
      0.0062591832494172,
      0.0062591832494172,
      0.0062591832494172,
      0.0062591832494172,
      0.0062591832494172,
      0.0062591832494172,
      0.0081551194038716,
      0.0081551194038716,
      0.0081551194038716,
      0.0018206702056403,
      0.0018206702056403,
      0.0018206702056403,
      0.0018206702056403,
      0.0018206702056403,
      0.0018206702056403,
      0.0040864392206135,
      0.0040864392206135,
      0.0040864392206135,
      0.0051565583176292,
      0.0051565583176292,
      0.0051565583176292,
      0.0028044235664156,
      0.0028044235664156,
      0.0028044235664156,
      0.0079566076420444,
      0.0079566076420444,
      0.0079566076420444,
      0.0003443363125209,
      0.0003443363125209,
      0.0003443363125209,
      0.0003443363125209,
      0.0003443363125209,
      0.0003443363125209}
    },
    {
      171,
      {{0.2590538845210674, 0.0478351231407726, 0.0},
      {0.6931109923381602, 0.2590538845210673, 0.0},
      {0.0478351231407726, 0.6931109923381603, 0.0},
      {0.6931109923381601, 0.0478351231407726, 0.0},
      {0.2590538845210674, 0.6931109923381602, 0.0},
      {0.0478351231407724, 0.2590538845210676, 0.0},
      {0.0033187249366444, 0.0033187249366447, 0.0},
      {0.9933625501267108, 0.0033187249366444, 0.0},
      {0.0033187249366448, 0.9933625501267112, 0.0},
      {0.3915721882912563, 0.0796595269316007, 0.0},
      {0.5287682847771430, 0.3915721882912564, 0.0},
      {0.0796595269316006, 0.5287682847771432, 0.0},
      {0.5287682847771430, 0.0796595269316007, 0.0},
      {0.3915721882912564, 0.5287682847771431, 0.0},
      {0.0796595269316005, 0.3915721882912566, 0.0},
      {0.3561402832962335, 0.0576934012738742, 0.0},
      {0.5861663154298922, 0.3561402832962335, 0.0},
      {0.0576934012738742, 0.5861663154298924, 0.0},
      {0.5861663154298922, 0.0576934012738742, 0.0},
      {0.3561402832962335, 0.5861663154298923, 0.0},
      {0.0576934012738741, 0.3561402832962338, 0.0},
      {0.0723724072246778, 0.0723724072246781, 0.0},
      {0.8552551855506439, 0.0723724072246778, 0.0},
      {0.0723724072246782, 0.8552551855506442, 0.0},
      {0.2830124973495888, 0.0772614375768841, 0.0},
      {0.6397260650735270, 0.2830124973495888, 0.0},
      {0.0772614375768841, 0.6397260650735274, 0.0},
      {0.6397260650735271, 0.0772614375768841, 0.0},
      {0.2830124973495887, 0.6397260650735274, 0.0},
      {0.0772614375768839, 0.2830124973495890, 0.0},
      {0.0471579102421718, 0.0471579102421721, 0.0},
      {0.9056841795156562, 0.0471579102421717, 0.0},
      {0.0471579102421721, 0.9056841795156564, 0.0},
      {0.2416137624515244, 0.0227583842950001, 0.0},
      {0.7356278532534756, 0.2416137624515244, 0.0},
      {0.0227583842950001, 0.7356278532534757, 0.0},
      {0.7356278532534756, 0.0227583842950001, 0.0},
      {0.2416137624515244, 0.7356278532534757, 0.0},
      {0.0227583842949999, 0.2416137624515247, 0.0},
      {0.2527812471879329, 0.1238119787706747, 0.0},
      {0.6234067740413924, 0.2527812471879330, 0.0},
      {0.1238119787706746, 0.6234067740413927, 0.0},
      {0.6234067740413924, 0.1238119787706747, 0.0},
      {0.2527812471879329, 0.6234067740413927, 0.0},
      {0.1238119787706745, 0.2527812471879332, 0.0},
      {0.1849061063839171, 0.1158819672361006, 0.0},
      {0.6992119263799822, 0.1849061063839172, 0.0},
      {0.1158819672361006, 0.6992119263799824, 0.0},
      {0.6992119263799822, 0.1158819672361006, 0.0},
      {0.1849061063839172, 0.6992119263799824, 0.0},
      {0.1158819672361004, 0.1849061063839174, 0.0},
      {0.4680301736511254, 0.0639396526977492, 0.0},
      {0.4680301736511254, 0.4680301736511254, 0.0},
      {0.0639396526977491, 0.4680301736511256, 0.0},
      {0.0126866046744676, 0.0126866046744679, 0.0},
      {0.9746267906510645, 0.0126866046744676, 0.0},
      {0.0126866046744680, 0.9746267906510646, 0.0},
      {0.1937043771536401, 0.0665174447818816, 0.0},
      {0.7397781780644782, 0.1937043771536401, 0.0},
      {0.0665174447818817, 0.7397781780644784, 0.0},
      {0.7397781780644782, 0.0665174447818816, 0.0},
      {0.1937043771536401, 0.7397781780644784, 0.0},
      {0.0665174447818814, 0.1937043771536404, 0.0},
      {0.0764012484393875, 0.0044387813770614, 0.0},
      {0.9191599701835511, 0.0764012484393875, 0.0},
      {0.0044387813770616, 0.9191599701835511, 0.0},
      {0.9191599701835510, 0.0044387813770614, 0.0},
      {0.0764012484393877, 0.9191599701835511, 0.0},
      {0.0044387813770611, 0.0764012484393879, 0.0},
      {0.1215915082227279, 0.1215915082227282, 0.0},
      {0.7568169835545440, 0.1215915082227279, 0.0},
      {0.1215915082227282, 0.7568169835545441, 0.0},
      {0.2090480874526896, 0.0046635793926886, 0.0},
      {0.7862883331546218, 0.2090480874526896, 0.0},
      {0.0046635793926887, 0.7862883331546219, 0.0},
      {0.7862883331546218, 0.0046635793926886, 0.0},
      {0.2090480874526897, 0.7862883331546219, 0.0},
      {0.0046635793926884, 0.2090480874526899, 0.0},
      {0.2985429940592427, 0.0047036817644770, 0.0},
      {0.6967533241762803, 0.2985429940592426, 0.0},
      {0.0047036817644771, 0.6967533241762806, 0.0},
      {0.6967533241762803, 0.0047036817644770, 0.0},
      {0.2985429940592427, 0.6967533241762803, 0.0},
      {0.0047036817644769, 0.2985429940592429, 0.0},
      {0.3343790400340311, 0.0251820667038687, 0.0},
      {0.6404388932621004, 0.3343790400340311, 0.0},
      {0.0251820667038687, 0.6404388932621005, 0.0},
      {0.6404388932621002, 0.0251820667038687, 0.0},
      {0.3343790400340311, 0.6404388932621004, 0.0},
      {0.0251820667038686, 0.3343790400340313, 0.0},
      {0.1232308023806951, 0.0657765738247429, 0.0},
      {0.8109926237945619, 0.1232308023806951, 0.0},
      {0.0657765738247430, 0.8109926237945622, 0.0},
      {0.8109926237945619, 0.0657765738247429, 0.0},
      {0.1232308023806952, 0.8109926237945622, 0.0},
      {0.0657765738247426, 0.1232308023806954, 0.0},
      {0.3385141624298457, 0.1261240949849895, 0.0},
      {0.5353617425851648, 0.3385141624298458, 0.0},
      {0.1261240949849894, 0.5353617425851650, 0.0},
      {0.5353617425851648, 0.1261240949849895, 0.0},
      {0.3385141624298457, 0.5353617425851650, 0.0},
      {0.1261240949849893, 0.3385141624298459, 0.0},
      {0.3544036021806874, 0.1948709509235185, 0.0},
      {0.4507254468957941, 0.3544036021806876, 0.0},
      {0.1948709509235184, 0.4507254468957942, 0.0},
      {0.4507254468957941, 0.1948709509235185, 0.0},
      {0.3544036021806874, 0.4507254468957941, 0.0},
      {0.1948709509235184, 0.3544036021806876, 0.0},
      {0.2630682977578083, 0.1910014245722831, 0.0},
      {0.5459302776699086, 0.2630682977578084, 0.0},
      {0.1910014245722831, 0.5459302776699086, 0.0},
      {0.5459302776699086, 0.1910014245722831, 0.0},
      {0.2630682977578084, 0.5459302776699086, 0.0},
      {0.1910014245722829, 0.2630682977578085, 0.0},
      {0.1824095615174529, 0.1824095615174531, 0.0},
      {0.6351808769650940, 0.1824095615174529, 0.0},
      {0.1824095615174532, 0.6351808769650941, 0.0},
      {0.4345661739646696, 0.0275334061245499, 0.0},
      {0.5379004199107804, 0.4345661739646697, 0.0},
      {0.0275334061245499, 0.5379004199107807, 0.0},
      {0.5379004199107805, 0.0275334061245499, 0.0},
      {0.4345661739646697, 0.5379004199107805, 0.0},
      {0.0275334061245498, 0.4345661739646698, 0.0},
      {0.1640709870698783, 0.0280639219813730, 0.0},
      {0.8078650909487486, 0.1640709870698784, 0.0},
      {0.0280639219813731, 0.8078650909487487, 0.0},
      {0.8078650909487487, 0.0280639219813730, 0.0},
      {0.1640709870698785, 0.8078650909487487, 0.0},
      {0.0280639219813727, 0.1640709870698787, 0.0},
      {0.3622872793529498, 0.2754254412941004, 0.0},
      {0.3622872793529498, 0.3622872793529499, 0.0},
      {0.2754254412941003, 0.3622872793529500, 0.0},
      {0.0426819997060804, 0.0159024162689347, 0.0},
      {0.9414155840249848, 0.0426819997060803, 0.0},
      {0.0159024162689350, 0.9414155840249849, 0.0},
      {0.9414155840249848, 0.0159024162689347, 0.0},
      {0.0426819997060806, 0.9414155840249849, 0.0},
      {0.0159024162689344, 0.0426819997060808, 0.0},
      {0.0939597946527299, 0.0272942306520958, 0.0},
      {0.8787459746951742, 0.0939597946527299, 0.0},
      {0.0272942306520959, 0.8787459746951745, 0.0},
      {0.8787459746951743, 0.0272942306520958, 0.0},
      {0.0939597946527300, 0.8787459746951745, 0.0},
      {0.0272942306520955, 0.0939597946527302, 0.0},
      {0.1354088535199345, 0.0056912114454161, 0.0},
      {0.8588999350346495, 0.1354088535199344, 0.0},
      {0.0056912114454161, 0.8588999350346497, 0.0},
      {0.8588999350346495, 0.0056912114454161, 0.0},
      {0.1354088535199345, 0.8588999350346497, 0.0},
      {0.0056912114454159, 0.1354088535199348, 0.0},
      {0.4367432485484602, 0.1265135029030796, 0.0},
      {0.4367432485484602, 0.4367432485484603, 0.0},
      {0.1265135029030796, 0.4367432485484604, 0.0},
      {0.2724280407839282, 0.2724280407839283, 0.0},
      {0.4551439184321434, 0.2724280407839283, 0.0},
      {0.2724280407839283, 0.4551439184321436, 0.0},
      {0.3962215147396591, 0.0051623470166213, 0.0},
      {0.5986161382437196, 0.3962215147396591, 0.0},
      {0.0051623470166213, 0.5986161382437198, 0.0},
      {0.5986161382437196, 0.0051623470166213, 0.0},
      {0.3962215147396591, 0.5986161382437197, 0.0},
      {0.0051623470166212, 0.3962215147396594, 0.0},
      {0.4973193390003086, 0.0053613219993829, 0.0},
      {0.4973193390003087, 0.4973193390003086, 0.0},
      {0.0053613219993828, 0.4973193390003088, 0.0},
      {0.0294840425976739, 0.0005337086606945, 0.0},
      {0.9699822487416315, 0.0294840425976739, 0.0},
      {0.0005337086606946, 0.9699822487416317, 0.0},
      {0.9699822487416316, 0.0005337086606945, 0.0},
      {0.0294840425976741, 0.9699822487416317, 0.0},
      {0.0005337086606942, 0.0294840425976742, 0.0}},
      {0.0021073792319562,
      0.0021073792319562,
      0.0021073792319562,
      0.0021073792319562,
      0.0021073792319562,
      0.0021073792319562,
      0.0000858649506855,
      0.0000858649506855,
      0.0000858649506855,
      0.0029624613725460,
      0.0029624613725460,
      0.0029624613725460,
      0.0029624613725460,
      0.0029624613725460,
      0.0029624613725460,
      0.0029721100922123,
      0.0029721100922123,
      0.0029721100922123,
      0.0029721100922123,
      0.0029721100922123,
      0.0029721100922123,
      0.0019009663342078,
      0.0019009663342078,
      0.0019009663342078,
      0.0034385921937679,
      0.0034385921937679,
      0.0034385921937679,
      0.0034385921937679,
      0.0034385921937679,
      0.0034385921937679,
      0.0014222168338437,
      0.0014222168338437,
      0.0014222168338437,
      0.0020462494841736,
      0.0020462494841736,
      0.0020462494841736,
      0.0020462494841736,
      0.0020462494841736,
      0.0020462494841736,
      0.0044363272530086,
      0.0044363272530086,
      0.0044363272530086,
      0.0044363272530086,
      0.0044363272530086,
      0.0044363272530086,
      0.0037671614647733,
      0.0037671614647733,
      0.0037671614647733,
      0.0037671614647733,
      0.0037671614647733,
      0.0037671614647733,
      0.0038920643661654,
      0.0038920643661654,
      0.0038920643661654,
      0.0004232750726645,
      0.0004232750726645,
      0.0004232750726645,
      0.0034020033780510,
      0.0034020033780510,
      0.0034020033780510,
      0.0034020033780510,
      0.0034020033780510,
      0.0034020033780510,
      0.0006033482842711,
      0.0006033482842711,
      0.0006033482842711,
      0.0006033482842711,
      0.0006033482842711,
      0.0006033482842711,
      0.0036934559174572,
      0.0036934559174572,
      0.0036934559174572,
      0.0009794742889466,
      0.0009794742889466,
      0.0009794742889466,
      0.0009794742889466,
      0.0009794742889466,
      0.0009794742889466,
      0.0011372229504994,
      0.0011372229504994,
      0.0011372229504994,
      0.0011372229504994,
      0.0011372229504994,
      0.0011372229504994,
      0.0026823424230931,
      0.0026823424230931,
      0.0026823424230931,
      0.0026823424230931,
      0.0026823424230931,
      0.0026823424230931,
      0.0029350349015327,
      0.0029350349015327,
      0.0029350349015327,
      0.0029350349015327,
      0.0029350349015327,
      0.0029350349015327,
      0.0056515117714687,
      0.0056515117714687,
      0.0056515117714687,
      0.0056515117714687,
      0.0056515117714687,
      0.0056515117714687,
      0.0071582426348338,
      0.0071582426348338,
      0.0071582426348338,
      0.0071582426348338,
      0.0071582426348338,
      0.0071582426348338,
      0.0064630117250592,
      0.0064630117250592,
      0.0064630117250592,
      0.0064630117250592,
      0.0064630117250592,
      0.0064630117250592,
      0.0053764254827164,
      0.0053764254827164,
      0.0053764254827164,
      0.0031407982904458,
      0.0031407982904458,
      0.0031407982904458,
      0.0031407982904458,
      0.0031407982904458,
      0.0031407982904458,
      0.0023509684970530,
      0.0023509684970530,
      0.0023509684970530,
      0.0023509684970530,
      0.0023509684970530,
      0.0023509684970530,
      0.0077980456629129,
      0.0077980456629129,
      0.0077980456629129,
      0.0009152534380744,
      0.0009152534380744,
      0.0009152534380744,
      0.0009152534380744,
      0.0009152534380744,
      0.0009152534380744,
      0.0019109402735475,
      0.0019109402735475,
      0.0019109402735475,
      0.0019109402735475,
      0.0019109402735475,
      0.0019109402735475,
      0.0009599402787344,
      0.0009599402787344,
      0.0009599402787344,
      0.0009599402787344,
      0.0009599402787344,
      0.0009599402787344,
      0.0062024795140730,
      0.0062024795140730,
      0.0062024795140730,
      0.0074572753813429,
      0.0074572753813429,
      0.0074572753813429,
      0.0013212839615816,
      0.0013212839615816,
      0.0013212839615816,
      0.0013212839615816,
      0.0013212839615816,
      0.0013212839615816,
      0.0013956590598704,
      0.0013956590598704,
      0.0013956590598704,
      0.0001678108557332,
      0.0001678108557332,
      0.0001678108557332,
      0.0001678108557332,
      0.0001678108557332,
      0.0001678108557332}
    },
    {
      181,
      {{0.3333333333333333, 0.3333333333333334, 0.0},
      {0.4861524679130598, 0.0276950641738804, 0.0},
      {0.4861524679130598, 0.4861524679130599, 0.0},
      {0.0276950641738803, 0.4861524679130600, 0.0},
      {0.1390906284720946, 0.1066696808640776, 0.0},
      {0.7542396906638279, 0.1390906284720946, 0.0},
      {0.1066696808640776, 0.7542396906638279, 0.0},
      {0.7542396906638279, 0.1066696808640776, 0.0},
      {0.1390906284720947, 0.7542396906638279, 0.0},
      {0.1066696808640773, 0.1390906284720949, 0.0},
      {0.4195880808642605, 0.1608238382714791, 0.0},
      {0.4195880808642605, 0.4195880808642605, 0.0},
      {0.1608238382714789, 0.4195880808642607, 0.0},
      {0.1972331307896530, 0.0032611556637283, 0.0},
      {0.7995057135466186, 0.1972331307896530, 0.0},
      {0.0032611556637284, 0.7995057135466188, 0.0},
      {0.7995057135466186, 0.0032611556637283, 0.0},
      {0.1972331307896531, 0.7995057135466188, 0.0},
      {0.0032611556637281, 0.1972331307896533, 0.0},
      {0.2385241509843100, 0.0131398178404037, 0.0},
      {0.7483360311752862, 0.2385241509843100, 0.0},
      {0.0131398178404037, 0.7483360311752866, 0.0},
      {0.7483360311752864, 0.0131398178404037, 0.0},
      {0.2385241509843100, 0.7483360311752866, 0.0},
      {0.0131398178404035, 0.2385241509843103, 0.0},
      {0.0792067965066985, 0.0792067965066988, 0.0},
      {0.8415864069866026, 0.0792067965066985, 0.0},
      {0.0792067965066989, 0.8415864069866028, 0.0},
      {0.0297507293859900, 0.0023442211286869, 0.0},
      {0.9679050494853231, 0.0297507293859899, 0.0},
      {0.0023442211286871, 0.9679050494853232, 0.0},
      {0.9679050494853232, 0.0023442211286869, 0.0},
      {0.0297507293859901, 0.9679050494853232, 0.0},
      {0.0023442211286867, 0.0297507293859903, 0.0},
      {0.0177313492497139, 0.0177313492497142, 0.0},
      {0.9645373015005718, 0.0177313492497139, 0.0},
      {0.0177313492497143, 0.9645373015005720, 0.0},
      {0.3957839332968854, 0.0019614003698950, 0.0},
      {0.6022546663332196, 0.3957839332968854, 0.0},
      {0.0019614003698950, 0.6022546663332198, 0.0},
      {0.6022546663332196, 0.0019614003698950, 0.0},
      {0.3957839332968854, 0.6022546663332197, 0.0},
      {0.0019614003698949, 0.3957839332968856, 0.0},
      {0.3516800587760809, 0.1526365657752198, 0.0},
      {0.4956833754486993, 0.3516800587760810, 0.0},
      {0.1526365657752197, 0.4956833754486996, 0.0},
      {0.4956833754486993, 0.1526365657752198, 0.0},
      {0.3516800587760809, 0.4956833754486994, 0.0},
      {0.1526365657752196, 0.3516800587760812, 0.0},
      {0.2024821942617583, 0.1134252783475029, 0.0},
      {0.6840925273907388, 0.2024821942617584, 0.0},
      {0.1134252783475029, 0.6840925273907389, 0.0},
      {0.6840925273907387, 0.1134252783475029, 0.0},
      {0.2024821942617585, 0.6840925273907389, 0.0},
      {0.1134252783475028, 0.2024821942617586, 0.0},
      {0.4225539552222151, 0.0545213844446831, 0.0},
      {0.5229246603331018, 0.4225539552222151, 0.0},
      {0.0545213844446831, 0.5229246603331020, 0.0},
      {0.5229246603331018, 0.0545213844446831, 0.0},
      {0.4225539552222151, 0.5229246603331019, 0.0},
      {0.0545213844446831, 0.4225539552222153, 0.0},
      {0.1246805088474440, 0.0606049291607722, 0.0},
      {0.8147145619917838, 0.1246805088474439, 0.0},
      {0.0606049291607723, 0.8147145619917839, 0.0},
      {0.8147145619917838, 0.0606049291607722, 0.0},
      {0.1246805088474441, 0.8147145619917839, 0.0},
      {0.0606049291607719, 0.1246805088474443, 0.0},
      {0.2555462364028573, 0.0375889521575178, 0.0},
      {0.7068648114396249, 0.2555462364028573, 0.0},
      {0.0375889521575178, 0.7068648114396252, 0.0},
      {0.7068648114396250, 0.0375889521575178, 0.0},
      {0.2555462364028574, 0.7068648114396251, 0.0},
      {0.0375889521575175, 0.2555462364028576, 0.0},
      {0.0745031299353863, 0.0043863301700617, 0.0},
      {0.9211105398945520, 0.0745031299353863, 0.0},
      {0.0043863301700618, 0.9211105398945522, 0.0},
      {0.9211105398945520, 0.0043863301700617, 0.0},
      {0.0745031299353864, 0.9211105398945522, 0.0},
      {0.0043863301700614, 0.0745031299353866, 0.0},
      {0.4168072755236505, 0.0172734143249371, 0.0},
      {0.5659193101514124, 0.4168072755236506, 0.0},
      {0.0172734143249371, 0.5659193101514125, 0.0},
      {0.5659193101514123, 0.0172734143249371, 0.0},
      {0.4168072755236506, 0.5659193101514124, 0.0},
      {0.0172734143249370, 0.4168072755236508, 0.0},
      {0.1928717230061985, 0.0625780220143109, 0.0},
      {0.7445502549794906, 0.1928717230061985, 0.0},
      {0.0625780220143110, 0.7445502549794908, 0.0},
      {0.7445502549794907, 0.0625780220143109, 0.0},
      {0.1928717230061984, 0.7445502549794908, 0.0},
      {0.0625780220143108, 0.1928717230061987, 0.0},
      {0.0473478189395525, 0.0184079470312829, 0.0},
      {0.9342442340291646, 0.0473478189395524, 0.0},
      {0.0184079470312831, 0.9342442340291647, 0.0},
      {0.9342442340291646, 0.0184079470312829, 0.0},
      {0.0473478189395526, 0.9342442340291647, 0.0},
      {0.0184079470312826, 0.0473478189395528, 0.0},
      {0.2184364967909280, 0.2184364967909282, 0.0},
      {0.5631270064181437, 0.2184364967909281, 0.0},
      {0.2184364967909281, 0.5631270064181439, 0.0},
      {0.1724507871838860, 0.1724507871838862, 0.0},
      {0.6550984256322278, 0.1724507871838861, 0.0},
      {0.1724507871838863, 0.6550984256322279, 0.0},
      {0.0510951808223813, 0.0510951808223816, 0.0},
      {0.8978096383552372, 0.0510951808223813, 0.0},
      {0.0510951808223817, 0.8978096383552373, 0.0},
      {0.3419050053475277, 0.0429032089427714, 0.0},
      {0.6151917857097009, 0.3419050053475277, 0.0},
      {0.0429032089427714, 0.6151917857097011, 0.0},
      {0.6151917857097009, 0.0429032089427714, 0.0},
      {0.3419050053475278, 0.6151917857097009, 0.0},
      {0.0429032089427713, 0.3419050053475279, 0.0},
      {0.3258139530575097, 0.0131890837790157, 0.0},
      {0.6609969631634746, 0.3258139530575097, 0.0},
      {0.0131890837790157, 0.6609969631634748, 0.0},
      {0.6609969631634747, 0.0131890837790157, 0.0},
      {0.3258139530575097, 0.6609969631634748, 0.0},
      {0.0131890837790155, 0.3258139530575100, 0.0},
      {0.2894259487571373, 0.0011968816307096, 0.0},
      {0.7093771696121531, 0.2894259487571373, 0.0},
      {0.0011968816307096, 0.7093771696121534, 0.0},
      {0.7093771696121531, 0.0011968816307096, 0.0},
      {0.2894259487571372, 0.7093771696121534, 0.0},
      {0.0011968816307094, 0.2894259487571375, 0.0},
      {0.2987897956760133, 0.2191327614301768, 0.0},
      {0.4820774428938099, 0.2987897956760134, 0.0},
      {0.2191327614301768, 0.4820774428938099, 0.0},
      {0.4820774428938098, 0.2191327614301768, 0.0},
      {0.2987897956760134, 0.4820774428938099, 0.0},
      {0.2191327614301767, 0.2987897956760135, 0.0},
      {0.2744757477427950, 0.0811298915693310, 0.0},
      {0.6443943606878740, 0.2744757477427950, 0.0},
      {0.0811298915693310, 0.6443943606878743, 0.0},
      {0.6443943606878740, 0.0811298915693310, 0.0},
      {0.2744757477427950, 0.6443943606878741, 0.0},
      {0.0811298915693308, 0.2744757477427952, 0.0},
      {0.3532298738914156, 0.0952781144820168, 0.0},
      {0.5514920116265676, 0.3532298738914157, 0.0},
      {0.0952781144820168, 0.5514920116265677, 0.0},
      {0.5514920116265676, 0.0952781144820168, 0.0},
      {0.3532298738914156, 0.5514920116265677, 0.0},
      {0.0952781144820166, 0.3532298738914159, 0.0},
      {0.4977486206264703, 0.0045027587470595, 0.0},
      {0.4977486206264703, 0.4977486206264704, 0.0},
      {0.0045027587470594, 0.4977486206264705, 0.0},
      {0.2663678824750718, 0.1468887080021119, 0.0},
      {0.5867434095228163, 0.2663678824750718, 0.0},
      {0.1468887080021118, 0.5867434095228166, 0.0},
      {0.5867434095228162, 0.1468887080021119, 0.0},
      {0.2663678824750718, 0.5867434095228163, 0.0},
      {0.1468887080021118, 0.2663678824750720, 0.0},
      {0.3851684692456800, 0.2296630615086399, 0.0},
      {0.3851684692456801, 0.3851684692456802, 0.0},
      {0.2296630615086398, 0.3851684692456802, 0.0},
      {0.2974914903679928, 0.2974914903679930, 0.0},
      {0.4050170192640142, 0.2974914903679929, 0.0},
      {0.2974914903679928, 0.4050170192640143, 0.0},
      {0.0960481521010511, 0.0275306300785875, 0.0},
      {0.8764212178203614, 0.0960481521010511, 0.0},
      {0.0275306300785876, 0.8764212178203615, 0.0},
      {0.8764212178203614, 0.0275306300785875, 0.0},
      {0.0960481521010512, 0.8764212178203615, 0.0},
      {0.0275306300785873, 0.0960481521010514, 0.0},
      {0.0050828094627277, 0.0050828094627280, 0.0},
      {0.9898343810745441, 0.0050828094627277, 0.0},
      {0.0050828094627282, 0.9898343810745442, 0.0},
      {0.1674078247783561, 0.0258751082199382, 0.0},
      {0.8067170670017056, 0.1674078247783561, 0.0},
      {0.0258751082199382, 0.8067170670017060, 0.0},
      {0.8067170670017058, 0.0258751082199382, 0.0},
      {0.1674078247783561, 0.8067170670017060, 0.0},
      {0.0258751082199379, 0.1674078247783564, 0.0},
      {0.4503409417390182, 0.0993181165219637, 0.0},
      {0.4503409417390182, 0.4503409417390183, 0.0},
      {0.0993181165219635, 0.4503409417390184, 0.0},
      {0.1306776337702883, 0.0057973186847398, 0.0},
      {0.8635250475449717, 0.1306776337702883, 0.0},
      {0.0057973186847399, 0.8635250475449719, 0.0},
      {0.8635250475449718, 0.0057973186847398, 0.0},
      {0.1306776337702885, 0.8635250475449719, 0.0},
      {0.0057973186847396, 0.1306776337702887, 0.0}},
      {0.0041715604069432,
      0.0020219941982597,
      0.0020219941982597,
      0.0020219941982597,
      0.0029031466241818,
      0.0029031466241818,
      0.0029031466241818,
      0.0029031466241818,
      0.0029031466241818,
      0.0029031466241818,
      0.0044215413215441,
      0.0044215413215441,
      0.0044215413215441,
      0.0006419258493849,
      0.0006419258493849,
      0.0006419258493849,
      0.0006419258493849,
      0.0006419258493849,
      0.0006419258493849,
      0.0014132571379638,
      0.0014132571379638,
      0.0014132571379638,
      0.0014132571379638,
      0.0014132571379638,
      0.0014132571379638,
      0.0021118649255516,
      0.0021118649255516,
      0.0021118649255516,
      0.0002527545234362,
      0.0002527545234362,
      0.0002527545234362,
      0.0002527545234362,
      0.0002527545234362,
      0.0002527545234362,
      0.0004934330725136,
      0.0004934330725136,
      0.0004934330725136,
      0.0006513477244868,
      0.0006513477244868,
      0.0006513477244868,
      0.0006513477244868,
      0.0006513477244868,
      0.0006513477244868,
      0.0050962897221707,
      0.0050962897221707,
      0.0050962897221707,
      0.0050962897221707,
      0.0050962897221707,
      0.0050962897221707,
      0.0039334352478690,
      0.0039334352478690,
      0.0039334352478690,
      0.0039334352478690,
      0.0039334352478690,
      0.0039334352478690,
      0.0035604046207094,
      0.0035604046207094,
      0.0035604046207094,
      0.0035604046207094,
      0.0035604046207094,
      0.0035604046207094,
      0.0024820572382538,
      0.0024820572382538,
      0.0024820572382538,
      0.0024820572382538,
      0.0024820572382538,
      0.0024820572382538,
      0.0027182187365068,
      0.0027182187365068,
      0.0027182187365068,
      0.0027182187365068,
      0.0027182187365068,
      0.0027182187365068,
      0.0005762914700421,
      0.0005762914700421,
      0.0005762914700421,
      0.0005762914700421,
      0.0005762914700421,
      0.0005762914700421,
      0.0020891663459108,
      0.0020891663459108,
      0.0020891663459108,
      0.0020891663459108,
      0.0020891663459108,
      0.0020891663459108,
      0.0031812115264334,
      0.0031812115264334,
      0.0031812115264334,
      0.0031812115264334,
      0.0031812115264334,
      0.0031812115264334,
      0.0009063846441420,
      0.0009063846441420,
      0.0009063846441420,
      0.0009063846441420,
      0.0009063846441420,
      0.0009063846441420,
      0.0056185017540066,
      0.0056185017540066,
      0.0056185017540066,
      0.0047795979833642,
      0.0047795979833642,
      0.0047795979833642,
      0.0016295433250929,
      0.0016295433250929,
      0.0016295433250929,
      0.0033449796225228,
      0.0033449796225228,
      0.0033449796225228,
      0.0033449796225228,
      0.0033449796225228,
      0.0033449796225228,
      0.0018198567424737,
      0.0018198567424737,
      0.0018198567424737,
      0.0018198567424737,
      0.0018198567424737,
      0.0018198567424737,
      0.0004646851688466,
      0.0004646851688466,
      0.0004646851688466,
      0.0004646851688466,
      0.0004646851688466,
      0.0004646851688466,
      0.0064477933939921,
      0.0064477933939921,
      0.0064477933939921,
      0.0064477933939921,
      0.0064477933939921,
      0.0064477933939921,
      0.0042379488585683,
      0.0042379488585683,
      0.0042379488585683,
      0.0042379488585683,
      0.0042379488585683,
      0.0042379488585683,
      0.0048111377658484,
      0.0048111377658484,
      0.0048111377658484,
      0.0048111377658484,
      0.0048111377658484,
      0.0048111377658484,
      0.0011638303210282,
      0.0011638303210282,
      0.0011638303210282,
      0.0057998345156928,
      0.0057998345156928,
      0.0057998345156928,
      0.0057998345156928,
      0.0057998345156928,
      0.0057998345156928,
      0.0068696975982841,
      0.0068696975982841,
      0.0068696975982841,
      0.0061795202546734,
      0.0061795202546734,
      0.0061795202546734,
      0.0018332933524634,
      0.0018332933524634,
      0.0018332933524634,
      0.0018332933524634,
      0.0018332933524634,
      0.0018332933524634,
      0.0001722006896363,
      0.0001722006896363,
      0.0001722006896363,
      0.0022226689956262,
      0.0022226689956262,
      0.0022226689956262,
      0.0022226689956262,
      0.0022226689956262,
      0.0022226689956262,
      0.0052169886478903,
      0.0052169886478903,
      0.0052169886478903,
      0.0009106263920612,
      0.0009106263920612,
      0.0009106263920612,
      0.0009106263920612,
      0.0009106263920612,
      0.0009106263920612}
    },
    {
      193,
      {{0.0014574454623699, 0.0014574454623702, 0.0},
      {0.9970851090752599, 0.0014574454623698, 0.0},
      {0.0014574454623703, 0.9970851090752600, 0.0},
      {0.2068316863159020, 0.1588679412290467, 0.0},
      {0.6343003724550512, 0.2068316863159021, 0.0},
      {0.1588679412290467, 0.6343003724550514, 0.0},
      {0.6343003724550512, 0.1588679412290467, 0.0},
      {0.2068316863159020, 0.6343003724550514, 0.0},
      {0.1588679412290465, 0.2068316863159023, 0.0},
      {0.4467614265331670, 0.0147664424308547, 0.0},
      {0.5384721310359784, 0.4467614265331671, 0.0},
      {0.0147664424308546, 0.5384721310359786, 0.0},
      {0.5384721310359784, 0.0147664424308547, 0.0},
      {0.4467614265331671, 0.5384721310359784, 0.0},
      {0.0147664424308546, 0.4467614265331673, 0.0},
      {0.4986508935788473, 0.0026982128423054, 0.0},
      {0.4986508935788473, 0.4986508935788474, 0.0},
      {0.0026982128423053, 0.4986508935788476, 0.0},
      {0.2324662076463787, 0.0514311272402331, 0.0},
      {0.7161026651133882, 0.2324662076463787, 0.0},
      {0.0514311272402331, 0.7161026651133885, 0.0},
      {0.7161026651133883, 0.0514311272402331, 0.0},
      {0.2324662076463787, 0.7161026651133884, 0.0},
      {0.0514311272402328, 0.2324662076463790, 0.0},
      {0.1168349267069579, 0.0195597403645924, 0.0},
      {0.8636053329284497, 0.1168349267069579, 0.0},
      {0.0195597403645925, 0.8636053329284498, 0.0},
      {0.8636053329284497, 0.0195597403645924, 0.0},
      {0.1168349267069580, 0.8636053329284498, 0.0},
      {0.0195597403645921, 0.1168349267069583, 0.0},
      {0.3282405458449340, 0.0126185667501572, 0.0},
      {0.6591408874049088, 0.3282405458449341, 0.0},
      {0.0126185667501573, 0.6591408874049089, 0.0},
      {0.6591408874049087, 0.0126185667501572, 0.0},
      {0.3282405458449340, 0.6591408874049089, 0.0},
      {0.0126185667501570, 0.3282405458449343, 0.0},
      {0.2593509320031201, 0.0210621566756392, 0.0},
      {0.7195869113212406, 0.2593509320031202, 0.0},
      {0.0210621566756393, 0.7195869113212408, 0.0},
      {0.7195869113212405, 0.0210621566756392, 0.0},
      {0.2593509320031203, 0.7195869113212406, 0.0},
      {0.0210621566756391, 0.2593509320031205, 0.0},
      {0.1654639982373495, 0.1175435092543480, 0.0},
      {0.7169924925083024, 0.1654639982373496, 0.0},
      {0.1175435092543480, 0.7169924925083025, 0.0},
      {0.7169924925083024, 0.1175435092543480, 0.0},
      {0.1654639982373496, 0.7169924925083027, 0.0},
      {0.1175435092543477, 0.1654639982373498, 0.0},
      {0.3824649073809372, 0.0292988505404852, 0.0},
      {0.5882362420785776, 0.3824649073809372, 0.0},
      {0.0292988505404852, 0.5882362420785777, 0.0},
      {0.5882362420785775, 0.0292988505404852, 0.0},
      {0.3824649073809373, 0.5882362420785776, 0.0},
      {0.0292988505404851, 0.3824649073809375, 0.0},
      {0.3978170726467208, 0.0034428216522744, 0.0},
      {0.5987401057010048, 0.3978170726467209, 0.0},
      {0.0034428216522743, 0.5987401057010050, 0.0},
      {0.5987401057010048, 0.0034428216522744, 0.0},
      {0.3978170726467208, 0.5987401057010049, 0.0},
      {0.0034428216522742, 0.3978170726467211, 0.0},
      {0.4811681159958368, 0.0376637680083265, 0.0},
      {0.4811681159958369, 0.4811681159958368, 0.0},
      {0.0376637680083264, 0.4811681159958370, 0.0},
      {0.3254174696466901, 0.0903702811448643, 0.0},
      {0.5842122492084456, 0.3254174696466902, 0.0},
      {0.0903702811448643, 0.5842122492084457, 0.0},
      {0.5842122492084456, 0.0903702811448643, 0.0},
      {0.3254174696466901, 0.5842122492084457, 0.0},
      {0.0903702811448641, 0.3254174696466903, 0.0},
      {0.4522857565116005, 0.0954284869767990, 0.0},
      {0.4522857565116005, 0.4522857565116006, 0.0},
      {0.0954284869767989, 0.4522857565116007, 0.0},
      {0.1133391914876031, 0.0506976359267405, 0.0},
      {0.8359631725856564, 0.1133391914876030, 0.0},
      {0.0506976359267406, 0.8359631725856566, 0.0},
      {0.8359631725856566, 0.0506976359267405, 0.0},
      {0.1133391914876031, 0.8359631725856566, 0.0},
      {0.0506976359267403, 0.1133391914876034, 0.0},
      {0.3110416328276417, 0.0479444789155618, 0.0},
      {0.6410138882567965, 0.3110416328276418, 0.0},
      {0.0479444789155618, 0.6410138882567966, 0.0},
      {0.6410138882567965, 0.0479444789155618, 0.0},
      {0.3110416328276417, 0.6410138882567966, 0.0},
      {0.0479444789155617, 0.3110416328276420, 0.0},
      {0.2988371353889130, 0.0012766573802798, 0.0},
      {0.6998862072308072, 0.2988371353889130, 0.0},
      {0.0012766573802799, 0.6998862072308073, 0.0},
      {0.6998862072308072, 0.0012766573802798, 0.0},
      {0.2988371353889130, 0.6998862072308073, 0.0},
      {0.0012766573802797, 0.2988371353889133, 0.0},
      {0.3763287580479190, 0.2473424839041621, 0.0},
      {0.3763287580479190, 0.3763287580479191, 0.0},
      {0.2473424839041620, 0.3763287580479191, 0.0},
      {0.1521122809437838, 0.0042625634151116, 0.0},
      {0.8436251556411046, 0.1521122809437838, 0.0},
      {0.0042625634151116, 0.8436251556411049, 0.0},
      {0.8436251556411045, 0.0042625634151116, 0.0},
      {0.1521122809437838, 0.8436251556411049, 0.0},
      {0.0042625634151113, 0.1521122809437842, 0.0},
      {0.0886617050339115, 0.0038525119887191, 0.0},
      {0.9074857829773694, 0.0886617050339115, 0.0},
      {0.0038525119887192, 0.9074857829773696, 0.0},
      {0.9074857829773694, 0.0038525119887191, 0.0},
      {0.0886617050339115, 0.9074857829773696, 0.0},
      {0.0038525119887188, 0.0886617050339118, 0.0},
      {0.2830182798495732, 0.2830182798495733, 0.0},
      {0.4339634403008535, 0.2830182798495732, 0.0},
      {0.2830182798495733, 0.4339634403008537, 0.0},
      {0.3191525805930605, 0.2019001486324767, 0.0},
      {0.4789472707744628, 0.3191525805930606, 0.0},
      {0.2019001486324766, 0.4789472707744629, 0.0},
      {0.4789472707744628, 0.2019001486324767, 0.0},
      {0.3191525805930605, 0.4789472707744629, 0.0},
      {0.2019001486324766, 0.3191525805930607, 0.0},
      {0.4136797843254134, 0.1726404313491733, 0.0},
      {0.4136797843254134, 0.4136797843254134, 0.0},
      {0.1726404313491732, 0.4136797843254136, 0.0},
      {0.1667556935821333, 0.0677103932212692, 0.0},
      {0.7655339131965976, 0.1667556935821333, 0.0},
      {0.0677103932212692, 0.7655339131965977, 0.0},
      {0.7655339131965975, 0.0677103932212692, 0.0},
      {0.1667556935821334, 0.7655339131965977, 0.0},
      {0.0677103932212689, 0.1667556935821336, 0.0},
      {0.0302691910404442, 0.0302691910404445, 0.0},
      {0.9394616179191112, 0.0302691910404442, 0.0},
      {0.0302691910404446, 0.9394616179191115, 0.0},
      {0.0128101964801118, 0.0128101964801121, 0.0},
      {0.9743796070397761, 0.0128101964801118, 0.0},
      {0.0128101964801123, 0.9743796070397761, 0.0},
      {0.2421379240194328, 0.0955724171106545, 0.0},
      {0.6622896588699126, 0.2421379240194328, 0.0},
      {0.0955724171106545, 0.6622896588699130, 0.0},
      {0.6622896588699126, 0.0955724171106545, 0.0},
      {0.2421379240194329, 0.6622896588699128, 0.0},
      {0.0955724171106544, 0.2421379240194330, 0.0},
      {0.2798617479978297, 0.1503708645734583, 0.0},
      {0.5697673874287119, 0.2798617479978298, 0.0},
      {0.1503708645734583, 0.5697673874287121, 0.0},
      {0.5697673874287120, 0.1503708645734583, 0.0},
      {0.2798617479978297, 0.5697673874287121, 0.0},
      {0.1503708645734581, 0.2798617479978299, 0.0},
      {0.0673984002072567, 0.0243498506845871, 0.0},
      {0.9082517491081563, 0.0673984002072566, 0.0},
      {0.0243498506845873, 0.9082517491081563, 0.0},
      {0.9082517491081562, 0.0243498506845871, 0.0},
      {0.0673984002072567, 0.9082517491081563, 0.0},
      {0.0243498506845868, 0.0673984002072569, 0.0},
      {0.3716879666934204, 0.1293878673669234, 0.0},
      {0.4989241659396562, 0.3716879666934204, 0.0},
      {0.1293878673669234, 0.4989241659396564, 0.0},
      {0.4989241659396562, 0.1293878673669234, 0.0},
      {0.3716879666934203, 0.4989241659396564, 0.0},
      {0.1293878673669233, 0.3716879666934205, 0.0},
      {0.4056281490658828, 0.0632274204565012, 0.0},
      {0.5311444304776159, 0.4056281490658829, 0.0},
      {0.0632274204565012, 0.5311444304776161, 0.0},
      {0.5311444304776159, 0.0632274204565012, 0.0},
      {0.4056281490658829, 0.5311444304776159, 0.0},
      {0.0632274204565011, 0.4056281490658831, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.0620106498262642, 0.0620106498262645, 0.0},
      {0.8759787003474712, 0.0620106498262642, 0.0},
      {0.0620106498262646, 0.8759787003474714, 0.0},
      {0.2219801657374810, 0.0052666330207658, 0.0},
      {0.7727532012417533, 0.2219801657374810, 0.0},
      {0.0052666330207657, 0.7727532012417535, 0.0},
      {0.7727532012417532, 0.0052666330207658, 0.0},
      {0.2219801657374811, 0.7727532012417533, 0.0},
      {0.0052666330207656, 0.2219801657374814, 0.0},
      {0.1783256820378629, 0.0261588819796510, 0.0},
      {0.7955154359824860, 0.1783256820378630, 0.0},
      {0.0261588819796511, 0.7955154359824862, 0.0},
      {0.7955154359824860, 0.0261588819796510, 0.0},
      {0.1783256820378630, 0.7955154359824862, 0.0},
      {0.0261588819796509, 0.1783256820378632, 0.0},
      {0.0426221905914095, 0.0055547678704322, 0.0},
      {0.9518230415381582, 0.0426221905914095, 0.0},
      {0.0055547678704324, 0.9518230415381584, 0.0},
      {0.9518230415381583, 0.0055547678704322, 0.0},
      {0.0426221905914095, 0.9518230415381584, 0.0},
      {0.0055547678704320, 0.0426221905914099, 0.0},
      {0.2292779969280699, 0.2292779969280701, 0.0},
      {0.5414440061438599, 0.2292779969280700, 0.0},
      {0.2292779969280701, 0.5414440061438600, 0.0},
      {0.1019588269002696, 0.1019588269002698, 0.0},
      {0.7960823461994605, 0.1019588269002696, 0.0},
      {0.1019588269002699, 0.7960823461994607, 0.0},
      {0.0143443003792804, 0.0006230980203451, 0.0},
      {0.9850326016003745, 0.0143443003792804, 0.0},
      {0.0006230980203453, 0.9850326016003745, 0.0},
      {0.9850326016003746, 0.0006230980203451, 0.0},
      {0.0143443003792806, 0.9850326016003745, 0.0},
      {0.0006230980203448, 0.0143443003792807, 0.0}},
      {0.0000266619511667,
      0.0000266619511667,
      0.0000266619511667,
      0.0041746095647689,
      0.0041746095647689,
      0.0041746095647689,
      0.0041746095647689,
      0.0041746095647689,
      0.0041746095647689,
      0.0015203261288865,
      0.0015203261288865,
      0.0015203261288865,
      0.0015203261288865,
      0.0015203261288865,
      0.0015203261288865,
      0.0007073136238314,
      0.0007073136238314,
      0.0007073136238314,
      0.0026430218012663,
      0.0026430218012663,
      0.0026430218012663,
      0.0026430218012663,
      0.0026430218012663,
      0.0026430218012663,
      0.0012466349623346,
      0.0012466349623346,
      0.0012466349623346,
      0.0012466349623346,
      0.0012466349623346,
      0.0012466349623346,
      0.0015102905283568,
      0.0015102905283568,
      0.0015102905283568,
      0.0015102905283568,
      0.0015102905283568,
      0.0015102905283568,
      0.0018220877884381,
      0.0018220877884381,
      0.0018220877884381,
      0.0018220877884381,
      0.0018220877884381,
      0.0018220877884381,
      0.0037405448495801,
      0.0037405448495801,
      0.0037405448495801,
      0.0037405448495801,
      0.0037405448495801,
      0.0037405448495801,
      0.0022840770677860,
      0.0022840770677860,
      0.0022840770677860,
      0.0022840770677860,
      0.0022840770677860,
      0.0022840770677860,
      0.0008426018202671,
      0.0008426018202671,
      0.0008426018202671,
      0.0008426018202671,
      0.0008426018202671,
      0.0008426018202671,
      0.0027900075712066,
      0.0027900075712066,
      0.0027900075712066,
      0.0041046033232397,
      0.0041046033232397,
      0.0041046033232397,
      0.0041046033232397,
      0.0041046033232397,
      0.0041046033232397,
      0.0041892953520631,
      0.0041892953520631,
      0.0041892953520631,
      0.0022328384238099,
      0.0022328384238099,
      0.0022328384238099,
      0.0022328384238099,
      0.0022328384238099,
      0.0022328384238099,
      0.0031014173607614,
      0.0031014173607614,
      0.0031014173607614,
      0.0031014173607614,
      0.0031014173607614,
      0.0031014173607614,
      0.0004444462715760,
      0.0004444462715760,
      0.0004444462715760,
      0.0004444462715760,
      0.0004444462715760,
      0.0004444462715760,
      0.0063306472802430,
      0.0063306472802430,
      0.0063306472802430,
      0.0007426146645926,
      0.0007426146645926,
      0.0007426146645926,
      0.0007426146645926,
      0.0007426146645926,
      0.0007426146645926,
      0.0005563531727953,
      0.0005563531727953,
      0.0005563531727953,
      0.0005563531727953,
      0.0005563531727953,
      0.0005563531727953,
      0.0064413665885877,
      0.0064413665885877,
      0.0064413665885877,
      0.0059866795512414,
      0.0059866795512414,
      0.0059866795512414,
      0.0059866795512414,
      0.0059866795512414,
      0.0059866795512414,
      0.0054402330134671,
      0.0054402330134671,
      0.0054402330134671,
      0.0029411580314099,
      0.0029411580314099,
      0.0029411580314099,
      0.0029411580314099,
      0.0029411580314099,
      0.0029411580314099,
      0.0009296391265970,
      0.0009296391265970,
      0.0009296391265970,
      0.0004008705258365,
      0.0004008705258365,
      0.0004008705258365,
      0.0042243717359755,
      0.0042243717359755,
      0.0042243717359755,
      0.0042243717359755,
      0.0042243717359755,
      0.0042243717359755,
      0.0051057279581831,
      0.0051057279581831,
      0.0051057279581831,
      0.0051057279581831,
      0.0051057279581831,
      0.0051057279581831,
      0.0012449795766083,
      0.0012449795766083,
      0.0012449795766083,
      0.0012449795766083,
      0.0012449795766083,
      0.0012449795766083,
      0.0049442528275112,
      0.0049442528275112,
      0.0049442528275112,
      0.0049442528275112,
      0.0049442528275112,
      0.0049442528275112,
      0.0037045565714682,
      0.0037045565714682,
      0.0037045565714682,
      0.0037045565714682,
      0.0037045565714682,
      0.0037045565714682,
      0.0065150354978813,
      0.0019881742974627,
      0.0019881742974627,
      0.0019881742974627,
      0.0009727966204916,
      0.0009727966204916,
      0.0009727966204916,
      0.0009727966204916,
      0.0009727966204916,
      0.0009727966204916,
      0.0021304182982501,
      0.0021304182982501,
      0.0021304182982501,
      0.0021304182982501,
      0.0021304182982501,
      0.0021304182982501,
      0.0005276871848088,
      0.0005276871848088,
      0.0005276871848088,
      0.0005276871848088,
      0.0005276871848088,
      0.0005276871848088,
      0.0062677988628244,
      0.0062677988628244,
      0.0062677988628244,
      0.0033146995575296,
      0.0033146995575296,
      0.0033146995575296,
      0.0000850441238713,
      0.0000850441238713,
      0.0000850441238713,
      0.0000850441238713,
      0.0000850441238713,
      0.0000850441238713}
    },
    {
      204,
      {{0.0866842399345696, 0.0866842399345699, 0.0},
      {0.8266315201308605, 0.0866842399345696, 0.0},
      {0.0866842399345699, 0.8266315201308606, 0.0},
      {0.4094279608026433, 0.1811440783947133, 0.0},
      {0.4094279608026433, 0.4094279608026434, 0.0},
      {0.1811440783947132, 0.4094279608026435, 0.0},
      {0.2581529287895067, 0.0869286359632870, 0.0},
      {0.6549184352472063, 0.2581529287895068, 0.0},
      {0.0869286359632870, 0.6549184352472064, 0.0},
      {0.6549184352472063, 0.0869286359632870, 0.0},
      {0.2581529287895068, 0.6549184352472064, 0.0},
      {0.0869286359632869, 0.2581529287895070, 0.0},
      {0.2849624019720829, 0.1140920353228314, 0.0},
      {0.6009455627050857, 0.2849624019720829, 0.0},
      {0.1140920353228314, 0.6009455627050859, 0.0},
      {0.6009455627050857, 0.1140920353228314, 0.0},
      {0.2849624019720829, 0.6009455627050859, 0.0},
      {0.1140920353228312, 0.2849624019720831, 0.0},
      {0.3783335382962427, 0.2433329234075146, 0.0},
      {0.3783335382962427, 0.3783335382962428, 0.0},
      {0.2433329234075145, 0.3783335382962429, 0.0},
      {0.4989797178600512, 0.0020405642798976, 0.0},
      {0.4989797178600512, 0.4989797178600513, 0.0},
      {0.0020405642798976, 0.4989797178600514, 0.0},
      {0.1340676338419341, 0.0916653314400512, 0.0},
      {0.7742670347180146, 0.1340676338419342, 0.0},
      {0.0916653314400512, 0.7742670347180148, 0.0},
      {0.7742670347180146, 0.0916653314400512, 0.0},
      {0.1340676338419342, 0.7742670347180148, 0.0},
      {0.0916653314400510, 0.1340676338419345, 0.0},
      {0.3269137502096716, 0.2159393912434157, 0.0},
      {0.4571468585469127, 0.3269137502096717, 0.0},
      {0.2159393912434156, 0.4571468585469129, 0.0},
      {0.4571468585469128, 0.2159393912434157, 0.0},
      {0.3269137502096715, 0.4571468585469129, 0.0},
      {0.2159393912434155, 0.3269137502096717, 0.0},
      {0.2596606680087211, 0.2596606680087213, 0.0},
      {0.4806786639825575, 0.2596606680087212, 0.0},
      {0.2596606680087213, 0.4806786639825577, 0.0},
      {0.1008457159827756, 0.0223903162865835, 0.0},
      {0.8767639677306408, 0.1008457159827756, 0.0},
      {0.0223903162865836, 0.8767639677306410, 0.0},
      {0.8767639677306409, 0.0223903162865835, 0.0},
      {0.1008457159827757, 0.8767639677306409, 0.0},
      {0.0223903162865833, 0.1008457159827759, 0.0},
      {0.3552328254047640, 0.0992419378880283, 0.0},
      {0.5455252367072077, 0.3552328254047640, 0.0},
      {0.0992419378880283, 0.5455252367072079, 0.0},
      {0.5455252367072078, 0.0992419378880283, 0.0},
      {0.3552328254047640, 0.5455252367072079, 0.0},
      {0.0992419378880282, 0.3552328254047642, 0.0},
      {0.1584027341957248, 0.0204436751441748, 0.0},
      {0.8211535906601003, 0.1584027341957248, 0.0},
      {0.0204436751441749, 0.8211535906601005, 0.0},
      {0.8211535906601004, 0.0204436751441748, 0.0},
      {0.1584027341957248, 0.8211535906601006, 0.0},
      {0.0204436751441747, 0.1584027341957251, 0.0},
      {0.4785119732563822, 0.0429760534872357, 0.0},
      {0.4785119732563822, 0.4785119732563822, 0.0},
      {0.0429760534872356, 0.4785119732563824, 0.0},
      {0.4012397484580547, 0.0586121184566509, 0.0},
      {0.5401481330852944, 0.4012397484580547, 0.0},
      {0.0586121184566509, 0.5401481330852946, 0.0},
      {0.5401481330852945, 0.0586121184566509, 0.0},
      {0.4012397484580547, 0.5401481330852945, 0.0},
      {0.0586121184566507, 0.4012397484580549, 0.0},
      {0.2273220131649666, 0.0194249101655772, 0.0},
      {0.7532530766694563, 0.2273220131649666, 0.0},
      {0.0194249101655773, 0.7532530766694563, 0.0},
      {0.7532530766694562, 0.0194249101655772, 0.0},
      {0.2273220131649666, 0.7532530766694563, 0.0},
      {0.0194249101655771, 0.2273220131649669, 0.0},
      {0.1040040001314270, 0.0508701921937650, 0.0},
      {0.8451258076748079, 0.1040040001314270, 0.0},
      {0.0508701921937651, 0.8451258076748080, 0.0},
      {0.8451258076748080, 0.0508701921937650, 0.0},
      {0.1040040001314272, 0.8451258076748080, 0.0},
      {0.0508701921937648, 0.1040040001314274, 0.0},
      {0.2516190193630367, 0.0038062521288191, 0.0},
      {0.7445747285081441, 0.2516190193630367, 0.0},
      {0.0038062521288191, 0.7445747285081444, 0.0},
      {0.7445747285081442, 0.0038062521288191, 0.0},
      {0.2516190193630368, 0.7445747285081442, 0.0},
      {0.0038062521288190, 0.2516190193630370, 0.0},
      {0.1946715273720462, 0.0939537103695900, 0.0},
      {0.7113747622583637, 0.1946715273720463, 0.0},
      {0.0939537103695900, 0.7113747622583638, 0.0},
      {0.7113747622583638, 0.0939537103695900, 0.0},
      {0.1946715273720463, 0.7113747622583639, 0.0},
      {0.0939537103695898, 0.1946715273720465, 0.0},
      {0.1756396314580331, 0.0036669776240121, 0.0},
      {0.8206933909179547, 0.1756396314580332, 0.0},
      {0.0036669776240122, 0.8206933909179549, 0.0},
      {0.8206933909179547, 0.0036669776240121, 0.0},
      {0.1756396314580332, 0.8206933909179549, 0.0},
      {0.0036669776240119, 0.1756396314580335, 0.0},
      {0.1654225039506263, 0.0504257035060658, 0.0},
      {0.7841517925433079, 0.1654225039506263, 0.0},
      {0.0504257035060658, 0.7841517925433081, 0.0},
      {0.7841517925433079, 0.0504257035060658, 0.0},
      {0.1654225039506264, 0.7841517925433080, 0.0},
      {0.0504257035060656, 0.1654225039506266, 0.0},
      {0.0604829240909603, 0.0041121482809412, 0.0},
      {0.9354049276280985, 0.0604829240909602, 0.0},
      {0.0041121482809413, 0.9354049276280987, 0.0},
      {0.9354049276280986, 0.0041121482809412, 0.0},
      {0.0604829240909603, 0.9354049276280987, 0.0},
      {0.0041121482809409, 0.0604829240909606, 0.0},
      {0.3321035653575233, 0.0043366708649929, 0.0},
      {0.6635597637774838, 0.3321035653575233, 0.0},
      {0.0043366708649930, 0.6635597637774840, 0.0},
      {0.6635597637774838, 0.0043366708649929, 0.0},
      {0.3321035653575233, 0.6635597637774838, 0.0},
      {0.0043366708649928, 0.3321035653575235, 0.0},
      {0.3794132168456387, 0.1437641762405430, 0.0},
      {0.4768226069138183, 0.3794132168456388, 0.0},
      {0.1437641762405430, 0.4768226069138184, 0.0},
      {0.4768226069138183, 0.1437641762405430, 0.0},
      {0.3794132168456387, 0.4768226069138183, 0.0},
      {0.1437641762405429, 0.3794132168456389, 0.0},
      {0.0235174430579130, 0.0235174430579133, 0.0},
      {0.9529651138841736, 0.0235174430579130, 0.0},
      {0.0235174430579135, 0.9529651138841736, 0.0},
      {0.0565688972489456, 0.0218658180283708, 0.0},
      {0.9215652847226834, 0.0565688972489457, 0.0},
      {0.0218658180283709, 0.9215652847226837, 0.0},
      {0.9215652847226835, 0.0218658180283708, 0.0},
      {0.0565688972489457, 0.9215652847226837, 0.0},
      {0.0218658180283706, 0.0565688972489460, 0.0},
      {0.3060238617603606, 0.3060238617603607, 0.0},
      {0.3879522764792787, 0.3060238617603607, 0.0},
      {0.3060238617603606, 0.3879522764792788, 0.0},
      {0.1464890507166427, 0.1464890507166429, 0.0},
      {0.7070218985667143, 0.1464890507166427, 0.0},
      {0.1464890507166430, 0.7070218985667144, 0.0},
      {0.0543792032794963, 0.0543792032794966, 0.0},
      {0.8912415934410072, 0.0543792032794963, 0.0},
      {0.0543792032794967, 0.8912415934410073, 0.0},
      {0.2909849162693301, 0.1683073313425024, 0.0},
      {0.5407077523881676, 0.2909849162693301, 0.0},
      {0.1683073313425023, 0.5407077523881677, 0.0},
      {0.5407077523881676, 0.1683073313425024, 0.0},
      {0.2909849162693300, 0.5407077523881677, 0.0},
      {0.1683073313425022, 0.2909849162693303, 0.0},
      {0.2368453944831515, 0.0485139799052084, 0.0},
      {0.7146406256116400, 0.2368453944831516, 0.0},
      {0.0485139799052084, 0.7146406256116403, 0.0},
      {0.7146406256116401, 0.0485139799052084, 0.0},
      {0.2368453944831516, 0.7146406256116402, 0.0},
      {0.0485139799052082, 0.2368453944831518, 0.0},
      {0.4145491529764430, 0.0047372398809322, 0.0},
      {0.5807136071426248, 0.4145491529764431, 0.0},
      {0.0047372398809322, 0.5807136071426250, 0.0},
      {0.5807136071426248, 0.0047372398809322, 0.0},
      {0.4145491529764430, 0.5807136071426249, 0.0},
      {0.0047372398809321, 0.4145491529764432, 0.0},
      {0.3075361882409410, 0.0221352139073682, 0.0},
      {0.6703285978516907, 0.3075361882409411, 0.0},
      {0.0221352139073682, 0.6703285978516910, 0.0},
      {0.6703285978516907, 0.0221352139073682, 0.0},
      {0.3075361882409411, 0.6703285978516909, 0.0},
      {0.0221352139073681, 0.3075361882409413, 0.0},
      {0.1109266644334704, 0.0043933951094667, 0.0},
      {0.8846799404570630, 0.1109266644334704, 0.0},
      {0.0043933951094668, 0.8846799404570631, 0.0},
      {0.8846799404570630, 0.0043933951094667, 0.0},
      {0.1109266644334704, 0.8846799404570631, 0.0},
      {0.0043933951094664, 0.1109266644334707, 0.0},
      {0.2181920158632539, 0.2181920158632541, 0.0},
      {0.5636159682734920, 0.2181920158632540, 0.0},
      {0.2181920158632540, 0.5636159682734921, 0.0},
      {0.3188631220563678, 0.0551602053562474, 0.0},
      {0.6259766725873849, 0.3188631220563678, 0.0},
      {0.0551602053562473, 0.6259766725873851, 0.0},
      {0.6259766725873848, 0.0551602053562474, 0.0},
      {0.3188631220563678, 0.6259766725873850, 0.0},
      {0.0551602053562472, 0.3188631220563680, 0.0},
      {0.4919913444565532, 0.0160173110868935, 0.0},
      {0.4919913444565532, 0.4919913444565533, 0.0},
      {0.0160173110868935, 0.4919913444565535, 0.0},
      {0.3953677591013828, 0.0243998884696387, 0.0},
      {0.5802323524289785, 0.3953677591013828, 0.0},
      {0.0243998884696386, 0.5802323524289787, 0.0},
      {0.5802323524289785, 0.0243998884696387, 0.0},
      {0.3953677591013828, 0.5802323524289786, 0.0},
      {0.0243998884696386, 0.3953677591013830, 0.0},
      {0.4533999616178053, 0.0932000767643895, 0.0},
      {0.4533999616178053, 0.4533999616178053, 0.0},
      {0.0932000767643894, 0.4533999616178054, 0.0},
      {0.2137578302973531, 0.1506349418291862, 0.0},
      {0.6356072278734608, 0.2137578302973531, 0.0},
      {0.1506349418291862, 0.6356072278734609, 0.0},
      {0.6356072278734608, 0.1506349418291862, 0.0},
      {0.2137578302973531, 0.6356072278734609, 0.0},
      {0.1506349418291860, 0.2137578302973533, 0.0},
      {0.0047266605105680, 0.0047266605105683, 0.0},
      {0.9905466789788636, 0.0047266605105680, 0.0},
      {0.0047266605105684, 0.9905466789788638, 0.0},
      {0.0247793702175406, 0.0045023852882974, 0.0},
      {0.9707182444941620, 0.0247793702175406, 0.0},
      {0.0045023852882976, 0.9707182444941621, 0.0},
      {0.9707182444941620, 0.0045023852882974, 0.0},
      {0.0247793702175407, 0.9707182444941621, 0.0},
      {0.0045023852882971, 0.0247793702175410, 0.0}},
      {0.0016011820062071,
      0.0016011820062071,
      0.0016011820062071,
      0.0034173399672340,
      0.0034173399672340,
      0.0034173399672340,
      0.0025054117282874,
      0.0025054117282874,
      0.0025054117282874,
      0.0025054117282874,
      0.0025054117282874,
      0.0025054117282874,
      0.0031333357572990,
      0.0031333357572990,
      0.0031333357572990,
      0.0031333357572990,
      0.0031333357572990,
      0.0031333357572990,
      0.0044670809040683,
      0.0044670809040683,
      0.0044670809040683,
      0.0005205456186432,
      0.0005205456186432,
      0.0005205456186432,
      0.0025530461509317,
      0.0025530461509317,
      0.0025530461509317,
      0.0025530461509317,
      0.0025530461509317,
      0.0025530461509317,
      0.0048196234294204,
      0.0048196234294204,
      0.0048196234294204,
      0.0048196234294204,
      0.0048196234294204,
      0.0048196234294204,
      0.0048694286183427,
      0.0048694286183427,
      0.0048694286183427,
      0.0012106190194343,
      0.0012106190194343,
      0.0012106190194343,
      0.0012106190194343,
      0.0012106190194343,
      0.0012106190194343,
      0.0038486733635274,
      0.0038486733635274,
      0.0038486733635274,
      0.0038486733635274,
      0.0038486733635274,
      0.0038486733635274,
      0.0014941223459745,
      0.0014941223459745,
      0.0014941223459745,
      0.0014941223459745,
      0.0014941223459745,
      0.0014941223459745,
      0.0027321628420230,
      0.0027321628420230,
      0.0027321628420230,
      0.0030994270129717,
      0.0030994270129717,
      0.0030994270129717,
      0.0030994270129717,
      0.0030994270129717,
      0.0030994270129717,
      0.0016584986469565,
      0.0016584986469565,
      0.0016584986469565,
      0.0016584986469565,
      0.0016584986469565,
      0.0016584986469565,
      0.0018544648062947,
      0.0018544648062947,
      0.0018544648062947,
      0.0018544648062947,
      0.0018544648062947,
      0.0018544648062947,
      0.0007713232592353,
      0.0007713232592353,
      0.0007713232592353,
      0.0007713232592353,
      0.0007713232592353,
      0.0007713232592353,
      0.0033060862055342,
      0.0033060862055342,
      0.0033060862055342,
      0.0033060862055342,
      0.0033060862055342,
      0.0033060862055342,
      0.0006775890978753,
      0.0006775890978753,
      0.0006775890978753,
      0.0006775890978753,
      0.0006775890978753,
      0.0006775890978753,
      0.0024518141717764,
      0.0024518141717764,
      0.0024518141717764,
      0.0024518141717764,
      0.0024518141717764,
      0.0024518141717764,
      0.0004578830785905,
      0.0004578830785905,
      0.0004578830785905,
      0.0004578830785905,
      0.0004578830785905,
      0.0004578830785905,
      0.0009109081322132,
      0.0009109081322132,
      0.0009109081322132,
      0.0009109081322132,
      0.0009109081322132,
      0.0009109081322132,
      0.0046211899508548,
      0.0046211899508548,
      0.0046211899508548,
      0.0046211899508548,
      0.0046211899508548,
      0.0046211899508548,
      0.0006906845315140,
      0.0006906845315140,
      0.0006906845315140,
      0.0009803156168466,
      0.0009803156168466,
      0.0009803156168466,
      0.0009803156168466,
      0.0009803156168466,
      0.0009803156168466,
      0.0060443699904583,
      0.0060443699904583,
      0.0060443699904583,
      0.0036887232483671,
      0.0036887232483671,
      0.0036887232483671,
      0.0016046752219708,
      0.0016046752219708,
      0.0016046752219708,
      0.0051049588931983,
      0.0051049588931983,
      0.0051049588931983,
      0.0051049588931983,
      0.0051049588931983,
      0.0051049588931983,
      0.0027318852186861,
      0.0027318852186861,
      0.0027318852186861,
      0.0027318852186861,
      0.0027318852186861,
      0.0027318852186861,
      0.0010055510048923,
      0.0010055510048923,
      0.0010055510048923,
      0.0010055510048923,
      0.0010055510048923,
      0.0010055510048923,
      0.0021341673513173,
      0.0021341673513173,
      0.0021341673513173,
      0.0021341673513173,
      0.0021341673513173,
      0.0021341673513173,
      0.0006439382044477,
      0.0006439382044477,
      0.0006439382044477,
      0.0006439382044477,
      0.0006439382044477,
      0.0006439382044477,
      0.0049631880897721,
      0.0049631880897721,
      0.0049631880897721,
      0.0032999416636284,
      0.0032999416636284,
      0.0032999416636284,
      0.0032999416636284,
      0.0032999416636284,
      0.0032999416636284,
      0.0019311810149830,
      0.0019311810149830,
      0.0019311810149830,
      0.0024737164044041,
      0.0024737164044041,
      0.0024737164044041,
      0.0024737164044041,
      0.0024737164044041,
      0.0024737164044041,
      0.0046668988848233,
      0.0046668988848233,
      0.0046668988848233,
      0.0045907407789636,
      0.0045907407789636,
      0.0045907407789636,
      0.0045907407789636,
      0.0045907407789636,
      0.0045907407789636,
      0.0001467413667890,
      0.0001467413667890,
      0.0001467413667890,
      0.0003220008871737,
      0.0003220008871737,
      0.0003220008871737,
      0.0003220008871737,
      0.0003220008871737,
      0.0003220008871737}
    },
    {
      214,
      {{0.3851262518382810, 0.0210967602701019, 0.0},
      {0.5937769878916170, 0.3851262518382811, 0.0},
      {0.0210967602701019, 0.5937769878916171, 0.0},
      {0.5937769878916170, 0.0210967602701019, 0.0},
      {0.3851262518382810, 0.5937769878916171, 0.0},
      {0.0210967602701018, 0.3851262518382813, 0.0},
      {0.0528200673454988, 0.0033881121503900, 0.0},
      {0.9437918205041110, 0.0528200673454988, 0.0},
      {0.0033881121503903, 0.9437918205041111, 0.0},
      {0.9437918205041111, 0.0033881121503900, 0.0},
      {0.0528200673454988, 0.9437918205041114, 0.0},
      {0.0033881121503898, 0.0528200673454991, 0.0},
      {0.0459825951541271, 0.0459825951541274, 0.0},
      {0.9080348096917454, 0.0459825951541271, 0.0},
      {0.0459825951541276, 0.9080348096917455, 0.0},
      {0.0508987652831316, 0.0182083287994724, 0.0},
      {0.9308929059173959, 0.0508987652831316, 0.0},
      {0.0182083287994725, 0.9308929059173960, 0.0},
      {0.9308929059173960, 0.0182083287994724, 0.0},
      {0.0508987652831318, 0.9308929059173960, 0.0},
      {0.0182083287994721, 0.0508987652831320, 0.0},
      {0.0208387475149684, 0.0208387475149687, 0.0},
      {0.9583225049700629, 0.0208387475149684, 0.0},
      {0.0208387475149688, 0.9583225049700631, 0.0},
      {0.1291243720726085, 0.0605107706433123, 0.0},
      {0.8103648572840793, 0.1291243720726085, 0.0},
      {0.0605107706433124, 0.8103648572840793, 0.0},
      {0.8103648572840793, 0.0605107706433123, 0.0},
      {0.1291243720726087, 0.8103648572840793, 0.0},
      {0.0605107706433121, 0.1291243720726088, 0.0},
      {0.0878179027775245, 0.0359198913635778, 0.0},
      {0.8762622058588977, 0.0878179027775245, 0.0},
      {0.0359198913635778, 0.8762622058588979, 0.0},
      {0.8762622058588977, 0.0359198913635778, 0.0},
      {0.0878179027775245, 0.8762622058588979, 0.0},
      {0.0359198913635776, 0.0878179027775248, 0.0},
      {0.0757621741867908, 0.0757621741867911, 0.0},
      {0.8484756516264180, 0.0757621741867908, 0.0},
      {0.0757621741867912, 0.8484756516264182, 0.0},
      {0.3766019805175457, 0.2467960389649086, 0.0},
      {0.3766019805175457, 0.3766019805175458, 0.0},
      {0.2467960389649085, 0.3766019805175458, 0.0},
      {0.2735537380248192, 0.1640610286702127, 0.0},
      {0.5623852333049681, 0.2735537380248192, 0.0},
      {0.1640610286702127, 0.5623852333049683, 0.0},
      {0.5623852333049681, 0.1640610286702127, 0.0},
      {0.2735537380248192, 0.5623852333049683, 0.0},
      {0.1640610286702125, 0.2735537380248194, 0.0},
      {0.0944770656323742, 0.0106832635235884, 0.0},
      {0.8948396708440374, 0.0944770656323742, 0.0},
      {0.0106832635235886, 0.8948396708440375, 0.0},
      {0.8948396708440374, 0.0106832635235884, 0.0},
      {0.0944770656323743, 0.8948396708440375, 0.0},
      {0.0106832635235882, 0.0944770656323745, 0.0},
      {0.2873558398221649, 0.2873558398221651, 0.0},
      {0.4252883203556701, 0.2873558398221650, 0.0},
      {0.2873558398221651, 0.4252883203556702, 0.0},
      {0.4111522333036005, 0.0311326803063737, 0.0},
      {0.5577150863900258, 0.4111522333036006, 0.0},
      {0.0311326803063737, 0.5577150863900260, 0.0},
      {0.5577150863900259, 0.0311326803063737, 0.0},
      {0.4111522333036006, 0.5577150863900259, 0.0},
      {0.0311326803063736, 0.4111522333036007, 0.0},
      {0.3594475694982553, 0.1331926232217058, 0.0},
      {0.5073598072800389, 0.3594475694982554, 0.0},
      {0.1331926232217058, 0.5073598072800390, 0.0},
      {0.5073598072800389, 0.1331926232217058, 0.0},
      {0.3594475694982553, 0.5073598072800389, 0.0},
      {0.1331926232217057, 0.3594475694982555, 0.0},
      {0.3884100853304371, 0.0742973192372942, 0.0},
      {0.5372925954322688, 0.3884100853304371, 0.0},
      {0.0742973192372942, 0.5372925954322689, 0.0},
      {0.5372925954322687, 0.0742973192372942, 0.0},
      {0.3884100853304370, 0.5372925954322688, 0.0},
      {0.0742973192372941, 0.3884100853304373, 0.0},
      {0.3253550433420577, 0.2051664978147620, 0.0},
      {0.4694784588431804, 0.3253550433420578, 0.0},
      {0.2051664978147619, 0.4694784588431805, 0.0},
      {0.4694784588431803, 0.2051664978147620, 0.0},
      {0.3253550433420578, 0.4694784588431805, 0.0},
      {0.2051664978147618, 0.3253550433420579, 0.0},
      {0.1747804794810684, 0.0902196061251755, 0.0},
      {0.7349999143937561, 0.1747804794810684, 0.0},
      {0.0902196061251755, 0.7349999143937562, 0.0},
      {0.7349999143937561, 0.0902196061251755, 0.0},
      {0.1747804794810685, 0.7349999143937561, 0.0},
      {0.0902196061251753, 0.1747804794810687, 0.0},
      {0.1941646181495633, 0.0439158729045114, 0.0},
      {0.7619195089459252, 0.1941646181495633, 0.0},
      {0.0439158729045115, 0.7619195089459254, 0.0},
      {0.7619195089459252, 0.0439158729045114, 0.0},
      {0.1941646181495633, 0.7619195089459254, 0.0},
      {0.0439158729045113, 0.1941646181495635, 0.0},
      {0.4744124989028456, 0.0511750021943088, 0.0},
      {0.4744124989028456, 0.4744124989028456, 0.0},
      {0.0511750021943088, 0.4744124989028458, 0.0},
      {0.2483278734045790, 0.0689638299646017, 0.0},
      {0.6827082966308193, 0.2483278734045790, 0.0},
      {0.0689638299646018, 0.6827082966308194, 0.0},
      {0.6827082966308193, 0.0689638299646017, 0.0},
      {0.2483278734045790, 0.6827082966308194, 0.0},
      {0.0689638299646015, 0.2483278734045792, 0.0},
      {0.2229920412612009, 0.1251817981428867, 0.0},
      {0.6518261605959125, 0.2229920412612009, 0.0},
      {0.1251817981428867, 0.6518261605959126, 0.0},
      {0.6518261605959125, 0.1251817981428867, 0.0},
      {0.2229920412612009, 0.6518261605959126, 0.0},
      {0.1251817981428865, 0.2229920412612011, 0.0},
      {0.1496044235489511, 0.0046702985612833, 0.0},
      {0.8457252778897656, 0.1496044235489511, 0.0},
      {0.0046702985612834, 0.8457252778897658, 0.0},
      {0.8457252778897657, 0.0046702985612833, 0.0},
      {0.1496044235489512, 0.8457252778897657, 0.0},
      {0.0046702985612830, 0.1496044235489514, 0.0},
      {0.1111727012858456, 0.1111727012858459, 0.0},
      {0.7776545974283084, 0.1111727012858457, 0.0},
      {0.1111727012858459, 0.7776545974283087, 0.0},
      {0.4918649619576897, 0.0162700760846207, 0.0},
      {0.4918649619576897, 0.4918649619576898, 0.0},
      {0.0162700760846206, 0.4918649619576899, 0.0},
      {0.3283914136921266, 0.0489259225038644, 0.0},
      {0.6226826638040091, 0.3283914136921267, 0.0},
      {0.0489259225038643, 0.6226826638040092, 0.0},
      {0.6226826638040089, 0.0489259225038644, 0.0},
      {0.3283914136921267, 0.6226826638040091, 0.0},
      {0.0489259225038643, 0.3283914136921268, 0.0},
      {0.1508877227628729, 0.1508877227628732, 0.0},
      {0.6982245544742539, 0.1508877227628729, 0.0},
      {0.1508877227628732, 0.6982245544742540, 0.0},
      {0.1423855003245428, 0.0246582497782073, 0.0},
      {0.8329562498972498, 0.1423855003245428, 0.0},
      {0.0246582497782074, 0.8329562498972500, 0.0},
      {0.8329562498972499, 0.0246582497782073, 0.0},
      {0.1423855003245429, 0.8329562498972499, 0.0},
      {0.0246582497782072, 0.1423855003245431, 0.0},
      {0.4238402997203689, 0.0059599432497443, 0.0},
      {0.5701997570298869, 0.4238402997203689, 0.0},
      {0.0059599432497442, 0.5701997570298871, 0.0},
      {0.5701997570298869, 0.0059599432497443, 0.0},
      {0.4238402997203689, 0.5701997570298870, 0.0},
      {0.0059599432497442, 0.4238402997203691, 0.0},
      {0.0041579573923189, 0.0041579573923192, 0.0},
      {0.9916840852153618, 0.0041579573923189, 0.0},
      {0.0041579573923194, 0.9916840852153619, 0.0},
      {0.2404349104138168, 0.2404349104138169, 0.0},
      {0.5191301791723664, 0.2404349104138168, 0.0},
      {0.2404349104138169, 0.5191301791723665, 0.0},
      {0.2676904193443056, 0.0285187228422854, 0.0},
      {0.7037908578134091, 0.2676904193443056, 0.0},
      {0.0285187228422855, 0.7037908578134091, 0.0},
      {0.7037908578134090, 0.0285187228422854, 0.0},
      {0.2676904193443056, 0.7037908578134091, 0.0},
      {0.0285187228422852, 0.2676904193443059, 0.0},
      {0.4155115967448372, 0.1689768065103257, 0.0},
      {0.4155115967448372, 0.4155115967448372, 0.0},
      {0.1689768065103256, 0.4155115967448373, 0.0},
      {0.4995421826615118, 0.0009156346769765, 0.0},
      {0.4995421826615117, 0.4995421826615118, 0.0},
      {0.0009156346769764, 0.4995421826615120, 0.0},
      {0.3454113789529011, 0.0155626433010155, 0.0},
      {0.6390259777460834, 0.3454113789529012, 0.0},
      {0.0155626433010154, 0.6390259777460836, 0.0},
      {0.6390259777460834, 0.0155626433010155, 0.0},
      {0.3454113789529011, 0.6390259777460835, 0.0},
      {0.0155626433010154, 0.3454113789529014, 0.0},
      {0.3534495196576637, 0.0009257182339338, 0.0},
      {0.6456247621084024, 0.3534495196576638, 0.0},
      {0.0009257182339338, 0.6456247621084026, 0.0},
      {0.6456247621084025, 0.0009257182339338, 0.0},
      {0.3534495196576638, 0.6456247621084026, 0.0},
      {0.0009257182339337, 0.3534495196576640, 0.0},
      {0.4485642658787082, 0.1028714682425835, 0.0},
      {0.4485642658787082, 0.4485642658787083, 0.0},
      {0.1028714682425834, 0.4485642658787085, 0.0},
      {0.3033356394717892, 0.0992862116185854, 0.0},
      {0.5973781489096254, 0.3033356394717893, 0.0},
      {0.0992862116185854, 0.5973781489096255, 0.0},
      {0.5973781489096254, 0.0992862116185854, 0.0},
      {0.3033356394717892, 0.5973781489096255, 0.0},
      {0.0992862116185853, 0.3033356394717894, 0.0},
      {0.0217555603201839, 0.0039745058725185, 0.0},
      {0.9742699338072974, 0.0217555603201839, 0.0},
      {0.0039745058725186, 0.9742699338072977, 0.0},
      {0.9742699338072975, 0.0039745058725185, 0.0},
      {0.0217555603201841, 0.9742699338072977, 0.0},
      {0.0039745058725182, 0.0217555603201842, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.2075468682735205, 0.0137796394053688, 0.0},
      {0.7786734923211107, 0.2075468682735205, 0.0},
      {0.0137796394053689, 0.7786734923211108, 0.0},
      {0.7786734923211107, 0.0137796394053688, 0.0},
      {0.2075468682735205, 0.7786734923211108, 0.0},
      {0.0137796394053686, 0.2075468682735207, 0.0},
      {0.1943865435789661, 0.1943865435789664, 0.0},
      {0.6112269128420674, 0.1943865435789662, 0.0},
      {0.1943865435789664, 0.6112269128420676, 0.0},
      {0.2135602395179478, 0.0007721391203270, 0.0},
      {0.7856676213617251, 0.2135602395179478, 0.0},
      {0.0007721391203271, 0.7856676213617252, 0.0},
      {0.7856676213617252, 0.0007721391203270, 0.0},
      {0.2135602395179479, 0.7856676213617252, 0.0},
      {0.0007721391203269, 0.2135602395179481, 0.0},
      {0.2793864317822237, 0.0054617908942934, 0.0},
      {0.7151517773234828, 0.2793864317822238, 0.0},
      {0.0054617908942934, 0.7151517773234830, 0.0},
      {0.7151517773234828, 0.0054617908942934, 0.0},
      {0.2793864317822238, 0.7151517773234830, 0.0},
      {0.0054617908942933, 0.2793864317822241, 0.0},
      {0.0968669225755472, 0.0000047096910185, 0.0},
      {0.9031283677334342, 0.0968669225755471, 0.0},
      {0.0000047096910187, 0.9031283677334344, 0.0},
      {0.9031283677334343, 0.0000047096910185, 0.0},
      {0.0968669225755472, 0.9031283677334344, 0.0},
      {0.0000047096910183, 0.0968669225755475, 0.0}},
      {0.0000508429861836,
      0.0000508429861836,
      0.0000508429861836,
      0.0000508429861836,
      0.0000508429861836,
      0.0000508429861836,
      0.0003270239832937,
      0.0003270239832937,
      0.0003270239832937,
      0.0003270239832937,
      0.0003270239832937,
      0.0003270239832937,
      0.0011494832844046,
      0.0011494832844046,
      0.0011494832844046,
      0.0007599058291308,
      0.0007599058291308,
      0.0007599058291308,
      0.0007599058291308,
      0.0007599058291308,
      0.0007599058291308,
      0.0005498402832316,
      0.0005498402832316,
      0.0005498402832316,
      0.0023348188419729,
      0.0023348188419729,
      0.0023348188419729,
      0.0023348188419729,
      0.0023348188419729,
      0.0023348188419729,
      0.0014760578152811,
      0.0014760578152811,
      0.0014760578152811,
      0.0014760578152811,
      0.0014760578152811,
      0.0014760578152811,
      0.0019986867268246,
      0.0019986867268246,
      0.0019986867268246,
      0.0058700032956288,
      0.0058700032956288,
      0.0058700032956288,
      0.0049304651564524,
      0.0049304651564524,
      0.0049304651564524,
      0.0049304651564524,
      0.0049304651564524,
      0.0049304651564524,
      0.0008631366294208,
      0.0008631366294208,
      0.0008631366294208,
      0.0008631366294208,
      0.0008631366294208,
      0.0008631366294208,
      0.0058806038197068,
      0.0058806038197068,
      0.0058806038197068,
      0.0025180912506669,
      0.0025180912506669,
      0.0025180912506669,
      0.0025180912506669,
      0.0025180912506669,
      0.0025180912506669,
      0.0048289401147234,
      0.0048289401147234,
      0.0048289401147234,
      0.0048289401147234,
      0.0048289401147234,
      0.0048289401147234,
      0.0038059575104430,
      0.0038059575104430,
      0.0038059575104430,
      0.0038059575104430,
      0.0038059575104430,
      0.0038059575104430,
      0.0055329319437996,
      0.0055329319437996,
      0.0055329319437996,
      0.0055329319437996,
      0.0055329319437996,
      0.0055329319437996,
      0.0032519198563248,
      0.0032519198563248,
      0.0032519198563248,
      0.0032519198563248,
      0.0032519198563248,
      0.0032519198563248,
      0.0024163436267425,
      0.0024163436267425,
      0.0024163436267425,
      0.0024163436267425,
      0.0024163436267425,
      0.0024163436267425,
      0.0032421126069192,
      0.0032421126069192,
      0.0032421126069192,
      0.0032967572338747,
      0.0032967572338747,
      0.0032967572338747,
      0.0032967572338747,
      0.0032967572338747,
      0.0032967572338747,
      0.0041426332565504,
      0.0041426332565504,
      0.0041426332565504,
      0.0041426332565504,
      0.0041426332565504,
      0.0041426332565504,
      0.0006997146927016,
      0.0006997146927016,
      0.0006997146927016,
      0.0006997146927016,
      0.0006997146927016,
      0.0006997146927016,
      0.0029167168397771,
      0.0029167168397771,
      0.0029167168397771,
      0.0018646854717091,
      0.0018646854717091,
      0.0018646854717091,
      0.0030476320162441,
      0.0030476320162441,
      0.0030476320162441,
      0.0030476320162441,
      0.0030476320162441,
      0.0030476320162441,
      0.0038436553429819,
      0.0038436553429819,
      0.0038436553429819,
      0.0015716030740796,
      0.0015716030740796,
      0.0015716030740796,
      0.0015716030740796,
      0.0015716030740796,
      0.0015716030740796,
      0.0011156658966767,
      0.0011156658966767,
      0.0011156658966767,
      0.0011156658966767,
      0.0011156658966767,
      0.0011156658966767,
      0.0001134806783855,
      0.0001134806783855,
      0.0001134806783855,
      0.0054068496581348,
      0.0054068496581348,
      0.0054068496581348,
      0.0022178654867210,
      0.0022178654867210,
      0.0022178654867210,
      0.0022178654867210,
      0.0022178654867210,
      0.0022178654867210,
      0.0053237224212445,
      0.0053237224212445,
      0.0053237224212445,
      0.0003804630831091,
      0.0003804630831091,
      0.0003804630831091,
      0.0017441636728719,
      0.0017441636728719,
      0.0017441636728719,
      0.0017441636728719,
      0.0017441636728719,
      0.0017441636728719,
      0.0003659229368350,
      0.0003659229368350,
      0.0003659229368350,
      0.0003659229368350,
      0.0003659229368350,
      0.0003659229368350,
      0.0044304696500765,
      0.0044304696500765,
      0.0044304696500765,
      0.0041253987153765,
      0.0041253987153765,
      0.0041253987153765,
      0.0041253987153765,
      0.0041253987153765,
      0.0041253987153765,
      0.0002489837865986,
      0.0002489837865986,
      0.0002489837865986,
      0.0002489837865986,
      0.0002489837865986,
      0.0002489837865986,
      0.0060504824650856,
      0.0014029090686569,
      0.0014029090686569,
      0.0014029090686569,
      0.0014029090686569,
      0.0014029090686569,
      0.0014029090686569,
      0.0047000705185992,
      0.0047000705185992,
      0.0047000705185992,
      0.0002888252871174,
      0.0002888252871174,
      0.0002888252871174,
      0.0002888252871174,
      0.0002888252871174,
      0.0002888252871174,
      0.0009813144027416,
      0.0009813144027416,
      0.0009813144027416,
      0.0009813144027416,
      0.0009813144027416,
      0.0009813144027416,
      0.0001436726773040,
      0.0001436726773040,
      0.0001436726773040,
      0.0001436726773040,
      0.0001436726773040,
      0.0001436726773040}
    },
    {
      228,
      {{0.3014351686173180, 0.3014351686173181, 0.0},
      {0.3971296627653639, 0.3014351686173181, 0.0},
      {0.3014351686173180, 0.3971296627653640, 0.0},
      {0.4916414039828907, 0.0167171920342186, 0.0},
      {0.4916414039828907, 0.4916414039828907, 0.0},
      {0.0167171920342186, 0.4916414039828909, 0.0},
      {0.3577931357985971, 0.2844137284028058, 0.0},
      {0.3577931357985971, 0.3577931357985972, 0.0},
      {0.2844137284028057, 0.3577931357985973, 0.0},
      {0.0550454796915316, 0.0020682202154059, 0.0},
      {0.9428863000930624, 0.0550454796915316, 0.0},
      {0.0020682202154061, 0.9428863000930624, 0.0},
      {0.9428863000930625, 0.0020682202154059, 0.0},
      {0.0550454796915318, 0.9428863000930624, 0.0},
      {0.0020682202154057, 0.0550454796915320, 0.0},
      {0.4453764649152975, 0.0089049941861278, 0.0},
      {0.5457185408985746, 0.4453764649152975, 0.0},
      {0.0089049941861278, 0.5457185408985749, 0.0},
      {0.5457185408985746, 0.0089049941861278, 0.0},
      {0.4453764649152976, 0.5457185408985747, 0.0},
      {0.0089049941861277, 0.4453764649152978, 0.0},
      {0.1523840025243152, 0.0021159775630399, 0.0},
      {0.8455000199126449, 0.1523840025243152, 0.0},
      {0.0021159775630399, 0.8455000199126450, 0.0},
      {0.8455000199126449, 0.0021159775630399, 0.0},
      {0.1523840025243154, 0.8455000199126450, 0.0},
      {0.0021159775630396, 0.1523840025243156, 0.0},
      {0.2624139601093454, 0.2624139601093456, 0.0},
      {0.4751720797813089, 0.2624139601093455, 0.0},
      {0.2624139601093456, 0.4751720797813091, 0.0},
      {0.3975166100282313, 0.0032612734291312, 0.0},
      {0.5992221165426375, 0.3975166100282314, 0.0},
      {0.0032612734291312, 0.5992221165426377, 0.0},
      {0.5992221165426374, 0.0032612734291312, 0.0},
      {0.3975166100282314, 0.5992221165426375, 0.0},
      {0.0032612734291311, 0.3975166100282316, 0.0},
      {0.4028164505857819, 0.1943670988284363, 0.0},
      {0.4028164505857819, 0.4028164505857819, 0.0},
      {0.1943670988284362, 0.4028164505857820, 0.0},
      {0.1560787694034761, 0.0451678948131287, 0.0},
      {0.7987533357833951, 0.1560787694034761, 0.0},
      {0.0451678948131288, 0.7987533357833954, 0.0},
      {0.7987533357833952, 0.0451678948131287, 0.0},
      {0.1560787694034762, 0.7987533357833954, 0.0},
      {0.0451678948131284, 0.1560787694034764, 0.0},
      {0.4730889655006641, 0.0538220689986717, 0.0},
      {0.4730889655006641, 0.4730889655006642, 0.0},
      {0.0538220689986716, 0.4730889655006644, 0.0},
      {0.1972956762765587, 0.0086060217057568, 0.0},
      {0.7940983020176844, 0.1972956762765587, 0.0},
      {0.0086060217057569, 0.7940983020176846, 0.0},
      {0.7940983020176845, 0.0086060217057568, 0.0},
      {0.1972956762765588, 0.7940983020176846, 0.0},
      {0.0086060217057566, 0.1972956762765590, 0.0},
      {0.0500938368999262, 0.0139743532515684, 0.0},
      {0.9359318098485054, 0.0500938368999262, 0.0},
      {0.0139743532515686, 0.9359318098485054, 0.0},
      {0.9359318098485054, 0.0139743532515684, 0.0},
      {0.0500938368999261, 0.9359318098485058, 0.0},
      {0.0139743532515681, 0.0500938368999265, 0.0},
      {0.2988590630143191, 0.1874713324998945, 0.0},
      {0.5136696044857865, 0.2988590630143192, 0.0},
      {0.1874713324998944, 0.5136696044857866, 0.0},
      {0.5136696044857865, 0.1874713324998945, 0.0},
      {0.2988590630143191, 0.5136696044857866, 0.0},
      {0.1874713324998943, 0.2988590630143193, 0.0},
      {0.0628680490440657, 0.0628680490440660, 0.0},
      {0.8742639019118683, 0.0628680490440657, 0.0},
      {0.0628680490440661, 0.8742639019118684, 0.0},
      {0.4415974337549303, 0.1168051324901393, 0.0},
      {0.4415974337549303, 0.4415974337549304, 0.0},
      {0.1168051324901392, 0.4415974337549305, 0.0},
      {0.0196271599834707, 0.0196271599834710, 0.0},
      {0.9607456800330582, 0.0196271599834707, 0.0},
      {0.0196271599834711, 0.9607456800330584, 0.0},
      {0.3347633432653314, 0.2327223440759485, 0.0},
      {0.4325143126587202, 0.3347633432653315, 0.0},
      {0.2327223440759484, 0.4325143126587203, 0.0},
      {0.4325143126587202, 0.2327223440759485, 0.0},
      {0.3347633432653314, 0.4325143126587202, 0.0},
      {0.2327223440759484, 0.3347633432653315, 0.0},
      {0.2093728249763754, 0.0290018846146845, 0.0},
      {0.7616252904089401, 0.2093728249763754, 0.0},
      {0.0290018846146846, 0.7616252904089403, 0.0},
      {0.7616252904089401, 0.0290018846146845, 0.0},
      {0.2093728249763754, 0.7616252904089402, 0.0},
      {0.0290018846146844, 0.2093728249763756, 0.0},
      {0.1065647448613834, 0.0535674828677535, 0.0},
      {0.8398677722708630, 0.1065647448613835, 0.0},
      {0.0535674828677536, 0.8398677722708632, 0.0},
      {0.8398677722708631, 0.0535674828677535, 0.0},
      {0.1065647448613836, 0.8398677722708632, 0.0},
      {0.0535674828677533, 0.1065647448613838, 0.0},
      {0.3213626276695433, 0.0034961943491435, 0.0},
      {0.6751411779813131, 0.3213626276695434, 0.0},
      {0.0034961943491436, 0.6751411779813132, 0.0},
      {0.6751411779813131, 0.0034961943491435, 0.0},
      {0.3213626276695434, 0.6751411779813132, 0.0},
      {0.0034961943491434, 0.3213626276695436, 0.0},
      {0.2415870263479875, 0.0011760427968418, 0.0},
      {0.7572369308551706, 0.2415870263479876, 0.0},
      {0.0011760427968418, 0.7572369308551709, 0.0},
      {0.7572369308551707, 0.0011760427968418, 0.0},
      {0.2415870263479876, 0.7572369308551709, 0.0},
      {0.0011760427968416, 0.2415870263479878, 0.0},
      {0.3331530997207648, 0.1139539287441484, 0.0},
      {0.5528929715350868, 0.3331530997207649, 0.0},
      {0.1139539287441483, 0.5528929715350870, 0.0},
      {0.5528929715350868, 0.1139539287441484, 0.0},
      {0.3331530997207648, 0.5528929715350869, 0.0},
      {0.1139539287441482, 0.3331530997207651, 0.0},
      {0.1551165548133034, 0.0849385995115550, 0.0},
      {0.7599448456751416, 0.1551165548133034, 0.0},
      {0.0849385995115551, 0.7599448456751416, 0.0},
      {0.7599448456751415, 0.0849385995115550, 0.0},
      {0.1551165548133034, 0.7599448456751416, 0.0},
      {0.0849385995115548, 0.1551165548133037, 0.0},
      {0.3710141516774619, 0.1525651863472164, 0.0},
      {0.4764206619753218, 0.3710141516774620, 0.0},
      {0.1525651863472163, 0.4764206619753220, 0.0},
      {0.4764206619753217, 0.1525651863472164, 0.0},
      {0.3710141516774620, 0.4764206619753218, 0.0},
      {0.1525651863472163, 0.3710141516774621, 0.0},
      {0.0402868688301934, 0.0402868688301937, 0.0},
      {0.9194262623396129, 0.0402868688301934, 0.0},
      {0.0402868688301939, 0.9194262623396129, 0.0},
      {0.2708836618077721, 0.0136186781090766, 0.0},
      {0.7154976600831513, 0.2708836618077721, 0.0},
      {0.0136186781090767, 0.7154976600831515, 0.0},
      {0.7154976600831513, 0.0136186781090766, 0.0},
      {0.2708836618077720, 0.7154976600831515, 0.0},
      {0.0136186781090765, 0.2708836618077723, 0.0},
      {0.3525067484154665, 0.0562263874435524, 0.0},
      {0.5912668641409812, 0.3525067484154665, 0.0},
      {0.0562263874435524, 0.5912668641409813, 0.0},
      {0.5912668641409811, 0.0562263874435524, 0.0},
      {0.3525067484154665, 0.5912668641409812, 0.0},
      {0.0562263874435522, 0.3525067484154668, 0.0},
      {0.4057201327781905, 0.0817534894396882, 0.0},
      {0.5125263777821213, 0.4057201327781905, 0.0},
      {0.0817534894396881, 0.5125263777821215, 0.0},
      {0.5125263777821213, 0.0817534894396882, 0.0},
      {0.4057201327781905, 0.5125263777821214, 0.0},
      {0.0817534894396881, 0.4057201327781906, 0.0},
      {0.2189496810149983, 0.0636424301897474, 0.0},
      {0.7174078887952542, 0.2189496810149984, 0.0},
      {0.0636424301897475, 0.7174078887952544, 0.0},
      {0.7174078887952543, 0.0636424301897474, 0.0},
      {0.2189496810149983, 0.7174078887952544, 0.0},
      {0.0636424301897472, 0.2189496810149986, 0.0},
      {0.2067092480855645, 0.1153942208360741, 0.0},
      {0.6778965310783613, 0.2067092480855645, 0.0},
      {0.1153942208360741, 0.6778965310783615, 0.0},
      {0.6778965310783613, 0.1153942208360741, 0.0},
      {0.2067092480855646, 0.6778965310783615, 0.0},
      {0.1153942208360739, 0.2067092480855648, 0.0},
      {0.0985261546687924, 0.0985261546687926, 0.0},
      {0.8029476906624150, 0.0985261546687924, 0.0},
      {0.0985261546687927, 0.8029476906624151, 0.0},
      {0.2566010878790922, 0.1481871162833107, 0.0},
      {0.5952117958375970, 0.2566010878790924, 0.0},
      {0.1481871162833107, 0.5952117958375971, 0.0},
      {0.5952117958375971, 0.1481871162833107, 0.0},
      {0.2566010878790923, 0.5952117958375971, 0.0},
      {0.1481871162833105, 0.2566010878790925, 0.0},
      {0.1382062952112500, 0.0185978710613814, 0.0},
      {0.8431958337273686, 0.1382062952112499, 0.0},
      {0.0185978710613815, 0.8431958337273687, 0.0},
      {0.8431958337273686, 0.0185978710613814, 0.0},
      {0.1382062952112501, 0.8431958337273687, 0.0},
      {0.0185978710613812, 0.1382062952112503, 0.0},
      {0.1395344629893073, 0.1395344629893075, 0.0},
      {0.7209310740213852, 0.1395344629893073, 0.0},
      {0.1395344629893076, 0.7209310740213853, 0.0},
      {0.2215360105096920, 0.2215360105096922, 0.0},
      {0.5569279789806159, 0.2215360105096921, 0.0},
      {0.2215360105096921, 0.5569279789806160, 0.0},
      {0.4258102085854810, 0.0329710199269464, 0.0},
      {0.5412187714875727, 0.4258102085854810, 0.0},
      {0.0329710199269463, 0.5412187714875728, 0.0},
      {0.5412187714875726, 0.0329710199269464, 0.0},
      {0.4258102085854810, 0.5412187714875727, 0.0},
      {0.0329710199269463, 0.4258102085854812, 0.0},
      {0.1807503221352996, 0.1807503221352998, 0.0},
      {0.6384993557294008, 0.1807503221352996, 0.0},
      {0.1807503221352998, 0.6384993557294009, 0.0},
      {0.2797259269522402, 0.0853795515729539, 0.0},
      {0.6348945214748060, 0.2797259269522402, 0.0},
      {0.0853795515729538, 0.6348945214748062, 0.0},
      {0.6348945214748060, 0.0853795515729539, 0.0},
      {0.2797259269522402, 0.6348945214748061, 0.0},
      {0.0853795515729537, 0.2797259269522404, 0.0},
      {0.0224666835802112, 0.0036330558429340, 0.0},
      {0.9739002605768547, 0.0224666835802112, 0.0},
      {0.0036330558429342, 0.9739002605768550, 0.0},
      {0.9739002605768547, 0.0036330558429340, 0.0},
      {0.0224666835802113, 0.9739002605768550, 0.0},
      {0.0036330558429336, 0.0224666835802115, 0.0},
      {0.0838622585244554, 0.0260200776623640, 0.0},
      {0.8901176638131806, 0.0838622585244554, 0.0},
      {0.0260200776623640, 0.8901176638131808, 0.0},
      {0.8901176638131806, 0.0260200776623640, 0.0},
      {0.0838622585244554, 0.8901176638131808, 0.0},
      {0.0260200776623637, 0.0838622585244558, 0.0},
      {0.2848003464144683, 0.0394569261564381, 0.0},
      {0.6757427274290935, 0.2848003464144683, 0.0},
      {0.0394569261564381, 0.6757427274290937, 0.0},
      {0.6757427274290936, 0.0394569261564381, 0.0},
      {0.2848003464144683, 0.6757427274290937, 0.0},
      {0.0394569261564380, 0.2848003464144686, 0.0},
      {0.3565941035365738, 0.0197707177044082, 0.0},
      {0.6236351787590180, 0.3565941035365738, 0.0},
      {0.0197707177044082, 0.6236351787590181, 0.0},
      {0.6236351787590180, 0.0197707177044082, 0.0},
      {0.3565941035365738, 0.6236351787590181, 0.0},
      {0.0197707177044081, 0.3565941035365741, 0.0},
      {0.0974242311151380, 0.0050286974156274, 0.0},
      {0.8975470714692345, 0.0974242311151380, 0.0},
      {0.0050286974156275, 0.8975470714692347, 0.0},
      {0.8975470714692346, 0.0050286974156274, 0.0},
      {0.0974242311151381, 0.8975470714692347, 0.0},
      {0.0050286974156271, 0.0974242311151384, 0.0},
      {0.0042772072684830, 0.0042772072684833, 0.0},
      {0.9914455854630337, 0.0042772072684829, 0.0},
      {0.0042772072684834, 0.9914455854630339, 0.0},
      {0.4999132224862251, 0.0001735550275498, 0.0},
      {0.4999132224862251, 0.4999132224862252, 0.0},
      {0.0001735550275497, 0.4999132224862253, 0.0}},
      {0.0028739670148351,
      0.0028739670148351,
      0.0028739670148351,
      0.0011481840460497,
      0.0011481840460497,
      0.0011481840460497,
      0.0039976754209322,
      0.0039976754209322,
      0.0039976754209322,
      0.0002270996911877,
      0.0002270996911877,
      0.0002270996911877,
      0.0002270996911877,
      0.0002270996911877,
      0.0002270996911877,
      0.0009858746654921,
      0.0009858746654921,
      0.0009858746654921,
      0.0009858746654921,
      0.0009858746654921,
      0.0009858746654921,
      0.0003913331527197,
      0.0003913331527197,
      0.0003913331527197,
      0.0003913331527197,
      0.0003913331527197,
      0.0003913331527197,
      0.0043950929466639,
      0.0043950929466639,
      0.0043950929466639,
      0.0006323465511986,
      0.0006323465511986,
      0.0006323465511986,
      0.0006323465511986,
      0.0006323465511986,
      0.0006323465511986,
      0.0041601833061628,
      0.0041601833061628,
      0.0041601833061628,
      0.0017786281304548,
      0.0017786281304548,
      0.0017786281304548,
      0.0017786281304548,
      0.0017786281304548,
      0.0017786281304548,
      0.0026375941425808,
      0.0026375941425808,
      0.0026375941425808,
      0.0009214163315877,
      0.0009214163315877,
      0.0009214163315877,
      0.0009214163315877,
      0.0009214163315877,
      0.0009214163315877,
      0.0006644899678271,
      0.0006644899678271,
      0.0006644899678271,
      0.0006644899678271,
      0.0006644899678271,
      0.0006644899678271,
      0.0045117064275531,
      0.0045117064275531,
      0.0045117064275531,
      0.0045117064275531,
      0.0045117064275531,
      0.0045117064275531,
      0.0014841033195258,
      0.0014841033195258,
      0.0014841033195258,
      0.0038025912652682,
      0.0038025912652682,
      0.0038025912652682,
      0.0005065605163691,
      0.0005065605163691,
      0.0005065605163691,
      0.0043027522470419,
      0.0043027522470419,
      0.0043027522470419,
      0.0043027522470419,
      0.0043027522470419,
      0.0043027522470419,
      0.0018052339984969,
      0.0018052339984969,
      0.0018052339984969,
      0.0018052339984969,
      0.0018052339984969,
      0.0018052339984969,
      0.0017617803836054,
      0.0017617803836054,
      0.0017617803836054,
      0.0017617803836054,
      0.0017617803836054,
      0.0017617803836054,
      0.0006890382610032,
      0.0006890382610032,
      0.0006890382610032,
      0.0006890382610032,
      0.0006890382610032,
      0.0006890382610032,
      0.0003475147929759,
      0.0003475147929759,
      0.0003475147929759,
      0.0003475147929759,
      0.0003475147929759,
      0.0003475147929759,
      0.0040648602990390,
      0.0040648602990390,
      0.0040648602990390,
      0.0040648602990390,
      0.0040648602990390,
      0.0040648602990390,
      0.0027448281799333,
      0.0027448281799333,
      0.0027448281799333,
      0.0027448281799333,
      0.0027448281799333,
      0.0027448281799333,
      0.0041179936614291,
      0.0041179936614291,
      0.0041179936614291,
      0.0041179936614291,
      0.0041179936614291,
      0.0041179936614291,
      0.0010324178132991,
      0.0010324178132991,
      0.0010324178132991,
      0.0013898867596813,
      0.0013898867596813,
      0.0013898867596813,
      0.0013898867596813,
      0.0013898867596813,
      0.0013898867596813,
      0.0031658909407417,
      0.0031658909407417,
      0.0031658909407417,
      0.0031658909407417,
      0.0031658909407417,
      0.0031658909407417,
      0.0034728553599708,
      0.0034728553599708,
      0.0034728553599708,
      0.0034728553599708,
      0.0034728553599708,
      0.0034728553599708,
      0.0027829705578431,
      0.0027829705578431,
      0.0027829705578431,
      0.0027829705578431,
      0.0027829705578431,
      0.0027829705578431,
      0.0036688050783813,
      0.0036688050783813,
      0.0036688050783813,
      0.0036688050783813,
      0.0036688050783813,
      0.0036688050783813,
      0.0025018198077787,
      0.0025018198077787,
      0.0025018198077787,
      0.0043557133117805,
      0.0043557133117805,
      0.0043557133117805,
      0.0043557133117805,
      0.0043557133117805,
      0.0043557133117805,
      0.0013683747163610,
      0.0013683747163610,
      0.0013683747163610,
      0.0013683747163610,
      0.0013683747163610,
      0.0013683747163610,
      0.0033595399062087,
      0.0033595399062087,
      0.0033595399062087,
      0.0045809544704687,
      0.0045809544704687,
      0.0045809544704687,
      0.0023508287904661,
      0.0023508287904661,
      0.0023508287904661,
      0.0023508287904661,
      0.0023508287904661,
      0.0023508287904661,
      0.0041859533185065,
      0.0041859533185065,
      0.0041859533185065,
      0.0035875044799664,
      0.0035875044799664,
      0.0035875044799664,
      0.0035875044799664,
      0.0035875044799664,
      0.0035875044799664,
      0.0002366234869003,
      0.0002366234869003,
      0.0002366234869003,
      0.0002366234869003,
      0.0002366234869003,
      0.0002366234869003,
      0.0012673310556839,
      0.0012673310556839,
      0.0012673310556839,
      0.0012673310556839,
      0.0012673310556839,
      0.0012673310556839,
      0.0025350786500057,
      0.0025350786500057,
      0.0025350786500057,
      0.0025350786500057,
      0.0025350786500057,
      0.0025350786500057,
      0.0020270360434420,
      0.0020270360434420,
      0.0020270360434420,
      0.0020270360434420,
      0.0020270360434420,
      0.0020270360434420,
      0.0006272391936205,
      0.0006272391936205,
      0.0006272391936205,
      0.0006272391936205,
      0.0006272391936205,
      0.0006272391936205,
      0.0001201668687024,
      0.0001201668687024,
      0.0001201668687024,
      0.0003137921705353,
      0.0003137921705353,
      0.0003137921705353}
    },
    {
      243,
      {{0.2550023360754195, 0.0322922639737694, 0.0},
      {0.7127053999508112, 0.2550023360754195, 0.0},
      {0.0322922639737694, 0.7127053999508113, 0.0},
      {0.7127053999508112, 0.0322922639737694, 0.0},
      {0.2550023360754196, 0.7127053999508113, 0.0},
      {0.0322922639737692, 0.2550023360754198, 0.0},
      {0.2711215136033400, 0.0941402098044082, 0.0},
      {0.6347382765922519, 0.2711215136033400, 0.0},
      {0.0941402098044082, 0.6347382765922520, 0.0},
      {0.6347382765922519, 0.0941402098044082, 0.0},
      {0.2711215136033400, 0.6347382765922520, 0.0},
      {0.0941402098044080, 0.2711215136033402, 0.0},
      {0.0926886044848755, 0.0649936721042402, 0.0},
      {0.8423177234108842, 0.0926886044848756, 0.0},
      {0.0649936721042403, 0.8423177234108843, 0.0},
      {0.8423177234108843, 0.0649936721042402, 0.0},
      {0.0926886044848757, 0.8423177234108843, 0.0},
      {0.0649936721042400, 0.0926886044848759, 0.0},
      {0.3294543765845505, 0.0737136644534188, 0.0},
      {0.5968319589620307, 0.3294543765845505, 0.0},
      {0.0737136644534188, 0.5968319589620309, 0.0},
      {0.5968319589620307, 0.0737136644534188, 0.0},
      {0.3294543765845505, 0.5968319589620308, 0.0},
      {0.0737136644534187, 0.3294543765845507, 0.0},
      {0.2883788411304420, 0.0500649881468919, 0.0},
      {0.6615561707226660, 0.2883788411304421, 0.0},
      {0.0500649881468920, 0.6615561707226660, 0.0},
      {0.6615561707226659, 0.0500649881468919, 0.0},
      {0.2883788411304421, 0.6615561707226660, 0.0},
      {0.0500649881468918, 0.2883788411304423, 0.0},
      {0.3921724164051739, 0.0658963384748947, 0.0},
      {0.5419312451199315, 0.3921724164051739, 0.0},
      {0.0658963384748946, 0.5419312451199317, 0.0},
      {0.5419312451199314, 0.0658963384748947, 0.0},
      {0.3921724164051738, 0.5419312451199316, 0.0},
      {0.0658963384748946, 0.3921724164051740, 0.0},
      {0.0076222500155876, 0.0012341185277633, 0.0},
      {0.9911436314566491, 0.0076222500155876, 0.0},
      {0.0012341185277635, 0.9911436314566491, 0.0},
      {0.9911436314566491, 0.0012341185277633, 0.0},
      {0.0076222500155876, 0.9911436314566494, 0.0},
      {0.0012341185277630, 0.0076222500155880, 0.0},
      {0.4123573403136303, 0.1752853193727394, 0.0},
      {0.4123573403136303, 0.4123573403136304, 0.0},
      {0.1752853193727393, 0.4123573403136305, 0.0},
      {0.1782079366279757, 0.1056958273276099, 0.0},
      {0.7160962360444145, 0.1782079366279757, 0.0},
      {0.1056958273276099, 0.7160962360444146, 0.0},
      {0.7160962360444145, 0.1056958273276099, 0.0},
      {0.1782079366279757, 0.7160962360444146, 0.0},
      {0.1056958273276097, 0.1782079366279760, 0.0},
      {0.3102489314599136, 0.1238716703955509, 0.0},
      {0.5658793981445355, 0.3102489314599137, 0.0},
      {0.1238716703955509, 0.5658793981445357, 0.0},
      {0.5658793981445355, 0.1238716703955509, 0.0},
      {0.3102489314599136, 0.5658793981445357, 0.0},
      {0.1238716703955508, 0.3102489314599138, 0.0},
      {0.4390483832458435, 0.1219032335083130, 0.0},
      {0.4390483832458435, 0.4390483832458436, 0.0},
      {0.1219032335083129, 0.4390483832458437, 0.0},
      {0.0829157021331252, 0.0028073976185661, 0.0},
      {0.9142769002483087, 0.0829157021331252, 0.0},
      {0.0028073976185662, 0.9142769002483089, 0.0},
      {0.9142769002483088, 0.0028073976185661, 0.0},
      {0.0829157021331253, 0.9142769002483089, 0.0},
      {0.0028073976185659, 0.0829157021331255, 0.0},
      {0.3796386793469431, 0.1106897783260212, 0.0},
      {0.5096715423270356, 0.3796386793469432, 0.0},
      {0.1106897783260212, 0.5096715423270358, 0.0},
      {0.5096715423270356, 0.1106897783260212, 0.0},
      {0.3796386793469431, 0.5096715423270358, 0.0},
      {0.1106897783260211, 0.3796386793469433, 0.0},
      {0.2350523254582137, 0.2350523254582139, 0.0},
      {0.5298953490835725, 0.2350523254582138, 0.0},
      {0.2350523254582138, 0.5298953490835726, 0.0},
      {0.1980682921145434, 0.0027564226137057, 0.0},
      {0.7991752852717510, 0.1980682921145434, 0.0},
      {0.0027564226137057, 0.7991752852717512, 0.0},
      {0.7991752852717511, 0.0027564226137057, 0.0},
      {0.1980682921145434, 0.7991752852717511, 0.0},
      {0.0027564226137055, 0.1980682921145436, 0.0},
      {0.1479479453220730, 0.0720516990960159, 0.0},
      {0.7800003555819109, 0.1479479453220730, 0.0},
      {0.0720516990960159, 0.7800003555819113, 0.0},
      {0.7800003555819111, 0.0720516990960159, 0.0},
      {0.1479479453220731, 0.7800003555819113, 0.0},
      {0.0720516990960157, 0.1479479453220733, 0.0},
      {0.2292027665534673, 0.0144286115706843, 0.0},
      {0.7563686218758484, 0.2292027665534674, 0.0},
      {0.0144286115706843, 0.7563686218758485, 0.0},
      {0.7563686218758484, 0.0144286115706843, 0.0},
      {0.2292027665534674, 0.7563686218758484, 0.0},
      {0.0144286115706841, 0.2292027665534676, 0.0},
      {0.3574738503443845, 0.0027425666974119, 0.0},
      {0.6397835829582036, 0.3574738503443846, 0.0},
      {0.0027425666974119, 0.6397835829582038, 0.0},
      {0.6397835829582036, 0.0027425666974119, 0.0},
      {0.3574738503443844, 0.6397835829582038, 0.0},
      {0.0027425666974117, 0.3574738503443848, 0.0},
      {0.1252496722294377, 0.0390243298422159, 0.0},
      {0.8357259979283462, 0.1252496722294378, 0.0},
      {0.0390243298422160, 0.8357259979283465, 0.0},
      {0.8357259979283463, 0.0390243298422159, 0.0},
      {0.1252496722294378, 0.8357259979283465, 0.0},
      {0.0390243298422157, 0.1252496722294380, 0.0},
      {0.0209805646004062, 0.0074746634659333, 0.0},
      {0.9715447719336604, 0.0209805646004062, 0.0},
      {0.0074746634659335, 0.9715447719336605, 0.0},
      {0.9715447719336604, 0.0074746634659333, 0.0},
      {0.0209805646004063, 0.9715447719336605, 0.0},
      {0.0074746634659331, 0.0209805646004066, 0.0},
      {0.2691316117440097, 0.1755717786015193, 0.0},
      {0.5552966096544709, 0.2691316117440098, 0.0},
      {0.1755717786015192, 0.5552966096544713, 0.0},
      {0.5552966096544710, 0.1755717786015193, 0.0},
      {0.2691316117440098, 0.5552966096544711, 0.0},
      {0.1755717786015191, 0.2691316117440100, 0.0},
      {0.1508217347798915, 0.1508217347798918, 0.0},
      {0.6983565304402166, 0.1508217347798916, 0.0},
      {0.1508217347798919, 0.6983565304402167, 0.0},
      {0.3462198724218936, 0.1712267588507529, 0.0},
      {0.4825533687273535, 0.3462198724218936, 0.0},
      {0.1712267588507529, 0.4825533687273537, 0.0},
      {0.4825533687273536, 0.1712267588507529, 0.0},
      {0.3462198724218935, 0.4825533687273537, 0.0},
      {0.1712267588507528, 0.3462198724218937, 0.0},
      {0.4423877426580685, 0.0343061434540937, 0.0},
      {0.5233061138878379, 0.4423877426580686, 0.0},
      {0.0343061434540937, 0.5233061138878380, 0.0},
      {0.5233061138878379, 0.0343061434540937, 0.0},
      {0.4423877426580686, 0.5233061138878379, 0.0},
      {0.0343061434540936, 0.4423877426580687, 0.0},
      {0.2746338837226588, 0.0030296355118471, 0.0},
      {0.7223364807654942, 0.2746338837226588, 0.0},
      {0.0030296355118472, 0.7223364807654942, 0.0},
      {0.7223364807654941, 0.0030296355118471, 0.0},
      {0.2746338837226588, 0.7223364807654942, 0.0},
      {0.0030296355118469, 0.2746338837226592, 0.0},
      {0.2237012846551441, 0.0724869584556441, 0.0},
      {0.7038117568892118, 0.2237012846551441, 0.0},
      {0.0724869584556441, 0.7038117568892120, 0.0},
      {0.7038117568892117, 0.0724869584556441, 0.0},
      {0.2237012846551442, 0.7038117568892119, 0.0},
      {0.0724869584556440, 0.2237012846551444, 0.0},
      {0.2264729475116298, 0.1338170233406639, 0.0},
      {0.6397100291477062, 0.2264729475116298, 0.0},
      {0.1338170233406640, 0.6397100291477065, 0.0},
      {0.6397100291477064, 0.1338170233406639, 0.0},
      {0.2264729475116297, 0.6397100291477065, 0.0},
      {0.1338170233406638, 0.2264729475116300, 0.0},
      {0.1633619668293704, 0.0176742202382503, 0.0},
      {0.8189638129323793, 0.1633619668293704, 0.0},
      {0.0176742202382503, 0.8189638129323794, 0.0},
      {0.8189638129323793, 0.0176742202382503, 0.0},
      {0.1633619668293705, 0.8189638129323794, 0.0},
      {0.0176742202382500, 0.1633619668293707, 0.0},
      {0.1907912728543789, 0.0422082170395821, 0.0},
      {0.7670005101060391, 0.1907912728543789, 0.0},
      {0.0422082170395822, 0.7670005101060392, 0.0},
      {0.7670005101060390, 0.0422082170395821, 0.0},
      {0.1907912728543789, 0.7670005101060391, 0.0},
      {0.0422082170395819, 0.1907912728543791, 0.0},
      {0.3581658706314307, 0.0344163203969896, 0.0},
      {0.6074178089715797, 0.3581658706314307, 0.0},
      {0.0344163203969896, 0.6074178089715798, 0.0},
      {0.6074178089715797, 0.0344163203969896, 0.0},
      {0.3581658706314307, 0.6074178089715798, 0.0},
      {0.0344163203969895, 0.3581658706314309, 0.0},
      {0.0993849892050522, 0.0174529902610205, 0.0},
      {0.8831620205339272, 0.0993849892050522, 0.0},
      {0.0174529902610207, 0.8831620205339273, 0.0},
      {0.8831620205339273, 0.0174529902610205, 0.0},
      {0.0993849892050523, 0.8831620205339273, 0.0},
      {0.0174529902610203, 0.0993849892050526, 0.0},
      {0.4502328978916524, 0.0012278348447231, 0.0},
      {0.5485392672636246, 0.4502328978916525, 0.0},
      {0.0012278348447230, 0.5485392672636248, 0.0},
      {0.5485392672636246, 0.0012278348447231, 0.0},
      {0.4502328978916525, 0.5485392672636246, 0.0},
      {0.0012278348447230, 0.4502328978916527, 0.0},
      {0.4642737776346700, 0.0714524447306600, 0.0},
      {0.4642737776346701, 0.4642737776346700, 0.0},
      {0.0714524447306599, 0.4642737776346703, 0.0},
      {0.3048602929866793, 0.2362631900663087, 0.0},
      {0.4588765169470121, 0.3048602929866794, 0.0},
      {0.2362631900663086, 0.4588765169470122, 0.0},
      {0.4588765169470121, 0.2362631900663087, 0.0},
      {0.3048602929866793, 0.4588765169470121, 0.0},
      {0.2362631900663085, 0.3048602929866794, 0.0},
      {0.3135836062706951, 0.0155229285287317, 0.0},
      {0.6708934652005731, 0.3135836062706952, 0.0},
      {0.0155229285287318, 0.6708934652005732, 0.0},
      {0.6708934652005731, 0.0155229285287317, 0.0},
      {0.3135836062706952, 0.6708934652005732, 0.0},
      {0.0155229285287316, 0.3135836062706954, 0.0},
      {0.1348062882666829, 0.0038480457708451, 0.0},
      {0.8613456659624720, 0.1348062882666829, 0.0},
      {0.0038480457708451, 0.8613456659624722, 0.0},
      {0.8613456659624721, 0.0038480457708451, 0.0},
      {0.1348062882666830, 0.8613456659624722, 0.0},
      {0.0038480457708449, 0.1348062882666833, 0.0},
      {0.1911613618773089, 0.1911613618773092, 0.0},
      {0.6176772762453819, 0.1911613618773090, 0.0},
      {0.1911613618773091, 0.6176772762453820, 0.0},
      {0.0658281924328199, 0.0380724693230131, 0.0},
      {0.8960993382441670, 0.0658281924328199, 0.0},
      {0.0380724693230132, 0.8960993382441672, 0.0},
      {0.8960993382441671, 0.0380724693230131, 0.0},
      {0.0658281924328200, 0.8960993382441672, 0.0},
      {0.0380724693230129, 0.0658281924328202, 0.0},
      {0.0293500490934269, 0.0293500490934272, 0.0},
      {0.9412999018131458, 0.0293500490934269, 0.0},
      {0.0293500490934273, 0.9412999018131460, 0.0},
      {0.3081452189910933, 0.3081452189910934, 0.0},
      {0.3837095620178134, 0.3081452189910934, 0.0},
      {0.3081452189910933, 0.3837095620178135, 0.0},
      {0.0531928439194577, 0.0121702541539402, 0.0},
      {0.9346369019266020, 0.0531928439194577, 0.0},
      {0.0121702541539404, 0.9346369019266021, 0.0},
      {0.9346369019266021, 0.0121702541539402, 0.0},
      {0.0531928439194579, 0.9346369019266021, 0.0},
      {0.0121702541539399, 0.0531928439194581, 0.0},
      {0.1111782647883809, 0.1111782647883812, 0.0},
      {0.7776434704232379, 0.1111782647883809, 0.0},
      {0.1111782647883812, 0.7776434704232381, 0.0},
      {0.0406479626359586, 0.0008585590446839, 0.0},
      {0.9584934783193574, 0.0406479626359585, 0.0},
      {0.0008585590446841, 0.9584934783193576, 0.0},
      {0.9584934783193575, 0.0008585590446839, 0.0},
      {0.0406479626359587, 0.9584934783193576, 0.0},
      {0.0008585590446836, 0.0406479626359589, 0.0},
      {0.4064475672122849, 0.0123358378103272, 0.0},
      {0.5812165949773879, 0.4064475672122849, 0.0},
      {0.0123358378103273, 0.5812165949773881, 0.0},
      {0.5812165949773879, 0.0123358378103272, 0.0},
      {0.4064475672122848, 0.5812165949773881, 0.0},
      {0.0123358378103271, 0.4064475672122851, 0.0},
      {0.3810023496034419, 0.2379953007931163, 0.0},
      {0.3810023496034418, 0.3810023496034420, 0.0},
      {0.2379953007931161, 0.3810023496034420, 0.0},
      {0.4944809967477004, 0.0110380065045991, 0.0},
      {0.4944809967477005, 0.4944809967477005, 0.0},
      {0.0110380065045990, 0.4944809967477007, 0.0}},
      {0.0015308415835341,
      0.0015308415835341,
      0.0015308415835341,
      0.0015308415835341,
      0.0015308415835341,
      0.0015308415835341,
      0.0024616359876450,
      0.0024616359876450,
      0.0024616359876450,
      0.0024616359876450,
      0.0024616359876450,
      0.0024616359876450,
      0.0016618604216976,
      0.0016618604216976,
      0.0016618604216976,
      0.0016618604216976,
      0.0016618604216976,
      0.0016618604216976,
      0.0023713682465169,
      0.0023713682465169,
      0.0023713682465169,
      0.0023713682465169,
      0.0023713682465169,
      0.0023713682465169,
      0.0020569006463248,
      0.0020569006463248,
      0.0020569006463248,
      0.0020569006463248,
      0.0020569006463248,
      0.0020569006463248,
      0.0026176712614469,
      0.0026176712614469,
      0.0026176712614469,
      0.0026176712614469,
      0.0026176712614469,
      0.0026176712614469,
      0.0000651699642263,
      0.0000651699642263,
      0.0000651699642263,
      0.0000651699642263,
      0.0000651699642263,
      0.0000651699642263,
      0.0038352472589543,
      0.0038352472589543,
      0.0038352472589543,
      0.0026138521419692,
      0.0026138521419692,
      0.0026138521419692,
      0.0026138521419692,
      0.0026138521419692,
      0.0026138521419692,
      0.0033155687707743,
      0.0033155687707743,
      0.0033155687707743,
      0.0033155687707743,
      0.0033155687707743,
      0.0033155687707743,
      0.0034381478546981,
      0.0034381478546981,
      0.0034381478546981,
      0.0003476173198586,
      0.0003476173198586,
      0.0003476173198586,
      0.0003476173198586,
      0.0003476173198586,
      0.0003476173198586,
      0.0034591862436963,
      0.0034591862436963,
      0.0034591862436963,
      0.0034591862436963,
      0.0034591862436963,
      0.0034591862436963,
      0.0040339988583191,
      0.0040339988583191,
      0.0040339988583191,
      0.0005088534246774,
      0.0005088534246774,
      0.0005088534246774,
      0.0005088534246774,
      0.0005088534246774,
      0.0005088534246774,
      0.0022654956485322,
      0.0022654956485322,
      0.0022654956485322,
      0.0022654956485322,
      0.0022654956485322,
      0.0022654956485322,
      0.0011896402478193,
      0.0011896402478193,
      0.0011896402478193,
      0.0011896402478193,
      0.0011896402478193,
      0.0011896402478193,
      0.0006084395548527,
      0.0006084395548527,
      0.0006084395548527,
      0.0006084395548527,
      0.0006084395548527,
      0.0006084395548527,
      0.0015565136201892,
      0.0015565136201892,
      0.0015565136201892,
      0.0015565136201892,
      0.0015565136201892,
      0.0015565136201892,
      0.0003240926664066,
      0.0003240926664066,
      0.0003240926664066,
      0.0003240926664066,
      0.0003240926664066,
      0.0003240926664066,
      0.0041725823730273,
      0.0041725823730273,
      0.0041725823730273,
      0.0041725823730273,
      0.0041725823730273,
      0.0041725823730273,
      0.0030855303407258,
      0.0030855303407258,
      0.0030855303407258,
      0.0044519014246259,
      0.0044519014246259,
      0.0044519014246259,
      0.0044519014246259,
      0.0044519014246259,
      0.0044519014246259,
      0.0023819069846942,
      0.0023819069846942,
      0.0023819069846942,
      0.0023819069846942,
      0.0023819069846942,
      0.0023819069846942,
      0.0006165717457187,
      0.0006165717457187,
      0.0006165717457187,
      0.0006165717457187,
      0.0006165717457187,
      0.0006165717457187,
      0.0026405181176831,
      0.0026405181176831,
      0.0026405181176831,
      0.0026405181176831,
      0.0026405181176831,
      0.0026405181176831,
      0.0036019790815255,
      0.0036019790815255,
      0.0036019790815255,
      0.0036019790815255,
      0.0036019790815255,
      0.0036019790815255,
      0.0012750232331756,
      0.0012750232331756,
      0.0012750232331756,
      0.0012750232331756,
      0.0012750232331756,
      0.0012750232331756,
      0.0020465605787014,
      0.0020465605787014,
      0.0020465605787014,
      0.0020465605787014,
      0.0020465605787014,
      0.0020465605787014,
      0.0022346958718063,
      0.0022346958718063,
      0.0022346958718063,
      0.0022346958718063,
      0.0022346958718063,
      0.0022346958718063,
      0.0010506112115176,
      0.0010506112115176,
      0.0010506112115176,
      0.0010506112115176,
      0.0010506112115176,
      0.0010506112115176,
      0.0004110445536571,
      0.0004110445536571,
      0.0004110445536571,
      0.0004110445536571,
      0.0004110445536571,
      0.0004110445536571,
      0.0033306168106238,
      0.0033306168106238,
      0.0033306168106238,
      0.0051066728077646,
      0.0051066728077646,
      0.0051066728077646,
      0.0051066728077646,
      0.0051066728077646,
      0.0051066728077646,
      0.0015201106774150,
      0.0015201106774150,
      0.0015201106774150,
      0.0015201106774150,
      0.0015201106774150,
      0.0015201106774150,
      0.0005548050768085,
      0.0005548050768085,
      0.0005548050768085,
      0.0005548050768085,
      0.0005548050768085,
      0.0005548050768085,
      0.0040466125378445,
      0.0040466125378445,
      0.0040466125378445,
      0.0013334505830018,
      0.0013334505830018,
      0.0013334505830018,
      0.0013334505830018,
      0.0013334505830018,
      0.0013334505830018,
      0.0007837118555754,
      0.0007837118555754,
      0.0007837118555754,
      0.0055041380808568,
      0.0055041380808568,
      0.0055041380808568,
      0.0006678043820325,
      0.0006678043820325,
      0.0006678043820325,
      0.0006678043820325,
      0.0006678043820325,
      0.0006678043820325,
      0.0027117744620663,
      0.0027117744620663,
      0.0027117744620663,
      0.0001330889119467,
      0.0001330889119467,
      0.0001330889119467,
      0.0001330889119467,
      0.0001330889119467,
      0.0001330889119467,
      0.0014438434956608,
      0.0014438434956608,
      0.0014438434956608,
      0.0014438434956608,
      0.0014438434956608,
      0.0014438434956608,
      0.0052531730034086,
      0.0052531730034086,
      0.0052531730034086,
      0.0014479578817343,
      0.0014479578817343,
      0.0014479578817343}
    },
    {
      252,
      {{0.1023039321577603, 0.1023039321577605, 0.0},
      {0.7953921356844791, 0.1023039321577603, 0.0},
      {0.1023039321577607, 0.7953921356844793, 0.0},
      {0.3160240964966399, 0.0338290906827213, 0.0},
      {0.6501468128206388, 0.3160240964966399, 0.0},
      {0.0338290906827213, 0.6501468128206390, 0.0},
      {0.6501468128206388, 0.0338290906827213, 0.0},
      {0.3160240964966400, 0.6501468128206389, 0.0},
      {0.0338290906827212, 0.3160240964966402, 0.0},
      {0.0350202358754684, 0.0350202358754687, 0.0},
      {0.9299595282490629, 0.0350202358754684, 0.0},
      {0.0350202358754688, 0.9299595282490630, 0.0},
      {0.1053594119602819, 0.0331459895561884, 0.0},
      {0.8614945984835296, 0.1053594119602819, 0.0},
      {0.0331459895561886, 0.8614945984835296, 0.0},
      {0.8614945984835296, 0.0331459895561884, 0.0},
      {0.1053594119602819, 0.8614945984835299, 0.0},
      {0.0331459895561881, 0.1053594119602822, 0.0},
      {0.0749810217841138, 0.0020334760761173, 0.0},
      {0.9229855021397688, 0.0749810217841138, 0.0},
      {0.0020334760761175, 0.9229855021397689, 0.0},
      {0.9229855021397688, 0.0020334760761173, 0.0},
      {0.0749810217841139, 0.9229855021397689, 0.0},
      {0.0020334760761171, 0.0749810217841142, 0.0},
      {0.1194865933803552, 0.0555104031624589, 0.0},
      {0.8250030034571860, 0.1194865933803552, 0.0},
      {0.0555104031624590, 0.8250030034571860, 0.0},
      {0.8250030034571859, 0.0555104031624589, 0.0},
      {0.1194865933803553, 0.8250030034571860, 0.0},
      {0.0555104031624586, 0.1194865933803555, 0.0},
      {0.3192351650459891, 0.2380608471170249, 0.0},
      {0.4427039878369861, 0.3192351650459893, 0.0},
      {0.2380608471170247, 0.4427039878369862, 0.0},
      {0.4427039878369861, 0.2380608471170249, 0.0},
      {0.3192351650459891, 0.4427039878369862, 0.0},
      {0.2380608471170247, 0.3192351650459893, 0.0},
      {0.3917882577236701, 0.0752705489375098, 0.0},
      {0.5329411933388201, 0.3917882577236702, 0.0},
      {0.0752705489375098, 0.5329411933388202, 0.0},
      {0.5329411933388201, 0.0752705489375098, 0.0},
      {0.3917882577236701, 0.5329411933388202, 0.0},
      {0.0752705489375098, 0.3917882577236704, 0.0},
      {0.2592434904565495, 0.2592434904565497, 0.0},
      {0.4815130190869007, 0.2592434904565497, 0.0},
      {0.2592434904565497, 0.4815130190869009, 0.0},
      {0.3175434072141983, 0.0536208316291131, 0.0},
      {0.6288357611566886, 0.3175434072141983, 0.0},
      {0.0536208316291132, 0.6288357611566887, 0.0},
      {0.6288357611566886, 0.0536208316291131, 0.0},
      {0.3175434072141983, 0.6288357611566887, 0.0},
      {0.0536208316291130, 0.3175434072141985, 0.0},
      {0.3267645245927318, 0.0912947670950428, 0.0},
      {0.5819407083122254, 0.3267645245927318, 0.0},
      {0.0912947670950428, 0.5819407083122257, 0.0},
      {0.5819407083122254, 0.0912947670950428, 0.0},
      {0.3267645245927318, 0.5819407083122256, 0.0},
      {0.0912947670950427, 0.3267645245927320, 0.0},
      {0.3187496668271801, 0.1351157490702847, 0.0},
      {0.5461345841025351, 0.3187496668271802, 0.0},
      {0.1351157490702847, 0.5461345841025353, 0.0},
      {0.5461345841025351, 0.1351157490702847, 0.0},
      {0.3187496668271802, 0.5461345841025353, 0.0},
      {0.1351157490702846, 0.3187496668271803, 0.0},
      {0.0674578790813868, 0.0407701237566787, 0.0},
      {0.8917719971619344, 0.0674578790813868, 0.0},
      {0.0407701237566788, 0.8917719971619346, 0.0},
      {0.8917719971619346, 0.0407701237566787, 0.0},
      {0.0674578790813870, 0.8917719971619346, 0.0},
      {0.0407701237566784, 0.0674578790813871, 0.0},
      {0.0183794409210812, 0.0029902778765639, 0.0},
      {0.9786302812023548, 0.0183794409210811, 0.0},
      {0.0029902778765640, 0.9786302812023551, 0.0},
      {0.9786302812023550, 0.0029902778765639, 0.0},
      {0.0183794409210812, 0.9786302812023551, 0.0},
      {0.0029902778765637, 0.0183794409210815, 0.0},
      {0.0476569358622159, 0.0172509604538075, 0.0},
      {0.9350921036839764, 0.0476569358622160, 0.0},
      {0.0172509604538077, 0.9350921036839765, 0.0},
      {0.9350921036839764, 0.0172509604538075, 0.0},
      {0.0476569358622161, 0.9350921036839765, 0.0},
      {0.0172509604538073, 0.0476569358622163, 0.0},
      {0.3357227870380101, 0.1811164091955320, 0.0},
      {0.4831608037664580, 0.3357227870380101, 0.0},
      {0.1811164091955319, 0.4831608037664581, 0.0},
      {0.4831608037664580, 0.1811164091955320, 0.0},
      {0.3357227870380101, 0.4831608037664581, 0.0},
      {0.1811164091955318, 0.3357227870380102, 0.0},
      {0.1394412358722927, 0.1394412358722930, 0.0},
      {0.7211175282554143, 0.1394412358722927, 0.0},
      {0.1394412358722930, 0.7211175282554145, 0.0},
      {0.4611684588908100, 0.0776630822183800, 0.0},
      {0.4611684588908100, 0.4611684588908101, 0.0},
      {0.0776630822183799, 0.4611684588908102, 0.0},
      {0.1356716079161176, 0.0901302718482901, 0.0},
      {0.7741981202355922, 0.1356716079161177, 0.0},
      {0.0901302718482901, 0.7741981202355923, 0.0},
      {0.7741981202355922, 0.0901302718482901, 0.0},
      {0.1356716079161178, 0.7741981202355923, 0.0},
      {0.0901302718482899, 0.1356716079161179, 0.0},
      {0.0417693614359698, 0.0040170970872313, 0.0},
      {0.9542135414767989, 0.0417693614359697, 0.0},
      {0.0040170970872315, 0.9542135414767990, 0.0},
      {0.9542135414767989, 0.0040170970872313, 0.0},
      {0.0417693614359699, 0.9542135414767990, 0.0},
      {0.0040170970872310, 0.0417693614359701, 0.0},
      {0.3791537179550378, 0.2416925640899244, 0.0},
      {0.3791537179550378, 0.3791537179550379, 0.0},
      {0.2416925640899243, 0.3791537179550379, 0.0},
      {0.1912637374380762, 0.0809653202292541, 0.0},
      {0.7277709423326696, 0.1912637374380763, 0.0},
      {0.0809653202292542, 0.7277709423326698, 0.0},
      {0.7277709423326697, 0.0809653202292541, 0.0},
      {0.1912637374380762, 0.7277709423326699, 0.0},
      {0.0809653202292539, 0.1912637374380765, 0.0},
      {0.4489203299566428, 0.0168814585101308, 0.0},
      {0.5341982115332263, 0.4489203299566429, 0.0},
      {0.0168814585101307, 0.5341982115332266, 0.0},
      {0.5341982115332264, 0.0168814585101308, 0.0},
      {0.4489203299566429, 0.5341982115332264, 0.0},
      {0.0168814585101307, 0.4489203299566431, 0.0},
      {0.3970833791349885, 0.0414766785336369, 0.0},
      {0.5614399423313745, 0.3970833791349886, 0.0},
      {0.0414766785336368, 0.5614399423313748, 0.0},
      {0.5614399423313746, 0.0414766785336369, 0.0},
      {0.3970833791349885, 0.5614399423313746, 0.0},
      {0.0414766785336368, 0.3970833791349888, 0.0},
      {0.0862263440628872, 0.0137345861622122, 0.0},
      {0.9000390697749006, 0.0862263440628872, 0.0},
      {0.0137345861622123, 0.9000390697749008, 0.0},
      {0.9000390697749007, 0.0137345861622122, 0.0},
      {0.0862263440628872, 0.9000390697749008, 0.0},
      {0.0137345861622120, 0.0862263440628875, 0.0},
      {0.0173411753639024, 0.0173411753639027, 0.0},
      {0.9653176492721949, 0.0173411753639024, 0.0},
      {0.0173411753639029, 0.9653176492721949, 0.0},
      {0.2814879599422820, 0.0179186043302918, 0.0},
      {0.7005934357274263, 0.2814879599422820, 0.0},
      {0.0179186043302919, 0.7005934357274264, 0.0},
      {0.7005934357274262, 0.0179186043302918, 0.0},
      {0.2814879599422819, 0.7005934357274264, 0.0},
      {0.0179186043302916, 0.2814879599422823, 0.0},
      {0.1923768347442593, 0.1284713900567929, 0.0},
      {0.6791517751989478, 0.1923768347442593, 0.0},
      {0.1284713900567929, 0.6791517751989480, 0.0},
      {0.6791517751989478, 0.1284713900567929, 0.0},
      {0.1923768347442593, 0.6791517751989480, 0.0},
      {0.1284713900567927, 0.1923768347442595, 0.0},
      {0.2517942475174331, 0.1373211597315956, 0.0},
      {0.6108845927509712, 0.2517942475174332, 0.0},
      {0.1373211597315956, 0.6108845927509714, 0.0},
      {0.6108845927509713, 0.1373211597315956, 0.0},
      {0.2517942475174332, 0.6108845927509713, 0.0},
      {0.1373211597315955, 0.2517942475174334, 0.0},
      {0.0779889301806426, 0.0779889301806429, 0.0},
      {0.8440221396387144, 0.0779889301806426, 0.0},
      {0.0779889301806430, 0.8440221396387146, 0.0},
      {0.3650347949363280, 0.0174418264943838, 0.0},
      {0.6175233785692882, 0.3650347949363281, 0.0},
      {0.0174418264943838, 0.6175233785692883, 0.0},
      {0.6175233785692882, 0.0174418264943838, 0.0},
      {0.3650347949363280, 0.6175233785692883, 0.0},
      {0.0174418264943837, 0.3650347949363282, 0.0},
      {0.4792449355569203, 0.0415101288861593, 0.0},
      {0.4792449355569204, 0.4792449355569204, 0.0},
      {0.0415101288861593, 0.4792449355569205, 0.0},
      {0.4120614054796895, 0.1758771890406211, 0.0},
      {0.4120614054796895, 0.4120614054796896, 0.0},
      {0.1758771890406210, 0.4120614054796897, 0.0},
      {0.2571566355502535, 0.0861981702781003, 0.0},
      {0.6566451941716462, 0.2571566355502535, 0.0},
      {0.0861981702781003, 0.6566451941716464, 0.0},
      {0.6566451941716462, 0.0861981702781003, 0.0},
      {0.2571566355502535, 0.6566451941716464, 0.0},
      {0.0861981702781001, 0.2571566355502538, 0.0},
      {0.4984105744139570, 0.0031788511720861, 0.0},
      {0.4984105744139570, 0.4984105744139570, 0.0},
      {0.0031788511720860, 0.4984105744139572, 0.0},
      {0.3091259486853505, 0.3091259486853506, 0.0},
      {0.3817481026292989, 0.3091259486853506, 0.0},
      {0.3091259486853505, 0.3817481026292990, 0.0},
      {0.1241816119341110, 0.0034112678008347, 0.0},
      {0.8724071202650543, 0.1241816119341110, 0.0},
      {0.0034112678008348, 0.8724071202650543, 0.0},
      {0.8724071202650543, 0.0034112678008347, 0.0},
      {0.1241816119341111, 0.8724071202650543, 0.0},
      {0.0034112678008344, 0.1241816119341114, 0.0},
      {0.1716507278093692, 0.0445555167117895, 0.0},
      {0.7837937554788413, 0.1716507278093692, 0.0},
      {0.0445555167117896, 0.7837937554788414, 0.0},
      {0.7837937554788413, 0.0445555167117895, 0.0},
      {0.1716507278093692, 0.7837937554788414, 0.0},
      {0.0445555167117893, 0.1716507278093694, 0.0},
      {0.3977231091982856, 0.1220975488845627, 0.0},
      {0.4801793419171518, 0.3977231091982856, 0.0},
      {0.1220975488845627, 0.4801793419171519, 0.0},
      {0.4801793419171518, 0.1220975488845627, 0.0},
      {0.3977231091982856, 0.4801793419171518, 0.0},
      {0.1220975488845626, 0.3977231091982857, 0.0},
      {0.0036715668219831, 0.0036715668219834, 0.0},
      {0.9926568663560335, 0.0036715668219831, 0.0},
      {0.0036715668219836, 0.9926568663560336, 0.0},
      {0.2578966456698565, 0.1957103712364281, 0.0},
      {0.5463929830937153, 0.2578966456698565, 0.0},
      {0.1957103712364281, 0.5463929830937155, 0.0},
      {0.5463929830937154, 0.1957103712364281, 0.0},
      {0.2578966456698565, 0.5463929830937155, 0.0},
      {0.1957103712364280, 0.2578966456698567, 0.0},
      {0.1441500744811936, 0.0183113919798340, 0.0},
      {0.8375385335389723, 0.1441500744811936, 0.0},
      {0.0183113919798341, 0.8375385335389725, 0.0},
      {0.8375385335389725, 0.0183113919798340, 0.0},
      {0.1441500744811937, 0.8375385335389725, 0.0},
      {0.0183113919798338, 0.1441500744811939, 0.0},
      {0.4122595610017910, 0.0032622580352038, 0.0},
      {0.5844781809630051, 0.4122595610017911, 0.0},
      {0.0032622580352039, 0.5844781809630053, 0.0},
      {0.5844781809630052, 0.0032622580352038, 0.0},
      {0.4122595610017911, 0.5844781809630052, 0.0},
      {0.0032622580352037, 0.4122595610017913, 0.0},
      {0.2524284647914363, 0.0034754415073157, 0.0},
      {0.7440960937012480, 0.2524284647914364, 0.0},
      {0.0034754415073158, 0.7440960937012481, 0.0},
      {0.7440960937012480, 0.0034754415073157, 0.0},
      {0.2524284647914364, 0.7440960937012481, 0.0},
      {0.0034754415073155, 0.2524284647914367, 0.0},
      {0.3293669274857745, 0.0035387900825818, 0.0},
      {0.6670942824316437, 0.3293669274857746, 0.0},
      {0.0035387900825819, 0.6670942824316438, 0.0},
      {0.6670942824316437, 0.0035387900825818, 0.0},
      {0.3293669274857746, 0.6670942824316437, 0.0},
      {0.0035387900825817, 0.3293669274857748, 0.0},
      {0.2095814817005154, 0.0188449390557523, 0.0},
      {0.7715735792437323, 0.2095814817005154, 0.0},
      {0.0188449390557522, 0.7715735792437326, 0.0},
      {0.7715735792437323, 0.0188449390557523, 0.0},
      {0.2095814817005155, 0.7715735792437324, 0.0},
      {0.0188449390557521, 0.2095814817005157, 0.0},
      {0.1836401694462322, 0.0036381021086963, 0.0},
      {0.8127217284450714, 0.1836401694462322, 0.0},
      {0.0036381021086965, 0.8127217284450715, 0.0},
      {0.8127217284450714, 0.0036381021086963, 0.0},
      {0.1836401694462323, 0.8127217284450715, 0.0},
      {0.0036381021086962, 0.1836401694462325, 0.0},
      {0.2416962324693375, 0.0457532869613174, 0.0},
      {0.7125504805693451, 0.2416962324693375, 0.0},
      {0.0457532869613175, 0.7125504805693452, 0.0},
      {0.7125504805693450, 0.0457532869613174, 0.0},
      {0.2416962324693376, 0.7125504805693452, 0.0},
      {0.0457532869613173, 0.2416962324693378, 0.0},
      {0.1893642612001962, 0.1893642612001964, 0.0},
      {0.6212714775996074, 0.1893642612001962, 0.0},
      {0.1893642612001963, 0.6212714775996077, 0.0}},
      {0.0007569948859290,
      0.0007569948859290,
      0.0007569948859290,
      0.0010495960486748,
      0.0010495960486748,
      0.0010495960486748,
      0.0010495960486748,
      0.0010495960486748,
      0.0010495960486748,
      0.0005523418345223,
      0.0005523418345223,
      0.0005523418345223,
      0.0009613125860911,
      0.0009613125860911,
      0.0009613125860911,
      0.0009613125860911,
      0.0009613125860911,
      0.0009613125860911,
      0.0002402215050010,
      0.0002402215050010,
      0.0002402215050010,
      0.0002402215050010,
      0.0002402215050010,
      0.0002402215050010,
      0.0015162737300447,
      0.0015162737300447,
      0.0015162737300447,
      0.0015162737300447,
      0.0015162737300447,
      0.0015162737300447,
      0.0036594531646208,
      0.0036594531646208,
      0.0036594531646208,
      0.0036594531646208,
      0.0036594531646208,
      0.0036594531646208,
      0.0026941594214905,
      0.0026941594214905,
      0.0026941594214905,
      0.0026941594214905,
      0.0026941594214905,
      0.0026941594214905,
      0.0037925461781852,
      0.0037925461781852,
      0.0037925461781852,
      0.0022846079594908,
      0.0022846079594908,
      0.0022846079594908,
      0.0022846079594908,
      0.0022846079594908,
      0.0022846079594908,
      0.0028601799532292,
      0.0028601799532292,
      0.0028601799532292,
      0.0028601799532292,
      0.0028601799532292,
      0.0028601799532292,
      0.0033560024245428,
      0.0033560024245428,
      0.0033560024245428,
      0.0033560024245428,
      0.0033560024245428,
      0.0033560024245428,
      0.0011585387546565,
      0.0011585387546565,
      0.0011585387546565,
      0.0011585387546565,
      0.0011585387546565,
      0.0011585387546565,
      0.0001543346309853,
      0.0001543346309853,
      0.0001543346309853,
      0.0001543346309853,
      0.0001543346309853,
      0.0001543346309853,
      0.0006023513847206,
      0.0006023513847206,
      0.0006023513847206,
      0.0006023513847206,
      0.0006023513847206,
      0.0006023513847206,
      0.0040818731260718,
      0.0040818731260718,
      0.0040818731260718,
      0.0040818731260718,
      0.0040818731260718,
      0.0040818731260718,
      0.0026250300771644,
      0.0026250300771644,
      0.0026250300771644,
      0.0029404173396389,
      0.0029404173396389,
      0.0029404173396389,
      0.0021822620140949,
      0.0021822620140949,
      0.0021822620140949,
      0.0021822620140949,
      0.0021822620140949,
      0.0021822620140949,
      0.0002748982753236,
      0.0002748982753236,
      0.0002748982753236,
      0.0002748982753236,
      0.0002748982753236,
      0.0002748982753236,
      0.0043222782870426,
      0.0043222782870426,
      0.0043222782870426,
      0.0026232190793461,
      0.0026232190793461,
      0.0026232190793461,
      0.0026232190793461,
      0.0026232190793461,
      0.0026232190793461,
      0.0016292447281957,
      0.0016292447281957,
      0.0016292447281957,
      0.0016292447281957,
      0.0016292447281957,
      0.0016292447281957,
      0.0023906174941902,
      0.0023906174941902,
      0.0023906174941902,
      0.0023906174941902,
      0.0023906174941902,
      0.0023906174941902,
      0.0008043859514434,
      0.0008043859514434,
      0.0008043859514434,
      0.0008043859514434,
      0.0008043859514434,
      0.0008043859514434,
      0.0004387936260754,
      0.0004387936260754,
      0.0004387936260754,
      0.0014912832526497,
      0.0014912832526497,
      0.0014912832526497,
      0.0014912832526497,
      0.0014912832526497,
      0.0014912832526497,
      0.0030469744217765,
      0.0030469744217765,
      0.0030469744217765,
      0.0030469744217765,
      0.0030469744217765,
      0.0030469744217765,
      0.0035436703730283,
      0.0035436703730283,
      0.0035436703730283,
      0.0035436703730283,
      0.0035436703730283,
      0.0035436703730283,
      0.0017931168013748,
      0.0017931168013748,
      0.0017931168013748,
      0.0015748763140406,
      0.0015748763140406,
      0.0015748763140406,
      0.0015748763140406,
      0.0015748763140406,
      0.0015748763140406,
      0.0024996214142379,
      0.0024996214142379,
      0.0024996214142379,
      0.0045394212527699,
      0.0045394212527699,
      0.0045394212527699,
      0.0031259349765605,
      0.0031259349765605,
      0.0031259349765605,
      0.0031259349765605,
      0.0031259349765605,
      0.0031259349765605,
      0.0007078390545973,
      0.0007078390545973,
      0.0007078390545973,
      0.0051033183245868,
      0.0051033183245868,
      0.0051033183245868,
      0.0004746526612736,
      0.0004746526612736,
      0.0004746526612736,
      0.0004746526612736,
      0.0004746526612736,
      0.0004746526612736,
      0.0020156260421464,
      0.0020156260421464,
      0.0020156260421464,
      0.0020156260421464,
      0.0020156260421464,
      0.0020156260421464,
      0.0040861087741152,
      0.0040861087741152,
      0.0040861087741152,
      0.0040861087741152,
      0.0040861087741152,
      0.0040861087741152,
      0.0000866476678905,
      0.0000866476678905,
      0.0000866476678905,
      0.0044793295143609,
      0.0044793295143609,
      0.0044793295143609,
      0.0044793295143609,
      0.0044793295143609,
      0.0044793295143609,
      0.0012310493707462,
      0.0012310493707462,
      0.0012310493707462,
      0.0012310493707462,
      0.0012310493707462,
      0.0012310493707462,
      0.0007106464606610,
      0.0007106464606610,
      0.0007106464606610,
      0.0007106464606610,
      0.0007106464606610,
      0.0007106464606610,
      0.0006525782479594,
      0.0006525782479594,
      0.0006525782479594,
      0.0006525782479594,
      0.0006525782479594,
      0.0006525782479594,
      0.0007229547638943,
      0.0007229547638943,
      0.0007229547638943,
      0.0007229547638943,
      0.0007229547638943,
      0.0007229547638943,
      0.0014672975290397,
      0.0014672975290397,
      0.0014672975290397,
      0.0014672975290397,
      0.0014672975290397,
      0.0014672975290397,
      0.0005982295026349,
      0.0005982295026349,
      0.0005982295026349,
      0.0005982295026349,
      0.0005982295026349,
      0.0005982295026349,
      0.0024367522099615,
      0.0024367522099615,
      0.0024367522099615,
      0.0024367522099615,
      0.0024367522099615,
      0.0024367522099615,
      0.0041453066285269,
      0.0041453066285269,
      0.0041453066285269}
    },
    {
      267,
      {{0.4590721360317646, 0.0818557279364709, 0.0},
      {0.4590721360317646, 0.4590721360317646, 0.0},
      {0.0818557279364708, 0.4590721360317648, 0.0},
      {0.2837655147017407, 0.0522971139020114, 0.0},
      {0.6639373713962480, 0.2837655147017407, 0.0},
      {0.0522971139020114, 0.6639373713962481, 0.0},
      {0.6639373713962479, 0.0522971139020114, 0.0},
      {0.2837655147017407, 0.6639373713962480, 0.0},
      {0.0522971139020114, 0.2837655147017409, 0.0},
      {0.3932680062553601, 0.2134639874892799, 0.0},
      {0.3932680062553601, 0.3932680062553601, 0.0},
      {0.2134639874892798, 0.3932680062553602, 0.0},
      {0.1285165649696939, 0.0990797627018413, 0.0},
      {0.7724036723284647, 0.1285165649696940, 0.0},
      {0.0990797627018414, 0.7724036723284649, 0.0},
      {0.7724036723284647, 0.0990797627018413, 0.0},
      {0.1285165649696941, 0.7724036723284649, 0.0},
      {0.0990797627018411, 0.1285165649696943, 0.0},
      {0.4681825892069357, 0.0636348215861286, 0.0},
      {0.4681825892069357, 0.4681825892069358, 0.0},
      {0.0636348215861285, 0.4681825892069360, 0.0},
      {0.0413075041428614, 0.0020644051493052, 0.0},
      {0.9566280907078333, 0.0413075041428614, 0.0},
      {0.0020644051493054, 0.9566280907078335, 0.0},
      {0.9566280907078334, 0.0020644051493052, 0.0},
      {0.0413075041428616, 0.9566280907078335, 0.0},
      {0.0020644051493049, 0.0413075041428617, 0.0},
      {0.0892719800534145, 0.0367301756493483, 0.0},
      {0.8739978442972371, 0.0892719800534144, 0.0},
      {0.0367301756493484, 0.8739978442972374, 0.0},
      {0.8739978442972371, 0.0367301756493483, 0.0},
      {0.0892719800534145, 0.8739978442972374, 0.0},
      {0.0367301756493481, 0.0892719800534148, 0.0},
      {0.3372052923179872, 0.2259458244877857, 0.0},
      {0.4368488831942271, 0.3372052923179873, 0.0},
      {0.2259458244877856, 0.4368488831942272, 0.0},
      {0.4368488831942271, 0.2259458244877857, 0.0},
      {0.3372052923179872, 0.4368488831942272, 0.0},
      {0.2259458244877856, 0.3372052923179874, 0.0},
      {0.2738513741241787, 0.0327011819900538, 0.0},
      {0.6934474438857674, 0.2738513741241788, 0.0},
      {0.0327011819900538, 0.6934474438857677, 0.0},
      {0.6934474438857675, 0.0327011819900538, 0.0},
      {0.2738513741241788, 0.6934474438857677, 0.0},
      {0.0327011819900536, 0.2738513741241790, 0.0},
      {0.0399162626294194, 0.0130695090443300, 0.0},
      {0.9470142283262505, 0.0399162626294194, 0.0},
      {0.0130695090443302, 0.9470142283262506, 0.0},
      {0.9470142283262506, 0.0130695090443300, 0.0},
      {0.0399162626294196, 0.9470142283262506, 0.0},
      {0.0130695090443297, 0.0399162626294198, 0.0},
      {0.0175045040462673, 0.0175045040462676, 0.0},
      {0.9649909919074650, 0.0175045040462673, 0.0},
      {0.0175045040462677, 0.9649909919074653, 0.0},
      {0.3081068327593843, 0.0737176145710159, 0.0},
      {0.6181755526695998, 0.3081068327593843, 0.0},
      {0.0737176145710159, 0.6181755526695999, 0.0},
      {0.6181755526695998, 0.0737176145710159, 0.0},
      {0.3081068327593843, 0.6181755526695998, 0.0},
      {0.0737176145710158, 0.3081068327593845, 0.0},
      {0.1454801146528607, 0.1454801146528609, 0.0},
      {0.7090397706942783, 0.1454801146528608, 0.0},
      {0.1454801146528609, 0.7090397706942786, 0.0},
      {0.2196247131384356, 0.0027038822145435, 0.0},
      {0.7776714046470208, 0.2196247131384356, 0.0},
      {0.0027038822145437, 0.7776714046470209, 0.0},
      {0.7776714046470208, 0.0027038822145435, 0.0},
      {0.2196247131384357, 0.7776714046470209, 0.0},
      {0.0027038822145434, 0.2196247131384359, 0.0},
      {0.2734646071142814, 0.2191582634460288, 0.0},
      {0.5073771294396898, 0.2734646071142815, 0.0},
      {0.2191582634460287, 0.5073771294396899, 0.0},
      {0.5073771294396898, 0.2191582634460288, 0.0},
      {0.2734646071142814, 0.5073771294396899, 0.0},
      {0.2191582634460286, 0.2734646071142816, 0.0},
      {0.0686830028555527, 0.0207228118815140, 0.0},
      {0.9105941852629332, 0.0686830028555526, 0.0},
      {0.0207228118815142, 0.9105941852629333, 0.0},
      {0.9105941852629332, 0.0207228118815140, 0.0},
      {0.0686830028555528, 0.9105941852629333, 0.0},
      {0.0207228118815138, 0.0686830028555530, 0.0},
      {0.1830135998401795, 0.1006217208920761, 0.0},
      {0.7163646792677445, 0.1830135998401795, 0.0},
      {0.1006217208920760, 0.7163646792677447, 0.0},
      {0.7163646792677445, 0.1006217208920761, 0.0},
      {0.1830135998401795, 0.7163646792677446, 0.0},
      {0.1006217208920759, 0.1830135998401798, 0.0},
      {0.0030471742663605, 0.0030471742663608, 0.0},
      {0.9939056514672787, 0.0030471742663604, 0.0},
      {0.0030471742663609, 0.9939056514672788, 0.0},
      {0.2287994832299253, 0.0694898178335093, 0.0},
      {0.7017106989365653, 0.2287994832299253, 0.0},
      {0.0694898178335093, 0.7017106989365657, 0.0},
      {0.7017106989365655, 0.0694898178335093, 0.0},
      {0.2287994832299252, 0.7017106989365657, 0.0},
      {0.0694898178335091, 0.2287994832299255, 0.0},
      {0.0587361601120220, 0.0587361601120223, 0.0},
      {0.8825276797759557, 0.0587361601120220, 0.0},
      {0.0587361601120223, 0.8825276797759559, 0.0},
      {0.1018827114869391, 0.0683773554225139, 0.0},
      {0.8297399330905469, 0.1018827114869391, 0.0},
      {0.0683773554225139, 0.8297399330905472, 0.0},
      {0.8297399330905471, 0.0683773554225139, 0.0},
      {0.1018827114869392, 0.8297399330905472, 0.0},
      {0.0683773554225136, 0.1018827114869394, 0.0},
      {0.2435636733958171, 0.0143204413633165, 0.0},
      {0.7421158852408665, 0.2435636733958171, 0.0},
      {0.0143204413633164, 0.7421158852408667, 0.0},
      {0.7421158852408665, 0.0143204413633165, 0.0},
      {0.2435636733958172, 0.7421158852408665, 0.0},
      {0.0143204413633164, 0.2435636733958174, 0.0},
      {0.2058877165179012, 0.0389678567422943, 0.0},
      {0.7551444267398045, 0.2058877165179012, 0.0},
      {0.0389678567422944, 0.7551444267398046, 0.0},
      {0.7551444267398045, 0.0389678567422943, 0.0},
      {0.2058877165179013, 0.7551444267398046, 0.0},
      {0.0389678567422941, 0.2058877165179015, 0.0},
      {0.2018695088876031, 0.1493964619179442, 0.0},
      {0.6487340291944527, 0.2018695088876032, 0.0},
      {0.1493964619179441, 0.6487340291944529, 0.0},
      {0.6487340291944528, 0.1493964619179442, 0.0},
      {0.2018695088876032, 0.6487340291944529, 0.0},
      {0.1493964619179440, 0.2018695088876034, 0.0},
      {0.1591160210105452, 0.0622282118189533, 0.0},
      {0.7786557671705014, 0.1591160210105452, 0.0},
      {0.0622282118189533, 0.7786557671705017, 0.0},
      {0.7786557671705016, 0.0622282118189533, 0.0},
      {0.1591160210105452, 0.7786557671705017, 0.0},
      {0.0622282118189531, 0.1591160210105455, 0.0},
      {0.1779476358555918, 0.0176192414704860, 0.0},
      {0.8044331226739222, 0.1779476358555918, 0.0},
      {0.0176192414704860, 0.8044331226739224, 0.0},
      {0.8044331226739222, 0.0176192414704860, 0.0},
      {0.1779476358555919, 0.8044331226739224, 0.0},
      {0.0176192414704858, 0.1779476358555921, 0.0},
      {0.1620367479496921, 0.0039477276652125, 0.0},
      {0.8340155243850955, 0.1620367479496921, 0.0},
      {0.0039477276652125, 0.8340155243850956, 0.0},
      {0.8340155243850954, 0.0039477276652125, 0.0},
      {0.1620367479496921, 0.8340155243850956, 0.0},
      {0.0039477276652123, 0.1620367479496924, 0.0},
      {0.2918549838585246, 0.0029311798861729, 0.0},
      {0.7052138362553024, 0.2918549838585247, 0.0},
      {0.0029311798861729, 0.7052138362553028, 0.0},
      {0.7052138362553025, 0.0029311798861729, 0.0},
      {0.2918549838585245, 0.7052138362553028, 0.0},
      {0.0029311798861727, 0.2918549838585249, 0.0},
      {0.3718786917495892, 0.0027779809824386, 0.0},
      {0.6253433272679723, 0.3718786917495892, 0.0},
      {0.0027779809824385, 0.6253433272679726, 0.0},
      {0.6253433272679723, 0.0027779809824386, 0.0},
      {0.3718786917495892, 0.6253433272679724, 0.0},
      {0.0027779809824384, 0.3718786917495893, 0.0},
      {0.1349693726524653, 0.0343221858358502, 0.0},
      {0.8307084415116845, 0.1349693726524653, 0.0},
      {0.0343221858358503, 0.8307084415116845, 0.0},
      {0.8307084415116845, 0.0343221858358502, 0.0},
      {0.1349693726524654, 0.8307084415116845, 0.0},
      {0.0343221858358500, 0.1349693726524656, 0.0},
      {0.0373835575017465, 0.0373835575017468, 0.0},
      {0.9252328849965067, 0.0373835575017464, 0.0},
      {0.0373835575017469, 0.9252328849965068, 0.0},
      {0.2855047782912640, 0.2855047782912642, 0.0},
      {0.4289904434174718, 0.2855047782912641, 0.0},
      {0.2855047782912640, 0.4289904434174719, 0.0},
      {0.2524534299064219, 0.1087240393419249, 0.0},
      {0.6388225307516531, 0.2524534299064219, 0.0},
      {0.1087240393419249, 0.6388225307516533, 0.0},
      {0.6388225307516532, 0.1087240393419249, 0.0},
      {0.2524534299064219, 0.6388225307516533, 0.0},
      {0.1087240393419247, 0.2524534299064221, 0.0},
      {0.4203335017700142, 0.1593329964599716, 0.0},
      {0.4203335017700143, 0.4203335017700143, 0.0},
      {0.1593329964599715, 0.4203335017700144, 0.0},
      {0.3242351641083045, 0.0149491079080454, 0.0},
      {0.6608157279836501, 0.3242351641083045, 0.0},
      {0.0149491079080454, 0.6608157279836503, 0.0},
      {0.6608157279836501, 0.0149491079080454, 0.0},
      {0.3242351641083045, 0.6608157279836503, 0.0},
      {0.0149491079080453, 0.3242351641083048, 0.0},
      {0.2076894216538629, 0.2076894216538631, 0.0},
      {0.5846211566922740, 0.2076894216538630, 0.0},
      {0.2076894216538630, 0.5846211566922742, 0.0},
      {0.4921962729553108, 0.0156074540893784, 0.0},
      {0.4921962729553108, 0.4921962729553109, 0.0},
      {0.0156074540893784, 0.4921962729553110, 0.0},
      {0.4374161906466257, 0.0373240771592635, 0.0},
      {0.5252597321941108, 0.4374161906466258, 0.0},
      {0.0373240771592634, 0.5252597321941110, 0.0},
      {0.5252597321941108, 0.0373240771592635, 0.0},
      {0.4374161906466257, 0.5252597321941109, 0.0},
      {0.0373240771592634, 0.4374161906466259, 0.0},
      {0.4559988036612093, 0.0029632719316370, 0.0},
      {0.5410379244071537, 0.4559988036612094, 0.0},
      {0.0029632719316369, 0.5410379244071539, 0.0},
      {0.5410379244071537, 0.0029632719316370, 0.0},
      {0.4559988036612093, 0.5410379244071538, 0.0},
      {0.0029632719316369, 0.4559988036612095, 0.0},
      {0.0754872857005683, 0.0042029781607735, 0.0},
      {0.9203097361386582, 0.0754872857005682, 0.0},
      {0.0042029781607736, 0.9203097361386584, 0.0},
      {0.9203097361386582, 0.0042029781607735, 0.0},
      {0.0754872857005683, 0.9203097361386584, 0.0},
      {0.0042029781607733, 0.0754872857005686, 0.0},
      {0.3260541266529466, 0.1142741873804544, 0.0},
      {0.5596716859665990, 0.3260541266529467, 0.0},
      {0.1142741873804544, 0.5596716859665991, 0.0},
      {0.5596716859665990, 0.1142741873804544, 0.0},
      {0.3260541266529467, 0.5596716859665990, 0.0},
      {0.1142741873804543, 0.3260541266529468, 0.0},
      {0.4011995885054882, 0.1128880853129215, 0.0},
      {0.4859123261815903, 0.4011995885054883, 0.0},
      {0.1128880853129214, 0.4859123261815905, 0.0},
      {0.4859123261815903, 0.1128880853129215, 0.0},
      {0.4011995885054881, 0.4859123261815904, 0.0},
      {0.1128880853129214, 0.4011995885054884, 0.0},
      {0.2687606603412640, 0.1601966778997559, 0.0},
      {0.5710426617589802, 0.2687606603412641, 0.0},
      {0.1601966778997558, 0.5710426617589803, 0.0},
      {0.5710426617589801, 0.1601966778997559, 0.0},
      {0.2687606603412641, 0.5710426617589802, 0.0},
      {0.1601966778997558, 0.2687606603412642, 0.0},
      {0.4060147025738772, 0.0151551286402894, 0.0},
      {0.5788301687858335, 0.4060147025738772, 0.0},
      {0.0151551286402894, 0.5788301687858336, 0.0},
      {0.5788301687858335, 0.0151551286402894, 0.0},
      {0.4060147025738773, 0.5788301687858335, 0.0},
      {0.0151551286402893, 0.4060147025738775, 0.0},
      {0.3565655939830261, 0.2868688120339477, 0.0},
      {0.3565655939830261, 0.3565655939830263, 0.0},
      {0.2868688120339477, 0.3565655939830263, 0.0},
      {0.3835103468343402, 0.0700421815004379, 0.0},
      {0.5464474716652220, 0.3835103468343402, 0.0},
      {0.0700421815004378, 0.5464474716652222, 0.0},
      {0.5464474716652219, 0.0700421815004379, 0.0},
      {0.3835103468343402, 0.5464474716652220, 0.0},
      {0.0700421815004378, 0.3835103468343404, 0.0},
      {0.0164470646369432, 0.0034429885015808, 0.0},
      {0.9801099468614760, 0.0164470646369432, 0.0},
      {0.0034429885015809, 0.9801099468614761, 0.0},
      {0.9801099468614760, 0.0034429885015808, 0.0},
      {0.0164470646369433, 0.9801099468614761, 0.0},
      {0.0034429885015805, 0.0164470646369435, 0.0},
      {0.3542977861626948, 0.0376676144786773, 0.0},
      {0.6080345993586279, 0.3542977861626948, 0.0},
      {0.0376676144786774, 0.6080345993586280, 0.0},
      {0.6080345993586279, 0.0376676144786773, 0.0},
      {0.3542977861626948, 0.6080345993586280, 0.0},
      {0.0376676144786772, 0.3542977861626950, 0.0},
      {0.3419951757983160, 0.1659984266767157, 0.0},
      {0.4920063975249682, 0.3419951757983162, 0.0},
      {0.1659984266767156, 0.4920063975249684, 0.0},
      {0.4920063975249683, 0.1659984266767157, 0.0},
      {0.3419951757983161, 0.4920063975249684, 0.0},
      {0.1659984266767155, 0.3419951757983163, 0.0},
      {0.1152888910792784, 0.0120325311111652, 0.0},
      {0.8726785778095564, 0.1152888910792784, 0.0},
      {0.0120325311111653, 0.8726785778095566, 0.0},
      {0.8726785778095565, 0.0120325311111652, 0.0},
      {0.1152888910792785, 0.8726785778095566, 0.0},
      {0.0120325311111650, 0.1152888910792787, 0.0},
      {0.1165281018124025, 0.0002982216815201, 0.0},
      {0.8831736765060774, 0.1165281018124025, 0.0},
      {0.0002982216815202, 0.8831736765060775, 0.0},
      {0.8831736765060774, 0.0002982216815201, 0.0},
      {0.1165281018124027, 0.8831736765060774, 0.0},
      {0.0002982216815200, 0.1165281018124029, 0.0}},
      {0.0015819822201420,
      0.0015819822201420,
      0.0015819822201420,
      0.0012425026155713,
      0.0012425026155713,
      0.0012425026155713,
      0.0012425026155713,
      0.0012425026155713,
      0.0012425026155713,
      0.0029308291095543,
      0.0029308291095543,
      0.0029308291095543,
      0.0018482445803534,
      0.0018482445803534,
      0.0018482445803534,
      0.0018482445803534,
      0.0018482445803534,
      0.0018482445803534,
      0.0019651084518506,
      0.0019651084518506,
      0.0019651084518506,
      0.0001738521994969,
      0.0001738521994969,
      0.0001738521994969,
      0.0001738521994969,
      0.0001738521994969,
      0.0001738521994969,
      0.0010369134369704,
      0.0010369134369704,
      0.0010369134369704,
      0.0010369134369704,
      0.0010369134369704,
      0.0010369134369704,
      0.0036855205450056,
      0.0036855205450056,
      0.0036855205450056,
      0.0036855205450056,
      0.0036855205450056,
      0.0036855205450056,
      0.0015626990572199,
      0.0015626990572199,
      0.0015626990572199,
      0.0015626990572199,
      0.0015626990572199,
      0.0015626990572199,
      0.0004530881072261,
      0.0004530881072261,
      0.0004530881072261,
      0.0004530881072261,
      0.0004530881072261,
      0.0004530881072261,
      0.0003497348836648,
      0.0003497348836648,
      0.0003497348836648,
      0.0024978138848734,
      0.0024978138848734,
      0.0024978138848734,
      0.0024978138848734,
      0.0024978138848734,
      0.0024978138848734,
      0.0026012112555219,
      0.0026012112555219,
      0.0026012112555219,
      0.0004628340994250,
      0.0004628340994250,
      0.0004628340994250,
      0.0004628340994250,
      0.0004628340994250,
      0.0004628340994250,
      0.0041410832313465,
      0.0041410832313465,
      0.0041410832313465,
      0.0041410832313465,
      0.0041410832313465,
      0.0041410832313465,
      0.0007684212329036,
      0.0007684212329036,
      0.0007684212329036,
      0.0007684212329036,
      0.0007684212329036,
      0.0007684212329036,
      0.0026113703694072,
      0.0026113703694072,
      0.0026113703694072,
      0.0026113703694072,
      0.0026113703694072,
      0.0026113703694072,
      0.0000620233313266,
      0.0000620233313266,
      0.0000620233313266,
      0.0023591963942099,
      0.0023591963942099,
      0.0023591963942099,
      0.0023591963942099,
      0.0023591963942099,
      0.0023591963942099,
      0.0012531911082573,
      0.0012531911082573,
      0.0012531911082573,
      0.0018010585024764,
      0.0018010585024764,
      0.0018010585024764,
      0.0018010585024764,
      0.0018010585024764,
      0.0018010585024764,
      0.0011851452336967,
      0.0011851452336967,
      0.0011851452336967,
      0.0011851452336967,
      0.0011851452336967,
      0.0011851452336967,
      0.0018052449166879,
      0.0018052449166879,
      0.0018052449166879,
      0.0018052449166879,
      0.0018052449166879,
      0.0018052449166879,
      0.0032702272653727,
      0.0032702272653727,
      0.0032702272653727,
      0.0032702272653727,
      0.0032702272653727,
      0.0032702272653727,
      0.0020396107401563,
      0.0020396107401563,
      0.0020396107401563,
      0.0020396107401563,
      0.0020396107401563,
      0.0020396107401563,
      0.0011628194590529,
      0.0011628194590529,
      0.0011628194590529,
      0.0011628194590529,
      0.0011628194590529,
      0.0011628194590529,
      0.0005028720110417,
      0.0005028720110417,
      0.0005028720110417,
      0.0005028720110417,
      0.0005028720110417,
      0.0005028720110417,
      0.0005715808857059,
      0.0005715808857059,
      0.0005715808857059,
      0.0005715808857059,
      0.0005715808857059,
      0.0005715808857059,
      0.0005942306612542,
      0.0005942306612542,
      0.0005942306612542,
      0.0005942306612542,
      0.0005942306612542,
      0.0005942306612542,
      0.0014526094726055,
      0.0014526094726055,
      0.0014526094726055,
      0.0014526094726055,
      0.0014526094726055,
      0.0014526094726055,
      0.0008537751019579,
      0.0008537751019579,
      0.0008537751019579,
      0.0044928047594025,
      0.0044928047594025,
      0.0044928047594025,
      0.0031633575987933,
      0.0031633575987933,
      0.0031633575987933,
      0.0031633575987933,
      0.0031633575987933,
      0.0031633575987933,
      0.0039688521375602,
      0.0039688521375602,
      0.0039688521375602,
      0.0013628431103600,
      0.0013628431103600,
      0.0013628431103600,
      0.0013628431103600,
      0.0013628431103600,
      0.0013628431103600,
      0.0038742765963338,
      0.0038742765963338,
      0.0038742765963338,
      0.0015179487772672,
      0.0015179487772672,
      0.0015179487772672,
      0.0022607957999026,
      0.0022607957999026,
      0.0022607957999026,
      0.0022607957999026,
      0.0022607957999026,
      0.0022607957999026,
      0.0006445871387147,
      0.0006445871387147,
      0.0006445871387147,
      0.0006445871387147,
      0.0006445871387147,
      0.0006445871387147,
      0.0004083419590864,
      0.0004083419590864,
      0.0004083419590864,
      0.0004083419590864,
      0.0004083419590864,
      0.0004083419590864,
      0.0034738896617761,
      0.0034738896617761,
      0.0034738896617761,
      0.0034738896617761,
      0.0034738896617761,
      0.0034738896617761,
      0.0035493089697451,
      0.0035493089697451,
      0.0035493089697451,
      0.0035493089697451,
      0.0035493089697451,
      0.0035493089697451,
      0.0038459601742889,
      0.0038459601742889,
      0.0038459601742889,
      0.0038459601742889,
      0.0038459601742889,
      0.0038459601742889,
      0.0014626095093621,
      0.0014626095093621,
      0.0014626095093621,
      0.0014626095093621,
      0.0014626095093621,
      0.0014626095093621,
      0.0049879968051913,
      0.0049879968051913,
      0.0049879968051913,
      0.0030283484982429,
      0.0030283484982429,
      0.0030283484982429,
      0.0030283484982429,
      0.0030283484982429,
      0.0030283484982429,
      0.0001668528165594,
      0.0001668528165594,
      0.0001668528165594,
      0.0001668528165594,
      0.0001668528165594,
      0.0001668528165594,
      0.0022647344604744,
      0.0022647344604744,
      0.0022647344604744,
      0.0022647344604744,
      0.0022647344604744,
      0.0022647344604744,
      0.0042012687071599,
      0.0042012687071599,
      0.0042012687071599,
      0.0042012687071599,
      0.0042012687071599,
      0.0042012687071599,
      0.0009054859262017,
      0.0009054859262017,
      0.0009054859262017,
      0.0009054859262017,
      0.0009054859262017,
      0.0009054859262017,
      0.0001461428315914,
      0.0001461428315914,
      0.0001461428315914,
      0.0001461428315914,
      0.0001461428315914,
      0.0001461428315914}
    },
    {
      282,
      {{0.2382752085845570, 0.0043744824382613, 0.0},
      {0.7573503089771818, 0.2382752085845570, 0.0},
      {0.0043744824382613, 0.7573503089771819, 0.0},
      {0.7573503089771818, 0.0043744824382613, 0.0},
      {0.2382752085845571, 0.7573503089771818, 0.0},
      {0.0043744824382611, 0.2382752085845573, 0.0},
      {0.5022902013149184, 0.2314370758890006, 0.0},
      {0.2662727227960811, 0.5022902013149185, 0.0},
      {0.2314370758890004, 0.2662727227960813, 0.0},
      {0.2662727227960811, 0.2314370758890006, 0.0},
      {0.5022902013149184, 0.2662727227960812, 0.0},
      {0.2314370758890005, 0.5022902013149185, 0.0},
      {0.3506827341571583, 0.2360994474365199, 0.0},
      {0.4132178184063219, 0.3506827341571583, 0.0},
      {0.2360994474365198, 0.4132178184063220, 0.0},
      {0.4132178184063219, 0.2360994474365199, 0.0},
      {0.3506827341571582, 0.4132178184063219, 0.0},
      {0.2360994474365199, 0.3506827341571584, 0.0},
      {0.2778089177192840, 0.0026623397478917, 0.0},
      {0.7195287425328243, 0.2778089177192841, 0.0},
      {0.0026623397478918, 0.7195287425328244, 0.0},
      {0.7195287425328242, 0.0026623397478917, 0.0},
      {0.2778089177192840, 0.7195287425328244, 0.0},
      {0.0026623397478915, 0.2778089177192843, 0.0},
      {0.4724094821505567, 0.0551810356988867, 0.0},
      {0.4724094821505566, 0.4724094821505568, 0.0},
      {0.0551810356988866, 0.4724094821505569, 0.0},
      {0.3212034826826168, 0.2074396119349455, 0.0},
      {0.4713569053824377, 0.3212034826826168, 0.0},
      {0.2074396119349455, 0.4713569053824378, 0.0},
      {0.4713569053824377, 0.2074396119349455, 0.0},
      {0.3212034826826168, 0.4713569053824377, 0.0},
      {0.2074396119349454, 0.3212034826826170, 0.0},
      {0.4575079686738243, 0.0849840626523515, 0.0},
      {0.4575079686738243, 0.4575079686738243, 0.0},
      {0.0849840626523514, 0.4575079686738245, 0.0},
      {0.4574242168529821, 0.0119091524767886, 0.0},
      {0.5306666306702293, 0.4574242168529821, 0.0},
      {0.0119091524767885, 0.5306666306702296, 0.0},
      {0.5306666306702292, 0.0119091524767886, 0.0},
      {0.4574242168529821, 0.5306666306702293, 0.0},
      {0.0119091524767886, 0.4574242168529823, 0.0},
      {0.1910383428521846, 0.0021569623388999, 0.0},
      {0.8068046948089155, 0.1910383428521846, 0.0},
      {0.0021569623388999, 0.8068046948089157, 0.0},
      {0.8068046948089155, 0.0021569623388999, 0.0},
      {0.1910383428521847, 0.8068046948089155, 0.0},
      {0.0021569623388998, 0.1910383428521849, 0.0},
      {0.3538472919168833, 0.2923054161662334, 0.0},
      {0.3538472919168833, 0.3538472919168835, 0.0},
      {0.2923054161662332, 0.3538472919168835, 0.0},
      {0.4991061498602990, 0.0017877002794021, 0.0},
      {0.4991061498602990, 0.4991061498602991, 0.0},
      {0.0017877002794020, 0.4991061498602992, 0.0},
      {0.2063519407328363, 0.1072574937516458, 0.0},
      {0.6863905655155179, 0.2063519407328363, 0.0},
      {0.1072574937516458, 0.6863905655155180, 0.0},
      {0.6863905655155178, 0.1072574937516458, 0.0},
      {0.2063519407328364, 0.6863905655155180, 0.0},
      {0.1072574937516456, 0.2063519407328365, 0.0},
      {0.2612383827874719, 0.1124383932914825, 0.0},
      {0.6263232239210454, 0.2612383827874720, 0.0},
      {0.1124383932914825, 0.6263232239210457, 0.0},
      {0.6263232239210456, 0.1124383932914825, 0.0},
      {0.2612383827874720, 0.6263232239210456, 0.0},
      {0.1124383932914823, 0.2612383827874722, 0.0},
      {0.1133316581963653, 0.1133316581963656, 0.0},
      {0.7733366836072691, 0.1133316581963653, 0.0},
      {0.1133316581963656, 0.7733366836072691, 0.0},
      {0.1593888996231203, 0.1140903249967197, 0.0},
      {0.7265207753801600, 0.1593888996231203, 0.0},
      {0.1140903249967197, 0.7265207753801601, 0.0},
      {0.7265207753801599, 0.1140903249967197, 0.0},
      {0.1593888996231204, 0.7265207753801601, 0.0},
      {0.1140903249967195, 0.1593888996231206, 0.0},
      {0.3988695132240669, 0.0619448419160625, 0.0},
      {0.5391856448598707, 0.3988695132240669, 0.0},
      {0.0619448419160624, 0.5391856448598709, 0.0},
      {0.5391856448598707, 0.0619448419160625, 0.0},
      {0.3988695132240669, 0.5391856448598707, 0.0},
      {0.0619448419160624, 0.3988695132240671, 0.0},
      {0.3122725792733609, 0.1219628832252846, 0.0},
      {0.5657645375013545, 0.3122725792733609, 0.0},
      {0.1219628832252846, 0.5657645375013547, 0.0},
      {0.5657645375013546, 0.1219628832252846, 0.0},
      {0.3122725792733608, 0.5657645375013547, 0.0},
      {0.1219628832252845, 0.3122725792733611, 0.0},
      {0.3644362623584764, 0.0386447963475402, 0.0},
      {0.5969189412939834, 0.3644362623584764, 0.0},
      {0.0386447963475403, 0.5969189412939835, 0.0},
      {0.5969189412939834, 0.0386447963475402, 0.0},
      {0.3644362623584764, 0.5969189412939835, 0.0},
      {0.0386447963475401, 0.3644362623584766, 0.0},
      {0.2305648342404432, 0.0172339717685666, 0.0},
      {0.7522011939909903, 0.2305648342404432, 0.0},
      {0.0172339717685666, 0.7522011939909904, 0.0},
      {0.7522011939909903, 0.0172339717685666, 0.0},
      {0.2305648342404432, 0.7522011939909904, 0.0},
      {0.0172339717685664, 0.2305648342404434, 0.0},
      {0.0157787047228343, 0.0157787047228346, 0.0},
      {0.9684425905543310, 0.0157787047228343, 0.0},
      {0.0157787047228348, 0.9684425905543311, 0.0},
      {0.0392682833818940, 0.0145623703030592, 0.0},
      {0.9461693463150468, 0.0392682833818940, 0.0},
      {0.0145623703030595, 0.9461693463150468, 0.0},
      {0.9461693463150468, 0.0145623703030592, 0.0},
      {0.0392682833818942, 0.9461693463150468, 0.0},
      {0.0145623703030590, 0.0392682833818944, 0.0},
      {0.3582577335477227, 0.1541611099973495, 0.0},
      {0.4875811564549278, 0.3582577335477228, 0.0},
      {0.1541611099973494, 0.4875811564549280, 0.0},
      {0.4875811564549278, 0.1541611099973495, 0.0},
      {0.3582577335477227, 0.4875811564549279, 0.0},
      {0.1541611099973493, 0.3582577335477229, 0.0},
      {0.3805078825483125, 0.1011401306789488, 0.0},
      {0.5183519867727387, 0.3805078825483126, 0.0},
      {0.1011401306789488, 0.5183519867727389, 0.0},
      {0.5183519867727387, 0.1011401306789488, 0.0},
      {0.3805078825483125, 0.5183519867727387, 0.0},
      {0.1011401306789487, 0.3805078825483127, 0.0},
      {0.1318865755866089, 0.0732858166481222, 0.0},
      {0.7948276077652688, 0.1318865755866089, 0.0},
      {0.0732858166481222, 0.7948276077652691, 0.0},
      {0.7948276077652688, 0.0732858166481222, 0.0},
      {0.1318865755866089, 0.7948276077652691, 0.0},
      {0.0732858166481220, 0.1318865755866092, 0.0},
      {0.1726302550688860, 0.0135174633972146, 0.0},
      {0.8138522815338993, 0.1726302550688860, 0.0},
      {0.0135174633972147, 0.8138522815338995, 0.0},
      {0.8138522815338993, 0.0135174633972146, 0.0},
      {0.1726302550688861, 0.8138522815338995, 0.0},
      {0.0135174633972144, 0.1726302550688863, 0.0},
      {0.0707756358900086, 0.0431486463812222, 0.0},
      {0.8860757177287693, 0.0707756358900086, 0.0},
      {0.0431486463812223, 0.8860757177287693, 0.0},
      {0.8860757177287693, 0.0431486463812222, 0.0},
      {0.0707756358900087, 0.8860757177287693, 0.0},
      {0.0431486463812219, 0.0707756358900089, 0.0},
      {0.0473824731204699, 0.0028483388383532, 0.0},
      {0.9497691880411768, 0.0473824731204699, 0.0},
      {0.0028483388383533, 0.9497691880411770, 0.0},
      {0.9497691880411769, 0.0028483388383532, 0.0},
      {0.0473824731204700, 0.9497691880411770, 0.0},
      {0.0028483388383529, 0.0473824731204703, 0.0},
      {0.1640091847666056, 0.1640091847666058, 0.0},
      {0.6719816304667885, 0.1640091847666057, 0.0},
      {0.1640091847666059, 0.6719816304667886, 0.0},
      {0.4084715607291340, 0.1830568785417320, 0.0},
      {0.4084715607291340, 0.4084715607291341, 0.0},
      {0.1830568785417320, 0.4084715607291342, 0.0},
      {0.2971932482957449, 0.0403974541969814, 0.0},
      {0.6624092975072737, 0.2971932482957449, 0.0},
      {0.0403974541969814, 0.6624092975072738, 0.0},
      {0.6624092975072737, 0.0403974541969814, 0.0},
      {0.2971932482957449, 0.6624092975072738, 0.0},
      {0.0403974541969812, 0.2971932482957452, 0.0},
      {0.4437652399535983, 0.0310025483435277, 0.0},
      {0.5252322117028740, 0.4437652399535983, 0.0},
      {0.0310025483435276, 0.5252322117028743, 0.0},
      {0.5252322117028740, 0.0310025483435277, 0.0},
      {0.4437652399535983, 0.5252322117028742, 0.0},
      {0.0310025483435276, 0.4437652399535986, 0.0},
      {0.2298548902498229, 0.0392854639000807, 0.0},
      {0.7308596458500963, 0.2298548902498229, 0.0},
      {0.0392854639000808, 0.7308596458500966, 0.0},
      {0.7308596458500963, 0.0392854639000807, 0.0},
      {0.2298548902498229, 0.7308596458500966, 0.0},
      {0.0392854639000805, 0.2298548902498232, 0.0},
      {0.4365416551079566, 0.1269166897840868, 0.0},
      {0.4365416551079566, 0.4365416551079567, 0.0},
      {0.1269166897840867, 0.4365416551079568, 0.0},
      {0.2515254271389071, 0.0715641320496677, 0.0},
      {0.6769104408114252, 0.2515254271389071, 0.0},
      {0.0715641320496676, 0.6769104408114255, 0.0},
      {0.6769104408114253, 0.0715641320496677, 0.0},
      {0.2515254271389073, 0.6769104408114252, 0.0},
      {0.0715641320496676, 0.2515254271389074, 0.0},
      {0.2206287919327525, 0.1543945917807747, 0.0},
      {0.6249766162864728, 0.2206287919327525, 0.0},
      {0.1543945917807746, 0.6249766162864731, 0.0},
      {0.6249766162864728, 0.1543945917807747, 0.0},
      {0.2206287919327525, 0.6249766162864731, 0.0},
      {0.1543945917807745, 0.2206287919327527, 0.0},
      {0.0370326513393936, 0.0370326513393939, 0.0},
      {0.9259346973212126, 0.0370326513393935, 0.0},
      {0.0370326513393939, 0.9259346973212128, 0.0},
      {0.4164031820468178, 0.0027734733727750, 0.0},
      {0.5808233445804072, 0.4164031820468178, 0.0},
      {0.0027734733727750, 0.5808233445804074, 0.0},
      {0.5808233445804072, 0.0027734733727750, 0.0},
      {0.4164031820468178, 0.5808233445804073, 0.0},
      {0.0027734733727749, 0.4164031820468180, 0.0},
      {0.2780372435840275, 0.1718458690352430, 0.0},
      {0.5501168873807294, 0.2780372435840276, 0.0},
      {0.1718458690352431, 0.5501168873807295, 0.0},
      {0.5501168873807294, 0.1718458690352430, 0.0},
      {0.2780372435840275, 0.5501168873807295, 0.0},
      {0.1718458690352429, 0.2780372435840278, 0.0},
      {0.3420758322145085, 0.0032524102580493, 0.0},
      {0.6546717575274422, 0.3420758322145085, 0.0},
      {0.0032524102580493, 0.6546717575274423, 0.0},
      {0.6546717575274422, 0.0032524102580493, 0.0},
      {0.3420758322145085, 0.6546717575274423, 0.0},
      {0.0032524102580491, 0.3420758322145088, 0.0},
      {0.2117317495651855, 0.2117317495651857, 0.0},
      {0.5765365008696287, 0.2117317495651856, 0.0},
      {0.2117317495651856, 0.5765365008696289, 0.0},
      {0.3766515227420243, 0.0160811994772277, 0.0},
      {0.6072672777807482, 0.3766515227420243, 0.0},
      {0.0160811994772276, 0.6072672777807483, 0.0},
      {0.6072672777807481, 0.0160811994772277, 0.0},
      {0.3766515227420243, 0.6072672777807482, 0.0},
      {0.0160811994772275, 0.3766515227420245, 0.0},
      {0.1122979167412356, 0.0436591987048876, 0.0},
      {0.8440428845538768, 0.1122979167412356, 0.0},
      {0.0436591987048877, 0.8440428845538770, 0.0},
      {0.8440428845538769, 0.0436591987048876, 0.0},
      {0.1122979167412357, 0.8440428845538770, 0.0},
      {0.0436591987048873, 0.1122979167412359, 0.0},
      {0.1861411558417737, 0.0682956923610528, 0.0},
      {0.7455631517971735, 0.1861411558417737, 0.0},
      {0.0682956923610529, 0.7455631517971736, 0.0},
      {0.7455631517971736, 0.0682956923610528, 0.0},
      {0.1861411558417738, 0.7455631517971735, 0.0},
      {0.0682956923610526, 0.1861411558417740, 0.0},
      {0.3217033982751871, 0.0747225031108798, 0.0},
      {0.6035740986139331, 0.3217033982751872, 0.0},
      {0.0747225031108797, 0.6035740986139333, 0.0},
      {0.6035740986139331, 0.0747225031108798, 0.0},
      {0.3217033982751872, 0.6035740986139331, 0.0},
      {0.0747225031108797, 0.3217033982751874, 0.0},
      {0.1660126851788667, 0.0361553743643339, 0.0},
      {0.7978319404567993, 0.1660126851788667, 0.0},
      {0.0361553743643340, 0.7978319404567995, 0.0},
      {0.7978319404567994, 0.0361553743643339, 0.0},
      {0.1660126851788668, 0.7978319404567995, 0.0},
      {0.0361553743643337, 0.1660126851788670, 0.0},
      {0.0193576952354285, 0.0028330906992412, 0.0},
      {0.9778092140653304, 0.0193576952354285, 0.0},
      {0.0028330906992414, 0.9778092140653304, 0.0},
      {0.9778092140653304, 0.0028330906992412, 0.0},
      {0.0193576952354286, 0.9778092140653306, 0.0},
      {0.0028330906992409, 0.0193576952354288, 0.0},
      {0.0719904092251254, 0.0179498835307188, 0.0},
      {0.9100597072441559, 0.0719904092251254, 0.0},
      {0.0179498835307189, 0.9100597072441560, 0.0},
      {0.9100597072441557, 0.0179498835307188, 0.0},
      {0.0719904092251255, 0.9100597072441560, 0.0},
      {0.0179498835307185, 0.0719904092251257, 0.0},
      {0.2869572884590686, 0.2869572884590688, 0.0},
      {0.4260854230818626, 0.2869572884590687, 0.0},
      {0.2869572884590688, 0.4260854230818627, 0.0},
      {0.1333327168798979, 0.0031925398705718, 0.0},
      {0.8634747432495302, 0.1333327168798978, 0.0},
      {0.0031925398705718, 0.8634747432495306, 0.0},
      {0.8634747432495303, 0.0031925398705718, 0.0},
      {0.1333327168798978, 0.8634747432495306, 0.0},
      {0.0031925398705716, 0.1333327168798982, 0.0},
      {0.2994455291729939, 0.0162924574190147, 0.0},
      {0.6842620134079914, 0.2994455291729939, 0.0},
      {0.0162924574190148, 0.6842620134079915, 0.0},
      {0.6842620134079914, 0.0162924574190147, 0.0},
      {0.2994455291729938, 0.6842620134079915, 0.0},
      {0.0162924574190146, 0.2994455291729942, 0.0},
      {0.1171078851891533, 0.0179153991813487, 0.0},
      {0.8649767156294978, 0.1171078851891533, 0.0},
      {0.0179153991813488, 0.8649767156294981, 0.0},
      {0.8649767156294980, 0.0179153991813487, 0.0},
      {0.1171078851891534, 0.8649767156294981, 0.0},
      {0.0179153991813485, 0.1171078851891536, 0.0},
      {0.0806765828829425, 0.0806765828829428, 0.0},
      {0.8386468342341147, 0.0806765828829425, 0.0},
      {0.0806765828829429, 0.8386468342341149, 0.0},
      {0.0853676016562655, 0.0035672493041342, 0.0},
      {0.9110651490396003, 0.0853676016562654, 0.0},
      {0.0035672493041343, 0.9110651490396005, 0.0},
      {0.9110651490396005, 0.0035672493041342, 0.0},
      {0.0853676016562656, 0.9110651490396005, 0.0},
      {0.0035672493041339, 0.0853676016562658, 0.0},
      {0.0036432199207299, 0.0036432199207302, 0.0},
      {0.9927135601585398, 0.0036432199207299, 0.0},
      {0.0036432199207303, 0.9927135601585401, 0.0}},
      {0.0003912015164198,
      0.0003912015164198,
      0.0003912015164198,
      0.0003912015164198,
      0.0003912015164198,
      0.0003912015164198,
      0.0027129629394763,
      0.0027129629394763,
      0.0027129629394763,
      0.0027129629394763,
      0.0027129629394763,
      0.0027129629394763,
      0.0032167824916104,
      0.0032167824916104,
      0.0032167824916104,
      0.0032167824916104,
      0.0032167824916104,
      0.0032167824916104,
      0.0003822553188158,
      0.0003822553188158,
      0.0003822553188158,
      0.0003822553188158,
      0.0003822553188158,
      0.0003822553188158,
      0.0019002364888004,
      0.0019002364888004,
      0.0019002364888004,
      0.0031297443405132,
      0.0031297443405132,
      0.0031297443405132,
      0.0031297443405132,
      0.0031297443405132,
      0.0031297443405132,
      0.0024045178700843,
      0.0024045178700843,
      0.0024045178700843,
      0.0010810437034472,
      0.0010810437034472,
      0.0010810437034472,
      0.0010810437034472,
      0.0010810437034472,
      0.0010810437034472,
      0.0003454269891611,
      0.0003454269891611,
      0.0003454269891611,
      0.0003454269891611,
      0.0003454269891611,
      0.0003454269891611,
      0.0040394972726736,
      0.0040394972726736,
      0.0040394972726736,
      0.0004192383409146,
      0.0004192383409146,
      0.0004192383409146,
      0.0021905480428541,
      0.0021905480428541,
      0.0021905480428541,
      0.0021905480428541,
      0.0021905480428541,
      0.0021905480428541,
      0.0024488695353958,
      0.0024488695353958,
      0.0024488695353958,
      0.0024488695353958,
      0.0024488695353958,
      0.0024488695353958,
      0.0018544678106881,
      0.0018544678106881,
      0.0018544678106881,
      0.0022154948254470,
      0.0022154948254470,
      0.0022154948254470,
      0.0022154948254470,
      0.0022154948254470,
      0.0022154948254470,
      0.0023499392143139,
      0.0023499392143139,
      0.0023499392143139,
      0.0023499392143139,
      0.0023499392143139,
      0.0023499392143139,
      0.0028474735794835,
      0.0028474735794835,
      0.0028474735794835,
      0.0028474735794835,
      0.0028474735794835,
      0.0028474735794835,
      0.0018616149943744,
      0.0018616149943744,
      0.0018616149943744,
      0.0018616149943744,
      0.0018616149943744,
      0.0018616149943744,
      0.0011355533110131,
      0.0011355533110131,
      0.0011355533110131,
      0.0011355533110131,
      0.0011355533110131,
      0.0011355533110131,
      0.0003343034558446,
      0.0003343034558446,
      0.0003343034558446,
      0.0004750928275739,
      0.0004750928275739,
      0.0004750928275739,
      0.0004750928275739,
      0.0004750928275739,
      0.0004750928275739,
      0.0037165503152853,
      0.0037165503152853,
      0.0037165503152853,
      0.0037165503152853,
      0.0037165503152853,
      0.0037165503152853,
      0.0030932152668710,
      0.0030932152668710,
      0.0030932152668710,
      0.0030932152668710,
      0.0030932152668710,
      0.0030932152668710,
      0.0019187111408719,
      0.0019187111408719,
      0.0019187111408719,
      0.0019187111408719,
      0.0019187111408719,
      0.0019187111408719,
      0.0009479364828765,
      0.0009479364828765,
      0.0009479364828765,
      0.0009479364828765,
      0.0009479364828765,
      0.0009479364828765,
      0.0011579385843162,
      0.0011579385843162,
      0.0011579385843162,
      0.0011579385843162,
      0.0011579385843162,
      0.0011579385843162,
      0.0002441353811019,
      0.0002441353811019,
      0.0002441353811019,
      0.0002441353811019,
      0.0002441353811019,
      0.0002441353811019,
      0.0029017689598018,
      0.0029017689598018,
      0.0029017689598018,
      0.0038496033161381,
      0.0038496033161381,
      0.0038496033161381,
      0.0019666235037674,
      0.0019666235037674,
      0.0019666235037674,
      0.0019666235037674,
      0.0019666235037674,
      0.0019666235037674,
      0.0018684915937171,
      0.0018684915937171,
      0.0018684915937171,
      0.0018684915937171,
      0.0018684915937171,
      0.0018684915937171,
      0.0017653134351678,
      0.0017653134351678,
      0.0017653134351678,
      0.0017653134351678,
      0.0017653134351678,
      0.0017653134351678,
      0.0033882516090427,
      0.0033882516090427,
      0.0033882516090427,
      0.0024366507453070,
      0.0024366507453070,
      0.0024366507453070,
      0.0024366507453070,
      0.0024366507453070,
      0.0024366507453070,
      0.0030348462109630,
      0.0030348462109630,
      0.0030348462109630,
      0.0030348462109630,
      0.0030348462109630,
      0.0030348462109630,
      0.0007960248160347,
      0.0007960248160347,
      0.0007960248160347,
      0.0005508817903754,
      0.0005508817903754,
      0.0005508817903754,
      0.0005508817903754,
      0.0005508817903754,
      0.0005508817903754,
      0.0036545914077611,
      0.0036545914077611,
      0.0036545914077611,
      0.0036545914077611,
      0.0036545914077611,
      0.0036545914077611,
      0.0005868097638695,
      0.0005868097638695,
      0.0005868097638695,
      0.0005868097638695,
      0.0005868097638695,
      0.0005868097638695,
      0.0035627819170746,
      0.0035627819170746,
      0.0035627819170746,
      0.0014138732102051,
      0.0014138732102051,
      0.0014138732102051,
      0.0014138732102051,
      0.0014138732102051,
      0.0014138732102051,
      0.0014059097790314,
      0.0014059097790314,
      0.0014059097790314,
      0.0014059097790314,
      0.0014059097790314,
      0.0014059097790314,
      0.0021888108864422,
      0.0021888108864422,
      0.0021888108864422,
      0.0021888108864422,
      0.0021888108864422,
      0.0021888108864422,
      0.0028898925874844,
      0.0028898925874844,
      0.0028898925874844,
      0.0028898925874844,
      0.0028898925874844,
      0.0028898925874844,
      0.0016608441575166,
      0.0016608441575166,
      0.0016608441575166,
      0.0016608441575166,
      0.0016608441575166,
      0.0016608441575166,
      0.0001617107192883,
      0.0001617107192883,
      0.0001617107192883,
      0.0001617107192883,
      0.0001617107192883,
      0.0001617107192883,
      0.0007799792558794,
      0.0007799792558794,
      0.0007799792558794,
      0.0007799792558794,
      0.0007799792558794,
      0.0007799792558794,
      0.0043643642549054,
      0.0043643642549054,
      0.0043643642549054,
      0.0004342298286508,
      0.0004342298286508,
      0.0004342298286508,
      0.0004342298286508,
      0.0004342298286508,
      0.0004342298286508,
      0.0013758391282983,
      0.0013758391282983,
      0.0013758391282983,
      0.0013758391282983,
      0.0013758391282983,
      0.0013758391282983,
      0.0010252931221566,
      0.0010252931221566,
      0.0010252931221566,
      0.0010252931221566,
      0.0010252931221566,
      0.0010252931221566,
      0.0018538426840327,
      0.0018538426840327,
      0.0018538426840327,
      0.0003920750628230,
      0.0003920750628230,
      0.0003920750628230,
      0.0003920750628230,
      0.0003920750628230,
      0.0003920750628230,
      0.0000874538907766,
      0.0000874538907766,
      0.0000874538907766}
    },
    {
      295,
      {{0.1671778383557105, 0.0015279490490132, 0.0},
      {0.8312942125952762, 0.1671778383557105, 0.0},
      {0.0015279490490133, 0.8312942125952765, 0.0},
      {0.8312942125952764, 0.0015279490490132, 0.0},
      {0.1671778383557105, 0.8312942125952765, 0.0},
      {0.0015279490490130, 0.1671778383557108, 0.0},
      {0.3714780994452375, 0.0149718224721411, 0.0},
      {0.6135500780826215, 0.3714780994452375, 0.0},
      {0.0149718224721411, 0.6135500780826216, 0.0},
      {0.6135500780826214, 0.0149718224721411, 0.0},
      {0.3714780994452375, 0.6135500780826215, 0.0},
      {0.0149718224721410, 0.3714780994452377, 0.0},
      {0.1590238847123736, 0.0068042587991278, 0.0},
      {0.8341718564884986, 0.1590238847123736, 0.0},
      {0.0068042587991279, 0.8341718564884987, 0.0},
      {0.8341718564884986, 0.0068042587991278, 0.0},
      {0.1590238847123738, 0.8341718564884987, 0.0},
      {0.0068042587991276, 0.1590238847123739, 0.0},
      {0.0462179329282665, 0.0353139747015546, 0.0},
      {0.9184680923701788, 0.0462179329282665, 0.0},
      {0.0353139747015547, 0.9184680923701790, 0.0},
      {0.9184680923701789, 0.0353139747015546, 0.0},
      {0.0462179329282665, 0.9184680923701790, 0.0},
      {0.0353139747015544, 0.0462179329282669, 0.0},
      {0.1148651673800807, 0.0015839864476047, 0.0},
      {0.8835508461723146, 0.1148651673800806, 0.0},
      {0.0015839864476049, 0.8835508461723147, 0.0},
      {0.8835508461723147, 0.0015839864476047, 0.0},
      {0.1148651673800808, 0.8835508461723147, 0.0},
      {0.0015839864476045, 0.1148651673800810, 0.0},
      {0.3342460081847144, 0.0215166720844818, 0.0},
      {0.6442373197308038, 0.3342460081847145, 0.0},
      {0.0215166720844818, 0.6442373197308039, 0.0},
      {0.6442373197308038, 0.0215166720844818, 0.0},
      {0.3342460081847144, 0.6442373197308039, 0.0},
      {0.0215166720844817, 0.3342460081847147, 0.0},
      {0.0723711649483900, 0.0020252616925631, 0.0},
      {0.9256035733590469, 0.0723711649483900, 0.0},
      {0.0020252616925632, 0.9256035733590470, 0.0},
      {0.9256035733590469, 0.0020252616925631, 0.0},
      {0.0723711649483901, 0.9256035733590470, 0.0},
      {0.0020252616925628, 0.0723711649483903, 0.0},
      {0.2236842466647887, 0.0153612037490869, 0.0},
      {0.7609545495861244, 0.2236842466647887, 0.0},
      {0.0153612037490869, 0.7609545495861246, 0.0},
      {0.7609545495861244, 0.0153612037490869, 0.0},
      {0.2236842466647888, 0.7609545495861246, 0.0},
      {0.0153612037490867, 0.2236842466647890, 0.0},
      {0.1111290169905849, 0.0101659439479150, 0.0},
      {0.8787050390615000, 0.1111290169905849, 0.0},
      {0.0101659439479151, 0.8787050390615002, 0.0},
      {0.8787050390615001, 0.0101659439479150, 0.0},
      {0.1111290169905850, 0.8787050390615003, 0.0},
      {0.0101659439479147, 0.1111290169905853, 0.0},
      {0.2764445418077909, 0.0300138537838293, 0.0},
      {0.6935416044083799, 0.2764445418077909, 0.0},
      {0.0300138537838293, 0.6935416044083801, 0.0},
      {0.6935416044083799, 0.0300138537838293, 0.0},
      {0.2764445418077908, 0.6935416044083801, 0.0},
      {0.0300138537838291, 0.2764445418077912, 0.0},
      {0.1692633554656758, 0.0189700426400216, 0.0},
      {0.8117666018943027, 0.1692633554656758, 0.0},
      {0.0189700426400217, 0.8117666018943028, 0.0},
      {0.8117666018943026, 0.0189700426400216, 0.0},
      {0.1692633554656758, 0.8117666018943028, 0.0},
      {0.0189700426400213, 0.1692633554656761, 0.0},
      {0.1196509552487583, 0.0267407842221270, 0.0},
      {0.8536082605291146, 0.1196509552487583, 0.0},
      {0.0267407842221271, 0.8536082605291148, 0.0},
      {0.8536082605291146, 0.0267407842221270, 0.0},
      {0.1196509552487584, 0.8536082605291149, 0.0},
      {0.0267407842221267, 0.1196509552487586, 0.0},
      {0.2830190971292686, 0.0008538186935375, 0.0},
      {0.7161270841771937, 0.2830190971292687, 0.0},
      {0.0008538186935376, 0.7161270841771940, 0.0},
      {0.7161270841771938, 0.0008538186935375, 0.0},
      {0.2830190971292687, 0.7161270841771940, 0.0},
      {0.0008538186935374, 0.2830190971292690, 0.0},
      {0.0704045501511468, 0.0122584700439817, 0.0},
      {0.9173369798048715, 0.0704045501511468, 0.0},
      {0.0122584700439818, 0.9173369798048717, 0.0},
      {0.9173369798048715, 0.0122584700439817, 0.0},
      {0.0704045501511468, 0.9173369798048717, 0.0},
      {0.0122584700439815, 0.0704045501511471, 0.0},
      {0.2164720987514121, 0.0371700471134970, 0.0},
      {0.7463578541350908, 0.2164720987514121, 0.0},
      {0.0371700471134971, 0.7463578541350910, 0.0},
      {0.7463578541350909, 0.0371700471134970, 0.0},
      {0.2164720987514122, 0.7463578541350910, 0.0},
      {0.0371700471134968, 0.2164720987514124, 0.0},
      {0.0769268815632591, 0.0321942006634705, 0.0},
      {0.8908789177732704, 0.0769268815632591, 0.0},
      {0.0321942006634706, 0.8908789177732706, 0.0},
      {0.8908789177732706, 0.0321942006634705, 0.0},
      {0.0769268815632591, 0.8908789177732707, 0.0},
      {0.0321942006634703, 0.0769268815632594, 0.0},
      {0.4292227320385513, 0.0089321527113154, 0.0},
      {0.5618451152501334, 0.4292227320385513, 0.0},
      {0.0089321527113154, 0.5618451152501336, 0.0},
      {0.5618451152501333, 0.0089321527113154, 0.0},
      {0.4292227320385513, 0.5618451152501334, 0.0},
      {0.0089321527113153, 0.4292227320385515, 0.0},
      {0.0157037453637930, 0.0157037453637933, 0.0},
      {0.9685925092724137, 0.0157037453637930, 0.0},
      {0.0157037453637935, 0.9685925092724137, 0.0},
      {0.2938899628974431, 0.2938899628974431, 0.0},
      {0.4122200742051138, 0.2938899628974431, 0.0},
      {0.2938899628974431, 0.4122200742051139, 0.0},
      {0.3701028684237416, 0.2597942631525167, 0.0},
      {0.3701028684237416, 0.3701028684237417, 0.0},
      {0.2597942631525166, 0.3701028684237418, 0.0},
      {0.3251479248852048, 0.2240452152274997, 0.0},
      {0.4508068598872954, 0.3251479248852049, 0.0},
      {0.2240452152274997, 0.4508068598872956, 0.0},
      {0.4508068598872955, 0.2240452152274997, 0.0},
      {0.3251479248852048, 0.4508068598872956, 0.0},
      {0.2240452152274996, 0.3251479248852050, 0.0},
      {0.2782727102869363, 0.1876262169906757, 0.0},
      {0.5341010727223879, 0.2782727102869364, 0.0},
      {0.1876262169906756, 0.5341010727223882, 0.0},
      {0.5341010727223879, 0.1876262169906757, 0.0},
      {0.2782727102869363, 0.5341010727223882, 0.0},
      {0.1876262169906754, 0.2782727102869366, 0.0},
      {0.2852102535187100, 0.0097585213238681, 0.0},
      {0.7050312251574219, 0.2852102535187100, 0.0},
      {0.0097585213238682, 0.7050312251574220, 0.0},
      {0.7050312251574219, 0.0097585213238681, 0.0},
      {0.2852102535187100, 0.7050312251574220, 0.0},
      {0.0097585213238680, 0.2852102535187103, 0.0},
      {0.2727256318298310, 0.0587062906216736, 0.0},
      {0.6685680775484955, 0.2727256318298311, 0.0},
      {0.0587062906216736, 0.6685680775484956, 0.0},
      {0.6685680775484953, 0.0587062906216736, 0.0},
      {0.2727256318298311, 0.6685680775484956, 0.0},
      {0.0587062906216735, 0.2727256318298313, 0.0},
      {0.4154695947712394, 0.0307419996861603, 0.0},
      {0.5537884055426003, 0.4154695947712395, 0.0},
      {0.0307419996861603, 0.5537884055426004, 0.0},
      {0.5537884055426003, 0.0307419996861603, 0.0},
      {0.4154695947712395, 0.5537884055426003, 0.0},
      {0.0307419996861602, 0.4154695947712397, 0.0},
      {0.2205774465337091, 0.0032567536371102, 0.0},
      {0.7761657998291807, 0.2205774465337091, 0.0},
      {0.0032567536371101, 0.7761657998291810, 0.0},
      {0.7761657998291808, 0.0032567536371102, 0.0},
      {0.2205774465337091, 0.7761657998291810, 0.0},
      {0.0032567536371099, 0.2205774465337094, 0.0},
      {0.3428856203943587, 0.0451876229604891, 0.0},
      {0.6119267566451521, 0.3428856203943588, 0.0},
      {0.0451876229604891, 0.6119267566451523, 0.0},
      {0.6119267566451522, 0.0451876229604891, 0.0},
      {0.3428856203943588, 0.6119267566451522, 0.0},
      {0.0451876229604890, 0.3428856203943590, 0.0},
      {0.2315038606116803, 0.1508185102240568, 0.0},
      {0.6176776291642629, 0.2315038606116803, 0.0},
      {0.1508185102240567, 0.6176776291642632, 0.0},
      {0.6176776291642631, 0.1508185102240568, 0.0},
      {0.2315038606116803, 0.6176776291642631, 0.0},
      {0.1508185102240566, 0.2315038606116805, 0.0},
      {0.0381017129649182, 0.0166736740374799, 0.0},
      {0.9452246129976019, 0.0381017129649183, 0.0},
      {0.0166736740374801, 0.9452246129976019, 0.0},
      {0.9452246129976020, 0.0166736740374799, 0.0},
      {0.0381017129649182, 0.9452246129976022, 0.0},
      {0.0166736740374796, 0.0381017129649186, 0.0},
      {0.3528911085101677, 0.1621052421455440, 0.0},
      {0.4850036493442883, 0.3528911085101678, 0.0},
      {0.1621052421455439, 0.4850036493442885, 0.0},
      {0.4850036493442883, 0.1621052421455440, 0.0},
      {0.3528911085101677, 0.4850036493442884, 0.0},
      {0.1621052421455439, 0.3528911085101679, 0.0},
      {0.4000760699195811, 0.0645258600999304, 0.0},
      {0.5353980699804886, 0.4000760699195811, 0.0},
      {0.0645258600999303, 0.5353980699804888, 0.0},
      {0.5353980699804887, 0.0645258600999304, 0.0},
      {0.4000760699195811, 0.5353980699804887, 0.0},
      {0.0645258600999302, 0.4000760699195813, 0.0},
      {0.1877843415465376, 0.1143796954685691, 0.0},
      {0.6978359629848933, 0.1877843415465376, 0.0},
      {0.1143796954685691, 0.6978359629848935, 0.0},
      {0.6978359629848933, 0.1143796954685691, 0.0},
      {0.1877843415465377, 0.6978359629848934, 0.0},
      {0.1143796954685689, 0.1877843415465379, 0.0},
      {0.2073769376903412, 0.0695719958527821, 0.0},
      {0.7230510664568768, 0.2073769376903412, 0.0},
      {0.0695719958527821, 0.7230510664568770, 0.0},
      {0.7230510664568768, 0.0695719958527821, 0.0},
      {0.2073769376903412, 0.7230510664568770, 0.0},
      {0.0695719958527819, 0.2073769376903414, 0.0},
      {0.1078038302263755, 0.0560343597005426, 0.0},
      {0.8361618100730819, 0.1078038302263755, 0.0},
      {0.0560343597005427, 0.8361618100730821, 0.0},
      {0.8361618100730819, 0.0560343597005426, 0.0},
      {0.1078038302263756, 0.8361618100730820, 0.0},
      {0.0560343597005424, 0.1078038302263758, 0.0},
      {0.1479014660731261, 0.0815896657841671, 0.0},
      {0.7705088681427068, 0.1479014660731261, 0.0},
      {0.0815896657841672, 0.7705088681427070, 0.0},
      {0.7705088681427068, 0.0815896657841671, 0.0},
      {0.1479014660731260, 0.7705088681427070, 0.0},
      {0.0815896657841669, 0.1479014660731263, 0.0},
      {0.1600426684113588, 0.0446292737712247, 0.0},
      {0.7953280578174164, 0.1600426684113588, 0.0},
      {0.0446292737712248, 0.7953280578174167, 0.0},
      {0.7953280578174164, 0.0446292737712247, 0.0},
      {0.1600426684113588, 0.7953280578174167, 0.0},
      {0.0446292737712245, 0.1600426684113591, 0.0},
      {0.3778766696887720, 0.1087918765992174, 0.0},
      {0.5133314537120106, 0.3778766696887721, 0.0},
      {0.1087918765992173, 0.5133314537120108, 0.0},
      {0.5133314537120106, 0.1087918765992174, 0.0},
      {0.3778766696887720, 0.5133314537120107, 0.0},
      {0.1087918765992173, 0.3778766696887723, 0.0},
      {0.2526963148522067, 0.2526963148522068, 0.0},
      {0.4946073702955864, 0.2526963148522067, 0.0},
      {0.2526963148522068, 0.4946073702955866, 0.0},
      {0.0392233696648084, 0.0032061485845242, 0.0},
      {0.9575704817506674, 0.0392233696648084, 0.0},
      {0.0032061485845244, 0.9575704817506674, 0.0},
      {0.9575704817506674, 0.0032061485845242, 0.0},
      {0.0392233696648085, 0.9575704817506674, 0.0},
      {0.0032061485845240, 0.0392233696648088, 0.0},
      {0.3243044450771381, 0.0829765480589191, 0.0},
      {0.5927190068639427, 0.3243044450771382, 0.0},
      {0.0829765480589191, 0.5927190068639430, 0.0},
      {0.5927190068639427, 0.0829765480589191, 0.0},
      {0.3243044450771381, 0.5927190068639429, 0.0},
      {0.0829765480589190, 0.3243044450771383, 0.0},
      {0.2105771295148042, 0.2105771295148044, 0.0},
      {0.5788457409703914, 0.2105771295148043, 0.0},
      {0.2105771295148043, 0.5788457409703917, 0.0},
      {0.0030575106285418, 0.0030575106285422, 0.0},
      {0.9938849787429160, 0.0030575106285418, 0.0},
      {0.0030575106285423, 0.9938849787429160, 0.0},
      {0.4032922222742628, 0.1934155554514744, 0.0},
      {0.4032922222742628, 0.4032922222742629, 0.0},
      {0.1934155554514743, 0.4032922222742630, 0.0},
      {0.4767516468002929, 0.0464967063994142, 0.0},
      {0.4767516468002928, 0.4767516468002930, 0.0},
      {0.0464967063994142, 0.4767516468002931, 0.0},
      {0.3509848476225047, 0.0035207350099882, 0.0},
      {0.6454944173675070, 0.3509848476225048, 0.0},
      {0.0035207350099882, 0.6454944173675073, 0.0},
      {0.6454944173675071, 0.0035207350099882, 0.0},
      {0.3509848476225048, 0.6454944173675070, 0.0},
      {0.0035207350099881, 0.3509848476225050, 0.0},
      {0.0648562451597025, 0.0648562451597028, 0.0},
      {0.8702875096805947, 0.0648562451597025, 0.0},
      {0.0648562451597030, 0.8702875096805948, 0.0},
      {0.1688963797829617, 0.1688963797829620, 0.0},
      {0.6622072404340762, 0.1688963797829618, 0.0},
      {0.1688963797829621, 0.6622072404340763, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.4208708276256937, 0.0005982947286368, 0.0},
      {0.5785308776456697, 0.4208708276256937, 0.0},
      {0.0005982947286367, 0.5785308776456698, 0.0},
      {0.5785308776456696, 0.0005982947286368, 0.0},
      {0.4208708276256937, 0.5785308776456696, 0.0},
      {0.0005982947286367, 0.4208708276256939, 0.0},
      {0.0962980341591576, 0.0962980341591579, 0.0},
      {0.8074039316816845, 0.0962980341591576, 0.0},
      {0.0962980341591579, 0.8074039316816847, 0.0},
      {0.2526560385602647, 0.0996837194601995, 0.0},
      {0.6476602419795359, 0.2526560385602647, 0.0},
      {0.0996837194601995, 0.6476602419795360, 0.0},
      {0.6476602419795359, 0.0996837194601995, 0.0},
      {0.2526560385602647, 0.6476602419795360, 0.0},
      {0.0996837194601994, 0.2526560385602649, 0.0},
      {0.4907015691302040, 0.0185968617395921, 0.0},
      {0.4907015691302040, 0.4907015691302040, 0.0},
      {0.0185968617395920, 0.4907015691302042, 0.0},
      {0.4984645792365976, 0.0030708415268048, 0.0},
      {0.4984645792365976, 0.4984645792365977, 0.0},
      {0.0030708415268047, 0.4984645792365979, 0.0},
      {0.3016703335605275, 0.1308239143750487, 0.0},
      {0.5675057520644238, 0.3016703335605276, 0.0},
      {0.1308239143750486, 0.5675057520644240, 0.0},
      {0.5675057520644238, 0.1308239143750487, 0.0},
      {0.3016703335605276, 0.5675057520644239, 0.0},
      {0.1308239143750485, 0.3016703335605278, 0.0},
      {0.4322719776068064, 0.1354560447863873, 0.0},
      {0.4322719776068064, 0.4322719776068065, 0.0},
      {0.1354560447863873, 0.4322719776068065, 0.0},
      {0.1302416069902091, 0.1302416069902094, 0.0},
      {0.7395167860195814, 0.1302416069902092, 0.0},
      {0.1302416069902094, 0.7395167860195816, 0.0},
      {0.4568717371543228, 0.0862565256913544, 0.0},
      {0.4568717371543228, 0.4568717371543228, 0.0},
      {0.0862565256913544, 0.4568717371543230, 0.0},
      {0.0160613879667907, 0.0029955877400127, 0.0},
      {0.9809430242931965, 0.0160613879667907, 0.0},
      {0.0029955877400129, 0.9809430242931966, 0.0},
      {0.9809430242931966, 0.0029955877400127, 0.0},
      {0.0160613879667909, 0.9809430242931966, 0.0},
      {0.0029955877400125, 0.0160613879667911, 0.0}},
      {0.0001887761839268,
      0.0001887761839268,
      0.0001887761839268,
      0.0001887761839268,
      0.0001887761839268,
      0.0001887761839268,
      0.0007758424849275,
      0.0007758424849275,
      0.0007758424849275,
      0.0007758424849275,
      0.0007758424849275,
      0.0007758424849275,
      0.0004322072803733,
      0.0004322072803733,
      0.0004322072803733,
      0.0004322072803733,
      0.0004322072803733,
      0.0004322072803733,
      0.0004985924659366,
      0.0004985924659366,
      0.0004985924659366,
      0.0004985924659366,
      0.0004985924659366,
      0.0004985924659366,
      0.0002045385156490,
      0.0002045385156490,
      0.0002045385156490,
      0.0002045385156490,
      0.0002045385156490,
      0.0002045385156490,
      0.0010218335800422,
      0.0010218335800422,
      0.0010218335800422,
      0.0010218335800422,
      0.0010218335800422,
      0.0010218335800422,
      0.0002063221663138,
      0.0002063221663138,
      0.0002063221663138,
      0.0002063221663138,
      0.0002063221663138,
      0.0002063221663138,
      0.0009573483771984,
      0.0009573483771984,
      0.0009573483771984,
      0.0009573483771984,
      0.0009573483771984,
      0.0009573483771984,
      0.0005820015679949,
      0.0005820015679949,
      0.0005820015679949,
      0.0005820015679949,
      0.0005820015679949,
      0.0005820015679949,
      0.0014593368553262,
      0.0014593368553262,
      0.0014593368553262,
      0.0014593368553262,
      0.0014593368553262,
      0.0014593368553262,
      0.0009711690861612,
      0.0009711690861612,
      0.0009711690861612,
      0.0009711690861612,
      0.0009711690861612,
      0.0009711690861612,
      0.0010218580321165,
      0.0010218580321165,
      0.0010218580321165,
      0.0010218580321165,
      0.0010218580321165,
      0.0010218580321165,
      0.0002280894292174,
      0.0002280894292174,
      0.0002280894292174,
      0.0002280894292174,
      0.0002280894292174,
      0.0002280894292174,
      0.0005590763253492,
      0.0005590763253492,
      0.0005590763253492,
      0.0005590763253492,
      0.0005590763253492,
      0.0005590763253492,
      0.0015182362058062,
      0.0015182362058062,
      0.0015182362058062,
      0.0015182362058062,
      0.0015182362058062,
      0.0015182362058062,
      0.0009700928395557,
      0.0009700928395557,
      0.0009700928395557,
      0.0009700928395557,
      0.0009700928395557,
      0.0009700928395557,
      0.0009574617421965,
      0.0009574617421965,
      0.0009574617421965,
      0.0009574617421965,
      0.0009574617421965,
      0.0009574617421965,
      0.0003128033098374,
      0.0003128033098374,
      0.0003128033098374,
      0.0042972644794934,
      0.0042972644794934,
      0.0042972644794934,
      0.0042996661707810,
      0.0042996661707810,
      0.0042996661707810,
      0.0041160670281138,
      0.0041160670281138,
      0.0041160670281138,
      0.0041160670281138,
      0.0041160670281138,
      0.0041160670281138,
      0.0037697602429351,
      0.0037697602429351,
      0.0037697602429351,
      0.0037697602429351,
      0.0037697602429351,
      0.0037697602429351,
      0.0009275873124364,
      0.0009275873124364,
      0.0009275873124364,
      0.0009275873124364,
      0.0009275873124364,
      0.0009275873124364,
      0.0022024106870840,
      0.0022024106870840,
      0.0022024106870840,
      0.0022024106870840,
      0.0022024106870840,
      0.0022024106870840,
      0.0019036025030930,
      0.0019036025030930,
      0.0019036025030930,
      0.0019036025030930,
      0.0019036025030930,
      0.0019036025030930,
      0.0004847222895272,
      0.0004847222895272,
      0.0004847222895272,
      0.0004847222895272,
      0.0004847222895272,
      0.0004847222895272,
      0.0021120401667404,
      0.0021120401667404,
      0.0021120401667404,
      0.0021120401667404,
      0.0021120401667404,
      0.0021120401667404,
      0.0033096627764702,
      0.0033096627764702,
      0.0033096627764702,
      0.0033096627764702,
      0.0033096627764702,
      0.0033096627764702,
      0.0005007036444134,
      0.0005007036444134,
      0.0005007036444134,
      0.0005007036444134,
      0.0005007036444134,
      0.0005007036444134,
      0.0037781555372607,
      0.0037781555372607,
      0.0037781555372607,
      0.0037781555372607,
      0.0037781555372607,
      0.0037781555372607,
      0.0026493525219968,
      0.0026493525219968,
      0.0026493525219968,
      0.0026493525219968,
      0.0026493525219968,
      0.0026493525219968,
      0.0027773448980642,
      0.0027773448980642,
      0.0027773448980642,
      0.0027773448980642,
      0.0027773448980642,
      0.0027773448980642,
      0.0022529922804747,
      0.0022529922804747,
      0.0022529922804747,
      0.0022529922804747,
      0.0022529922804747,
      0.0022529922804747,
      0.0015565675694602,
      0.0015565675694602,
      0.0015565675694602,
      0.0015565675694602,
      0.0015565675694602,
      0.0015565675694602,
      0.0021949191081013,
      0.0021949191081013,
      0.0021949191081013,
      0.0021949191081013,
      0.0021949191081013,
      0.0021949191081013,
      0.0016192761860415,
      0.0016192761860415,
      0.0016192761860415,
      0.0016192761860415,
      0.0016192761860415,
      0.0016192761860415,
      0.0032871490558048,
      0.0032871490558048,
      0.0032871490558048,
      0.0032871490558048,
      0.0032871490558048,
      0.0032871490558048,
      0.0040195677996607,
      0.0040195677996607,
      0.0040195677996607,
      0.0002309687901874,
      0.0002309687901874,
      0.0002309687901874,
      0.0002309687901874,
      0.0002309687901874,
      0.0002309687901874,
      0.0028101649311348,
      0.0028101649311348,
      0.0028101649311348,
      0.0028101649311348,
      0.0028101649311348,
      0.0028101649311348,
      0.0036059117974544,
      0.0036059117974544,
      0.0036059117974544,
      0.0000614819425667,
      0.0000614819425667,
      0.0000614819425667,
      0.0040334024837260,
      0.0040334024837260,
      0.0040334024837260,
      0.0023394881142265,
      0.0023394881142265,
      0.0023394881142265,
      0.0006163847784648,
      0.0006163847784648,
      0.0006163847784648,
      0.0006163847784648,
      0.0006163847784648,
      0.0006163847784648,
      0.0013586492859624,
      0.0013586492859624,
      0.0013586492859624,
      0.0031199710152651,
      0.0031199710152651,
      0.0031199710152651,
      0.0043927422264316,
      0.0002245051190981,
      0.0002245051190981,
      0.0002245051190981,
      0.0002245051190981,
      0.0002245051190981,
      0.0002245051190981,
      0.0019454546898025,
      0.0019454546898025,
      0.0019454546898025,
      0.0028440916326783,
      0.0028440916326783,
      0.0028440916326783,
      0.0028440916326783,
      0.0028440916326783,
      0.0028440916326783,
      0.0015144845531387,
      0.0015144845531387,
      0.0015144845531387,
      0.0006025140311277,
      0.0006025140311277,
      0.0006025140311277,
      0.0033739544480612,
      0.0033739544480612,
      0.0033739544480612,
      0.0033739544480612,
      0.0033739544480612,
      0.0033739544480612,
      0.0036124428674794,
      0.0036124428674794,
      0.0036124428674794,
      0.0025681958630702,
      0.0025681958630702,
      0.0025681958630702,
      0.0030422269377476,
      0.0030422269377476,
      0.0030422269377476,
      0.0001392803125243,
      0.0001392803125243,
      0.0001392803125243,
      0.0001392803125243,
      0.0001392803125243,
      0.0001392803125243}
    },
    {
      309,
      {{0.0018100550930815, 0.0018100550930818, 0.0},
      {0.9963798898138365, 0.0018100550930815, 0.0},
      {0.0018100550930819, 0.9963798898138368, 0.0},
      {0.3146077513838778, 0.3146077513838779, 0.0},
      {0.3707844972322443, 0.3146077513838779, 0.0},
      {0.3146077513838778, 0.3707844972322444, 0.0},
      {0.3718324578394357, 0.2563350843211284, 0.0},
      {0.3718324578394359, 0.3718324578394359, 0.0},
      {0.2563350843211283, 0.3718324578394359, 0.0},
      {0.9338478958692606, 0.0267728022087155, 0.0},
      {0.0393793019220240, 0.9338478958692606, 0.0},
      {0.0267728022087153, 0.0393793019220242, 0.0},
      {0.0393793019220239, 0.0267728022087155, 0.0},
      {0.9338478958692605, 0.0393793019220238, 0.0},
      {0.0267728022087157, 0.9338478958692606, 0.0},
      {0.0708749727266859, 0.0214673861123130, 0.0},
      {0.9076576411610010, 0.0708749727266859, 0.0},
      {0.0214673861123131, 0.9076576411610012, 0.0},
      {0.9076576411610010, 0.0214673861123130, 0.0},
      {0.0708749727266859, 0.9076576411610013, 0.0},
      {0.0214673861123128, 0.0708749727266862, 0.0},
      {0.0227676589099524, 0.0119851112427548, 0.0},
      {0.9652472298472927, 0.0227676589099523, 0.0},
      {0.0119851112427549, 0.9652472298472929, 0.0},
      {0.9652472298472928, 0.0119851112427548, 0.0},
      {0.0227676589099525, 0.9652472298472929, 0.0},
      {0.0119851112427545, 0.0227676589099527, 0.0},
      {0.2127026585190979, 0.0964112026242690, 0.0},
      {0.6908861388566330, 0.2127026585190980, 0.0},
      {0.0964112026242691, 0.6908861388566332, 0.0},
      {0.6908861388566331, 0.0964112026242690, 0.0},
      {0.2127026585190979, 0.6908861388566333, 0.0},
      {0.0964112026242688, 0.2127026585190982, 0.0},
      {0.1970799949415202, 0.1279782228464328, 0.0},
      {0.6749417822120469, 0.1970799949415202, 0.0},
      {0.1279782228464328, 0.6749417822120471, 0.0},
      {0.6749417822120470, 0.1279782228464328, 0.0},
      {0.1970799949415203, 0.6749417822120471, 0.0},
      {0.1279782228464326, 0.1970799949415204, 0.0},
      {0.4818489601758344, 0.0363020796483313, 0.0},
      {0.4818489601758343, 0.4818489601758345, 0.0},
      {0.0363020796483312, 0.4818489601758346, 0.0},
      {0.0639130414360007, 0.0423991066111562, 0.0},
      {0.8936878519528431, 0.0639130414360007, 0.0},
      {0.0423991066111563, 0.8936878519528434, 0.0},
      {0.8936878519528432, 0.0423991066111562, 0.0},
      {0.0639130414360007, 0.8936878519528434, 0.0},
      {0.0423991066111560, 0.0639130414360010, 0.0},
      {0.0916254632757577, 0.0116869974011747, 0.0},
      {0.8966875393230677, 0.0916254632757577, 0.0},
      {0.0116869974011748, 0.8966875393230677, 0.0},
      {0.8966875393230677, 0.0116869974011747, 0.0},
      {0.0916254632757578, 0.8966875393230677, 0.0},
      {0.0116869974011745, 0.0916254632757580, 0.0},
      {0.0889175403557380, 0.0632415329385832, 0.0},
      {0.8478409267056788, 0.0889175403557380, 0.0},
      {0.0632415329385833, 0.8478409267056790, 0.0},
      {0.8478409267056788, 0.0632415329385832, 0.0},
      {0.0889175403557381, 0.8478409267056790, 0.0},
      {0.0632415329385829, 0.0889175403557383, 0.0},
      {0.4592305589036990, 0.0148409448407795, 0.0},
      {0.5259284962555215, 0.4592305589036991, 0.0},
      {0.0148409448407795, 0.5259284962555217, 0.0},
      {0.5259284962555214, 0.0148409448407795, 0.0},
      {0.4592305589036990, 0.5259284962555215, 0.0},
      {0.0148409448407795, 0.4592305589036992, 0.0},
      {0.0338843822348439, 0.0018400396156131, 0.0},
      {0.9642755781495430, 0.0338843822348439, 0.0},
      {0.0018400396156133, 0.9642755781495430, 0.0},
      {0.9642755781495431, 0.0018400396156131, 0.0},
      {0.0338843822348439, 0.9642755781495433, 0.0},
      {0.0018400396156128, 0.0338843822348442, 0.0},
      {0.3200752104066154, 0.2580571137398717, 0.0},
      {0.4218676758535129, 0.3200752104066155, 0.0},
      {0.2580571137398716, 0.4218676758535131, 0.0},
      {0.4218676758535129, 0.2580571137398717, 0.0},
      {0.3200752104066154, 0.4218676758535130, 0.0},
      {0.2580571137398715, 0.3200752104066156, 0.0},
      {0.1422819983084207, 0.1422819983084210, 0.0},
      {0.7154360033831583, 0.1422819983084208, 0.0},
      {0.1422819983084210, 0.7154360033831585, 0.0},
      {0.4220684925318889, 0.0362192254340186, 0.0},
      {0.5417122820340925, 0.4220684925318890, 0.0},
      {0.0362192254340186, 0.5417122820340926, 0.0},
      {0.5417122820340925, 0.0362192254340186, 0.0},
      {0.4220684925318889, 0.5417122820340925, 0.0},
      {0.0362192254340186, 0.4220684925318892, 0.0},
      {0.0478764410729284, 0.0105439660443664, 0.0},
      {0.9415795928827051, 0.0478764410729283, 0.0},
      {0.0105439660443666, 0.9415795928827053, 0.0},
      {0.9415795928827051, 0.0105439660443664, 0.0},
      {0.0478764410729285, 0.9415795928827053, 0.0},
      {0.0105439660443662, 0.0478764410729287, 0.0},
      {0.1746336148663416, 0.1746336148663418, 0.0},
      {0.6507327702673166, 0.1746336148663417, 0.0},
      {0.1746336148663419, 0.6507327702673166, 0.0},
      {0.4251980784151399, 0.1496038431697201, 0.0},
      {0.4251980784151399, 0.4251980784151401, 0.0},
      {0.1496038431697200, 0.4251980784151402, 0.0},
      {0.3617990227060762, 0.1510268874995656, 0.0},
      {0.4871740897943582, 0.3617990227060763, 0.0},
      {0.1510268874995655, 0.4871740897943584, 0.0},
      {0.4871740897943583, 0.1510268874995656, 0.0},
      {0.3617990227060763, 0.4871740897943583, 0.0},
      {0.1510268874995655, 0.3617990227060764, 0.0},
      {0.2099773723379177, 0.2099773723379179, 0.0},
      {0.5800452553241643, 0.2099773723379178, 0.0},
      {0.2099773723379179, 0.5800452553241644, 0.0},
      {0.0684569430922210, 0.0024820609017324, 0.0},
      {0.9290609960060465, 0.0684569430922209, 0.0},
      {0.0024820609017325, 0.9290609960060469, 0.0},
      {0.9290609960060466, 0.0024820609017324, 0.0},
      {0.0684569430922210, 0.9290609960060469, 0.0},
      {0.0024820609017321, 0.0684569430922213, 0.0},
      {0.2362884725964560, 0.1549577670219629, 0.0},
      {0.6087537603815811, 0.2362884725964561, 0.0},
      {0.1549577670219628, 0.6087537603815814, 0.0},
      {0.6087537603815811, 0.1549577670219629, 0.0},
      {0.2362884725964560, 0.6087537603815814, 0.0},
      {0.1549577670219627, 0.2362884725964562, 0.0},
      {0.2978954189850496, 0.1519615536650989, 0.0},
      {0.5501430273498515, 0.2978954189850497, 0.0},
      {0.1519615536650988, 0.5501430273498518, 0.0},
      {0.5501430273498515, 0.1519615536650989, 0.0},
      {0.2978954189850496, 0.5501430273498517, 0.0},
      {0.1519615536650987, 0.2978954189850498, 0.0},
      {0.1125572515239218, 0.0023775457303722, 0.0},
      {0.8850652027457061, 0.1125572515239218, 0.0},
      {0.0023775457303723, 0.8850652027457062, 0.0},
      {0.8850652027457061, 0.0023775457303722, 0.0},
      {0.1125572515239218, 0.8850652027457062, 0.0},
      {0.0023775457303719, 0.1125572515239221, 0.0},
      {0.0120658629713076, 0.0030492380858382, 0.0},
      {0.9848848989428542, 0.0120658629713075, 0.0},
      {0.0030492380858384, 0.9848848989428544, 0.0},
      {0.9848848989428541, 0.0030492380858382, 0.0},
      {0.0120658629713077, 0.9848848989428544, 0.0},
      {0.0030492380858379, 0.0120658629713080, 0.0},
      {0.3906530970863169, 0.0150754657505243, 0.0},
      {0.5942714371631589, 0.3906530970863170, 0.0},
      {0.0150754657505243, 0.5942714371631590, 0.0},
      {0.5942714371631588, 0.0150754657505243, 0.0},
      {0.3906530970863170, 0.5942714371631588, 0.0},
      {0.0150754657505242, 0.3906530970863172, 0.0},
      {0.4274107273739702, 0.0028845541075685, 0.0},
      {0.5697047185184613, 0.4274107273739702, 0.0},
      {0.0028845541075684, 0.5697047185184615, 0.0},
      {0.5697047185184614, 0.0028845541075685, 0.0},
      {0.4274107273739702, 0.5697047185184614, 0.0},
      {0.0028845541075684, 0.4274107273739705, 0.0},
      {0.2627399736253427, 0.2627399736253429, 0.0},
      {0.4745200527493144, 0.2627399736253428, 0.0},
      {0.2627399736253428, 0.4745200527493146, 0.0},
      {0.4118455220977197, 0.1047651377258457, 0.0},
      {0.4833893401764346, 0.4118455220977198, 0.0},
      {0.1047651377258456, 0.4833893401764348, 0.0},
      {0.4833893401764346, 0.1047651377258457, 0.0},
      {0.4118455220977197, 0.4833893401764347, 0.0},
      {0.1047651377258456, 0.4118455220977200, 0.0},
      {0.4986036392311143, 0.0027927215377714, 0.0},
      {0.4986036392311143, 0.4986036392311144, 0.0},
      {0.0027927215377713, 0.4986036392311146, 0.0},
      {0.3222211945168437, 0.0148317675755722, 0.0},
      {0.6629470379075842, 0.3222211945168437, 0.0},
      {0.0148317675755723, 0.6629470379075844, 0.0},
      {0.6629470379075841, 0.0148317675755722, 0.0},
      {0.3222211945168438, 0.6629470379075841, 0.0},
      {0.0148317675755720, 0.3222211945168439, 0.0},
      {0.3545249016308813, 0.0367451935646272, 0.0},
      {0.6087299048044915, 0.3545249016308813, 0.0},
      {0.0367451935646271, 0.6087299048044917, 0.0},
      {0.6087299048044915, 0.0367451935646272, 0.0},
      {0.3545249016308814, 0.6087299048044916, 0.0},
      {0.0367451935646270, 0.3545249016308816, 0.0},
      {0.3320276172987797, 0.2028329770562543, 0.0},
      {0.4651394056449660, 0.3320276172987798, 0.0},
      {0.2028329770562542, 0.4651394056449663, 0.0},
      {0.4651394056449660, 0.2028329770562543, 0.0},
      {0.3320276172987797, 0.4651394056449661, 0.0},
      {0.2028329770562542, 0.3320276172987799, 0.0},
      {0.2530297390847723, 0.0655893176896967, 0.0},
      {0.6813809432255311, 0.2530297390847723, 0.0},
      {0.0655893176896967, 0.6813809432255312, 0.0},
      {0.6813809432255310, 0.0655893176896967, 0.0},
      {0.2530297390847723, 0.6813809432255311, 0.0},
      {0.0655893176896965, 0.2530297390847726, 0.0},
      {0.3570399048945097, 0.0028334078818125, 0.0},
      {0.6401266872236777, 0.3570399048945098, 0.0},
      {0.0028334078818126, 0.6401266872236778, 0.0},
      {0.6401266872236777, 0.0028334078818125, 0.0},
      {0.3570399048945097, 0.6401266872236778, 0.0},
      {0.0028334078818124, 0.3570399048945100, 0.0},
      {0.2870606892116554, 0.0360525157666744, 0.0},
      {0.6768867950216703, 0.2870606892116554, 0.0},
      {0.0360525157666745, 0.6768867950216704, 0.0},
      {0.6768867950216702, 0.0360525157666744, 0.0},
      {0.2870606892116554, 0.6768867950216704, 0.0},
      {0.0360525157666742, 0.2870606892116557, 0.0},
      {0.1361313516311361, 0.0141561971271275, 0.0},
      {0.8497124512417364, 0.1361313516311361, 0.0},
      {0.0141561971271276, 0.8497124512417367, 0.0},
      {0.8497124512417363, 0.0141561971271275, 0.0},
      {0.1361313516311361, 0.8497124512417367, 0.0},
      {0.0141561971271272, 0.1361313516311364, 0.0},
      {0.1322241256617750, 0.0637728096721328, 0.0},
      {0.8040030646660921, 0.1322241256617750, 0.0},
      {0.0637728096721329, 0.8040030646660923, 0.0},
      {0.8040030646660922, 0.0637728096721328, 0.0},
      {0.1322241256617751, 0.8040030646660923, 0.0},
      {0.0637728096721326, 0.1322241256617753, 0.0},
      {0.1079242198395523, 0.0335534395458731, 0.0},
      {0.8585223406145746, 0.1079242198395524, 0.0},
      {0.0335534395458731, 0.8585223406145748, 0.0},
      {0.8585223406145744, 0.0335534395458731, 0.0},
      {0.1079242198395524, 0.8585223406145748, 0.0},
      {0.0335534395458729, 0.1079242198395527, 0.0},
      {0.1528869021413866, 0.0995790446035428, 0.0},
      {0.7475340532550706, 0.1528869021413866, 0.0},
      {0.0995790446035428, 0.7475340532550708, 0.0},
      {0.7475340532550706, 0.0995790446035428, 0.0},
      {0.1528869021413866, 0.7475340532550708, 0.0},
      {0.0995790446035426, 0.1528869021413868, 0.0},
      {0.1881017719296066, 0.0646491520250394, 0.0},
      {0.7472490760453540, 0.1881017719296067, 0.0},
      {0.0646491520250393, 0.7472490760453542, 0.0},
      {0.7472490760453540, 0.0646491520250394, 0.0},
      {0.1881017719296067, 0.7472490760453542, 0.0},
      {0.0646491520250392, 0.1881017719296069, 0.0},
      {0.2549824291508879, 0.0148987268708080, 0.0},
      {0.7301188439783041, 0.2549824291508879, 0.0},
      {0.0148987268708080, 0.7301188439783043, 0.0},
      {0.7301188439783041, 0.0148987268708080, 0.0},
      {0.2549824291508879, 0.7301188439783043, 0.0},
      {0.0148987268708078, 0.2549824291508882, 0.0},
      {0.2230780804812777, 0.0028544472780139, 0.0},
      {0.7740674722407084, 0.2230780804812777, 0.0},
      {0.0028544472780139, 0.7740674722407086, 0.0},
      {0.7740674722407084, 0.0028544472780139, 0.0},
      {0.2230780804812778, 0.7740674722407086, 0.0},
      {0.0028544472780137, 0.2230780804812780, 0.0},
      {0.3995665255269703, 0.2008669489460595, 0.0},
      {0.3995665255269703, 0.3995665255269704, 0.0},
      {0.2008669489460594, 0.3995665255269705, 0.0},
      {0.1640507360210325, 0.0028307718211818, 0.0},
      {0.8331184921577857, 0.1640507360210326, 0.0},
      {0.0028307718211819, 0.8331184921577858, 0.0},
      {0.8331184921577858, 0.0028307718211818, 0.0},
      {0.1640507360210327, 0.8331184921577858, 0.0},
      {0.0028307718211816, 0.1640507360210328, 0.0},
      {0.2881511174189886, 0.0028359467965890, 0.0},
      {0.7090129357844225, 0.2881511174189886, 0.0},
      {0.0028359467965890, 0.7090129357844227, 0.0},
      {0.7090129357844225, 0.0028359467965890, 0.0},
      {0.2881511174189885, 0.7090129357844227, 0.0},
      {0.0028359467965888, 0.2881511174189888, 0.0},
      {0.3409577888740615, 0.1056326862781779, 0.0},
      {0.5534095248477606, 0.3409577888740616, 0.0},
      {0.1056326862781778, 0.5534095248477607, 0.0},
      {0.5534095248477606, 0.1056326862781779, 0.0},
      {0.3409577888740616, 0.5534095248477606, 0.0},
      {0.1056326862781778, 0.3409577888740618, 0.0},
      {0.2682934480692839, 0.2054088957827467, 0.0},
      {0.5262976561479694, 0.2682934480692840, 0.0},
      {0.2054088957827467, 0.5262976561479695, 0.0},
      {0.5262976561479694, 0.2054088957827467, 0.0},
      {0.2682934480692839, 0.5262976561479695, 0.0},
      {0.2054088957827466, 0.2682934480692841, 0.0},
      {0.3194090035283247, 0.0674786310679263, 0.0},
      {0.6131123654037490, 0.3194090035283247, 0.0},
      {0.0674786310679263, 0.6131123654037492, 0.0},
      {0.6131123654037490, 0.0674786310679263, 0.0},
      {0.3194090035283247, 0.6131123654037491, 0.0},
      {0.0674786310679262, 0.3194090035283249, 0.0},
      {0.1922954651859523, 0.0149331073629516, 0.0},
      {0.7927714274510961, 0.1922954651859523, 0.0},
      {0.0149331073629516, 0.7927714274510964, 0.0},
      {0.7927714274510962, 0.0149331073629516, 0.0},
      {0.1922954651859524, 0.7927714274510963, 0.0},
      {0.0149331073629514, 0.1922954651859526, 0.0},
      {0.1604140534029236, 0.0353852934054427, 0.0},
      {0.8042006531916337, 0.1604140534029236, 0.0},
      {0.0353852934054428, 0.8042006531916338, 0.0},
      {0.8042006531916338, 0.0353852934054427, 0.0},
      {0.1604140534029238, 0.8042006531916338, 0.0},
      {0.0353852934054425, 0.1604140534029239, 0.0},
      {0.2212534989804217, 0.0362519958338577, 0.0},
      {0.7424945051857205, 0.2212534989804218, 0.0},
      {0.0362519958338579, 0.7424945051857206, 0.0},
      {0.7424945051857206, 0.0362519958338577, 0.0},
      {0.2212534989804218, 0.7424945051857206, 0.0},
      {0.0362519958338575, 0.2212534989804221, 0.0},
      {0.2711942240688135, 0.1065996969730374, 0.0},
      {0.6222060789581492, 0.2711942240688135, 0.0},
      {0.1065996969730373, 0.6222060789581494, 0.0},
      {0.6222060789581491, 0.1065996969730374, 0.0},
      {0.2711942240688137, 0.6222060789581492, 0.0},
      {0.1065996969730372, 0.2711942240688138, 0.0},
      {0.3923234112997704, 0.0665697076536746, 0.0},
      {0.5411068810465550, 0.3923234112997704, 0.0},
      {0.0665697076536745, 0.5411068810465552, 0.0},
      {0.5411068810465550, 0.0665697076536746, 0.0},
      {0.3923234112997703, 0.5411068810465551, 0.0},
      {0.0665697076536745, 0.3923234112997706, 0.0},
      {0.4667610577855150, 0.0664778844289701, 0.0},
      {0.4667610577855150, 0.4667610577855150, 0.0},
      {0.0664778844289700, 0.4667610577855152, 0.0},
      {0.1030954855760305, 0.1030954855760308, 0.0},
      {0.7938090288479388, 0.1030954855760305, 0.0},
      {0.1030954855760309, 0.7938090288479389, 0.0}},
      {0.0000261046615627,
      0.0000261046615627,
      0.0000261046615627,
      0.0032216348354413,
      0.0032216348354413,
      0.0032216348354413,
      0.0028231018862523,
      0.0028231018862523,
      0.0028231018862523,
      0.0004733169982265,
      0.0004733169982265,
      0.0004733169982265,
      0.0004733169982265,
      0.0004733169982265,
      0.0004733169982265,
      0.0005389191646936,
      0.0005389191646936,
      0.0005389191646936,
      0.0005389191646936,
      0.0005389191646936,
      0.0005389191646936,
      0.0002767757031523,
      0.0002767757031523,
      0.0002767757031523,
      0.0002767757031523,
      0.0002767757031523,
      0.0002767757031523,
      0.0018723604576400,
      0.0018723604576400,
      0.0018723604576400,
      0.0018723604576400,
      0.0018723604576400,
      0.0018723604576400,
      0.0020368696048112,
      0.0020368696048112,
      0.0020368696048112,
      0.0020368696048112,
      0.0020368696048112,
      0.0020368696048112,
      0.0014875534123712,
      0.0014875534123712,
      0.0014875534123712,
      0.0008943772853323,
      0.0008943772853323,
      0.0008943772853323,
      0.0008943772853323,
      0.0008943772853323,
      0.0008943772853323,
      0.0005157123690718,
      0.0005157123690718,
      0.0005157123690718,
      0.0005157123690718,
      0.0005157123690718,
      0.0005157123690718,
      0.0012661036341635,
      0.0012661036341635,
      0.0012661036341635,
      0.0012661036341635,
      0.0012661036341635,
      0.0012661036341635,
      0.0011262392789623,
      0.0011262392789623,
      0.0011262392789623,
      0.0011262392789623,
      0.0011262392789623,
      0.0011262392789623,
      0.0001409187542329,
      0.0001409187542329,
      0.0001409187542329,
      0.0001409187542329,
      0.0001409187542329,
      0.0001409187542329,
      0.0030546865348241,
      0.0030546865348241,
      0.0030546865348241,
      0.0030546865348241,
      0.0030546865348241,
      0.0030546865348241,
      0.0021224778885556,
      0.0021224778885556,
      0.0021224778885556,
      0.0016544856076512,
      0.0016544856076512,
      0.0016544856076512,
      0.0016544856076512,
      0.0016544856076512,
      0.0016544856076512,
      0.0003751701870503,
      0.0003751701870503,
      0.0003751701870503,
      0.0003751701870503,
      0.0003751701870503,
      0.0003751701870503,
      0.0025181258544882,
      0.0025181258544882,
      0.0025181258544882,
      0.0030050502385811,
      0.0030050502385811,
      0.0030050502385811,
      0.0031118296166048,
      0.0031118296166048,
      0.0031118296166048,
      0.0031118296166048,
      0.0031118296166048,
      0.0031118296166048,
      0.0030386144984887,
      0.0030386144984887,
      0.0030386144984887,
      0.0002452104355071,
      0.0002452104355071,
      0.0002452104355071,
      0.0002452104355071,
      0.0002452104355071,
      0.0002452104355071,
      0.0029453221328521,
      0.0029453221328521,
      0.0029453221328521,
      0.0029453221328521,
      0.0029453221328521,
      0.0029453221328521,
      0.0031190600595570,
      0.0031190600595570,
      0.0031190600595570,
      0.0031190600595570,
      0.0031190600595570,
      0.0031190600595570,
      0.0002944590940641,
      0.0002944590940641,
      0.0002944590940641,
      0.0002944590940641,
      0.0002944590940641,
      0.0002944590940641,
      0.0001195257809024,
      0.0001195257809024,
      0.0001195257809024,
      0.0001195257809024,
      0.0001195257809024,
      0.0001195257809024,
      0.0011690668799345,
      0.0011690668799345,
      0.0011690668799345,
      0.0011690668799345,
      0.0011690668799345,
      0.0011690668799345,
      0.0005227164853878,
      0.0005227164853878,
      0.0005227164853878,
      0.0005227164853878,
      0.0005227164853878,
      0.0005227164853878,
      0.0033835701042113,
      0.0033835701042113,
      0.0033835701042113,
      0.0029950945601548,
      0.0029950945601548,
      0.0029950945601548,
      0.0029950945601548,
      0.0029950945601548,
      0.0029950945601548,
      0.0005118199851502,
      0.0005118199851502,
      0.0005118199851502,
      0.0011441247936306,
      0.0011441247936306,
      0.0011441247936306,
      0.0011441247936306,
      0.0011441247936306,
      0.0011441247936306,
      0.0018052435282779,
      0.0018052435282779,
      0.0018052435282779,
      0.0018052435282779,
      0.0018052435282779,
      0.0018052435282779,
      0.0035293289042751,
      0.0035293289042751,
      0.0035293289042751,
      0.0035293289042751,
      0.0035293289042751,
      0.0035293289042751,
      0.0021892762868514,
      0.0021892762868514,
      0.0021892762868514,
      0.0021892762868514,
      0.0021892762868514,
      0.0021892762868514,
      0.0005085691794751,
      0.0005085691794751,
      0.0005085691794751,
      0.0005085691794751,
      0.0005085691794751,
      0.0005085691794751,
      0.0017277422623603,
      0.0017277422623603,
      0.0017277422623603,
      0.0017277422623603,
      0.0017277422623603,
      0.0017277422623603,
      0.0008313096207886,
      0.0008313096207886,
      0.0008313096207886,
      0.0008313096207886,
      0.0008313096207886,
      0.0008313096207886,
      0.0016755853266344,
      0.0016755853266344,
      0.0016755853266344,
      0.0016755853266344,
      0.0016755853266344,
      0.0016755853266344,
      0.0011932823252664,
      0.0011932823252664,
      0.0011932823252664,
      0.0011932823252664,
      0.0011932823252664,
      0.0011932823252664,
      0.0020876894965844,
      0.0020876894965844,
      0.0020876894965844,
      0.0020876894965844,
      0.0020876894965844,
      0.0020876894965844,
      0.0019355607442713,
      0.0019355607442713,
      0.0019355607442713,
      0.0019355607442713,
      0.0019355607442713,
      0.0019355607442713,
      0.0010961695980220,
      0.0010961695980220,
      0.0010961695980220,
      0.0010961695980220,
      0.0010961695980220,
      0.0010961695980220,
      0.0004563661682607,
      0.0004563661682607,
      0.0004563661682607,
      0.0004563661682607,
      0.0004563661682607,
      0.0004563661682607,
      0.0036150548664759,
      0.0036150548664759,
      0.0036150548664759,
      0.0004014173263077,
      0.0004014173263077,
      0.0004014173263077,
      0.0004014173263077,
      0.0004014173263077,
      0.0004014173263077,
      0.0004896798007893,
      0.0004896798007893,
      0.0004896798007893,
      0.0004896798007893,
      0.0004896798007893,
      0.0004896798007893,
      0.0029775433804440,
      0.0029775433804440,
      0.0029775433804440,
      0.0029775433804440,
      0.0029775433804440,
      0.0029775433804440,
      0.0033762576693587,
      0.0033762576693587,
      0.0033762576693587,
      0.0033762576693587,
      0.0033762576693587,
      0.0033762576693587,
      0.0024663373041229,
      0.0024663373041229,
      0.0024663373041229,
      0.0024663373041229,
      0.0024663373041229,
      0.0024663373041229,
      0.0009997585618220,
      0.0009997585618220,
      0.0009997585618220,
      0.0009997585618220,
      0.0009997585618220,
      0.0009997585618220,
      0.0014290849464138,
      0.0014290849464138,
      0.0014290849464138,
      0.0014290849464138,
      0.0014290849464138,
      0.0014290849464138,
      0.0016286914747247,
      0.0016286914747247,
      0.0016286914747247,
      0.0016286914747247,
      0.0016286914747247,
      0.0016286914747247,
      0.0029361408291862,
      0.0029361408291862,
      0.0029361408291862,
      0.0029361408291862,
      0.0029361408291862,
      0.0029361408291862,
      0.0025475392237502,
      0.0025475392237502,
      0.0025475392237502,
      0.0025475392237502,
      0.0025475392237502,
      0.0025475392237502,
      0.0025669351626192,
      0.0025669351626192,
      0.0025669351626192,
      0.0019847845196798,
      0.0019847845196798,
      0.0019847845196798}
    },
    {
      324,
      {{0.4023719251097932, 0.1952561497804135, 0.0},
      {0.4023719251097932, 0.4023719251097934, 0.0},
      {0.1952561497804134, 0.4023719251097935, 0.0},
      {0.0013653925759540, 0.0013653925759543, 0.0},
      {0.9972692148480916, 0.0013653925759540, 0.0},
      {0.0013653925759544, 0.9972692148480918, 0.0},
      {0.3305538536843531, 0.0015704256014629, 0.0},
      {0.6678757207141839, 0.3305538536843532, 0.0},
      {0.0015704256014629, 0.6678757207141842, 0.0},
      {0.6678757207141840, 0.0015704256014629, 0.0},
      {0.3305538536843532, 0.6678757207141840, 0.0},
      {0.0015704256014628, 0.3305538536843535, 0.0},
      {0.4975317296252054, 0.0049365407495893, 0.0},
      {0.4975317296252054, 0.4975317296252054, 0.0},
      {0.0049365407495892, 0.4975317296252056, 0.0},
      {0.9699539651466832, 0.0096146691207101, 0.0},
      {0.0204313657326067, 0.9699539651466834, 0.0},
      {0.0096146691207099, 0.0204313657326070, 0.0},
      {0.0204313657326067, 0.0096146691207101, 0.0},
      {0.9699539651466832, 0.0204313657326066, 0.0},
      {0.0096146691207102, 0.9699539651466834, 0.0},
      {0.0571983061939757, 0.0219035586156294, 0.0},
      {0.9208981351903950, 0.0571983061939757, 0.0},
      {0.0219035586156294, 0.9208981351903952, 0.0},
      {0.9208981351903950, 0.0219035586156294, 0.0},
      {0.0571983061939758, 0.9208981351903952, 0.0},
      {0.0219035586156291, 0.0571983061939760, 0.0},
      {0.4799002957544191, 0.0401994084911617, 0.0},
      {0.4799002957544192, 0.4799002957544192, 0.0},
      {0.0401994084911617, 0.4799002957544193, 0.0},
      {0.0664663290557836, 0.0664663290557839, 0.0},
      {0.8670673418884325, 0.0664663290557836, 0.0},
      {0.0664663290557839, 0.8670673418884327, 0.0},
      {0.4532709716379456, 0.0014935512902910, 0.0},
      {0.5452354770717635, 0.4532709716379457, 0.0},
      {0.0014935512902909, 0.5452354770717637, 0.0},
      {0.5452354770717635, 0.0014935512902910, 0.0},
      {0.4532709716379456, 0.5452354770717636, 0.0},
      {0.0014935512902909, 0.4532709716379458, 0.0},
      {0.1407324739067203, 0.0554414996743645, 0.0},
      {0.8038260264189151, 0.1407324739067203, 0.0},
      {0.0554414996743646, 0.8038260264189153, 0.0},
      {0.8038260264189151, 0.0554414996743645, 0.0},
      {0.1407324739067203, 0.8038260264189153, 0.0},
      {0.0554414996743643, 0.1407324739067206, 0.0},
      {0.1837886303019410, 0.1186935694213950, 0.0},
      {0.6975178002766640, 0.1837886303019410, 0.0},
      {0.1186935694213951, 0.6975178002766641, 0.0},
      {0.6975178002766639, 0.1186935694213950, 0.0},
      {0.1837886303019411, 0.6975178002766641, 0.0},
      {0.1186935694213948, 0.1837886303019413, 0.0},
      {0.0432363893877460, 0.0432363893877463, 0.0},
      {0.9135272212245076, 0.0432363893877460, 0.0},
      {0.0432363893877464, 0.9135272212245078, 0.0},
      {0.0321121524868778, 0.0013930572171862, 0.0},
      {0.9664947902959359, 0.0321121524868778, 0.0},
      {0.0013930572171864, 0.9664947902959360, 0.0},
      {0.9664947902959361, 0.0013930572171862, 0.0},
      {0.0321121524868780, 0.9664947902959360, 0.0},
      {0.0013930572171859, 0.0321121524868782, 0.0},
      {0.4903050142016467, 0.0193899715967067, 0.0},
      {0.4903050142016467, 0.4903050142016467, 0.0},
      {0.0193899715967066, 0.4903050142016469, 0.0},
      {0.0429640722006761, 0.0102994751818414, 0.0},
      {0.9467364526174824, 0.0429640722006761, 0.0},
      {0.0102994751818416, 0.9467364526174826, 0.0},
      {0.9467364526174824, 0.0102994751818414, 0.0},
      {0.0429640722006762, 0.9467364526174826, 0.0},
      {0.0102994751818411, 0.0429640722006765, 0.0},
      {0.0105248427300749, 0.0025847099922882, 0.0},
      {0.9868904472776370, 0.0105248427300749, 0.0},
      {0.0025847099922883, 0.9868904472776371, 0.0},
      {0.9868904472776370, 0.0025847099922882, 0.0},
      {0.0105248427300750, 0.9868904472776371, 0.0},
      {0.0025847099922879, 0.0105248427300752, 0.0},
      {0.0753790285629703, 0.0394743127161577, 0.0},
      {0.8851466587208721, 0.0753790285629703, 0.0},
      {0.0394743127161577, 0.8851466587208722, 0.0},
      {0.8851466587208721, 0.0394743127161577, 0.0},
      {0.0753790285629704, 0.8851466587208722, 0.0},
      {0.0394743127161574, 0.0753790285629706, 0.0},
      {0.4363333620899560, 0.0122938385245525, 0.0},
      {0.5513727993854914, 0.4363333620899560, 0.0},
      {0.0122938385245525, 0.5513727993854917, 0.0},
      {0.5513727993854914, 0.0122938385245525, 0.0},
      {0.4363333620899560, 0.5513727993854916, 0.0},
      {0.0122938385245524, 0.4363333620899563, 0.0},
      {0.2704907465149770, 0.0025898092478457, 0.0},
      {0.7269194442371774, 0.2704907465149770, 0.0},
      {0.0025898092478457, 0.7269194442371776, 0.0},
      {0.7269194442371772, 0.0025898092478457, 0.0},
      {0.2704907465149770, 0.7269194442371776, 0.0},
      {0.0025898092478455, 0.2704907465149773, 0.0},
      {0.3508136433810974, 0.0375563888722969, 0.0},
      {0.6116299677466058, 0.3508136433810974, 0.0},
      {0.0375563888722968, 0.6116299677466061, 0.0},
      {0.6116299677466057, 0.0375563888722969, 0.0},
      {0.3508136433810974, 0.6116299677466058, 0.0},
      {0.0375563888722968, 0.3508136433810976, 0.0},
      {0.3727655119742921, 0.0170945467207073, 0.0},
      {0.6101399413050006, 0.3727655119742921, 0.0},
      {0.0170945467207073, 0.6101399413050007, 0.0},
      {0.6101399413050006, 0.0170945467207073, 0.0},
      {0.3727655119742921, 0.6101399413050007, 0.0},
      {0.0170945467207072, 0.3727655119742924, 0.0},
      {0.2171073522633415, 0.0857583162943159, 0.0},
      {0.6971343314423426, 0.2171073522633416, 0.0},
      {0.0857583162943159, 0.6971343314423428, 0.0},
      {0.6971343314423426, 0.0857583162943159, 0.0},
      {0.2171073522633415, 0.6971343314423428, 0.0},
      {0.0857583162943157, 0.2171073522633418, 0.0},
      {0.1927563650090058, 0.0579856449641144, 0.0},
      {0.7492579900268798, 0.1927563650090059, 0.0},
      {0.0579856449641145, 0.7492579900268799, 0.0},
      {0.7492579900268798, 0.0579856449641144, 0.0},
      {0.1927563650090059, 0.7492579900268799, 0.0},
      {0.0579856449641142, 0.1927563650090061, 0.0},
      {0.4206850306864163, 0.0338481728829903, 0.0},
      {0.5454667964305935, 0.4206850306864163, 0.0},
      {0.0338481728829903, 0.5454667964305936, 0.0},
      {0.5454667964305935, 0.0338481728829903, 0.0},
      {0.4206850306864162, 0.5454667964305936, 0.0},
      {0.0338481728829902, 0.4206850306864165, 0.0},
      {0.0269988226283730, 0.0269988226283733, 0.0},
      {0.9460023547432537, 0.0269988226283730, 0.0},
      {0.0269988226283734, 0.9460023547432538, 0.0},
      {0.2124851854370911, 0.0025266913852086, 0.0},
      {0.7849881231777003, 0.2124851854370911, 0.0},
      {0.0025266913852087, 0.7849881231777005, 0.0},
      {0.7849881231777003, 0.0025266913852086, 0.0},
      {0.2124851854370911, 0.7849881231777005, 0.0},
      {0.0025266913852085, 0.2124851854370913, 0.0},
      {0.3192572087291873, 0.0106565421155276, 0.0},
      {0.6700862491552849, 0.3192572087291874, 0.0},
      {0.0106565421155276, 0.6700862491552853, 0.0},
      {0.6700862491552850, 0.0106565421155276, 0.0},
      {0.3192572087291874, 0.6700862491552851, 0.0},
      {0.0106565421155275, 0.3192572087291877, 0.0},
      {0.1385048069103228, 0.1385048069103230, 0.0},
      {0.7229903861793542, 0.1385048069103228, 0.0},
      {0.1385048069103231, 0.7229903861793542, 0.0},
      {0.4670132073557640, 0.0659735852884721, 0.0},
      {0.4670132073557640, 0.4670132073557641, 0.0},
      {0.0659735852884720, 0.4670132073557642, 0.0},
      {0.2516382355125726, 0.0134296732422585, 0.0},
      {0.7349320912451689, 0.2516382355125726, 0.0},
      {0.0134296732422586, 0.7349320912451690, 0.0},
      {0.7349320912451689, 0.0134296732422585, 0.0},
      {0.2516382355125727, 0.7349320912451689, 0.0},
      {0.0134296732422583, 0.2516382355125729, 0.0},
      {0.1522119814568094, 0.0883417782932651, 0.0},
      {0.7594462402499256, 0.1522119814568094, 0.0},
      {0.0883417782932651, 0.7594462402499257, 0.0},
      {0.7594462402499256, 0.0883417782932651, 0.0},
      {0.1522119814568094, 0.7594462402499258, 0.0},
      {0.0883417782932648, 0.1522119814568096, 0.0},
      {0.1111420947940333, 0.0329221182111608, 0.0},
      {0.8559357869948059, 0.1111420947940333, 0.0},
      {0.0329221182111609, 0.8559357869948060, 0.0},
      {0.8559357869948060, 0.0329221182111608, 0.0},
      {0.1111420947940334, 0.8559357869948060, 0.0},
      {0.0329221182111606, 0.1111420947940336, 0.0},
      {0.3581500588512503, 0.1948007607773277, 0.0},
      {0.4470491803714221, 0.3581500588512503, 0.0},
      {0.1948007607773276, 0.4470491803714222, 0.0},
      {0.4470491803714221, 0.1948007607773277, 0.0},
      {0.3581500588512503, 0.4470491803714221, 0.0},
      {0.1948007607773276, 0.3581500588512505, 0.0},
      {0.1637998569561501, 0.0314610159618918, 0.0},
      {0.8047391270819582, 0.1637998569561501, 0.0},
      {0.0314610159618919, 0.8047391270819583, 0.0},
      {0.8047391270819582, 0.0314610159618918, 0.0},
      {0.1637998569561501, 0.8047391270819583, 0.0},
      {0.0314610159618916, 0.1637998569561504, 0.0},
      {0.1021127733366743, 0.0673683101645259, 0.0},
      {0.8305189164987998, 0.1021127733366743, 0.0},
      {0.0673683101645260, 0.8305189164987999, 0.0},
      {0.8305189164987997, 0.0673683101645259, 0.0},
      {0.1021127733366744, 0.8305189164987999, 0.0},
      {0.0673683101645256, 0.1021127733366746, 0.0},
      {0.2335557030193516, 0.1865652441999728, 0.0},
      {0.5798790527806756, 0.2335557030193517, 0.0},
      {0.1865652441999727, 0.5798790527806759, 0.0},
      {0.5798790527806756, 0.1865652441999728, 0.0},
      {0.2335557030193516, 0.5798790527806759, 0.0},
      {0.1865652441999726, 0.2335557030193518, 0.0},
      {0.3214164628317113, 0.0638834834123953, 0.0},
      {0.6147000537558933, 0.3214164628317114, 0.0},
      {0.0638834834123954, 0.6147000537558934, 0.0},
      {0.6147000537558933, 0.0638834834123953, 0.0},
      {0.3214164628317114, 0.6147000537558934, 0.0},
      {0.0638834834123952, 0.3214164628317117, 0.0},
      {0.2233361848957126, 0.0325712217949499, 0.0},
      {0.7440925933093376, 0.2233361848957126, 0.0},
      {0.0325712217949499, 0.7440925933093379, 0.0},
      {0.7440925933093376, 0.0325712217949499, 0.0},
      {0.2233361848957126, 0.7440925933093377, 0.0},
      {0.0325712217949497, 0.2233361848957128, 0.0},
      {0.1905881189463203, 0.0132734827561578, 0.0},
      {0.7961383982975219, 0.1905881189463202, 0.0},
      {0.0132734827561579, 0.7961383982975221, 0.0},
      {0.7961383982975220, 0.0132734827561578, 0.0},
      {0.1905881189463203, 0.7961383982975221, 0.0},
      {0.0132734827561576, 0.1905881189463206, 0.0},
      {0.2273558402454795, 0.1362337755091281, 0.0},
      {0.6364103842453923, 0.2273558402454796, 0.0},
      {0.1362337755091281, 0.6364103842453925, 0.0},
      {0.6364103842453924, 0.1362337755091281, 0.0},
      {0.2273558402454797, 0.6364103842453924, 0.0},
      {0.1362337755091279, 0.2273558402454798, 0.0},
      {0.1065804258588588, 0.1065804258588591, 0.0},
      {0.7868391482822822, 0.1065804258588588, 0.0},
      {0.1065804258588591, 0.7868391482822823, 0.0},
      {0.1567679227871600, 0.0026153775338892, 0.0},
      {0.8406166996789507, 0.1567679227871601, 0.0},
      {0.0026153775338892, 0.8406166996789509, 0.0},
      {0.8406166996789507, 0.0026153775338892, 0.0},
      {0.1567679227871602, 0.8406166996789507, 0.0},
      {0.0026153775338890, 0.1567679227871604, 0.0},
      {0.1340333349867785, 0.0134812112051568, 0.0},
      {0.8524854538080647, 0.1340333349867785, 0.0},
      {0.0134812112051569, 0.8524854538080648, 0.0},
      {0.8524854538080647, 0.0134812112051568, 0.0},
      {0.1340333349867785, 0.8524854538080648, 0.0},
      {0.0134812112051566, 0.1340333349867788, 0.0},
      {0.2957211757454536, 0.1918841043522063, 0.0},
      {0.5123947199023402, 0.2957211757454537, 0.0},
      {0.1918841043522062, 0.5123947199023403, 0.0},
      {0.5123947199023402, 0.1918841043522063, 0.0},
      {0.2957211757454536, 0.5123947199023402, 0.0},
      {0.1918841043522061, 0.2957211757454539, 0.0},
      {0.1067192718130004, 0.0025472816321698, 0.0},
      {0.8907334465548298, 0.1067192718130003, 0.0},
      {0.0025472816321699, 0.8907334465548301, 0.0},
      {0.8907334465548298, 0.0025472816321698, 0.0},
      {0.1067192718130003, 0.8907334465548301, 0.0},
      {0.0025472816321696, 0.1067192718130007, 0.0},
      {0.4125664194839165, 0.1010372943557597, 0.0},
      {0.4863962861603238, 0.4125664194839166, 0.0},
      {0.1010372943557596, 0.4863962861603239, 0.0},
      {0.4863962861603238, 0.1010372943557597, 0.0},
      {0.4125664194839165, 0.4863962861603238, 0.0},
      {0.1010372943557596, 0.4125664194839167, 0.0},
      {0.3078519829928861, 0.2497507411652888, 0.0},
      {0.4423972758418252, 0.3078519829928861, 0.0},
      {0.2497507411652887, 0.4423972758418253, 0.0},
      {0.4423972758418251, 0.2497507411652888, 0.0},
      {0.3078519829928861, 0.4423972758418253, 0.0},
      {0.2497507411652886, 0.3078519829928862, 0.0},
      {0.0870205486996966, 0.0137126252885580, 0.0},
      {0.8992668260117453, 0.0870205486996967, 0.0},
      {0.0137126252885581, 0.8992668260117456, 0.0},
      {0.8992668260117453, 0.0137126252885580, 0.0},
      {0.0870205486996967, 0.8992668260117456, 0.0},
      {0.0137126252885578, 0.0870205486996970, 0.0},
      {0.3894283316812197, 0.0033445954334026, 0.0},
      {0.6072270728853777, 0.3894283316812198, 0.0},
      {0.0033445954334026, 0.6072270728853779, 0.0},
      {0.6072270728853777, 0.0033445954334026, 0.0},
      {0.3894283316812197, 0.6072270728853778, 0.0},
      {0.0033445954334026, 0.3894283316812200, 0.0},
      {0.3117353910725873, 0.3117353910725875, 0.0},
      {0.3765292178548252, 0.3117353910725874, 0.0},
      {0.3117353910725873, 0.3765292178548253, 0.0},
      {0.2612939703141770, 0.0556746304306052, 0.0},
      {0.6830313992552178, 0.2612939703141770, 0.0},
      {0.0556746304306052, 0.6830313992552181, 0.0},
      {0.6830313992552177, 0.0556746304306052, 0.0},
      {0.2612939703141770, 0.6830313992552179, 0.0},
      {0.0556746304306051, 0.2612939703141772, 0.0},
      {0.0649755331098119, 0.0028037585935403, 0.0},
      {0.9322207082966478, 0.0649755331098119, 0.0},
      {0.0028037585935404, 0.9322207082966479, 0.0},
      {0.9322207082966478, 0.0028037585935403, 0.0},
      {0.0649755331098119, 0.9322207082966479, 0.0},
      {0.0028037585935401, 0.0649755331098122, 0.0},
      {0.2719500060981310, 0.0963505000772398, 0.0},
      {0.6316994938246291, 0.2719500060981311, 0.0},
      {0.0963505000772399, 0.6316994938246292, 0.0},
      {0.6316994938246292, 0.0963505000772398, 0.0},
      {0.2719500060981311, 0.6316994938246292, 0.0},
      {0.0963505000772396, 0.2719500060981313, 0.0},
      {0.2930903642725462, 0.0291991066004360, 0.0},
      {0.6777105291270178, 0.2930903642725463, 0.0},
      {0.0291991066004360, 0.6777105291270180, 0.0},
      {0.6777105291270178, 0.0291991066004360, 0.0},
      {0.2930903642725463, 0.6777105291270179, 0.0},
      {0.0291991066004358, 0.2930903642725465, 0.0},
      {0.3946648265878192, 0.0635745506621692, 0.0},
      {0.5417606227500117, 0.3946648265878193, 0.0},
      {0.0635745506621692, 0.5417606227500118, 0.0},
      {0.5417606227500116, 0.0635745506621692, 0.0},
      {0.3946648265878193, 0.5417606227500117, 0.0},
      {0.0635745506621691, 0.3946648265878194, 0.0},
      {0.3568522181929828, 0.1439285800655438, 0.0},
      {0.4992192017414733, 0.3568522181929829, 0.0},
      {0.1439285800655438, 0.4992192017414736, 0.0},
      {0.4992192017414734, 0.1439285800655438, 0.0},
      {0.3568522181929829, 0.4992192017414734, 0.0},
      {0.1439285800655437, 0.3568522181929830, 0.0},
      {0.2448605758688817, 0.2448605758688819, 0.0},
      {0.5102788482622364, 0.2448605758688818, 0.0},
      {0.2448605758688818, 0.5102788482622366, 0.0},
      {0.3413613516388763, 0.0992086997532998, 0.0},
      {0.5594299486078240, 0.3413613516388764, 0.0},
      {0.0992086997532998, 0.5594299486078241, 0.0},
      {0.5594299486078240, 0.0992086997532998, 0.0},
      {0.3413613516388763, 0.5594299486078240, 0.0},
      {0.0992086997532997, 0.3413613516388765, 0.0},
      {0.2895129410795146, 0.1403889214214629, 0.0},
      {0.5700981374990226, 0.2895129410795146, 0.0},
      {0.1403889214214628, 0.5700981374990227, 0.0},
      {0.5700981374990226, 0.1403889214214629, 0.0},
      {0.2895129410795146, 0.5700981374990226, 0.0},
      {0.1403889214214627, 0.2895129410795148, 0.0},
      {0.1748351458111906, 0.1748351458111909, 0.0},
      {0.6503297083776184, 0.1748351458111907, 0.0},
      {0.1748351458111909, 0.6503297083776187, 0.0},
      {0.4275693341501904, 0.1448613316996193, 0.0},
      {0.4275693341501904, 0.4275693341501904, 0.0},
      {0.1448613316996191, 0.4275693341501906, 0.0},
      {0.3742701566784477, 0.2514596866431047, 0.0},
      {0.3742701566784477, 0.3742701566784478, 0.0},
      {0.2514596866431046, 0.3742701566784478, 0.0}},
      {0.0017732200079886,
      0.0017732200079886,
      0.0017732200079886,
      0.0000173597677183,
      0.0000173597677183,
      0.0000173597677183,
      0.0002663658471702,
      0.0002663658471702,
      0.0002663658471702,
      0.0002663658471702,
      0.0002663658471702,
      0.0002663658471702,
      0.0005070848214163,
      0.0005070848214163,
      0.0005070848214163,
      0.0002195385135849,
      0.0002195385135849,
      0.0002195385135849,
      0.0002195385135849,
      0.0002195385135849,
      0.0002195385135849,
      0.0005185125232677,
      0.0005185125232677,
      0.0005185125232677,
      0.0005185125232677,
      0.0005185125232677,
      0.0005185125232677,
      0.0014220929785061,
      0.0014220929785061,
      0.0014220929785061,
      0.0009354907690114,
      0.0009354907690114,
      0.0009354907690114,
      0.0002797759751140,
      0.0002797759751140,
      0.0002797759751140,
      0.0002797759751140,
      0.0002797759751140,
      0.0002797759751140,
      0.0012689602532324,
      0.0012689602532324,
      0.0012689602532324,
      0.0012689602532324,
      0.0012689602532324,
      0.0012689602532324,
      0.0020290788990570,
      0.0020290788990570,
      0.0020290788990570,
      0.0020290788990570,
      0.0020290788990570,
      0.0020290788990570,
      0.0006663125002040,
      0.0006663125002040,
      0.0006663125002040,
      0.0001133212876038,
      0.0001133212876038,
      0.0001133212876038,
      0.0001133212876038,
      0.0001133212876038,
      0.0001133212876038,
      0.0010963408407929,
      0.0010963408407929,
      0.0010963408407929,
      0.0003409266715707,
      0.0003409266715707,
      0.0003409266715707,
      0.0003409266715707,
      0.0003409266715707,
      0.0003409266715707,
      0.0000935465343078,
      0.0000935465343078,
      0.0000935465343078,
      0.0000935465343078,
      0.0000935465343078,
      0.0000935465343078,
      0.0008433148349119,
      0.0008433148349119,
      0.0008433148349119,
      0.0008433148349119,
      0.0008433148349119,
      0.0008433148349119,
      0.0009512748982289,
      0.0009512748982289,
      0.0009512748982289,
      0.0009512748982289,
      0.0009512748982289,
      0.0009512748982289,
      0.0003866835126499,
      0.0003866835126499,
      0.0003866835126499,
      0.0003866835126499,
      0.0003866835126499,
      0.0003866835126499,
      0.0015861939454501,
      0.0015861939454501,
      0.0015861939454501,
      0.0015861939454501,
      0.0015861939454501,
      0.0015861939454501,
      0.0010842623549309,
      0.0010842623549309,
      0.0010842623549309,
      0.0010842623549309,
      0.0010842623549309,
      0.0010842623549309,
      0.0020370587368203,
      0.0020370587368203,
      0.0020370587368203,
      0.0020370587368203,
      0.0020370587368203,
      0.0020370587368203,
      0.0016860445071981,
      0.0016860445071981,
      0.0016860445071981,
      0.0016860445071981,
      0.0016860445071981,
      0.0016860445071981,
      0.0016687327076747,
      0.0016687327076747,
      0.0016687327076747,
      0.0016687327076747,
      0.0016687327076747,
      0.0016687327076747,
      0.0004762029778274,
      0.0004762029778274,
      0.0004762029778274,
      0.0003721502303561,
      0.0003721502303561,
      0.0003721502303561,
      0.0003721502303561,
      0.0003721502303561,
      0.0003721502303561,
      0.0008745028287546,
      0.0008745028287546,
      0.0008745028287546,
      0.0008745028287546,
      0.0008745028287546,
      0.0008745028287546,
      0.0021799250209463,
      0.0021799250209463,
      0.0021799250209463,
      0.0022662661275164,
      0.0022662661275164,
      0.0022662661275164,
      0.0009515841977190,
      0.0009515841977190,
      0.0009515841977190,
      0.0009515841977190,
      0.0009515841977190,
      0.0009515841977190,
      0.0019790722453783,
      0.0019790722453783,
      0.0019790722453783,
      0.0019790722453783,
      0.0019790722453783,
      0.0019790722453783,
      0.0010816650059588,
      0.0010816650059588,
      0.0010816650059588,
      0.0010816650059588,
      0.0010816650059588,
      0.0010816650059588,
      0.0031333706982709,
      0.0031333706982709,
      0.0031333706982709,
      0.0031333706982709,
      0.0031333706982709,
      0.0031333706982709,
      0.0012157354723764,
      0.0012157354723764,
      0.0012157354723764,
      0.0012157354723764,
      0.0012157354723764,
      0.0012157354723764,
      0.0014132890488013,
      0.0014132890488013,
      0.0014132890488013,
      0.0014132890488013,
      0.0014132890488013,
      0.0014132890488013,
      0.0032376781127417,
      0.0032376781127417,
      0.0032376781127417,
      0.0032376781127417,
      0.0032376781127417,
      0.0032376781127417,
      0.0021242412256271,
      0.0021242412256271,
      0.0021242412256271,
      0.0021242412256271,
      0.0021242412256271,
      0.0021242412256271,
      0.0014550429545312,
      0.0014550429545312,
      0.0014550429545312,
      0.0014550429545312,
      0.0014550429545312,
      0.0014550429545312,
      0.0008771971654657,
      0.0008771971654657,
      0.0008771971654657,
      0.0008771971654657,
      0.0008771971654657,
      0.0008771971654657,
      0.0026639071817663,
      0.0026639071817663,
      0.0026639071817663,
      0.0026639071817663,
      0.0026639071817663,
      0.0026639071817663,
      0.0018480353307769,
      0.0018480353307769,
      0.0018480353307769,
      0.0003558677575267,
      0.0003558677575267,
      0.0003558677575267,
      0.0003558677575267,
      0.0003558677575267,
      0.0003558677575267,
      0.0007842598090717,
      0.0007842598090717,
      0.0007842598090717,
      0.0007842598090717,
      0.0007842598090717,
      0.0007842598090717,
      0.0034548418516927,
      0.0034548418516927,
      0.0034548418516927,
      0.0034548418516927,
      0.0034548418516927,
      0.0034548418516927,
      0.0003038834446692,
      0.0003038834446692,
      0.0003038834446692,
      0.0003038834446692,
      0.0003038834446692,
      0.0003038834446692,
      0.0029299530252759,
      0.0029299530252759,
      0.0029299530252759,
      0.0029299530252759,
      0.0029299530252759,
      0.0029299530252759,
      0.0038825930203148,
      0.0038825930203148,
      0.0038825930203148,
      0.0038825930203148,
      0.0038825930203148,
      0.0038825930203148,
      0.0006550204164930,
      0.0006550204164930,
      0.0006550204164930,
      0.0006550204164930,
      0.0006550204164930,
      0.0006550204164930,
      0.0005285486245862,
      0.0005285486245862,
      0.0005285486245862,
      0.0005285486245862,
      0.0005285486245862,
      0.0005285486245862,
      0.0040486688045142,
      0.0040486688045142,
      0.0040486688045142,
      0.0020205988253753,
      0.0020205988253753,
      0.0020205988253753,
      0.0020205988253753,
      0.0020205988253753,
      0.0020205988253753,
      0.0002621239456730,
      0.0002621239456730,
      0.0002621239456730,
      0.0002621239456730,
      0.0002621239456730,
      0.0002621239456730,
      0.0026024340031601,
      0.0026024340031601,
      0.0026024340031601,
      0.0026024340031601,
      0.0026024340031601,
      0.0026024340031601,
      0.0015166556631349,
      0.0015166556631349,
      0.0015166556631349,
      0.0015166556631349,
      0.0015166556631349,
      0.0015166556631349,
      0.0024342761801989,
      0.0024342761801989,
      0.0024342761801989,
      0.0024342761801989,
      0.0024342761801989,
      0.0024342761801989,
      0.0033014363090445,
      0.0033014363090445,
      0.0033014363090445,
      0.0033014363090445,
      0.0033014363090445,
      0.0033014363090445,
      0.0036453005263934,
      0.0036453005263934,
      0.0036453005263934,
      0.0028265683441562,
      0.0028265683441562,
      0.0028265683441562,
      0.0028265683441562,
      0.0028265683441562,
      0.0028265683441562,
      0.0030959645061719,
      0.0030959645061719,
      0.0030959645061719,
      0.0030959645061719,
      0.0030959645061719,
      0.0030959645061719,
      0.0030277633910127,
      0.0030277633910127,
      0.0030277633910127,
      0.0033794723196149,
      0.0033794723196149,
      0.0033794723196149,
      0.0039610222882957,
      0.0039610222882957,
      0.0039610222882957}
    },
    {
      339,
      {{0.4117801708599338, 0.1764396582801324, 0.0},
      {0.4117801708599338, 0.4117801708599339, 0.0},
      {0.1764396582801323, 0.4117801708599340, 0.0},
      {0.4755758670991315, 0.0488482658017371, 0.0},
      {0.4755758670991315, 0.4755758670991315, 0.0},
      {0.0488482658017370, 0.4755758670991317, 0.0},
      {0.2297394365272492, 0.1454537953878326, 0.0},
      {0.6248067680849181, 0.2297394365272493, 0.0},
      {0.1454537953878327, 0.6248067680849183, 0.0},
      {0.6248067680849182, 0.1454537953878326, 0.0},
      {0.2297394365272493, 0.6248067680849183, 0.0},
      {0.1454537953878325, 0.2297394365272494, 0.0},
      {0.4991798367344780, 0.0016403265310440, 0.0},
      {0.4991798367344780, 0.4991798367344781, 0.0},
      {0.0016403265310439, 0.4991798367344782, 0.0},
      {0.2962479881688698, 0.2962479881688700, 0.0},
      {0.4075040236622601, 0.2962479881688700, 0.0},
      {0.2962479881688698, 0.4075040236622604, 0.0},
      {0.3605580760887838, 0.0324154748417785, 0.0},
      {0.6070264490694377, 0.3605580760887839, 0.0},
      {0.0324154748417785, 0.6070264490694378, 0.0},
      {0.6070264490694376, 0.0324154748417785, 0.0},
      {0.3605580760887839, 0.6070264490694377, 0.0},
      {0.0324154748417784, 0.3605580760887841, 0.0},
      {0.0959172276669437, 0.0124129082244384, 0.0},
      {0.8916698641086179, 0.0959172276669437, 0.0},
      {0.0124129082244385, 0.8916698641086181, 0.0},
      {0.8916698641086179, 0.0124129082244384, 0.0},
      {0.0959172276669437, 0.8916698641086181, 0.0},
      {0.0124129082244381, 0.0959172276669440, 0.0},
      {0.1898015101110885, 0.1378581178914931, 0.0},
      {0.6723403719974184, 0.1898015101110885, 0.0},
      {0.1378581178914931, 0.6723403719974186, 0.0},
      {0.6723403719974185, 0.1378581178914931, 0.0},
      {0.1898015101110885, 0.6723403719974186, 0.0},
      {0.1378581178914929, 0.1898015101110887, 0.0},
      {0.4010584512926051, 0.1353361311535962, 0.0},
      {0.4636054175537986, 0.4010584512926053, 0.0},
      {0.1353361311535962, 0.4636054175537988, 0.0},
      {0.4636054175537986, 0.1353361311535962, 0.0},
      {0.4010584512926052, 0.4636054175537987, 0.0},
      {0.1353361311535961, 0.4010584512926054, 0.0},
      {0.2775825322761021, 0.1435267918924767, 0.0},
      {0.5788906758314212, 0.2775825322761022, 0.0},
      {0.1435267918924767, 0.5788906758314213, 0.0},
      {0.5788906758314212, 0.1435267918924767, 0.0},
      {0.2775825322761021, 0.5788906758314213, 0.0},
      {0.1435267918924765, 0.2775825322761024, 0.0},
      {0.3531888048870007, 0.2936223902259986, 0.0},
      {0.3531888048870007, 0.3531888048870008, 0.0},
      {0.2936223902259985, 0.3531888048870009, 0.0},
      {0.3061183766903350, 0.0354515915965142, 0.0},
      {0.6584300317131508, 0.3061183766903350, 0.0},
      {0.0354515915965141, 0.6584300317131511, 0.0},
      {0.6584300317131508, 0.0354515915965142, 0.0},
      {0.3061183766903349, 0.6584300317131511, 0.0},
      {0.0354515915965140, 0.3061183766903353, 0.0},
      {0.4951059808027699, 0.0097880383944602, 0.0},
      {0.4951059808027699, 0.4951059808027700, 0.0},
      {0.0097880383944601, 0.4951059808027701, 0.0},
      {0.4182891701734248, 0.0354185331313222, 0.0},
      {0.5462922966952530, 0.4182891701734248, 0.0},
      {0.0354185331313221, 0.5462922966952534, 0.0},
      {0.5462922966952531, 0.0354185331313222, 0.0},
      {0.4182891701734248, 0.5462922966952531, 0.0},
      {0.0354185331313221, 0.4182891701734250, 0.0},
      {0.0329098159605346, 0.0329098159605348, 0.0},
      {0.9341803680789306, 0.0329098159605345, 0.0},
      {0.0329098159605350, 0.9341803680789307, 0.0},
      {0.0965099243357524, 0.0557938681257981, 0.0},
      {0.8476962075384494, 0.0965099243357524, 0.0},
      {0.0557938681257983, 0.8476962075384495, 0.0},
      {0.8476962075384494, 0.0557938681257981, 0.0},
      {0.0965099243357524, 0.8476962075384497, 0.0},
      {0.0557938681257979, 0.0965099243357527, 0.0},
      {0.1328397411105292, 0.0639762347898086, 0.0},
      {0.8031840240996622, 0.1328397411105292, 0.0},
      {0.0639762347898087, 0.8031840240996623, 0.0},
      {0.8031840240996622, 0.0639762347898086, 0.0},
      {0.1328397411105293, 0.8031840240996623, 0.0},
      {0.0639762347898084, 0.1328397411105295, 0.0},
      {0.0969643297980663, 0.0302625797417711, 0.0},
      {0.8727730904601626, 0.0969643297980663, 0.0},
      {0.0302625797417711, 0.8727730904601628, 0.0},
      {0.8727730904601626, 0.0302625797417711, 0.0},
      {0.0969643297980663, 0.8727730904601628, 0.0},
      {0.0302625797417709, 0.0969643297980667, 0.0},
      {0.4865637511306883, 0.0268724977386233, 0.0},
      {0.4865637511306884, 0.4865637511306884, 0.0},
      {0.0268724977386232, 0.4865637511306886, 0.0},
      {0.4287537536910129, 0.0146376163589685, 0.0},
      {0.5566086299500186, 0.4287537536910130, 0.0},
      {0.0146376163589685, 0.5566086299500187, 0.0},
      {0.5566086299500185, 0.0146376163589685, 0.0},
      {0.4287537536910130, 0.5566086299500186, 0.0},
      {0.0146376163589684, 0.4287537536910132, 0.0},
      {0.0627150483897617, 0.0627150483897619, 0.0},
      {0.8745699032204765, 0.0627150483897616, 0.0},
      {0.0627150483897621, 0.8745699032204766, 0.0},
      {0.1368534453260830, 0.0136052509293985, 0.0},
      {0.8495413037445184, 0.1368534453260830, 0.0},
      {0.0136052509293986, 0.8495413037445185, 0.0},
      {0.8495413037445184, 0.0136052509293985, 0.0},
      {0.1368534453260831, 0.8495413037445186, 0.0},
      {0.0136052509293983, 0.1368534453260834, 0.0},
      {0.3635619090992528, 0.0024760406221923, 0.0},
      {0.6339620502785548, 0.3635619090992529, 0.0},
      {0.0024760406221923, 0.6339620502785551, 0.0},
      {0.6339620502785549, 0.0024760406221923, 0.0},
      {0.3635619090992528, 0.6339620502785550, 0.0},
      {0.0024760406221921, 0.3635619090992531, 0.0},
      {0.1447435597702214, 0.1447435597702217, 0.0},
      {0.7105128804595570, 0.1447435597702214, 0.0},
      {0.1447435597702216, 0.7105128804595572, 0.0},
      {0.3361435723276197, 0.1429259733779294, 0.0},
      {0.5209304542944508, 0.3361435723276198, 0.0},
      {0.1429259733779294, 0.5209304542944510, 0.0},
      {0.5209304542944510, 0.1429259733779294, 0.0},
      {0.3361435723276198, 0.5209304542944510, 0.0},
      {0.1429259733779293, 0.3361435723276199, 0.0},
      {0.0133646206082616, 0.0133646206082619, 0.0},
      {0.9732707587834765, 0.0133646206082616, 0.0},
      {0.0133646206082620, 0.9732707587834766, 0.0},
      {0.0940988534499003, 0.0024207777457973, 0.0},
      {0.9034803688043024, 0.0940988534499003, 0.0},
      {0.0024207777457974, 0.9034803688043025, 0.0},
      {0.9034803688043024, 0.0024207777457973, 0.0},
      {0.0940988534499004, 0.9034803688043025, 0.0},
      {0.0024207777457970, 0.0940988534499007, 0.0},
      {0.2427105866699005, 0.1951503399895134, 0.0},
      {0.5621390733405861, 0.2427105866699006, 0.0},
      {0.1951503399895134, 0.5621390733405862, 0.0},
      {0.5621390733405861, 0.1951503399895134, 0.0},
      {0.2427105866699005, 0.5621390733405862, 0.0},
      {0.1951503399895132, 0.2427105866699007, 0.0},
      {0.4252384428310576, 0.0640551018216570, 0.0},
      {0.5107064553472853, 0.4252384428310577, 0.0},
      {0.0640551018216570, 0.5107064553472855, 0.0},
      {0.5107064553472853, 0.0640551018216570, 0.0},
      {0.4252384428310577, 0.5107064553472854, 0.0},
      {0.0640551018216569, 0.4252384428310579, 0.0},
      {0.3642781269741873, 0.0131876112324832, 0.0},
      {0.6225342617933295, 0.3642781269741873, 0.0},
      {0.0131876112324832, 0.6225342617933296, 0.0},
      {0.6225342617933295, 0.0131876112324832, 0.0},
      {0.3642781269741873, 0.6225342617933296, 0.0},
      {0.0131876112324831, 0.3642781269741875, 0.0},
      {0.1776637264896365, 0.0618730821070127, 0.0},
      {0.7604631914033508, 0.1776637264896365, 0.0},
      {0.0618730821070127, 0.7604631914033511, 0.0},
      {0.7604631914033508, 0.0618730821070127, 0.0},
      {0.1776637264896365, 0.7604631914033509, 0.0},
      {0.0618730821070125, 0.1776637264896368, 0.0},
      {0.0588456479492051, 0.0026124723427688, 0.0},
      {0.9385418797080261, 0.0588456479492050, 0.0},
      {0.0026124723427690, 0.9385418797080262, 0.0},
      {0.9385418797080262, 0.0026124723427688, 0.0},
      {0.0588456479492050, 0.9385418797080264, 0.0},
      {0.0026124723427685, 0.0588456479492054, 0.0},
      {0.2500996019098293, 0.0341904035600950, 0.0},
      {0.7157099945300758, 0.2500996019098293, 0.0},
      {0.0341904035600950, 0.7157099945300760, 0.0},
      {0.7157099945300757, 0.0341904035600950, 0.0},
      {0.2500996019098293, 0.7157099945300759, 0.0},
      {0.0341904035600948, 0.2500996019098295, 0.0},
      {0.0319725865365346, 0.0026508313305474, 0.0},
      {0.9653765821329179, 0.0319725865365346, 0.0},
      {0.0026508313305476, 0.9653765821329181, 0.0},
      {0.9653765821329179, 0.0026508313305474, 0.0},
      {0.0319725865365347, 0.9653765821329181, 0.0},
      {0.0026508313305472, 0.0319725865365349, 0.0},
      {0.2417239613576790, 0.0025915690897240, 0.0},
      {0.7556844695525968, 0.2417239613576791, 0.0},
      {0.0025915690897241, 0.7556844695525970, 0.0},
      {0.7556844695525969, 0.0025915690897240, 0.0},
      {0.2417239613576790, 0.7556844695525970, 0.0},
      {0.0025915690897238, 0.2417239613576793, 0.0},
      {0.1844047074392018, 0.0142733464909807, 0.0},
      {0.8013219460698174, 0.1844047074392018, 0.0},
      {0.0142733464909807, 0.8013219460698178, 0.0},
      {0.8013219460698174, 0.0142733464909807, 0.0},
      {0.1844047074392019, 0.8013219460698175, 0.0},
      {0.0142733464909806, 0.1844047074392021, 0.0},
      {0.0612563890152720, 0.0138564076854345, 0.0},
      {0.9248872032992935, 0.0612563890152719, 0.0},
      {0.0138564076854347, 0.9248872032992936, 0.0},
      {0.9248872032992935, 0.0138564076854345, 0.0},
      {0.0612563890152721, 0.9248872032992936, 0.0},
      {0.0138564076854343, 0.0612563890152723, 0.0},
      {0.0132571416922719, 0.0024938262659979, 0.0},
      {0.9842490320417303, 0.0132571416922719, 0.0},
      {0.0024938262659979, 0.9842490320417305, 0.0},
      {0.9842490320417302, 0.0024938262659979, 0.0},
      {0.0132571416922719, 0.9842490320417305, 0.0},
      {0.0024938262659976, 0.0132571416922722, 0.0},
      {0.0025649413700201, 0.0025649413700205, 0.0},
      {0.9948701172599593, 0.0025649413700201, 0.0},
      {0.0025649413700206, 0.9948701172599594, 0.0},
      {0.1877722427785784, 0.1877722427785786, 0.0},
      {0.6244555144428430, 0.1877722427785785, 0.0},
      {0.1877722427785786, 0.6244555144428431, 0.0},
      {0.1379731400814552, 0.1004409310583719, 0.0},
      {0.7615859288601728, 0.1379731400814553, 0.0},
      {0.1004409310583720, 0.7615859288601730, 0.0},
      {0.7615859288601727, 0.1004409310583719, 0.0},
      {0.1379731400814552, 0.7615859288601730, 0.0},
      {0.1004409310583717, 0.1379731400814555, 0.0},
      {0.0332056799273113, 0.0136974423527237, 0.0},
      {0.9530968777199650, 0.0332056799273113, 0.0},
      {0.0136974423527239, 0.9530968777199650, 0.0},
      {0.9530968777199650, 0.0136974423527237, 0.0},
      {0.0332056799273114, 0.9530968777199653, 0.0},
      {0.0136974423527234, 0.0332056799273116, 0.0},
      {0.3825643087869355, 0.2348713824261290, 0.0},
      {0.3825643087869355, 0.3825643087869356, 0.0},
      {0.2348713824261289, 0.3825643087869356, 0.0},
      {0.1944661967811065, 0.0341431958895440, 0.0},
      {0.7713906073293495, 0.1944661967811065, 0.0},
      {0.0341431958895441, 0.7713906073293497, 0.0},
      {0.7713906073293495, 0.0341431958895440, 0.0},
      {0.1944661967811065, 0.7713906073293496, 0.0},
      {0.0341431958895439, 0.1944661967811067, 0.0},
      {0.2394826350852045, 0.0138046252946413, 0.0},
      {0.7467127396201542, 0.2394826350852046, 0.0},
      {0.0138046252946413, 0.7467127396201543, 0.0},
      {0.7467127396201542, 0.0138046252946413, 0.0},
      {0.2394826350852046, 0.7467127396201543, 0.0},
      {0.0138046252946411, 0.2394826350852048, 0.0},
      {0.3645542277763408, 0.1864972142995551, 0.0},
      {0.4489485579241042, 0.3645542277763408, 0.0},
      {0.1864972142995550, 0.4489485579241043, 0.0},
      {0.4489485579241042, 0.1864972142995551, 0.0},
      {0.3645542277763408, 0.4489485579241042, 0.0},
      {0.1864972142995550, 0.3645542277763409, 0.0},
      {0.1373564023661998, 0.0025417593646838, 0.0},
      {0.8601018382691165, 0.1373564023661998, 0.0},
      {0.0025417593646838, 0.8601018382691167, 0.0},
      {0.8601018382691165, 0.0025417593646838, 0.0},
      {0.1373564023661998, 0.8601018382691167, 0.0},
      {0.0025417593646835, 0.1373564023662001, 0.0},
      {0.2322320373188904, 0.0630349337046810, 0.0},
      {0.7047330289764286, 0.2322320373188904, 0.0},
      {0.0630349337046811, 0.7047330289764286, 0.0},
      {0.7047330289764286, 0.0630349337046810, 0.0},
      {0.2322320373188905, 0.7047330289764286, 0.0},
      {0.0630349337046808, 0.2322320373188907, 0.0},
      {0.2926613950722469, 0.0638769125793660, 0.0},
      {0.6434616923483870, 0.2926613950722469, 0.0},
      {0.0638769125793660, 0.6434616923483873, 0.0},
      {0.6434616923483871, 0.0638769125793660, 0.0},
      {0.2926613950722470, 0.6434616923483871, 0.0},
      {0.0638769125793659, 0.2926613950722471, 0.0},
      {0.3176481163981567, 0.2419528451754752, 0.0},
      {0.4403990384263681, 0.3176481163981568, 0.0},
      {0.2419528451754751, 0.4403990384263682, 0.0},
      {0.4403990384263682, 0.2419528451754752, 0.0},
      {0.3176481163981567, 0.4403990384263682, 0.0},
      {0.2419528451754750, 0.3176481163981569, 0.0},
      {0.3003015966157783, 0.0145992712589763, 0.0},
      {0.6850991321252453, 0.3003015966157783, 0.0},
      {0.0145992712589764, 0.6850991321252455, 0.0},
      {0.6850991321252453, 0.0145992712589763, 0.0},
      {0.3003015966157784, 0.6850991321252453, 0.0},
      {0.0145992712589763, 0.3003015966157786, 0.0},
      {0.3005613537168110, 0.0028181145969261, 0.0},
      {0.6966205316862629, 0.3005613537168110, 0.0},
      {0.0028181145969260, 0.6966205316862633, 0.0},
      {0.6966205316862629, 0.0028181145969261, 0.0},
      {0.3005613537168111, 0.6966205316862629, 0.0},
      {0.0028181145969259, 0.3005613537168113, 0.0},
      {0.0605239793987312, 0.0341344995593797, 0.0},
      {0.9053415210418889, 0.0605239793987312, 0.0},
      {0.0341344995593799, 0.9053415210418891, 0.0},
      {0.9053415210418890, 0.0341344995593797, 0.0},
      {0.0605239793987313, 0.9053415210418891, 0.0},
      {0.0341344995593795, 0.0605239793987315, 0.0},
      {0.3007859192338894, 0.1933554772764183, 0.0},
      {0.5058586034896924, 0.3007859192338894, 0.0},
      {0.1933554772764183, 0.5058586034896925, 0.0},
      {0.5058586034896924, 0.1933554772764183, 0.0},
      {0.3007859192338894, 0.5058586034896925, 0.0},
      {0.1933554772764182, 0.3007859192338895, 0.0},
      {0.1895507039776852, 0.0973629697501509, 0.0},
      {0.7130863262721638, 0.1895507039776853, 0.0},
      {0.0973629697501509, 0.7130863262721642, 0.0},
      {0.7130863262721640, 0.0973629697501509, 0.0},
      {0.1895507039776853, 0.7130863262721641, 0.0},
      {0.0973629697501507, 0.1895507039776855, 0.0},
      {0.4301399033621489, 0.0027883024051114, 0.0},
      {0.5670717942327397, 0.4301399033621490, 0.0},
      {0.0027883024051114, 0.5670717942327399, 0.0},
      {0.5670717942327397, 0.0027883024051114, 0.0},
      {0.4301399033621489, 0.5670717942327398, 0.0},
      {0.0027883024051113, 0.4301399033621491, 0.0},
      {0.1870185722176647, 0.0027559624464837, 0.0},
      {0.8102254653358516, 0.1870185722176647, 0.0},
      {0.0027559624464837, 0.8102254653358518, 0.0},
      {0.8102254653358516, 0.0027559624464837, 0.0},
      {0.1870185722176648, 0.8102254653358517, 0.0},
      {0.0027559624464835, 0.1870185722176650, 0.0},
      {0.3791895225090074, 0.0978271071588678, 0.0},
      {0.5229833703321247, 0.3791895225090076, 0.0},
      {0.0978271071588677, 0.5229833703321249, 0.0},
      {0.5229833703321248, 0.0978271071588678, 0.0},
      {0.3791895225090075, 0.5229833703321249, 0.0},
      {0.0978271071588676, 0.3791895225090077, 0.0},
      {0.3129669769821593, 0.0995736856368710, 0.0},
      {0.5874593373809698, 0.3129669769821593, 0.0},
      {0.0995736856368709, 0.5874593373809700, 0.0},
      {0.5874593373809698, 0.0995736856368710, 0.0},
      {0.3129669769821594, 0.5874593373809698, 0.0},
      {0.0995736856368708, 0.3129669769821595, 0.0},
      {0.1418044870172890, 0.0341143516662649, 0.0},
      {0.8240811613164462, 0.1418044870172890, 0.0},
      {0.0341143516662649, 0.8240811613164464, 0.0},
      {0.8240811613164462, 0.0341143516662649, 0.0},
      {0.1418044870172891, 0.8240811613164463, 0.0},
      {0.0341143516662646, 0.1418044870172893, 0.0},
      {0.4528922608459510, 0.0942154783080980, 0.0},
      {0.4528922608459510, 0.4528922608459511, 0.0},
      {0.0942154783080979, 0.4528922608459512, 0.0},
      {0.2481814600155885, 0.1004974432729284, 0.0},
      {0.6513210967114831, 0.2481814600155886, 0.0},
      {0.1004974432729285, 0.6513210967114832, 0.0},
      {0.6513210967114831, 0.1004974432729284, 0.0},
      {0.2481814600155886, 0.6513210967114832, 0.0},
      {0.1004974432729282, 0.2481814600155887, 0.0},
      {0.3587978354079987, 0.0616738036965517, 0.0},
      {0.5795283608954496, 0.3587978354079987, 0.0},
      {0.0616738036965517, 0.5795283608954498, 0.0},
      {0.5795283608954497, 0.0616738036965517, 0.0},
      {0.3587978354079988, 0.5795283608954497, 0.0},
      {0.0616738036965515, 0.3587978354079990, 0.0},
      {0.0928670462140479, 0.0928670462140481, 0.0},
      {0.8142659075719040, 0.0928670462140479, 0.0},
      {0.0928670462140481, 0.8142659075719042, 0.0},
      {0.2524849326490907, 0.2524849326490909, 0.0},
      {0.4950301347018184, 0.2524849326490908, 0.0},
      {0.2524849326490909, 0.4950301347018185, 0.0}},
      {0.0015399827845908,
      0.0015399827845908,
      0.0015399827845908,
      0.0012436741101233,
      0.0012436741101233,
      0.0012436741101233,
      0.0019563191704088,
      0.0019563191704088,
      0.0019563191704088,
      0.0019563191704088,
      0.0019563191704088,
      0.0019563191704088,
      0.0003043078610904,
      0.0003043078610904,
      0.0003043078610904,
      0.0029390411668796,
      0.0029390411668796,
      0.0029390411668796,
      0.0013287798860386,
      0.0013287798860386,
      0.0013287798860386,
      0.0013287798860386,
      0.0013287798860386,
      0.0013287798860386,
      0.0005201929288082,
      0.0005201929288082,
      0.0005201929288082,
      0.0005201929288082,
      0.0005201929288082,
      0.0005201929288082,
      0.0019344111227621,
      0.0019344111227621,
      0.0019344111227621,
      0.0019344111227621,
      0.0019344111227621,
      0.0019344111227621,
      0.0027180039607926,
      0.0027180039607926,
      0.0027180039607926,
      0.0027180039607926,
      0.0027180039607926,
      0.0027180039607926,
      0.0025665105385067,
      0.0025665105385067,
      0.0025665105385067,
      0.0025665105385067,
      0.0025665105385067,
      0.0025665105385067,
      0.0034342444832575,
      0.0034342444832575,
      0.0034342444832575,
      0.0013624426302032,
      0.0013624426302032,
      0.0013624426302032,
      0.0013624426302032,
      0.0013624426302032,
      0.0013624426302032,
      0.0008071536219579,
      0.0008071536219579,
      0.0008071536219579,
      0.0015409639138513,
      0.0015409639138513,
      0.0015409639138513,
      0.0015409639138513,
      0.0015409639138513,
      0.0015409639138513,
      0.0005418112146355,
      0.0005418112146355,
      0.0005418112146355,
      0.0010980073304482,
      0.0010980073304482,
      0.0010980073304482,
      0.0010980073304482,
      0.0010980073304482,
      0.0010980073304482,
      0.0013535858145947,
      0.0013535858145947,
      0.0013535858145947,
      0.0013535858145947,
      0.0013535858145947,
      0.0013535858145947,
      0.0008866252831515,
      0.0008866252831515,
      0.0008866252831515,
      0.0008866252831515,
      0.0008866252831515,
      0.0008866252831515,
      0.0013583025341476,
      0.0013583025341476,
      0.0013583025341476,
      0.0010597047660428,
      0.0010597047660428,
      0.0010597047660428,
      0.0010597047660428,
      0.0010597047660428,
      0.0010597047660428,
      0.0010121646522749,
      0.0010121646522749,
      0.0010121646522749,
      0.0006911329136998,
      0.0006911329136998,
      0.0006911329136998,
      0.0006911329136998,
      0.0006911329136998,
      0.0006911329136998,
      0.0004143898011380,
      0.0004143898011380,
      0.0004143898011380,
      0.0004143898011380,
      0.0004143898011380,
      0.0004143898011380,
      0.0021227159605322,
      0.0021227159605322,
      0.0021227159605322,
      0.0028804649851842,
      0.0028804649851842,
      0.0028804649851842,
      0.0028804649851842,
      0.0028804649851842,
      0.0028804649851842,
      0.0002360510527896,
      0.0002360510527896,
      0.0002360510527896,
      0.0002433203267894,
      0.0002433203267894,
      0.0002433203267894,
      0.0002433203267894,
      0.0002433203267894,
      0.0002433203267894,
      0.0030305853022484,
      0.0030305853022484,
      0.0030305853022484,
      0.0030305853022484,
      0.0030305853022484,
      0.0030305853022484,
      0.0019795801765006,
      0.0019795801765006,
      0.0019795801765006,
      0.0019795801765006,
      0.0019795801765006,
      0.0019795801765006,
      0.0009757652807424,
      0.0009757652807424,
      0.0009757652807424,
      0.0009757652807424,
      0.0009757652807424,
      0.0009757652807424,
      0.0015879320097892,
      0.0015879320097892,
      0.0015879320097892,
      0.0015879320097892,
      0.0015879320097892,
      0.0015879320097892,
      0.0002088844892013,
      0.0002088844892013,
      0.0002088844892013,
      0.0002088844892013,
      0.0002088844892013,
      0.0002088844892013,
      0.0014018992565018,
      0.0014018992565018,
      0.0014018992565018,
      0.0014018992565018,
      0.0014018992565018,
      0.0014018992565018,
      0.0001537161500621,
      0.0001537161500621,
      0.0001537161500621,
      0.0001537161500621,
      0.0001537161500621,
      0.0001537161500621,
      0.0003790239372578,
      0.0003790239372578,
      0.0003790239372578,
      0.0003790239372578,
      0.0003790239372578,
      0.0003790239372578,
      0.0008142800935957,
      0.0008142800935957,
      0.0008142800935957,
      0.0008142800935957,
      0.0008142800935957,
      0.0008142800935957,
      0.0005007907575658,
      0.0005007907575658,
      0.0005007907575658,
      0.0005007907575658,
      0.0005007907575658,
      0.0005007907575658,
      0.0000948517546105,
      0.0000948517546105,
      0.0000948517546105,
      0.0000948517546105,
      0.0000948517546105,
      0.0000948517546105,
      0.0000428662883399,
      0.0000428662883399,
      0.0000428662883399,
      0.0026676376711981,
      0.0026676376711981,
      0.0026676376711981,
      0.0019612968594132,
      0.0019612968594132,
      0.0019612968594132,
      0.0019612968594132,
      0.0019612968594132,
      0.0019612968594132,
      0.0003676322245355,
      0.0003676322245355,
      0.0003676322245355,
      0.0003676322245355,
      0.0003676322245355,
      0.0003676322245355,
      0.0034748404345283,
      0.0034748404345283,
      0.0034748404345283,
      0.0013138750491333,
      0.0013138750491333,
      0.0013138750491333,
      0.0013138750491333,
      0.0013138750491333,
      0.0013138750491333,
      0.0009208169595140,
      0.0009208169595140,
      0.0009208169595140,
      0.0009208169595140,
      0.0009208169595140,
      0.0009208169595140,
      0.0029625163863369,
      0.0029625163863369,
      0.0029625163863369,
      0.0029625163863369,
      0.0029625163863369,
      0.0029625163863369,
      0.0003061694204072,
      0.0003061694204072,
      0.0003061694204072,
      0.0003061694204072,
      0.0003061694204072,
      0.0003061694204072,
      0.0019124817242395,
      0.0019124817242395,
      0.0019124817242395,
      0.0019124817242395,
      0.0019124817242395,
      0.0019124817242395,
      0.0020641516111032,
      0.0020641516111032,
      0.0020641516111032,
      0.0020641516111032,
      0.0020641516111032,
      0.0020641516111032,
      0.0033460160208706,
      0.0033460160208706,
      0.0033460160208706,
      0.0033460160208706,
      0.0033460160208706,
      0.0033460160208706,
      0.0010236414736965,
      0.0010236414736965,
      0.0010236414736965,
      0.0010236414736965,
      0.0010236414736965,
      0.0010236414736965,
      0.0004390078395826,
      0.0004390078395826,
      0.0004390078395826,
      0.0004390078395826,
      0.0004390078395826,
      0.0004390078395826,
      0.0007888925592730,
      0.0007888925592730,
      0.0007888925592730,
      0.0007888925592730,
      0.0007888925592730,
      0.0007888925592730,
      0.0031537179554942,
      0.0031537179554942,
      0.0031537179554942,
      0.0031537179554942,
      0.0031537179554942,
      0.0031537179554942,
      0.0021363199105266,
      0.0021363199105266,
      0.0021363199105266,
      0.0021363199105266,
      0.0021363199105266,
      0.0021363199105266,
      0.0004849186731197,
      0.0004849186731197,
      0.0004849186731197,
      0.0004849186731197,
      0.0004849186731197,
      0.0004849186731197,
      0.0003690091352290,
      0.0003690091352290,
      0.0003690091352290,
      0.0003690091352290,
      0.0003690091352290,
      0.0003690091352290,
      0.0026309406800638,
      0.0026309406800638,
      0.0026309406800638,
      0.0026309406800638,
      0.0026309406800638,
      0.0026309406800638,
      0.0026501359874991,
      0.0026501359874991,
      0.0026501359874991,
      0.0026501359874991,
      0.0026501359874991,
      0.0026501359874991,
      0.0012279005051396,
      0.0012279005051396,
      0.0012279005051396,
      0.0012279005051396,
      0.0012279005051396,
      0.0012279005051396,
      0.0026664027102005,
      0.0026664027102005,
      0.0026664027102005,
      0.0025278051975789,
      0.0025278051975789,
      0.0025278051975789,
      0.0025278051975789,
      0.0025278051975789,
      0.0025278051975789,
      0.0022170954845651,
      0.0022170954845651,
      0.0022170954845651,
      0.0022170954845651,
      0.0022170954845651,
      0.0022170954845651,
      0.0016453530729270,
      0.0016453530729270,
      0.0016453530729270,
      0.0036571045715601,
      0.0036571045715601,
      0.0036571045715601}
    },
    {
      354,
      {{0.0017123075799637, 0.0017123075799640, 0.0},
      {0.9965753848400722, 0.0017123075799637, 0.0},
      {0.0017123075799641, 0.9965753848400725, 0.0},
      {0.0197551676157197, 0.0107567195566843, 0.0},
      {0.9694881128275959, 0.0197551676157197, 0.0},
      {0.0107567195566844, 0.9694881128275962, 0.0},
      {0.9694881128275961, 0.0107567195566843, 0.0},
      {0.0197551676157198, 0.9694881128275962, 0.0},
      {0.0107567195566840, 0.0197551676157201, 0.0},
      {0.0995165813818446, 0.0007805858998619, 0.0},
      {0.8997028327182934, 0.0995165813818446, 0.0},
      {0.0007805858998620, 0.8997028327182935, 0.0},
      {0.8997028327182934, 0.0007805858998619, 0.0},
      {0.0995165813818448, 0.8997028327182935, 0.0},
      {0.0007805858998617, 0.0995165813818450, 0.0},
      {0.1534945447552139, 0.0019383329127906, 0.0},
      {0.8445671223319955, 0.1534945447552139, 0.0},
      {0.0019383329127907, 0.8445671223319957, 0.0},
      {0.8445671223319955, 0.0019383329127906, 0.0},
      {0.1534945447552140, 0.8445671223319957, 0.0},
      {0.0019383329127904, 0.1534945447552142, 0.0},
      {0.4606074520039989, 0.0017861424805254, 0.0},
      {0.5376064055154758, 0.4606074520039989, 0.0},
      {0.0017861424805254, 0.5376064055154759, 0.0},
      {0.5376064055154757, 0.0017861424805254, 0.0},
      {0.4606074520039989, 0.5376064055154758, 0.0},
      {0.0017861424805253, 0.4606074520039992, 0.0},
      {0.0296420604264444, 0.0017542251710528, 0.0},
      {0.9686037144025027, 0.0296420604264444, 0.0},
      {0.0017542251710529, 0.9686037144025029, 0.0},
      {0.9686037144025029, 0.0017542251710528, 0.0},
      {0.0296420604264444, 0.9686037144025031, 0.0},
      {0.0017542251710524, 0.0296420604264447, 0.0},
      {0.3338406248248118, 0.0005172942596601, 0.0},
      {0.6656420809155281, 0.3338406248248119, 0.0},
      {0.0005172942596601, 0.6656420809155283, 0.0},
      {0.6656420809155281, 0.0005172942596601, 0.0},
      {0.3338406248248119, 0.6656420809155282, 0.0},
      {0.0005172942596599, 0.3338406248248121, 0.0},
      {0.0291673514653005, 0.0291673514653008, 0.0},
      {0.9416652970693987, 0.0291673514653005, 0.0},
      {0.0291673514653009, 0.9416652970693989, 0.0},
      {0.0495499879123024, 0.0495499879123027, 0.0},
      {0.9009000241753948, 0.0495499879123024, 0.0},
      {0.0495499879123029, 0.9009000241753949, 0.0},
      {0.1194266872274120, 0.0075458709257061, 0.0},
      {0.8730274418468819, 0.1194266872274120, 0.0},
      {0.0075458709257062, 0.8730274418468821, 0.0},
      {0.8730274418468820, 0.0075458709257061, 0.0},
      {0.1194266872274120, 0.8730274418468821, 0.0},
      {0.0075458709257059, 0.1194266872274123, 0.0},
      {0.1733118066994289, 0.0114925464528053, 0.0},
      {0.8151956468477657, 0.1733118066994289, 0.0},
      {0.0114925464528055, 0.8151956468477658, 0.0},
      {0.8151956468477658, 0.0114925464528053, 0.0},
      {0.1733118066994291, 0.8151956468477658, 0.0},
      {0.0114925464528051, 0.1733118066994292, 0.0},
      {0.0593978633857566, 0.0242629677211045, 0.0},
      {0.9163391688931388, 0.0593978633857566, 0.0},
      {0.0242629677211046, 0.9163391688931390, 0.0},
      {0.9163391688931388, 0.0242629677211045, 0.0},
      {0.0593978633857567, 0.9163391688931390, 0.0},
      {0.0242629677211044, 0.0593978633857569, 0.0},
      {0.2594461646278431, 0.0097976933024596, 0.0},
      {0.7307561420696972, 0.2594461646278432, 0.0},
      {0.0097976933024596, 0.7307561420696975, 0.0},
      {0.7307561420696972, 0.0097976933024596, 0.0},
      {0.2594461646278433, 0.7307561420696972, 0.0},
      {0.0097976933024595, 0.2594461646278435, 0.0},
      {0.2696648795311178, 0.0018991508282083, 0.0},
      {0.7284359696406738, 0.2696648795311178, 0.0},
      {0.0018991508282084, 0.7284359696406740, 0.0},
      {0.7284359696406739, 0.0018991508282083, 0.0},
      {0.2696648795311180, 0.7284359696406739, 0.0},
      {0.0018991508282082, 0.2696648795311181, 0.0},
      {0.3234284104109564, 0.0076746798561443, 0.0},
      {0.6688969097328992, 0.3234284104109565, 0.0},
      {0.0076746798561443, 0.6688969097328994, 0.0},
      {0.6688969097328992, 0.0076746798561443, 0.0},
      {0.3234284104109564, 0.6688969097328994, 0.0},
      {0.0076746798561442, 0.3234284104109568, 0.0},
      {0.2093976094351925, 0.0027072043077686, 0.0},
      {0.7878951862570389, 0.2093976094351925, 0.0},
      {0.0027072043077686, 0.7878951862570391, 0.0},
      {0.7878951862570389, 0.0027072043077686, 0.0},
      {0.2093976094351925, 0.7878951862570391, 0.0},
      {0.0027072043077684, 0.2093976094351928, 0.0},
      {0.0107937373867395, 0.0027063965571699, 0.0},
      {0.9864998660560905, 0.0107937373867395, 0.0},
      {0.0027063965571701, 0.9864998660560906, 0.0},
      {0.9864998660560905, 0.0027063965571699, 0.0},
      {0.0107937373867397, 0.9864998660560906, 0.0},
      {0.0027063965571697, 0.0107937373867399, 0.0},
      {0.0416577600266789, 0.0104627340296395, 0.0},
      {0.9478795059436815, 0.0416577600266789, 0.0},
      {0.0104627340296397, 0.9478795059436816, 0.0},
      {0.9478795059436816, 0.0104627340296395, 0.0},
      {0.0416577600266790, 0.9478795059436816, 0.0},
      {0.0104627340296393, 0.0416577600266793, 0.0},
      {0.1393007744184822, 0.0215128602849199, 0.0},
      {0.8391863652965978, 0.1393007744184823, 0.0},
      {0.0215128602849200, 0.8391863652965980, 0.0},
      {0.8391863652965978, 0.0215128602849199, 0.0},
      {0.1393007744184823, 0.8391863652965980, 0.0},
      {0.0215128602849197, 0.1393007744184826, 0.0},
      {0.4877239805831534, 0.0245520388336933, 0.0},
      {0.4877239805831534, 0.4877239805831535, 0.0},
      {0.0245520388336932, 0.4877239805831536, 0.0},
      {0.0971407574555291, 0.0228245356349520, 0.0},
      {0.8800347069095190, 0.0971407574555291, 0.0},
      {0.0228245356349520, 0.8800347069095191, 0.0},
      {0.8800347069095190, 0.0228245356349520, 0.0},
      {0.0971407574555291, 0.8800347069095191, 0.0},
      {0.0228245356349517, 0.0971407574555294, 0.0},
      {0.1511248097389706, 0.1000885303047900, 0.0},
      {0.7487866599562394, 0.1511248097389706, 0.0},
      {0.1000885303047901, 0.7487866599562395, 0.0},
      {0.7487866599562394, 0.1000885303047900, 0.0},
      {0.1511248097389706, 0.7487866599562395, 0.0},
      {0.1000885303047898, 0.1511248097389709, 0.0},
      {0.4960618044654749, 0.0078763910690501, 0.0},
      {0.4960618044654750, 0.4960618044654750, 0.0},
      {0.0078763910690500, 0.4960618044654752, 0.0},
      {0.3939323955351545, 0.0031595695366355, 0.0},
      {0.6029080349282100, 0.3939323955351545, 0.0},
      {0.0031595695366355, 0.6029080349282102, 0.0},
      {0.6029080349282100, 0.0031595695366355, 0.0},
      {0.3939323955351544, 0.6029080349282102, 0.0},
      {0.0031595695366353, 0.3939323955351547, 0.0},
      {0.1276374594173248, 0.0435675752808352, 0.0},
      {0.8287949653018399, 0.1276374594173248, 0.0},
      {0.0435675752808352, 0.8287949653018403, 0.0},
      {0.8287949653018402, 0.0435675752808352, 0.0},
      {0.1276374594173247, 0.8287949653018403, 0.0},
      {0.0435675752808349, 0.1276374594173251, 0.0},
      {0.2153427859170523, 0.0628153762458396, 0.0},
      {0.7218418378371082, 0.2153427859170523, 0.0},
      {0.0628153762458395, 0.7218418378371083, 0.0},
      {0.7218418378371081, 0.0628153762458396, 0.0},
      {0.2153427859170524, 0.7218418378371083, 0.0},
      {0.0628153762458394, 0.2153427859170526, 0.0},
      {0.1805238872560285, 0.0346911866169312, 0.0},
      {0.7847849261270403, 0.1805238872560285, 0.0},
      {0.0346911866169312, 0.7847849261270404, 0.0},
      {0.7847849261270403, 0.0346911866169312, 0.0},
      {0.1805238872560286, 0.7847849261270404, 0.0},
      {0.0346911866169310, 0.1805238872560288, 0.0},
      {0.3706974787497283, 0.0159521357680312, 0.0},
      {0.6133503854822405, 0.3706974787497284, 0.0},
      {0.0159521357680312, 0.6133503854822406, 0.0},
      {0.6133503854822404, 0.0159521357680312, 0.0},
      {0.3706974787497284, 0.6133503854822405, 0.0},
      {0.0159521357680310, 0.3706974787497286, 0.0},
      {0.2389043466345939, 0.0365656909809640, 0.0},
      {0.7245299623844421, 0.2389043466345939, 0.0},
      {0.0365656909809640, 0.7245299623844423, 0.0},
      {0.7245299623844421, 0.0365656909809640, 0.0},
      {0.2389043466345939, 0.7245299623844423, 0.0},
      {0.0365656909809638, 0.2389043466345942, 0.0},
      {0.0765961156051737, 0.0765961156051741, 0.0},
      {0.8468077687896521, 0.0765961156051737, 0.0},
      {0.0765961156051742, 0.8468077687896522, 0.0},
      {0.4035445047649169, 0.1228672484069712, 0.0},
      {0.4735882468281119, 0.4035445047649170, 0.0},
      {0.1228672484069711, 0.4735882468281121, 0.0},
      {0.4735882468281119, 0.1228672484069712, 0.0},
      {0.4035445047649169, 0.4735882468281120, 0.0},
      {0.1228672484069711, 0.4035445047649171, 0.0},
      {0.2510886184010617, 0.1220611920335108, 0.0},
      {0.6268501895654275, 0.2510886184010618, 0.0},
      {0.1220611920335108, 0.6268501895654277, 0.0},
      {0.6268501895654275, 0.1220611920335108, 0.0},
      {0.2510886184010617, 0.6268501895654277, 0.0},
      {0.1220611920335106, 0.2510886184010619, 0.0},
      {0.3100924603530019, 0.0878278870328068, 0.0},
      {0.6020796526141913, 0.3100924603530020, 0.0},
      {0.0878278870328068, 0.6020796526141915, 0.0},
      {0.6020796526141913, 0.0878278870328068, 0.0},
      {0.3100924603530019, 0.6020796526141915, 0.0},
      {0.0878278870328066, 0.3100924603530022, 0.0},
      {0.4171889370681111, 0.0656811030561965, 0.0},
      {0.5171299598756924, 0.4171889370681111, 0.0},
      {0.0656811030561965, 0.5171299598756927, 0.0},
      {0.5171299598756924, 0.0656811030561965, 0.0},
      {0.4171889370681111, 0.5171299598756924, 0.0},
      {0.0656811030561964, 0.4171889370681113, 0.0},
      {0.4759518460527712, 0.0480963078944575, 0.0},
      {0.4759518460527714, 0.4759518460527713, 0.0},
      {0.0480963078944574, 0.4759518460527715, 0.0},
      {0.0842771975158688, 0.0457201471848306, 0.0},
      {0.8700026552993007, 0.0842771975158688, 0.0},
      {0.0457201471848306, 0.8700026552993009, 0.0},
      {0.8700026552993007, 0.0457201471848306, 0.0},
      {0.0842771975158688, 0.8700026552993009, 0.0},
      {0.0457201471848303, 0.0842771975158691, 0.0},
      {0.2875172818904952, 0.0528034108950670, 0.0},
      {0.6596793072144378, 0.2875172818904953, 0.0},
      {0.0528034108950670, 0.6596793072144379, 0.0},
      {0.6596793072144378, 0.0528034108950670, 0.0},
      {0.2875172818904952, 0.6596793072144379, 0.0},
      {0.0528034108950669, 0.2875172818904955, 0.0},
      {0.1090955419572377, 0.1090955419572379, 0.0},
      {0.7818089160855244, 0.1090955419572376, 0.0},
      {0.1090955419572380, 0.7818089160855246, 0.0},
      {0.3421421849069924, 0.0332627075502786, 0.0},
      {0.6245951075427290, 0.3421421849069925, 0.0},
      {0.0332627075502785, 0.6245951075427292, 0.0},
      {0.6245951075427290, 0.0332627075502786, 0.0},
      {0.3421421849069924, 0.6245951075427292, 0.0},
      {0.0332627075502785, 0.3421421849069927, 0.0},
      {0.2312470036938409, 0.2312470036938411, 0.0},
      {0.5375059926123180, 0.2312470036938410, 0.0},
      {0.2312470036938411, 0.5375059926123180, 0.0},
      {0.1959081982236505, 0.1959081982236507, 0.0},
      {0.6081836035526987, 0.1959081982236506, 0.0},
      {0.1959081982236506, 0.6081836035526990, 0.0},
      {0.2907813356274843, 0.0230909511650298, 0.0},
      {0.6861277132074860, 0.2907813356274843, 0.0},
      {0.0230909511650299, 0.6861277132074860, 0.0},
      {0.6861277132074859, 0.0230909511650298, 0.0},
      {0.2907813356274844, 0.6861277132074860, 0.0},
      {0.0230909511650297, 0.2907813356274845, 0.0},
      {0.0777385297161003, 0.0084121228886458, 0.0},
      {0.9138493473952539, 0.0777385297161002, 0.0},
      {0.0084121228886461, 0.9138493473952539, 0.0},
      {0.9138493473952539, 0.0084121228886458, 0.0},
      {0.0777385297161003, 0.9138493473952539, 0.0},
      {0.0084121228886456, 0.0777385297161006, 0.0},
      {0.0583538087981567, 0.0018558446904391, 0.0},
      {0.9397903465114041, 0.0583538087981567, 0.0},
      {0.0018558446904393, 0.9397903465114043, 0.0},
      {0.9397903465114043, 0.0018558446904391, 0.0},
      {0.0583538087981568, 0.9397903465114043, 0.0},
      {0.0018558446904388, 0.0583538087981571, 0.0},
      {0.3537362173911874, 0.1518612214991534, 0.0},
      {0.4944025611096592, 0.3537362173911875, 0.0},
      {0.1518612214991533, 0.4944025611096594, 0.0},
      {0.4944025611096592, 0.1518612214991534, 0.0},
      {0.3537362173911875, 0.4944025611096593, 0.0},
      {0.1518612214991532, 0.3537362173911877, 0.0},
      {0.1941036834219221, 0.1466938464955038, 0.0},
      {0.6592024700825740, 0.1941036834219222, 0.0},
      {0.1466938464955038, 0.6592024700825742, 0.0},
      {0.6592024700825740, 0.1466938464955038, 0.0},
      {0.1941036834219222, 0.6592024700825742, 0.0},
      {0.1466938464955036, 0.1941036834219224, 0.0},
      {0.1152446802620743, 0.0730582724234848, 0.0},
      {0.8116970473144409, 0.1152446802620743, 0.0},
      {0.0730582724234849, 0.8116970473144410, 0.0},
      {0.8116970473144409, 0.0730582724234848, 0.0},
      {0.1152446802620743, 0.8116970473144410, 0.0},
      {0.0730582724234846, 0.1152446802620745, 0.0},
      {0.2526489188568274, 0.0833205145800467, 0.0},
      {0.6640305665631259, 0.2526489188568274, 0.0},
      {0.0833205145800467, 0.6640305665631261, 0.0},
      {0.6640305665631260, 0.0833205145800467, 0.0},
      {0.2526489188568274, 0.6640305665631261, 0.0},
      {0.0833205145800466, 0.2526489188568276, 0.0},
      {0.3706381279110351, 0.0976700995193538, 0.0},
      {0.5316917725696112, 0.3706381279110351, 0.0},
      {0.0976700995193537, 0.5316917725696113, 0.0},
      {0.5316917725696112, 0.0976700995193538, 0.0},
      {0.3706381279110351, 0.5316917725696113, 0.0},
      {0.0976700995193536, 0.3706381279110353, 0.0},
      {0.1975107662353285, 0.1021274732358202, 0.0},
      {0.7003617605288512, 0.1975107662353285, 0.0},
      {0.1021274732358203, 0.7003617605288514, 0.0},
      {0.7003617605288514, 0.1021274732358202, 0.0},
      {0.1975107662353285, 0.7003617605288514, 0.0},
      {0.1021274732358200, 0.1975107662353287, 0.0},
      {0.2191021787766396, 0.0169401226467935, 0.0},
      {0.7639576985765668, 0.2191021787766396, 0.0},
      {0.0169401226467936, 0.7639576985765670, 0.0},
      {0.7639576985765668, 0.0169401226467935, 0.0},
      {0.2191021787766396, 0.7639576985765670, 0.0},
      {0.0169401226467933, 0.2191021787766399, 0.0},
      {0.1647494714555341, 0.0646074146543942, 0.0},
      {0.7706431138900716, 0.1647494714555342, 0.0},
      {0.0646074146543943, 0.7706431138900717, 0.0},
      {0.7706431138900717, 0.0646074146543942, 0.0},
      {0.1647494714555343, 0.7706431138900717, 0.0},
      {0.0646074146543940, 0.1647494714555344, 0.0},
      {0.3487568874306149, 0.1942934918619254, 0.0},
      {0.4569496207074597, 0.3487568874306151, 0.0},
      {0.1942934918619253, 0.4569496207074598, 0.0},
      {0.4569496207074596, 0.1942934918619254, 0.0},
      {0.3487568874306151, 0.4569496207074597, 0.0},
      {0.1942934918619253, 0.3487568874306152, 0.0},
      {0.3558533300586623, 0.0587895877753212, 0.0},
      {0.5853570821660166, 0.3558533300586624, 0.0},
      {0.0587895877753212, 0.5853570821660167, 0.0},
      {0.5853570821660165, 0.0587895877753212, 0.0},
      {0.3558533300586624, 0.5853570821660166, 0.0},
      {0.0587895877753211, 0.3558533300586625, 0.0},
      {0.3487165289854694, 0.2435849758851731, 0.0},
      {0.4076984951293575, 0.3487165289854695, 0.0},
      {0.2435849758851730, 0.4076984951293576, 0.0},
      {0.4076984951293575, 0.2435849758851731, 0.0},
      {0.3487165289854693, 0.4076984951293576, 0.0},
      {0.2435849758851730, 0.3487165289854696, 0.0},
      {0.4223218082071514, 0.1553563835856973, 0.0},
      {0.4223218082071514, 0.4223218082071514, 0.0},
      {0.1553563835856973, 0.4223218082071515, 0.0},
      {0.4548817499595059, 0.0902365000809881, 0.0},
      {0.4548817499595060, 0.4548817499595060, 0.0},
      {0.0902365000809880, 0.4548817499595061, 0.0},
      {0.2907043355496960, 0.1861259659561633, 0.0},
      {0.5231696984941406, 0.2907043355496962, 0.0},
      {0.1861259659561632, 0.5231696984941409, 0.0},
      {0.5231696984941406, 0.1861259659561633, 0.0},
      {0.2907043355496960, 0.5231696984941409, 0.0},
      {0.1861259659561631, 0.2907043355496963, 0.0},
      {0.1439001195404208, 0.1439001195404211, 0.0},
      {0.7121997609191582, 0.1439001195404208, 0.0},
      {0.1439001195404210, 0.7121997609191584, 0.0},
      {0.2898629348171137, 0.2383570195525562, 0.0},
      {0.4717800456303300, 0.2898629348171138, 0.0},
      {0.2383570195525562, 0.4717800456303302, 0.0},
      {0.4717800456303302, 0.2383570195525562, 0.0},
      {0.2898629348171137, 0.4717800456303302, 0.0},
      {0.2383570195525561, 0.2898629348171138, 0.0},
      {0.4023404478547354, 0.1953191042905293, 0.0},
      {0.4023404478547353, 0.4023404478547354, 0.0},
      {0.1953191042905292, 0.4023404478547355, 0.0},
      {0.3093835754454580, 0.1312296167919730, 0.0},
      {0.5593868077625690, 0.3093835754454580, 0.0},
      {0.1312296167919730, 0.5593868077625692, 0.0},
      {0.5593868077625690, 0.1312296167919730, 0.0},
      {0.3093835754454579, 0.5593868077625692, 0.0},
      {0.1312296167919728, 0.3093835754454582, 0.0},
      {0.2469795390992444, 0.1684589785524792, 0.0},
      {0.5845614823482763, 0.2469795390992445, 0.0},
      {0.1684589785524792, 0.5845614823482764, 0.0},
      {0.5845614823482763, 0.1684589785524792, 0.0},
      {0.2469795390992445, 0.5845614823482764, 0.0},
      {0.1684589785524790, 0.2469795390992447, 0.0},
      {0.4353451951214652, 0.0131254695112235, 0.0},
      {0.5515293353673113, 0.4353451951214652, 0.0},
      {0.0131254695112235, 0.5515293353673115, 0.0},
      {0.5515293353673113, 0.0131254695112235, 0.0},
      {0.4353451951214652, 0.5515293353673113, 0.0},
      {0.0131254695112233, 0.4353451951214655, 0.0},
      {0.2942957656908339, 0.2942957656908340, 0.0},
      {0.4114084686183320, 0.2942957656908340, 0.0},
      {0.2942957656908339, 0.4114084686183322, 0.0},
      {0.3516157887957689, 0.2967684224084621, 0.0},
      {0.3516157887957689, 0.3516157887957691, 0.0},
      {0.2967684224084619, 0.3516157887957691, 0.0},
      {0.4137012893148831, 0.0344134405185802, 0.0},
      {0.5518852701665368, 0.4137012893148831, 0.0},
      {0.0344134405185802, 0.5518852701665369, 0.0},
      {0.5518852701665367, 0.0344134405185802, 0.0},
      {0.4137012893148831, 0.5518852701665368, 0.0},
      {0.0344134405185801, 0.4137012893148834, 0.0}},
      {0.0000221520454571,
      0.0000221520454571,
      0.0000221520454571,
      0.0002059878232185,
      0.0002059878232185,
      0.0002059878232185,
      0.0002059878232185,
      0.0002059878232185,
      0.0002059878232185,
      0.0001366204230018,
      0.0001366204230018,
      0.0001366204230018,
      0.0001366204230018,
      0.0001366204230018,
      0.0001366204230018,
      0.0002696642219650,
      0.0002696642219650,
      0.0002696642219650,
      0.0002696642219650,
      0.0002696642219650,
      0.0002696642219650,
      0.0003354894852910,
      0.0003354894852910,
      0.0003354894852910,
      0.0003354894852910,
      0.0003354894852910,
      0.0003354894852910,
      0.0001129657680570,
      0.0001129657680570,
      0.0001129657680570,
      0.0001129657680570,
      0.0001129657680570,
      0.0001129657680570,
      0.0001610331547091,
      0.0001610331547091,
      0.0001610331547091,
      0.0001610331547091,
      0.0001610331547091,
      0.0001610331547091,
      0.0005614886914119,
      0.0005614886914119,
      0.0005614886914119,
      0.0008725757827664,
      0.0008725757827664,
      0.0008725757827664,
      0.0004838836513975,
      0.0004838836513975,
      0.0004838836513975,
      0.0004838836513975,
      0.0004838836513975,
      0.0004838836513975,
      0.0006733324460909,
      0.0006733324460909,
      0.0006733324460909,
      0.0006733324460909,
      0.0006733324460909,
      0.0006733324460909,
      0.0006687541419906,
      0.0006687541419906,
      0.0006687541419906,
      0.0006687541419906,
      0.0006687541419906,
      0.0006687541419906,
      0.0006047140183441,
      0.0006047140183441,
      0.0006047140183441,
      0.0006047140183441,
      0.0006047140183441,
      0.0006047140183441,
      0.0002986880708810,
      0.0002986880708810,
      0.0002986880708810,
      0.0002986880708810,
      0.0002986880708810,
      0.0002986880708810,
      0.0007341005318668,
      0.0007341005318668,
      0.0007341005318668,
      0.0007341005318668,
      0.0007341005318668,
      0.0007341005318668,
      0.0004005272426006,
      0.0004005272426006,
      0.0004005272426006,
      0.0004005272426006,
      0.0004005272426006,
      0.0004005272426006,
      0.0000927835211689,
      0.0000927835211689,
      0.0000927835211689,
      0.0000927835211689,
      0.0000927835211689,
      0.0000927835211689,
      0.0003607110132131,
      0.0003607110132131,
      0.0003607110132131,
      0.0003607110132131,
      0.0003607110132131,
      0.0003607110132131,
      0.0007660833620237,
      0.0007660833620237,
      0.0007660833620237,
      0.0007660833620237,
      0.0007660833620237,
      0.0007660833620237,
      0.0013648263582449,
      0.0013648263582449,
      0.0013648263582449,
      0.0007545133102141,
      0.0007545133102141,
      0.0007545133102141,
      0.0007545133102141,
      0.0007545133102141,
      0.0007545133102141,
      0.0017008612364745,
      0.0017008612364745,
      0.0017008612364745,
      0.0017008612364745,
      0.0017008612364745,
      0.0017008612364745,
      0.0006022530067059,
      0.0006022530067059,
      0.0006022530067059,
      0.0005179940819032,
      0.0005179940819032,
      0.0005179940819032,
      0.0005179940819032,
      0.0005179940819032,
      0.0005179940819032,
      0.0011887438463900,
      0.0011887438463900,
      0.0011887438463900,
      0.0011887438463900,
      0.0011887438463900,
      0.0011887438463900,
      0.0016976956170924,
      0.0016976956170924,
      0.0016976956170924,
      0.0016976956170924,
      0.0016976956170924,
      0.0016976956170924,
      0.0013520441912409,
      0.0013520441912409,
      0.0013520441912409,
      0.0013520441912409,
      0.0013520441912409,
      0.0013520441912409,
      0.0010085016951456,
      0.0010085016951456,
      0.0010085016951456,
      0.0010085016951456,
      0.0010085016951456,
      0.0010085016951456,
      0.0013959405662129,
      0.0013959405662129,
      0.0013959405662129,
      0.0013959405662129,
      0.0013959405662129,
      0.0013959405662129,
      0.0011828966553896,
      0.0011828966553896,
      0.0011828966553896,
      0.0021502294328653,
      0.0021502294328653,
      0.0021502294328653,
      0.0021502294328653,
      0.0021502294328653,
      0.0021502294328653,
      0.0024919282645969,
      0.0024919282645969,
      0.0024919282645969,
      0.0024919282645969,
      0.0024919282645969,
      0.0024919282645969,
      0.0023266481378335,
      0.0023266481378335,
      0.0023266481378335,
      0.0023266481378335,
      0.0023266481378335,
      0.0023266481378335,
      0.0020639380211283,
      0.0020639380211283,
      0.0020639380211283,
      0.0020639380211283,
      0.0020639380211283,
      0.0020639380211283,
      0.0017555971341706,
      0.0017555971341706,
      0.0017555971341706,
      0.0010289108811763,
      0.0010289108811763,
      0.0010289108811763,
      0.0010289108811763,
      0.0010289108811763,
      0.0010289108811763,
      0.0019469102292144,
      0.0019469102292144,
      0.0019469102292144,
      0.0019469102292144,
      0.0019469102292144,
      0.0019469102292144,
      0.0014717340235979,
      0.0014717340235979,
      0.0014717340235979,
      0.0013853806982777,
      0.0013853806982777,
      0.0013853806982777,
      0.0013853806982777,
      0.0013853806982777,
      0.0013853806982777,
      0.0033616214492365,
      0.0033616214492365,
      0.0033616214492365,
      0.0024899294062594,
      0.0024899294062594,
      0.0024899294062594,
      0.0012400719992534,
      0.0012400719992534,
      0.0012400719992534,
      0.0012400719992534,
      0.0012400719992534,
      0.0012400719992534,
      0.0004314510904168,
      0.0004314510904168,
      0.0004314510904168,
      0.0004314510904168,
      0.0004314510904168,
      0.0004314510904168,
      0.0001604665887872,
      0.0001604665887872,
      0.0001604665887872,
      0.0001604665887872,
      0.0001604665887872,
      0.0001604665887872,
      0.0024905043053087,
      0.0024905043053087,
      0.0024905043053087,
      0.0024905043053087,
      0.0024905043053087,
      0.0024905043053087,
      0.0025097889590407,
      0.0025097889590407,
      0.0025097889590407,
      0.0025097889590407,
      0.0025097889590407,
      0.0025097889590407,
      0.0013846164091468,
      0.0013846164091468,
      0.0013846164091468,
      0.0013846164091468,
      0.0013846164091468,
      0.0013846164091468,
      0.0018507903298394,
      0.0018507903298394,
      0.0018507903298394,
      0.0018507903298394,
      0.0018507903298394,
      0.0018507903298394,
      0.0025375187295810,
      0.0025375187295810,
      0.0025375187295810,
      0.0025375187295810,
      0.0025375187295810,
      0.0025375187295810,
      0.0021338383457741,
      0.0021338383457741,
      0.0021338383457741,
      0.0021338383457741,
      0.0021338383457741,
      0.0021338383457741,
      0.0008646136519940,
      0.0008646136519940,
      0.0008646136519940,
      0.0008646136519940,
      0.0008646136519940,
      0.0008646136519940,
      0.0016123699561220,
      0.0016123699561220,
      0.0016123699561220,
      0.0016123699561220,
      0.0016123699561220,
      0.0016123699561220,
      0.0026439738564809,
      0.0026439738564809,
      0.0026439738564809,
      0.0026439738564809,
      0.0026439738564809,
      0.0026439738564809,
      0.0020097921431825,
      0.0020097921431825,
      0.0020097921431825,
      0.0020097921431825,
      0.0020097921431825,
      0.0020097921431825,
      0.0030304166036500,
      0.0030304166036500,
      0.0030304166036500,
      0.0030304166036500,
      0.0030304166036500,
      0.0030304166036500,
      0.0024604186226234,
      0.0024604186226234,
      0.0024604186226234,
      0.0025107375142988,
      0.0025107375142988,
      0.0025107375142988,
      0.0028995685292817,
      0.0028995685292817,
      0.0028995685292817,
      0.0028995685292817,
      0.0028995685292817,
      0.0028995685292817,
      0.0022232966203807,
      0.0022232966203807,
      0.0022232966203807,
      0.0031965737605550,
      0.0031965737605550,
      0.0031965737605550,
      0.0031965737605550,
      0.0031965737605550,
      0.0031965737605550,
      0.0023233063822922,
      0.0023233063822922,
      0.0023233063822922,
      0.0028273375044942,
      0.0028273375044942,
      0.0028273375044942,
      0.0028273375044942,
      0.0028273375044942,
      0.0028273375044942,
      0.0026555247084500,
      0.0026555247084500,
      0.0026555247084500,
      0.0026555247084500,
      0.0026555247084500,
      0.0026555247084500,
      0.0009866152107380,
      0.0009866152107380,
      0.0009866152107380,
      0.0009866152107380,
      0.0009866152107380,
      0.0009866152107380,
      0.0032127360250872,
      0.0032127360250872,
      0.0032127360250872,
      0.0031327939418134,
      0.0031327939418134,
      0.0031327939418134,
      0.0017777397357834,
      0.0017777397357834,
      0.0017777397357834,
      0.0017777397357834,
      0.0017777397357834,
      0.0017777397357834}
    },
    {
      370,
      {{0.0126048038307362, 0.0072924295508434, 0.0},
      {0.9801027666184203, 0.0126048038307362, 0.0},
      {0.0072924295508435, 0.9801027666184206, 0.0},
      {0.9801027666184204, 0.0072924295508434, 0.0},
      {0.0126048038307363, 0.9801027666184206, 0.0},
      {0.0072924295508431, 0.0126048038307365, 0.0},
      {0.0046054766314413, 0.0013499971821105, 0.0},
      {0.9940445261864481, 0.0046054766314413, 0.0},
      {0.0013499971821107, 0.9940445261864482, 0.0},
      {0.9940445261864482, 0.0013499971821105, 0.0},
      {0.0046054766314413, 0.9940445261864485, 0.0},
      {0.0013499971821102, 0.0046054766314416, 0.0},
      {0.0679464966039112, 0.0068591759380605, 0.0},
      {0.9251943274580282, 0.0679464966039111, 0.0},
      {0.0068591759380607, 0.9251943274580284, 0.0},
      {0.9251943274580283, 0.0068591759380605, 0.0},
      {0.0679464966039112, 0.9251943274580284, 0.0},
      {0.0068591759380603, 0.0679464966039115, 0.0},
      {0.0693380043634408, 0.0012297679959349, 0.0},
      {0.9294322276406242, 0.0693380043634408, 0.0},
      {0.0012297679959350, 0.9294322276406244, 0.0},
      {0.9294322276406243, 0.0012297679959349, 0.0},
      {0.0693380043634408, 0.9294322276406244, 0.0},
      {0.0012297679959347, 0.0693380043634411, 0.0},
      {0.4682708598709480, 0.0087801002006948, 0.0},
      {0.5229490399283573, 0.4682708598709480, 0.0},
      {0.0087801002006947, 0.5229490399283575, 0.0},
      {0.5229490399283572, 0.0087801002006948, 0.0},
      {0.4682708598709480, 0.5229490399283574, 0.0},
      {0.0087801002006947, 0.4682708598709482, 0.0},
      {0.2504951703352215, 0.0016830263611803, 0.0},
      {0.7478218033035982, 0.2504951703352215, 0.0},
      {0.0016830263611803, 0.7478218033035984, 0.0},
      {0.7478218033035983, 0.0016830263611803, 0.0},
      {0.2504951703352215, 0.7478218033035984, 0.0},
      {0.0016830263611800, 0.2504951703352218, 0.0},
      {0.0179231013398374, 0.0017579725835351, 0.0},
      {0.9803189260766274, 0.0179231013398374, 0.0},
      {0.0017579725835354, 0.9803189260766274, 0.0},
      {0.9803189260766274, 0.0017579725835351, 0.0},
      {0.0179231013398374, 0.9803189260766277, 0.0},
      {0.0017579725835349, 0.0179231013398377, 0.0},
      {0.3594568419950528, 0.0014756136531629, 0.0},
      {0.6390675443517843, 0.3594568419950528, 0.0},
      {0.0014756136531630, 0.6390675443517844, 0.0},
      {0.6390675443517843, 0.0014756136531629, 0.0},
      {0.3594568419950528, 0.6390675443517843, 0.0},
      {0.0014756136531628, 0.3594568419950530, 0.0},
      {0.2211539847849475, 0.1536631274625925, 0.0},
      {0.6251828877524599, 0.2211539847849475, 0.0},
      {0.1536631274625926, 0.6251828877524600, 0.0},
      {0.6251828877524600, 0.1536631274625925, 0.0},
      {0.2211539847849475, 0.6251828877524601, 0.0},
      {0.1536631274625923, 0.2211539847849477, 0.0},
      {0.2389731948786961, 0.2276252161372493, 0.0},
      {0.5334015889840547, 0.2389731948786962, 0.0},
      {0.2276252161372492, 0.5334015889840548, 0.0},
      {0.5334015889840547, 0.2276252161372493, 0.0},
      {0.2389731948786961, 0.5334015889840548, 0.0},
      {0.2276252161372491, 0.2389731948786963, 0.0},
      {0.1298066533002767, 0.1298066533002769, 0.0},
      {0.7403866933994464, 0.1298066533002767, 0.0},
      {0.1298066533002770, 0.7403866933994465, 0.0},
      {0.2265228859811758, 0.1142030377515119, 0.0},
      {0.6592740762673123, 0.2265228859811758, 0.0},
      {0.1142030377515119, 0.6592740762673125, 0.0},
      {0.6592740762673124, 0.1142030377515119, 0.0},
      {0.2265228859811758, 0.6592740762673125, 0.0},
      {0.1142030377515117, 0.2265228859811760, 0.0},
      {0.2395905240377902, 0.0803750594685514, 0.0},
      {0.6800344164936584, 0.2395905240377902, 0.0},
      {0.0803750594685514, 0.6800344164936586, 0.0},
      {0.6800344164936585, 0.0803750594685514, 0.0},
      {0.2395905240377902, 0.6800344164936585, 0.0},
      {0.0803750594685512, 0.2395905240377905, 0.0},
      {0.1782914029925923, 0.0306183636824900, 0.0},
      {0.7910902333249177, 0.1782914029925923, 0.0},
      {0.0306183636824901, 0.7910902333249178, 0.0},
      {0.7910902333249177, 0.0306183636824900, 0.0},
      {0.1782914029925923, 0.7910902333249178, 0.0},
      {0.0306183636824899, 0.1782914029925926, 0.0},
      {0.2063496656308830, 0.2063496656308833, 0.0},
      {0.5873006687382336, 0.2063496656308831, 0.0},
      {0.2063496656308832, 0.5873006687382338, 0.0},
      {0.1750845888069398, 0.1750845888069401, 0.0},
      {0.6498308223861201, 0.1750845888069399, 0.0},
      {0.1750845888069402, 0.6498308223861201, 0.0},
      {0.2825866891204175, 0.0269421027814231, 0.0},
      {0.6904712080981594, 0.2825866891204175, 0.0},
      {0.0269421027814231, 0.6904712080981597, 0.0},
      {0.6904712080981593, 0.0269421027814231, 0.0},
      {0.2825866891204175, 0.6904712080981594, 0.0},
      {0.0269421027814229, 0.2825866891204177, 0.0},
      {0.1372571102587296, 0.0327972135469327, 0.0},
      {0.8299456761943377, 0.1372571102587296, 0.0},
      {0.0327972135469328, 0.8299456761943379, 0.0},
      {0.8299456761943378, 0.0327972135469327, 0.0},
      {0.1372571102587296, 0.8299456761943380, 0.0},
      {0.0327972135469324, 0.1372571102587299, 0.0},
      {0.4995045768679594, 0.0009908462640811, 0.0},
      {0.4995045768679594, 0.4995045768679595, 0.0},
      {0.0009908462640811, 0.4995045768679597, 0.0},
      {0.1002544152365696, 0.0327271127336490, 0.0},
      {0.8670184720297813, 0.1002544152365696, 0.0},
      {0.0327271127336490, 0.8670184720297816, 0.0},
      {0.8670184720297813, 0.0327271127336490, 0.0},
      {0.1002544152365697, 0.8670184720297816, 0.0},
      {0.0327271127336487, 0.1002544152365700, 0.0},
      {0.0517074765599653, 0.0151601013138726, 0.0},
      {0.9331324221261621, 0.0517074765599653, 0.0},
      {0.0151601013138727, 0.9331324221261622, 0.0},
      {0.9331324221261622, 0.0151601013138726, 0.0},
      {0.0517074765599654, 0.9331324221261622, 0.0},
      {0.0151601013138723, 0.0517074765599656, 0.0},
      {0.2606148551376569, 0.1708846529529776, 0.0},
      {0.5685004919093655, 0.2606148551376569, 0.0},
      {0.1708846529529776, 0.5685004919093657, 0.0},
      {0.5685004919093656, 0.1708846529529776, 0.0},
      {0.2606148551376569, 0.5685004919093657, 0.0},
      {0.1708846529529774, 0.2606148551376571, 0.0},
      {0.2272143249599573, 0.0286632862041672, 0.0},
      {0.7441223888358753, 0.2272143249599573, 0.0},
      {0.0286632862041673, 0.7441223888358756, 0.0},
      {0.7441223888358756, 0.0286632862041672, 0.0},
      {0.2272143249599574, 0.7441223888358756, 0.0},
      {0.0286632862041670, 0.2272143249599576, 0.0},
      {0.3110404872350509, 0.0126956715659609, 0.0},
      {0.6762638411989882, 0.3110404872350509, 0.0},
      {0.0126956715659609, 0.6762638411989884, 0.0},
      {0.6762638411989881, 0.0126956715659609, 0.0},
      {0.3110404872350510, 0.6762638411989882, 0.0},
      {0.0126956715659608, 0.3110404872350512, 0.0},
      {0.0552204190605028, 0.0552204190605031, 0.0},
      {0.8895591618789941, 0.0552204190605028, 0.0},
      {0.0552204190605031, 0.8895591618789943, 0.0},
      {0.1856238762990532, 0.0879015352488811, 0.0},
      {0.7264745884520657, 0.1856238762990532, 0.0},
      {0.0879015352488812, 0.7264745884520658, 0.0},
      {0.7264745884520657, 0.0879015352488811, 0.0},
      {0.1856238762990533, 0.7264745884520658, 0.0},
      {0.0879015352488808, 0.1856238762990535, 0.0},
      {0.0658626437608178, 0.0307203461767424, 0.0},
      {0.9034170100624399, 0.0658626437608177, 0.0},
      {0.0307203461767425, 0.9034170100624400, 0.0},
      {0.9034170100624398, 0.0307203461767424, 0.0},
      {0.0658626437608178, 0.9034170100624400, 0.0},
      {0.0307203461767422, 0.0658626437608181, 0.0},
      {0.3015904700953623, 0.0028145506398764, 0.0},
      {0.6955949792647613, 0.3015904700953623, 0.0},
      {0.0028145506398765, 0.6955949792647615, 0.0},
      {0.6955949792647613, 0.0028145506398764, 0.0},
      {0.3015904700953622, 0.6955949792647615, 0.0},
      {0.0028145506398762, 0.3015904700953626, 0.0},
      {0.2925426576542363, 0.0798394163722916, 0.0},
      {0.6276179259734721, 0.2925426576542363, 0.0},
      {0.0798394163722916, 0.6276179259734722, 0.0},
      {0.6276179259734721, 0.0798394163722916, 0.0},
      {0.2925426576542364, 0.6276179259734722, 0.0},
      {0.0798394163722914, 0.2925426576542366, 0.0},
      {0.0273024847528849, 0.0133211979027671, 0.0},
      {0.9593763173443479, 0.0273024847528849, 0.0},
      {0.0133211979027673, 0.9593763173443480, 0.0},
      {0.9593763173443479, 0.0133211979027671, 0.0},
      {0.0273024847528850, 0.9593763173443480, 0.0},
      {0.0133211979027669, 0.0273024847528853, 0.0},
      {0.2991527658365939, 0.2991527658365941, 0.0},
      {0.4016944683268121, 0.2991527658365941, 0.0},
      {0.2991527658365939, 0.4016944683268122, 0.0},
      {0.3715719205775442, 0.0096021860181156, 0.0},
      {0.6188258934043402, 0.3715719205775442, 0.0},
      {0.0096021860181155, 0.6188258934043405, 0.0},
      {0.6188258934043402, 0.0096021860181156, 0.0},
      {0.3715719205775442, 0.6188258934043404, 0.0},
      {0.0096021860181154, 0.3715719205775445, 0.0},
      {0.2010073039936871, 0.0025024373852327, 0.0},
      {0.7964902586210802, 0.2010073039936871, 0.0},
      {0.0025024373852327, 0.7964902586210805, 0.0},
      {0.7964902586210802, 0.0025024373852327, 0.0},
      {0.2010073039936872, 0.7964902586210802, 0.0},
      {0.0025024373852325, 0.2010073039936874, 0.0},
      {0.2931831293722664, 0.2017308280659965, 0.0},
      {0.5050860425617371, 0.2931831293722665, 0.0},
      {0.2017308280659965, 0.5050860425617372, 0.0},
      {0.5050860425617371, 0.2017308280659965, 0.0},
      {0.2931831293722664, 0.5050860425617372, 0.0},
      {0.2017308280659964, 0.2931831293722666, 0.0},
      {0.2333318911036226, 0.0526776823006486, 0.0},
      {0.7139904265957288, 0.2333318911036226, 0.0},
      {0.0526776823006487, 0.7139904265957290, 0.0},
      {0.7139904265957289, 0.0526776823006486, 0.0},
      {0.2333318911036226, 0.7139904265957289, 0.0},
      {0.0526776823006484, 0.2333318911036229, 0.0},
      {0.1792991418970043, 0.0557691461107686, 0.0},
      {0.7649317119922271, 0.1792991418970043, 0.0},
      {0.0557691461107687, 0.7649317119922272, 0.0},
      {0.7649317119922271, 0.0557691461107686, 0.0},
      {0.1792991418970044, 0.7649317119922272, 0.0},
      {0.0557691461107684, 0.1792991418970046, 0.0},
      {0.3669822772564582, 0.2660354454870836, 0.0},
      {0.3669822772564582, 0.3669822772564583, 0.0},
      {0.2660354454870835, 0.3669822772564584, 0.0},
      {0.2934353311290930, 0.0482367013092172, 0.0},
      {0.6583279675616898, 0.2934353311290930, 0.0},
      {0.0482367013092171, 0.6583279675616901, 0.0},
      {0.6583279675616899, 0.0482367013092172, 0.0},
      {0.2934353311290930, 0.6583279675616900, 0.0},
      {0.0482367013092171, 0.2934353311290932, 0.0},
      {0.3298405962756192, 0.2336975835531222, 0.0},
      {0.4364618201712586, 0.3298405962756193, 0.0},
      {0.2336975835531220, 0.4364618201712588, 0.0},
      {0.4364618201712586, 0.2336975835531222, 0.0},
      {0.3298405962756192, 0.4364618201712587, 0.0},
      {0.2336975835531221, 0.3298405962756194, 0.0},
      {0.1381756936115276, 0.0926621506087761, 0.0},
      {0.7691621557796962, 0.1381756936115276, 0.0},
      {0.0926621506087763, 0.7691621557796964, 0.0},
      {0.7691621557796963, 0.0926621506087761, 0.0},
      {0.1381756936115277, 0.7691621557796964, 0.0},
      {0.0926621506087759, 0.1381756936115279, 0.0},
      {0.3585128657761011, 0.1734547681030724, 0.0},
      {0.4680323661208265, 0.3585128657761012, 0.0},
      {0.1734547681030723, 0.4680323661208267, 0.0},
      {0.4680323661208265, 0.1734547681030724, 0.0},
      {0.3585128657761011, 0.4680323661208267, 0.0},
      {0.1734547681030723, 0.3585128657761013, 0.0},
      {0.4203377834665488, 0.0178508936465828, 0.0},
      {0.5618113228868684, 0.4203377834665488, 0.0},
      {0.0178508936465827, 0.5618113228868687, 0.0},
      {0.5618113228868684, 0.0178508936465828, 0.0},
      {0.4203377834665488, 0.5618113228868685, 0.0},
      {0.0178508936465827, 0.4203377834665490, 0.0},
      {0.2790511330034806, 0.1199466924958472, 0.0},
      {0.6010021745006722, 0.2790511330034807, 0.0},
      {0.1199466924958471, 0.6010021745006724, 0.0},
      {0.6010021745006722, 0.1199466924958472, 0.0},
      {0.2790511330034807, 0.6010021745006723, 0.0},
      {0.1199466924958470, 0.2790511330034809, 0.0},
      {0.0934626722493711, 0.0139846172517609, 0.0},
      {0.8925527104988681, 0.0934626722493711, 0.0},
      {0.0139846172517610, 0.8925527104988682, 0.0},
      {0.8925527104988681, 0.0139846172517609, 0.0},
      {0.0934626722493711, 0.8925527104988682, 0.0},
      {0.0139846172517607, 0.0934626722493714, 0.0},
      {0.3874089876890199, 0.1193577643150775, 0.0},
      {0.4932332479959026, 0.3874089876890200, 0.0},
      {0.1193577643150775, 0.4932332479959028, 0.0},
      {0.4932332479959027, 0.1193577643150775, 0.0},
      {0.3874089876890199, 0.4932332479959027, 0.0},
      {0.1193577643150774, 0.3874089876890201, 0.0},
      {0.4071846956615202, 0.0735846549853618, 0.0},
      {0.5192306493531181, 0.4071846956615202, 0.0},
      {0.0735846549853617, 0.5192306493531184, 0.0},
      {0.5192306493531181, 0.0735846549853618, 0.0},
      {0.4071846956615202, 0.5192306493531181, 0.0},
      {0.0735846549853617, 0.4071846956615204, 0.0},
      {0.1876405833897097, 0.0127167478485408, 0.0},
      {0.7996426687617495, 0.1876405833897097, 0.0},
      {0.0127167478485408, 0.7996426687617497, 0.0},
      {0.7996426687617495, 0.0127167478485408, 0.0},
      {0.1876405833897097, 0.7996426687617497, 0.0},
      {0.0127167478485406, 0.1876405833897100, 0.0},
      {0.2649216423903805, 0.2649216423903806, 0.0},
      {0.4701567152192389, 0.2649216423903806, 0.0},
      {0.2649216423903806, 0.4701567152192390, 0.0},
      {0.1522255260382763, 0.0024403804076879, 0.0},
      {0.8453340935540358, 0.1522255260382763, 0.0},
      {0.0024403804076879, 0.8453340935540361, 0.0},
      {0.8453340935540359, 0.0024403804076879, 0.0},
      {0.1522255260382763, 0.8453340935540361, 0.0},
      {0.0024403804076877, 0.1522255260382766, 0.0},
      {0.1741420017865634, 0.1298906259772945, 0.0},
      {0.6959673722361421, 0.1741420017865634, 0.0},
      {0.1298906259772944, 0.6959673722361424, 0.0},
      {0.6959673722361421, 0.1298906259772945, 0.0},
      {0.1741420017865634, 0.6959673722361424, 0.0},
      {0.1298906259772943, 0.1741420017865636, 0.0},
      {0.3984690684139450, 0.2030618631721099, 0.0},
      {0.3984690684139450, 0.3984690684139451, 0.0},
      {0.2030618631721098, 0.3984690684139452, 0.0},
      {0.3214081551755261, 0.1438806303873045, 0.0},
      {0.5347112144371693, 0.3214081551755262, 0.0},
      {0.1438806303873045, 0.5347112144371695, 0.0},
      {0.5347112144371693, 0.1438806303873045, 0.0},
      {0.3214081551755262, 0.5347112144371694, 0.0},
      {0.1438806303873043, 0.3214081551755264, 0.0},
      {0.0888043361734265, 0.0588598092883568, 0.0},
      {0.8523358545382167, 0.0888043361734265, 0.0},
      {0.0588598092883569, 0.8523358545382169, 0.0},
      {0.8523358545382167, 0.0588598092883568, 0.0},
      {0.0888043361734265, 0.8523358545382169, 0.0},
      {0.0588598092883565, 0.0888043361734268, 0.0},
      {0.1304289155517122, 0.0602973413848094, 0.0},
      {0.8092737430634784, 0.1304289155517122, 0.0},
      {0.0602973413848094, 0.8092737430634785, 0.0},
      {0.8092737430634784, 0.0602973413848094, 0.0},
      {0.1304289155517123, 0.8092737430634785, 0.0},
      {0.0602973413848091, 0.1304289155517125, 0.0},
      {0.4526201907514942, 0.0947596184970116, 0.0},
      {0.4526201907514942, 0.4526201907514942, 0.0},
      {0.0947596184970115, 0.4526201907514944, 0.0},
      {0.1368379127222100, 0.0132254525993588, 0.0},
      {0.8499366346784312, 0.1368379127222100, 0.0},
      {0.0132254525993589, 0.8499366346784313, 0.0},
      {0.8499366346784312, 0.0132254525993588, 0.0},
      {0.1368379127222101, 0.8499366346784313, 0.0},
      {0.0132254525993586, 0.1368379127222104, 0.0},
      {0.3528533591796706, 0.0572712595019887, 0.0},
      {0.5898753813183408, 0.3528533591796706, 0.0},
      {0.0572712595019886, 0.5898753813183411, 0.0},
      {0.5898753813183407, 0.0572712595019887, 0.0},
      {0.3528533591796706, 0.5898753813183409, 0.0},
      {0.0572712595019886, 0.3528533591796708, 0.0},
      {0.2463869002171422, 0.0109336475862860, 0.0},
      {0.7426794521965718, 0.2463869002171422, 0.0},
      {0.0109336475862860, 0.7426794521965722, 0.0},
      {0.7426794521965718, 0.0109336475862860, 0.0},
      {0.2463869002171423, 0.7426794521965718, 0.0},
      {0.0109336475862858, 0.2463869002171425, 0.0},
      {0.4268610495333074, 0.1462779009333851, 0.0},
      {0.4268610495333074, 0.4268610495333075, 0.0},
      {0.1462779009333850, 0.4268610495333077, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.3435200440594713, 0.0959251650895476, 0.0},
      {0.5605547908509811, 0.3435200440594713, 0.0},
      {0.0959251650895475, 0.5605547908509814, 0.0},
      {0.5605547908509811, 0.0959251650895476, 0.0},
      {0.3435200440594713, 0.5605547908509813, 0.0},
      {0.0959251650895474, 0.3435200440594715, 0.0},
      {0.4162089636364642, 0.0396704297157568, 0.0},
      {0.5441206066477789, 0.4162089636364643, 0.0},
      {0.0396704297157567, 0.5441206066477793, 0.0},
      {0.5441206066477789, 0.0396704297157568, 0.0},
      {0.4162089636364643, 0.5441206066477792, 0.0},
      {0.0396704297157567, 0.4162089636364644, 0.0},
      {0.0945724371334944, 0.0945724371334947, 0.0},
      {0.8108551257330109, 0.0945724371334944, 0.0},
      {0.0945724371334948, 0.8108551257330111, 0.0},
      {0.4729894708688973, 0.0540210582622053, 0.0},
      {0.4729894708688974, 0.4729894708688974, 0.0},
      {0.0540210582622052, 0.4729894708688976, 0.0},
      {0.0390118106999858, 0.0027950477505294, 0.0},
      {0.9581931415494848, 0.0390118106999857, 0.0},
      {0.0027950477505296, 0.9581931415494850, 0.0},
      {0.9581931415494848, 0.0027950477505294, 0.0},
      {0.0390118106999859, 0.9581931415494850, 0.0},
      {0.0027950477505291, 0.0390118106999862, 0.0},
      {0.1071807141346061, 0.0026666116526769, 0.0},
      {0.8901526742127170, 0.1071807141346060, 0.0},
      {0.0026666116526771, 0.8901526742127170, 0.0},
      {0.8901526742127172, 0.0026666116526769, 0.0},
      {0.1071807141346060, 0.8901526742127174, 0.0},
      {0.0026666116526766, 0.1071807141346064, 0.0},
      {0.4264238197063261, 0.0025395300762578, 0.0},
      {0.5710366502174161, 0.4264238197063261, 0.0},
      {0.0025395300762577, 0.5710366502174163, 0.0},
      {0.5710366502174161, 0.0025395300762578, 0.0},
      {0.4264238197063261, 0.5710366502174162, 0.0},
      {0.0025395300762577, 0.4264238197063264, 0.0},
      {0.0340851411567991, 0.0340851411567994, 0.0},
      {0.9318297176864013, 0.0340851411567991, 0.0},
      {0.0340851411567996, 0.9318297176864015, 0.0},
      {0.3519601313638296, 0.0287313755582888, 0.0},
      {0.6193084930778816, 0.3519601313638296, 0.0},
      {0.0287313755582888, 0.6193084930778818, 0.0},
      {0.6193084930778816, 0.0287313755582888, 0.0},
      {0.3519601313638297, 0.6193084930778817, 0.0},
      {0.0287313755582886, 0.3519601313638299, 0.0},
      {0.4869989968257815, 0.0260020063484371, 0.0},
      {0.4869989968257815, 0.4869989968257816, 0.0},
      {0.0260020063484370, 0.4869989968257817, 0.0}},
      {0.0000971532123878,
      0.0000971532123878,
      0.0000971532123878,
      0.0000971532123878,
      0.0000971532123878,
      0.0000971532123878,
      0.0000296832401489,
      0.0000296832401489,
      0.0000296832401489,
      0.0000296832401489,
      0.0000296832401489,
      0.0000296832401489,
      0.0002403443537513,
      0.0002403443537513,
      0.0002403443537513,
      0.0002403443537513,
      0.0002403443537513,
      0.0002403443537513,
      0.0001150624667501,
      0.0001150624667501,
      0.0001150624667501,
      0.0001150624667501,
      0.0001150624667501,
      0.0001150624667501,
      0.0006472313585340,
      0.0006472313585340,
      0.0006472313585340,
      0.0006472313585340,
      0.0006472313585340,
      0.0006472313585340,
      0.0002302604491453,
      0.0002302604491453,
      0.0002302604491453,
      0.0002302604491453,
      0.0002302604491453,
      0.0002302604491453,
      0.0000775345040014,
      0.0000775345040014,
      0.0000775345040014,
      0.0000775345040014,
      0.0000775345040014,
      0.0000775345040014,
      0.0002632160084499,
      0.0002632160084499,
      0.0002632160084499,
      0.0002632160084499,
      0.0002632160084499,
      0.0002632160084499,
      0.0020388014004386,
      0.0020388014004386,
      0.0020388014004386,
      0.0020388014004386,
      0.0020388014004386,
      0.0020388014004386,
      0.0013636685777816,
      0.0013636685777816,
      0.0013636685777816,
      0.0013636685777816,
      0.0013636685777816,
      0.0013636685777816,
      0.0016012085751763,
      0.0016012085751763,
      0.0016012085751763,
      0.0019089191800209,
      0.0019089191800209,
      0.0019089191800209,
      0.0019089191800209,
      0.0019089191800209,
      0.0019089191800209,
      0.0016217949642443,
      0.0016217949642443,
      0.0016217949642443,
      0.0016217949642443,
      0.0016217949642443,
      0.0016217949642443,
      0.0009424828591924,
      0.0009424828591924,
      0.0009424828591924,
      0.0009424828591924,
      0.0009424828591924,
      0.0009424828591924,
      0.0023186744205610,
      0.0023186744205610,
      0.0023186744205610,
      0.0020435302327436,
      0.0020435302327436,
      0.0020435302327436,
      0.0010903611938035,
      0.0010903611938035,
      0.0010903611938035,
      0.0010903611938035,
      0.0010903611938035,
      0.0010903611938035,
      0.0009172906195805,
      0.0009172906195805,
      0.0009172906195805,
      0.0009172906195805,
      0.0009172906195805,
      0.0009172906195805,
      0.0002415567615160,
      0.0002415567615160,
      0.0002415567615160,
      0.0008136698977163,
      0.0008136698977163,
      0.0008136698977163,
      0.0008136698977163,
      0.0008136698977163,
      0.0008136698977163,
      0.0004142664105864,
      0.0004142664105864,
      0.0004142664105864,
      0.0004142664105864,
      0.0004142664105864,
      0.0004142664105864,
      0.0025537450433838,
      0.0025537450433838,
      0.0025537450433838,
      0.0025537450433838,
      0.0025537450433838,
      0.0025537450433838,
      0.0011195566794458,
      0.0011195566794458,
      0.0011195566794458,
      0.0011195566794458,
      0.0011195566794458,
      0.0011195566794458,
      0.0008200861822105,
      0.0008200861822105,
      0.0008200861822105,
      0.0008200861822105,
      0.0008200861822105,
      0.0008200861822105,
      0.0008463050702776,
      0.0008463050702776,
      0.0008463050702776,
      0.0018358356965889,
      0.0018358356965889,
      0.0018358356965889,
      0.0018358356965889,
      0.0018358356965889,
      0.0018358356965889,
      0.0007065024707997,
      0.0007065024707997,
      0.0007065024707997,
      0.0007065024707997,
      0.0007065024707997,
      0.0007065024707997,
      0.0003822210004311,
      0.0003822210004311,
      0.0003822210004311,
      0.0003822210004311,
      0.0003822210004311,
      0.0003822210004311,
      0.0019770158130266,
      0.0019770158130266,
      0.0019770158130266,
      0.0019770158130266,
      0.0019770158130266,
      0.0019770158130266,
      0.0003224167724785,
      0.0003224167724785,
      0.0003224167724785,
      0.0003224167724785,
      0.0003224167724785,
      0.0003224167724785,
      0.0034168776878490,
      0.0034168776878490,
      0.0034168776878490,
      0.0007773784512691,
      0.0007773784512691,
      0.0007773784512691,
      0.0007773784512691,
      0.0007773784512691,
      0.0007773784512691,
      0.0003132398962406,
      0.0003132398962406,
      0.0003132398962406,
      0.0003132398962406,
      0.0003132398962406,
      0.0003132398962406,
      0.0029959028750869,
      0.0029959028750869,
      0.0029959028750869,
      0.0029959028750869,
      0.0029959028750869,
      0.0029959028750869,
      0.0015292089072820,
      0.0015292089072820,
      0.0015292089072820,
      0.0015292089072820,
      0.0015292089072820,
      0.0015292089072820,
      0.0014982039520766,
      0.0014982039520766,
      0.0014982039520766,
      0.0014982039520766,
      0.0014982039520766,
      0.0014982039520766,
      0.0034462523992935,
      0.0034462523992935,
      0.0034462523992935,
      0.0016166867916586,
      0.0016166867916586,
      0.0016166867916586,
      0.0016166867916586,
      0.0016166867916586,
      0.0016166867916586,
      0.0033231821824900,
      0.0033231821824900,
      0.0033231821824900,
      0.0033231821824900,
      0.0033231821824900,
      0.0033231821824900,
      0.0015974524664545,
      0.0015974524664545,
      0.0015974524664545,
      0.0015974524664545,
      0.0015974524664545,
      0.0015974524664545,
      0.0030809958414888,
      0.0030809958414888,
      0.0030809958414888,
      0.0030809958414888,
      0.0030809958414888,
      0.0030809958414888,
      0.0010919088656449,
      0.0010919088656449,
      0.0010919088656449,
      0.0010919088656449,
      0.0010919088656449,
      0.0010919088656449,
      0.0024191143767986,
      0.0024191143767986,
      0.0024191143767986,
      0.0024191143767986,
      0.0024191143767986,
      0.0024191143767986,
      0.0005854717597540,
      0.0005854717597540,
      0.0005854717597540,
      0.0005854717597540,
      0.0005854717597540,
      0.0005854717597540,
      0.0027037484282199,
      0.0027037484282199,
      0.0027037484282199,
      0.0027037484282199,
      0.0027037484282199,
      0.0027037484282199,
      0.0022666945618745,
      0.0022666945618745,
      0.0022666945618745,
      0.0022666945618745,
      0.0022666945618745,
      0.0022666945618745,
      0.0007634222222683,
      0.0007634222222683,
      0.0007634222222683,
      0.0007634222222683,
      0.0007634222222683,
      0.0007634222222683,
      0.0031573284965164,
      0.0031573284965164,
      0.0031573284965164,
      0.0002992778793760,
      0.0002992778793760,
      0.0002992778793760,
      0.0002992778793760,
      0.0002992778793760,
      0.0002992778793760,
      0.0021106687157036,
      0.0021106687157036,
      0.0021106687157036,
      0.0021106687157036,
      0.0021106687157036,
      0.0021106687157036,
      0.0032791537500989,
      0.0032791537500989,
      0.0032791537500989,
      0.0027572811156698,
      0.0027572811156698,
      0.0027572811156698,
      0.0027572811156698,
      0.0027572811156698,
      0.0027572811156698,
      0.0011685889049553,
      0.0011685889049553,
      0.0011685889049553,
      0.0011685889049553,
      0.0011685889049553,
      0.0011685889049553,
      0.0013939944763398,
      0.0013939944763398,
      0.0013939944763398,
      0.0013939944763398,
      0.0013939944763398,
      0.0013939944763398,
      0.0025084297429841,
      0.0025084297429841,
      0.0025084297429841,
      0.0007152874976853,
      0.0007152874976853,
      0.0007152874976853,
      0.0007152874976853,
      0.0007152874976853,
      0.0007152874976853,
      0.0019860211209387,
      0.0019860211209387,
      0.0019860211209387,
      0.0019860211209387,
      0.0019860211209387,
      0.0019860211209387,
      0.0008115107146956,
      0.0008115107146956,
      0.0008115107146956,
      0.0008115107146956,
      0.0008115107146956,
      0.0008115107146956,
      0.0029412545630877,
      0.0029412545630877,
      0.0029412545630877,
      0.0034707348271925,
      0.0024154274389476,
      0.0024154274389476,
      0.0024154274389476,
      0.0024154274389476,
      0.0024154274389476,
      0.0024154274389476,
      0.0017698394080732,
      0.0017698394080732,
      0.0017698394080732,
      0.0017698394080732,
      0.0017698394080732,
      0.0017698394080732,
      0.0015685717310822,
      0.0015685717310822,
      0.0015685717310822,
      0.0020465325739098,
      0.0020465325739098,
      0.0020465325739098,
      0.0001835329947090,
      0.0001835329947090,
      0.0001835329947090,
      0.0001835329947090,
      0.0001835329947090,
      0.0001835329947090,
      0.0002854849547021,
      0.0002854849547021,
      0.0002854849547021,
      0.0002854849547021,
      0.0002854849547021,
      0.0002854849547021,
      0.0004380567695844,
      0.0004380567695844,
      0.0004380567695844,
      0.0004380567695844,
      0.0004380567695844,
      0.0004380567695844,
      0.0006256573426602,
      0.0006256573426602,
      0.0006256573426602,
      0.0015435554874621,
      0.0015435554874621,
      0.0015435554874621,
      0.0015435554874621,
      0.0015435554874621,
      0.0015435554874621,
      0.0015279028851494,
      0.0015279028851494,
      0.0015279028851494}
    },
    {
      385,
      {{0.4791627094810211, 0.0416745810379578, 0.0},
      {0.4791627094810211, 0.4791627094810212, 0.0},
      {0.0416745810379577, 0.4791627094810214, 0.0},
      {0.3979697646407098, 0.1541231491660773, 0.0},
      {0.4479070861932129, 0.3979697646407100, 0.0},
      {0.1541231491660771, 0.4479070861932131, 0.0},
      {0.4479070861932128, 0.1541231491660773, 0.0},
      {0.3979697646407099, 0.4479070861932130, 0.0},
      {0.1541231491660771, 0.3979697646407100, 0.0},
      {0.4143076079881758, 0.1131367059240234, 0.0},
      {0.4725556860878008, 0.4143076079881759, 0.0},
      {0.1131367059240233, 0.4725556860878009, 0.0},
      {0.4725556860878007, 0.1131367059240234, 0.0},
      {0.4143076079881759, 0.4725556860878009, 0.0},
      {0.1131367059240234, 0.4143076079881760, 0.0},
      {0.3297282498766237, 0.0730183277797033, 0.0},
      {0.5972534223436730, 0.3297282498766238, 0.0},
      {0.0730183277797033, 0.5972534223436732, 0.0},
      {0.5972534223436731, 0.0730183277797033, 0.0},
      {0.3297282498766237, 0.5972534223436731, 0.0},
      {0.0730183277797032, 0.3297282498766239, 0.0},
      {0.2257592423300578, 0.1098033768073636, 0.0},
      {0.6644373808625786, 0.2257592423300578, 0.0},
      {0.1098033768073635, 0.6644373808625790, 0.0},
      {0.6644373808625786, 0.1098033768073636, 0.0},
      {0.2257592423300578, 0.6644373808625788, 0.0},
      {0.1098033768073634, 0.2257592423300580, 0.0},
      {0.0486830795833199, 0.0486830795833202, 0.0},
      {0.9026338408333598, 0.0486830795833199, 0.0},
      {0.0486830795833202, 0.9026338408333601, 0.0},
      {0.4868494344525785, 0.0263011310948430, 0.0},
      {0.4868494344525786, 0.4868494344525785, 0.0},
      {0.0263011310948430, 0.4868494344525787, 0.0},
      {0.3286993475605237, 0.1028358753710730, 0.0},
      {0.5684647770684034, 0.3286993475605237, 0.0},
      {0.1028358753710730, 0.5684647770684035, 0.0},
      {0.5684647770684034, 0.1028358753710730, 0.0},
      {0.3286993475605237, 0.5684647770684035, 0.0},
      {0.1028358753710729, 0.3286993475605239, 0.0},
      {0.0570245183292809, 0.0104550153444101, 0.0},
      {0.9325204663263090, 0.0570245183292809, 0.0},
      {0.0104550153444102, 0.9325204663263091, 0.0},
      {0.9325204663263090, 0.0104550153444101, 0.0},
      {0.0570245183292810, 0.9325204663263091, 0.0},
      {0.0104550153444098, 0.0570245183292812, 0.0},
      {0.2181534906276071, 0.1475222615460564, 0.0},
      {0.6343242478263365, 0.2181534906276071, 0.0},
      {0.1475222615460565, 0.6343242478263366, 0.0},
      {0.6343242478263365, 0.1475222615460564, 0.0},
      {0.2181534906276072, 0.6343242478263366, 0.0},
      {0.1475222615460563, 0.2181534906276073, 0.0},
      {0.2023063458834636, 0.0521260968074301, 0.0},
      {0.7455675573091064, 0.2023063458834636, 0.0},
      {0.0521260968074301, 0.7455675573091065, 0.0},
      {0.7455675573091063, 0.0521260968074301, 0.0},
      {0.2023063458834636, 0.7455675573091065, 0.0},
      {0.0521260968074299, 0.2023063458834639, 0.0},
      {0.2132832098088080, 0.0800841957661689, 0.0},
      {0.7066325944250231, 0.2132832098088080, 0.0},
      {0.0800841957661689, 0.7066325944250232, 0.0},
      {0.7066325944250231, 0.0800841957661689, 0.0},
      {0.2132832098088080, 0.7066325944250232, 0.0},
      {0.0800841957661687, 0.2132832098088083, 0.0},
      {0.0082133588712220, 0.0021607803546683, 0.0},
      {0.9896258607741097, 0.0082133588712219, 0.0},
      {0.0021607803546684, 0.9896258607741099, 0.0},
      {0.9896258607741097, 0.0021607803546683, 0.0},
      {0.0082133588712220, 0.9896258607741099, 0.0},
      {0.0021607803546680, 0.0082133588712223, 0.0},
      {0.4615153616123003, 0.0113638023497273, 0.0},
      {0.5271208360379724, 0.4615153616123004, 0.0},
      {0.0113638023497272, 0.5271208360379727, 0.0},
      {0.5271208360379723, 0.0113638023497273, 0.0},
      {0.4615153616123003, 0.5271208360379724, 0.0},
      {0.0113638023497272, 0.4615153616123006, 0.0},
      {0.3546224673127057, 0.1297785236176510, 0.0},
      {0.5155990090696432, 0.3546224673127058, 0.0},
      {0.1297785236176510, 0.5155990090696434, 0.0},
      {0.5155990090696433, 0.1297785236176510, 0.0},
      {0.3546224673127058, 0.5155990090696433, 0.0},
      {0.1297785236176509, 0.3546224673127060, 0.0},
      {0.3109206435860316, 0.0489372786072454, 0.0},
      {0.6401420778067229, 0.3109206435860317, 0.0},
      {0.0489372786072454, 0.6401420778067232, 0.0},
      {0.6401420778067229, 0.0489372786072454, 0.0},
      {0.3109206435860316, 0.6401420778067232, 0.0},
      {0.0489372786072453, 0.3109206435860319, 0.0},
      {0.3049557841332307, 0.1584175590074660, 0.0},
      {0.5366266568593034, 0.3049557841332307, 0.0},
      {0.1584175590074660, 0.5366266568593036, 0.0},
      {0.5366266568593034, 0.1584175590074660, 0.0},
      {0.3049557841332306, 0.5366266568593036, 0.0},
      {0.1584175590074658, 0.3049557841332309, 0.0},
      {0.2507944375639041, 0.1755000854384091, 0.0},
      {0.5737054769976868, 0.2507944375639042, 0.0},
      {0.1755000854384091, 0.5737054769976869, 0.0},
      {0.5737054769976868, 0.1755000854384091, 0.0},
      {0.2507944375639042, 0.5737054769976868, 0.0},
      {0.1755000854384089, 0.2507944375639043, 0.0},
      {0.1962948092294495, 0.1962948092294497, 0.0},
      {0.6074103815411007, 0.1962948092294495, 0.0},
      {0.1962948092294496, 0.6074103815411009, 0.0},
      {0.1657987165670959, 0.1657987165670962, 0.0},
      {0.6684025668658078, 0.1657987165670960, 0.0},
      {0.1657987165670963, 0.6684025668658079, 0.0},
      {0.4338692218644805, 0.0531263813104963, 0.0},
      {0.5130043968250232, 0.4338692218644806, 0.0},
      {0.0531263813104962, 0.5130043968250234, 0.0},
      {0.5130043968250232, 0.0531263813104963, 0.0},
      {0.4338692218644806, 0.5130043968250232, 0.0},
      {0.0531263813104962, 0.4338692218644807, 0.0},
      {0.0770376986253196, 0.0509089773583296, 0.0},
      {0.8720533240163508, 0.0770376986253196, 0.0},
      {0.0509089773583296, 0.8720533240163509, 0.0},
      {0.8720533240163508, 0.0509089773583296, 0.0},
      {0.0770376986253197, 0.8720533240163509, 0.0},
      {0.0509089773583293, 0.0770376986253199, 0.0},
      {0.2768361860195224, 0.0111213995971377, 0.0},
      {0.7120424143833398, 0.2768361860195225, 0.0},
      {0.0111213995971378, 0.7120424143833399, 0.0},
      {0.7120424143833398, 0.0111213995971377, 0.0},
      {0.2768361860195226, 0.7120424143833398, 0.0},
      {0.0111213995971376, 0.2768361860195228, 0.0},
      {0.1736371621034214, 0.0117797536807180, 0.0},
      {0.8145830842158606, 0.1736371621034214, 0.0},
      {0.0117797536807180, 0.8145830842158608, 0.0},
      {0.8145830842158606, 0.0117797536807180, 0.0},
      {0.1736371621034215, 0.8145830842158608, 0.0},
      {0.0117797536807177, 0.1736371621034218, 0.0},
      {0.3354336886613770, 0.0118081263036934, 0.0},
      {0.6527581850349298, 0.3354336886613770, 0.0},
      {0.0118081263036934, 0.6527581850349299, 0.0},
      {0.6527581850349297, 0.0118081263036934, 0.0},
      {0.3354336886613770, 0.6527581850349298, 0.0},
      {0.0118081263036932, 0.3354336886613772, 0.0},
      {0.3750642184523966, 0.0512438839908936, 0.0},
      {0.5736918975567098, 0.3750642184523967, 0.0},
      {0.0512438839908936, 0.5736918975567099, 0.0},
      {0.5736918975567098, 0.0512438839908936, 0.0},
      {0.3750642184523967, 0.5736918975567098, 0.0},
      {0.0512438839908935, 0.3750642184523969, 0.0},
      {0.3467112861299004, 0.1825412564761328, 0.0},
      {0.4707474573939668, 0.3467112861299005, 0.0},
      {0.1825412564761328, 0.4707474573939669, 0.0},
      {0.4707474573939667, 0.1825412564761328, 0.0},
      {0.3467112861299005, 0.4707474573939668, 0.0},
      {0.1825412564761327, 0.3467112861299007, 0.0},
      {0.2559685112606641, 0.0539018714502929, 0.0},
      {0.6901296172890430, 0.2559685112606642, 0.0},
      {0.0539018714502929, 0.6901296172890432, 0.0},
      {0.6901296172890431, 0.0539018714502929, 0.0},
      {0.2559685112606642, 0.6901296172890431, 0.0},
      {0.0539018714502927, 0.2559685112606644, 0.0},
      {0.2920069207885316, 0.0273433134615537, 0.0},
      {0.6806497657499146, 0.2920069207885316, 0.0},
      {0.0273433134615538, 0.6806497657499148, 0.0},
      {0.6806497657499146, 0.0273433134615537, 0.0},
      {0.2920069207885317, 0.6806497657499148, 0.0},
      {0.0273433134615535, 0.2920069207885319, 0.0},
      {0.2364172395795756, 0.0297423818792803, 0.0},
      {0.7338403785411441, 0.2364172395795756, 0.0},
      {0.0297423818792803, 0.7338403785411444, 0.0},
      {0.7338403785411441, 0.0297423818792803, 0.0},
      {0.2364172395795756, 0.7338403785411444, 0.0},
      {0.0297423818792801, 0.2364172395795759, 0.0},
      {0.4661508981294354, 0.0021912946810574, 0.0},
      {0.5316578071895072, 0.4661508981294356, 0.0},
      {0.0021912946810573, 0.5316578071895074, 0.0},
      {0.5316578071895072, 0.0021912946810574, 0.0},
      {0.4661508981294354, 0.5316578071895073, 0.0},
      {0.0021912946810573, 0.4661508981294357, 0.0},
      {0.2234826291455856, 0.0125736658366043, 0.0},
      {0.7639437050178102, 0.2234826291455856, 0.0},
      {0.0125736658366042, 0.7639437050178104, 0.0},
      {0.7639437050178102, 0.0125736658366043, 0.0},
      {0.2234826291455856, 0.7639437050178103, 0.0},
      {0.0125736658366041, 0.2234826291455859, 0.0},
      {0.3973680138780006, 0.0115196691295125, 0.0},
      {0.5911123169924868, 0.3973680138780006, 0.0},
      {0.0115196691295125, 0.5911123169924871, 0.0},
      {0.5911123169924869, 0.0115196691295125, 0.0},
      {0.3973680138780006, 0.5911123169924870, 0.0},
      {0.0115196691295124, 0.3973680138780009, 0.0},
      {0.2856792432449899, 0.2106680376667663, 0.0},
      {0.5036527190882436, 0.2856792432449901, 0.0},
      {0.2106680376667663, 0.5036527190882438, 0.0},
      {0.5036527190882436, 0.2106680376667663, 0.0},
      {0.2856792432449899, 0.5036527190882438, 0.0},
      {0.2106680376667662, 0.2856792432449901, 0.0},
      {0.2290071177870446, 0.2290071177870448, 0.0},
      {0.5419857644259105, 0.2290071177870447, 0.0},
      {0.2290071177870448, 0.5419857644259106, 0.0},
      {0.2998981018797286, 0.2998981018797288, 0.0},
      {0.4002037962405427, 0.2998981018797287, 0.0},
      {0.2998981018797286, 0.4002037962405428, 0.0},
      {0.4009919122964068, 0.0021708502824680, 0.0},
      {0.5968372374211253, 0.4009919122964068, 0.0},
      {0.0021708502824680, 0.5968372374211254, 0.0},
      {0.5968372374211253, 0.0021708502824680, 0.0},
      {0.4009919122964067, 0.5968372374211253, 0.0},
      {0.0021708502824680, 0.4009919122964070, 0.0},
      {0.1562735104040767, 0.0557664223797979, 0.0},
      {0.7879600672161253, 0.1562735104040768, 0.0},
      {0.0557664223797980, 0.7879600672161254, 0.0},
      {0.7879600672161253, 0.0557664223797979, 0.0},
      {0.1562735104040768, 0.7879600672161254, 0.0},
      {0.0557664223797977, 0.1562735104040771, 0.0},
      {0.0477858936846818, 0.0020979082703016, 0.0},
      {0.9501161980450165, 0.0477858936846818, 0.0},
      {0.0020979082703018, 0.9501161980450167, 0.0},
      {0.9501161980450166, 0.0020979082703016, 0.0},
      {0.0477858936846819, 0.9501161980450167, 0.0},
      {0.0020979082703013, 0.0477858936846822, 0.0},
      {0.1822491599313741, 0.0290781975795925, 0.0},
      {0.7886726424890335, 0.1822491599313741, 0.0},
      {0.0290781975795925, 0.7886726424890337, 0.0},
      {0.7886726424890333, 0.0290781975795925, 0.0},
      {0.1822491599313742, 0.7886726424890335, 0.0},
      {0.0290781975795923, 0.1822491599313744, 0.0},
      {0.3660812614114254, 0.2678374771771493, 0.0},
      {0.3660812614114254, 0.3660812614114254, 0.0},
      {0.2678374771771491, 0.3660812614114256, 0.0},
      {0.0893475681919859, 0.0115829668440253, 0.0},
      {0.8990694649639888, 0.0893475681919859, 0.0},
      {0.0115829668440254, 0.8990694649639890, 0.0},
      {0.8990694649639889, 0.0115829668440253, 0.0},
      {0.0893475681919860, 0.8990694649639890, 0.0},
      {0.0115829668440250, 0.0893475681919862, 0.0},
      {0.1652198812399128, 0.0874623616502728, 0.0},
      {0.7473177571098144, 0.1652198812399128, 0.0},
      {0.0874623616502729, 0.7473177571098146, 0.0},
      {0.7473177571098144, 0.0874623616502728, 0.0},
      {0.1652198812399128, 0.7473177571098146, 0.0},
      {0.0874623616502726, 0.1652198812399130, 0.0},
      {0.0805280047391684, 0.0020785035982193, 0.0},
      {0.9173934916626123, 0.0805280047391684, 0.0},
      {0.0020785035982195, 0.9173934916626123, 0.0},
      {0.9173934916626122, 0.0020785035982193, 0.0},
      {0.0805280047391685, 0.9173934916626123, 0.0},
      {0.0020785035982190, 0.0805280047391688, 0.0},
      {0.3305415015675990, 0.2364777837447838, 0.0},
      {0.4329807146876172, 0.3305415015675991, 0.0},
      {0.2364777837447837, 0.4329807146876173, 0.0},
      {0.4329807146876172, 0.2364777837447838, 0.0},
      {0.3305415015675991, 0.4329807146876173, 0.0},
      {0.2364777837447836, 0.3305415015675992, 0.0},
      {0.1750740132522936, 0.1228090180537129, 0.0},
      {0.7021169686939935, 0.1750740132522937, 0.0},
      {0.1228090180537130, 0.7021169686939935, 0.0},
      {0.7021169686939934, 0.1228090180537129, 0.0},
      {0.1750740132522937, 0.7021169686939936, 0.0},
      {0.1228090180537126, 0.1750740132522939, 0.0},
      {0.1133949730294224, 0.0550050284444838, 0.0},
      {0.8315999985260938, 0.1133949730294223, 0.0},
      {0.0550050284444840, 0.8315999985260939, 0.0},
      {0.8315999985260938, 0.0550050284444838, 0.0},
      {0.1133949730294224, 0.8315999985260939, 0.0},
      {0.0550050284444836, 0.1133949730294226, 0.0},
      {0.0945442284064488, 0.0287935332271121, 0.0},
      {0.8766622383664391, 0.0945442284064488, 0.0},
      {0.0287935332271122, 0.8766622383664393, 0.0},
      {0.8766622383664391, 0.0287935332271121, 0.0},
      {0.0945442284064488, 0.8766622383664393, 0.0},
      {0.0287935332271118, 0.0945442284064491, 0.0},
      {0.0589589423608463, 0.0262933317918678, 0.0},
      {0.9147477258472858, 0.0589589423608463, 0.0},
      {0.0262933317918678, 0.9147477258472861, 0.0},
      {0.9147477258472859, 0.0262933317918678, 0.0},
      {0.0589589423608463, 0.9147477258472861, 0.0},
      {0.0262933317918676, 0.0589589423608466, 0.0},
      {0.3554461171023697, 0.0283073273852236, 0.0},
      {0.6162465555124068, 0.3554461171023697, 0.0},
      {0.0283073273852236, 0.6162465555124070, 0.0},
      {0.6162465555124067, 0.0283073273852236, 0.0},
      {0.3554461171023696, 0.6162465555124069, 0.0},
      {0.0283073273852235, 0.3554461171023699, 0.0},
      {0.4206278444454289, 0.0281269388071389, 0.0},
      {0.5512452167474322, 0.4206278444454289, 0.0},
      {0.0281269388071388, 0.5512452167474324, 0.0},
      {0.5512452167474322, 0.0281269388071389, 0.0},
      {0.4206278444454289, 0.5512452167474323, 0.0},
      {0.0281269388071388, 0.4206278444454292, 0.0},
      {0.0307693439520818, 0.0307693439520821, 0.0},
      {0.9384613120958362, 0.0307693439520817, 0.0},
      {0.0307693439520821, 0.9384613120958364, 0.0},
      {0.2763424771664457, 0.0021009010930693, 0.0},
      {0.7215566217404850, 0.2763424771664457, 0.0},
      {0.0021009010930694, 0.7215566217404851, 0.0},
      {0.7215566217404851, 0.0021009010930693, 0.0},
      {0.2763424771664458, 0.7215566217404851, 0.0},
      {0.0021009010930691, 0.2763424771664461, 0.0},
      {0.1358344051381373, 0.0305221004022599, 0.0},
      {0.8336434944596028, 0.1358344051381373, 0.0},
      {0.0305221004022600, 0.8336434944596030, 0.0},
      {0.8336434944596030, 0.0305221004022599, 0.0},
      {0.1358344051381374, 0.8336434944596030, 0.0},
      {0.0305221004022596, 0.1358344051381376, 0.0},
      {0.1214324449203873, 0.0881430419815303, 0.0},
      {0.7904245130980823, 0.1214324449203873, 0.0},
      {0.0881430419815304, 0.7904245130980825, 0.0},
      {0.7904245130980824, 0.0881430419815303, 0.0},
      {0.1214324449203874, 0.7904245130980825, 0.0},
      {0.0881430419815301, 0.1214324449203876, 0.0},
      {0.0824986937635583, 0.0824986937635586, 0.0},
      {0.8350026124728831, 0.0824986937635583, 0.0},
      {0.0824986937635587, 0.8350026124728833, 0.0},
      {0.1292544813329482, 0.0125808072722039, 0.0},
      {0.8581647113948480, 0.1292544813329482, 0.0},
      {0.0125808072722039, 0.8581647113948481, 0.0},
      {0.8581647113948480, 0.0125808072722039, 0.0},
      {0.1292544813329483, 0.8581647113948480, 0.0},
      {0.0125808072722036, 0.1292544813329486, 0.0},
      {0.2729393183303263, 0.0838760556578322, 0.0},
      {0.6431846260118416, 0.2729393183303264, 0.0},
      {0.0838760556578322, 0.6431846260118417, 0.0},
      {0.6431846260118415, 0.0838760556578322, 0.0},
      {0.2729393183303264, 0.6431846260118417, 0.0},
      {0.0838760556578320, 0.2729393183303265, 0.0},
      {0.4608230043475881, 0.0783539913048238, 0.0},
      {0.4608230043475881, 0.4608230043475882, 0.0},
      {0.0783539913048237, 0.4608230043475884, 0.0},
      {0.2762244316742870, 0.1247916004447048, 0.0},
      {0.5989839678810083, 0.2762244316742871, 0.0},
      {0.1247916004447048, 0.5989839678810084, 0.0},
      {0.5989839678810083, 0.1247916004447048, 0.0},
      {0.2762244316742870, 0.5989839678810084, 0.0},
      {0.1247916004447046, 0.2762244316742872, 0.0},
      {0.1665310619918011, 0.0022302492894395, 0.0},
      {0.8312386887187595, 0.1665310619918011, 0.0},
      {0.0022302492894395, 0.8312386887187597, 0.0},
      {0.8312386887187594, 0.0022302492894395, 0.0},
      {0.1665310619918010, 0.8312386887187597, 0.0},
      {0.0022302492894393, 0.1665310619918013, 0.0},
      {0.3373238618789556, 0.0022846671110010, 0.0},
      {0.6603914710100434, 0.3373238618789556, 0.0},
      {0.0022846671110011, 0.6603914710100435, 0.0},
      {0.6603914710100434, 0.0022846671110010, 0.0},
      {0.3373238618789556, 0.6603914710100435, 0.0},
      {0.0022846671110009, 0.3373238618789559, 0.0},
      {0.2666157086976967, 0.2666157086976969, 0.0},
      {0.4667685826046064, 0.2666157086976969, 0.0},
      {0.2666157086976969, 0.4667685826046065, 0.0},
      {0.1284394349475123, 0.1284394349475125, 0.0},
      {0.7431211301049753, 0.1284394349475123, 0.0},
      {0.1284394349475125, 0.7431211301049754, 0.0},
      {0.3899848376803788, 0.0831933025800659, 0.0},
      {0.5268218597395553, 0.3899848376803789, 0.0},
      {0.0831933025800658, 0.5268218597395556, 0.0},
      {0.5268218597395553, 0.0831933025800659, 0.0},
      {0.3899848376803788, 0.5268218597395554, 0.0},
      {0.0831933025800657, 0.3899848376803791, 0.0},
      {0.3333333333333333, 0.3333333333333334, 0.0},
      {0.2190574020296069, 0.0024070310918326, 0.0},
      {0.7785355668785605, 0.2190574020296069, 0.0},
      {0.0024070310918327, 0.7785355668785606, 0.0},
      {0.7785355668785605, 0.0024070310918326, 0.0},
      {0.2190574020296069, 0.7785355668785606, 0.0},
      {0.0024070310918324, 0.2190574020296072, 0.0},
      {0.1203413424430903, 0.0024170636184783, 0.0},
      {0.8772415939384313, 0.1203413424430903, 0.0},
      {0.0024170636184784, 0.8772415939384315, 0.0},
      {0.8772415939384313, 0.0024170636184783, 0.0},
      {0.1203413424430904, 0.8772415939384315, 0.0},
      {0.0024170636184781, 0.1203413424430907, 0.0},
      {0.3975115754020483, 0.2049768491959034, 0.0},
      {0.3975115754020484, 0.3975115754020485, 0.0},
      {0.2049768491959033, 0.3975115754020485, 0.0},
      {0.0321434597854268, 0.0128868718184530, 0.0},
      {0.9549696683961201, 0.0321434597854268, 0.0},
      {0.0128868718184532, 0.9549696683961202, 0.0},
      {0.9549696683961202, 0.0128868718184530, 0.0},
      {0.0321434597854269, 0.9549696683961204, 0.0},
      {0.0128868718184528, 0.0321434597854271, 0.0},
      {0.0128796984358850, 0.0128796984358853, 0.0},
      {0.9742406031282296, 0.0128796984358850, 0.0},
      {0.0128796984358854, 0.9742406031282298, 0.0},
      {0.0239783625881974, 0.0025873987120107, 0.0},
      {0.9734342386997918, 0.0239783625881974, 0.0},
      {0.0025873987120110, 0.9734342386997918, 0.0},
      {0.9734342386997918, 0.0025873987120107, 0.0},
      {0.0239783625881975, 0.9734342386997921, 0.0},
      {0.0025873987120104, 0.0239783625881978, 0.0},
      {0.0004929222424058, 0.0004929222424061, 0.0},
      {0.9990141555151880, 0.0004929222424058, 0.0},
      {0.0004929222424063, 0.9990141555151881, 0.0}},
      {0.0007531811939498,
      0.0007531811939498,
      0.0007531811939498,
      0.0022268540934665,
      0.0022268540934665,
      0.0022268540934665,
      0.0022268540934665,
      0.0022268540934665,
      0.0022268540934665,
      0.0021600047623208,
      0.0021600047623208,
      0.0021600047623208,
      0.0021600047623208,
      0.0021600047623208,
      0.0021600047623208,
      0.0015434398279694,
      0.0015434398279694,
      0.0015434398279694,
      0.0015434398279694,
      0.0015434398279694,
      0.0015434398279694,
      0.0017151129077401,
      0.0017151129077401,
      0.0017151129077401,
      0.0017151129077401,
      0.0017151129077401,
      0.0017151129077401,
      0.0006237345908226,
      0.0006237345908226,
      0.0006237345908226,
      0.0010947581916500,
      0.0010947581916500,
      0.0010947581916500,
      0.0018913809911957,
      0.0018913809911957,
      0.0018913809911957,
      0.0018913809911957,
      0.0018913809911957,
      0.0018913809911957,
      0.0003352153200313,
      0.0003352153200313,
      0.0003352153200313,
      0.0003352153200313,
      0.0003352153200313,
      0.0003352153200313,
      0.0020870573836702,
      0.0020870573836702,
      0.0020870573836702,
      0.0020870573836702,
      0.0020870573836702,
      0.0020870573836702,
      0.0012791833284880,
      0.0012791833284880,
      0.0012791833284880,
      0.0012791833284880,
      0.0012791833284880,
      0.0012791833284880,
      0.0015919934760617,
      0.0015919934760617,
      0.0015919934760617,
      0.0015919934760617,
      0.0015919934760617,
      0.0015919934760617,
      0.0000658585463977,
      0.0000658585463977,
      0.0000658585463977,
      0.0000658585463977,
      0.0000658585463977,
      0.0000658585463977,
      0.0008228361915530,
      0.0008228361915530,
      0.0008228361915530,
      0.0008228361915530,
      0.0008228361915530,
      0.0008228361915530,
      0.0022977415466293,
      0.0022977415466293,
      0.0022977415466293,
      0.0022977415466293,
      0.0022977415466293,
      0.0022977415466293,
      0.0014344300784078,
      0.0014344300784078,
      0.0014344300784078,
      0.0014344300784078,
      0.0014344300784078,
      0.0014344300784078,
      0.0023633670304601,
      0.0023633670304601,
      0.0023633670304601,
      0.0023633670304601,
      0.0023633670304601,
      0.0023633670304601,
      0.0024043658844148,
      0.0024043658844148,
      0.0024043658844148,
      0.0024043658844148,
      0.0024043658844148,
      0.0024043658844148,
      0.0022663082591709,
      0.0022663082591709,
      0.0022663082591709,
      0.0020560667756081,
      0.0020560667756081,
      0.0020560667756081,
      0.0015963835418468,
      0.0015963835418468,
      0.0015963835418468,
      0.0015963835418468,
      0.0015963835418468,
      0.0015963835418468,
      0.0008743825251447,
      0.0008743825251447,
      0.0008743825251447,
      0.0008743825251447,
      0.0008743825251447,
      0.0008743825251447,
      0.0007153351661915,
      0.0007153351661915,
      0.0007153351661915,
      0.0007153351661915,
      0.0007153351661915,
      0.0007153351661915,
      0.0006340876394677,
      0.0006340876394677,
      0.0006340876394677,
      0.0006340876394677,
      0.0006340876394677,
      0.0006340876394677,
      0.0007895168719558,
      0.0007895168719558,
      0.0007895168719558,
      0.0007895168719558,
      0.0007895168719558,
      0.0007895168719558,
      0.0016077226042161,
      0.0016077226042161,
      0.0016077226042161,
      0.0016077226042161,
      0.0016077226042161,
      0.0016077226042161,
      0.0026988529110085,
      0.0026988529110085,
      0.0026988529110085,
      0.0026988529110085,
      0.0026988529110085,
      0.0026988529110085,
      0.0015313526015054,
      0.0015313526015054,
      0.0015313526015054,
      0.0015313526015054,
      0.0015313526015054,
      0.0015313526015054,
      0.0011596092048188,
      0.0011596092048188,
      0.0011596092048188,
      0.0011596092048188,
      0.0011596092048188,
      0.0011596092048188,
      0.0011480501450886,
      0.0011480501450886,
      0.0011480501450886,
      0.0011480501450886,
      0.0011480501450886,
      0.0011480501450886,
      0.0003669051660356,
      0.0003669051660356,
      0.0003669051660356,
      0.0003669051660356,
      0.0003669051660356,
      0.0003669051660356,
      0.0007218801383704,
      0.0007218801383704,
      0.0007218801383704,
      0.0007218801383704,
      0.0007218801383704,
      0.0007218801383704,
      0.0008293331619822,
      0.0008293331619822,
      0.0008293331619822,
      0.0008293331619822,
      0.0008293331619822,
      0.0008293331619822,
      0.0027853646540566,
      0.0027853646540566,
      0.0027853646540566,
      0.0027853646540566,
      0.0027853646540566,
      0.0027853646540566,
      0.0026681419989316,
      0.0026681419989316,
      0.0026681419989316,
      0.0032378109639113,
      0.0032378109639113,
      0.0032378109639113,
      0.0003607187172323,
      0.0003607187172323,
      0.0003607187172323,
      0.0003607187172323,
      0.0003607187172323,
      0.0003607187172323,
      0.0013110639122074,
      0.0013110639122074,
      0.0013110639122074,
      0.0013110639122074,
      0.0013110639122074,
      0.0013110639122074,
      0.0001517898016862,
      0.0001517898016862,
      0.0001517898016862,
      0.0001517898016862,
      0.0001517898016862,
      0.0001517898016862,
      0.0010500435005742,
      0.0010500435005742,
      0.0010500435005742,
      0.0010500435005742,
      0.0010500435005742,
      0.0010500435005742,
      0.0032641584727157,
      0.0032641584727157,
      0.0032641584727157,
      0.0004831493093707,
      0.0004831493093707,
      0.0004831493093707,
      0.0004831493093707,
      0.0004831493093707,
      0.0004831493093707,
      0.0016023361851145,
      0.0016023361851145,
      0.0016023361851145,
      0.0016023361851145,
      0.0016023361851145,
      0.0016023361851145,
      0.0001961475226937,
      0.0001961475226937,
      0.0001961475226937,
      0.0001961475226937,
      0.0001961475226937,
      0.0001961475226937,
      0.0031287906040183,
      0.0031287906040183,
      0.0031287906040183,
      0.0031287906040183,
      0.0031287906040183,
      0.0031287906040183,
      0.0018906202570455,
      0.0018906202570455,
      0.0018906202570455,
      0.0018906202570455,
      0.0018906202570455,
      0.0018906202570455,
      0.0011531809823861,
      0.0011531809823861,
      0.0011531809823861,
      0.0011531809823861,
      0.0011531809823861,
      0.0011531809823861,
      0.0007982151686494,
      0.0007982151686494,
      0.0007982151686494,
      0.0007982151686494,
      0.0007982151686494,
      0.0007982151686494,
      0.0006244180067970,
      0.0006244180067970,
      0.0006244180067970,
      0.0006244180067970,
      0.0006244180067970,
      0.0006244180067970,
      0.0012778142041473,
      0.0012778142041473,
      0.0012778142041473,
      0.0012778142041473,
      0.0012778142041473,
      0.0012778142041473,
      0.0013391424658453,
      0.0013391424658453,
      0.0013391424658453,
      0.0013391424658453,
      0.0013391424658453,
      0.0013391424658453,
      0.0004858548500685,
      0.0004858548500685,
      0.0004858548500685,
      0.0003206950079169,
      0.0003206950079169,
      0.0003206950079169,
      0.0003206950079169,
      0.0003206950079169,
      0.0003206950079169,
      0.0009553586770292,
      0.0009553586770292,
      0.0009553586770292,
      0.0009553586770292,
      0.0009553586770292,
      0.0009553586770292,
      0.0015142824283245,
      0.0015142824283245,
      0.0015142824283245,
      0.0015142824283245,
      0.0015142824283245,
      0.0015142824283245,
      0.0012193489507377,
      0.0012193489507377,
      0.0012193489507377,
      0.0005970097372569,
      0.0005970097372569,
      0.0005970097372569,
      0.0005970097372569,
      0.0005970097372569,
      0.0005970097372569,
      0.0019948930721100,
      0.0019948930721100,
      0.0019948930721100,
      0.0019948930721100,
      0.0019948930721100,
      0.0019948930721100,
      0.0020941020358882,
      0.0020941020358882,
      0.0020941020358882,
      0.0023870799382237,
      0.0023870799382237,
      0.0023870799382237,
      0.0023870799382237,
      0.0023870799382237,
      0.0023870799382237,
      0.0002840506487653,
      0.0002840506487653,
      0.0002840506487653,
      0.0002840506487653,
      0.0002840506487653,
      0.0002840506487653,
      0.0003646448949458,
      0.0003646448949458,
      0.0003646448949458,
      0.0003646448949458,
      0.0003646448949458,
      0.0003646448949458,
      0.0029931371482376,
      0.0029931371482376,
      0.0029931371482376,
      0.0018867400782665,
      0.0018867400782665,
      0.0018867400782665,
      0.0021997780849930,
      0.0021997780849930,
      0.0021997780849930,
      0.0021997780849930,
      0.0021997780849930,
      0.0021997780849930,
      0.0033175235616036,
      0.0003394794992886,
      0.0003394794992886,
      0.0003394794992886,
      0.0003394794992886,
      0.0003394794992886,
      0.0003394794992886,
      0.0002657220567031,
      0.0002657220567031,
      0.0002657220567031,
      0.0002657220567031,
      0.0002657220567031,
      0.0002657220567031,
      0.0032408236272432,
      0.0032408236272432,
      0.0032408236272432,
      0.0003237623317067,
      0.0003237623317067,
      0.0003237623317067,
      0.0003237623317067,
      0.0003237623317067,
      0.0003237623317067,
      0.0002246733551400,
      0.0002246733551400,
      0.0002246733551400,
      0.0001306639730011,
      0.0001306639730011,
      0.0001306639730011,
      0.0001306639730011,
      0.0001306639730011,
      0.0001306639730011,
      0.0000070476100680,
      0.0000070476100680,
      0.0000070476100680}
    },
    {
      399,
      {{0.3491775041543910, 0.3016449916912181, 0.0},
      {0.3491775041543910, 0.3491775041543910, 0.0},
      {0.3016449916912180, 0.3491775041543911, 0.0},
      {0.0132182347742615, 0.0069261081189947, 0.0},
      {0.9798556571067439, 0.0132182347742614, 0.0},
      {0.0069261081189949, 0.9798556571067439, 0.0},
      {0.9798556571067438, 0.0069261081189947, 0.0},
      {0.0132182347742616, 0.9798556571067439, 0.0},
      {0.0069261081189944, 0.0132182347742619, 0.0},
      {0.1820741432615592, 0.0816358471109855, 0.0},
      {0.7362900096274552, 0.1820741432615593, 0.0},
      {0.0816358471109855, 0.7362900096274555, 0.0},
      {0.7362900096274553, 0.0816358471109855, 0.0},
      {0.1820741432615592, 0.7362900096274555, 0.0},
      {0.0816358471109853, 0.1820741432615595, 0.0},
      {0.1505809273177014, 0.0861963659887331, 0.0},
      {0.7632227066935655, 0.1505809273177014, 0.0},
      {0.0861963659887331, 0.7632227066935657, 0.0},
      {0.7632227066935656, 0.0861963659887331, 0.0},
      {0.1505809273177015, 0.7632227066935657, 0.0},
      {0.0861963659887328, 0.1505809273177016, 0.0},
      {0.3028965441179143, 0.3028965441179145, 0.0},
      {0.3942069117641712, 0.3028965441179144, 0.0},
      {0.3028965441179143, 0.3942069117641713, 0.0},
      {0.0764329826567075, 0.0502931805301914, 0.0},
      {0.8732738368131011, 0.0764329826567075, 0.0},
      {0.0502931805301914, 0.8732738368131013, 0.0},
      {0.8732738368131011, 0.0502931805301914, 0.0},
      {0.0764329826567076, 0.8732738368131013, 0.0},
      {0.0502931805301912, 0.0764329826567078, 0.0},
      {0.0138326160976293, 0.0010488989499175, 0.0},
      {0.9851184849524532, 0.0138326160976293, 0.0},
      {0.0010488989499177, 0.9851184849524532, 0.0},
      {0.9851184849524532, 0.0010488989499175, 0.0},
      {0.0138326160976294, 0.9851184849524532, 0.0},
      {0.0010488989499173, 0.0138326160976297, 0.0},
      {0.0267452272430761, 0.0100204975653523, 0.0},
      {0.9632342751915716, 0.0267452272430761, 0.0},
      {0.0100204975653523, 0.9632342751915718, 0.0},
      {0.9632342751915717, 0.0100204975653523, 0.0},
      {0.0267452272430762, 0.9632342751915718, 0.0},
      {0.0100204975653520, 0.0267452272430765, 0.0},
      {0.0505319024623014, 0.0505319024623017, 0.0},
      {0.8989361950753969, 0.0505319024623013, 0.0},
      {0.0505319024623018, 0.8989361950753971, 0.0},
      {0.3711902367525145, 0.2036742530451799, 0.0},
      {0.4251355102023056, 0.3711902367525146, 0.0},
      {0.2036742530451798, 0.4251355102023057, 0.0},
      {0.4251355102023055, 0.2036742530451799, 0.0},
      {0.3711902367525146, 0.4251355102023057, 0.0},
      {0.2036742530451798, 0.3711902367525147, 0.0},
      {0.3688949727921184, 0.1593266838739337, 0.0},
      {0.4717783433339480, 0.3688949727921184, 0.0},
      {0.1593266838739336, 0.4717783433339481, 0.0},
      {0.4717783433339479, 0.1593266838739337, 0.0},
      {0.3688949727921184, 0.4717783433339481, 0.0},
      {0.1593266838739335, 0.3688949727921186, 0.0},
      {0.4215093928787094, 0.1569812142425812, 0.0},
      {0.4215093928787095, 0.4215093928787095, 0.0},
      {0.1569812142425811, 0.4215093928787095, 0.0},
      {0.1070745798931199, 0.0533687025646960, 0.0},
      {0.8395567175421841, 0.1070745798931199, 0.0},
      {0.0533687025646961, 0.8395567175421843, 0.0},
      {0.8395567175421841, 0.0533687025646960, 0.0},
      {0.1070745798931200, 0.8395567175421843, 0.0},
      {0.0533687025646957, 0.1070745798931203, 0.0},
      {0.1152179121673602, 0.0852960644923164, 0.0},
      {0.7994860233403234, 0.1152179121673602, 0.0},
      {0.0852960644923165, 0.7994860233403235, 0.0},
      {0.7994860233403234, 0.0852960644923164, 0.0},
      {0.1152179121673603, 0.7994860233403235, 0.0},
      {0.0852960644923162, 0.1152179121673605, 0.0},
      {0.4746270243429588, 0.0507459513140824, 0.0},
      {0.4746270243429588, 0.4746270243429588, 0.0},
      {0.0507459513140823, 0.4746270243429590, 0.0},
      {0.4179628539698995, 0.0514822979290668, 0.0},
      {0.5305548481010338, 0.4179628539698995, 0.0},
      {0.0514822979290668, 0.5305548481010339, 0.0},
      {0.5305548481010337, 0.0514822979290668, 0.0},
      {0.4179628539698995, 0.5305548481010338, 0.0},
      {0.0514822979290667, 0.4179628539698997, 0.0},
      {0.0795728420692438, 0.0795728420692441, 0.0},
      {0.8408543158615120, 0.0795728420692439, 0.0},
      {0.0795728420692443, 0.8408543158615122, 0.0},
      {0.0462992599159319, 0.0275536314394735, 0.0},
      {0.9261471086445946, 0.0462992599159318, 0.0},
      {0.0275536314394737, 0.9261471086445947, 0.0},
      {0.9261471086445946, 0.0275536314394735, 0.0},
      {0.0462992599159320, 0.9261471086445947, 0.0},
      {0.0275536314394733, 0.0462992599159322, 0.0},
      {0.3250299373159994, 0.1187020271689408, 0.0},
      {0.5562680355150597, 0.3250299373159995, 0.0},
      {0.1187020271689408, 0.5562680355150599, 0.0},
      {0.5562680355150598, 0.1187020271689408, 0.0},
      {0.3250299373159994, 0.5562680355150599, 0.0},
      {0.1187020271689407, 0.3250299373159996, 0.0},
      {0.2230214568534538, 0.0850615771696930, 0.0},
      {0.6919169659768531, 0.2230214568534538, 0.0},
      {0.0850615771696930, 0.6919169659768534, 0.0},
      {0.6919169659768531, 0.0850615771696930, 0.0},
      {0.2230214568534538, 0.6919169659768533, 0.0},
      {0.0850615771696929, 0.2230214568534540, 0.0},
      {0.3822400802517831, 0.1181275084411924, 0.0},
      {0.4996324113070245, 0.3822400802517831, 0.0},
      {0.1181275084411924, 0.4996324113070247, 0.0},
      {0.4996324113070245, 0.1181275084411924, 0.0},
      {0.3822400802517831, 0.4996324113070246, 0.0},
      {0.1181275084411923, 0.3822400802517833, 0.0},
      {0.3186810168040575, 0.2527083506527417, 0.0},
      {0.4286106325432008, 0.3186810168040576, 0.0},
      {0.2527083506527416, 0.4286106325432010, 0.0},
      {0.4286106325432009, 0.2527083506527417, 0.0},
      {0.3186810168040575, 0.4286106325432010, 0.0},
      {0.2527083506527416, 0.3186810168040576, 0.0},
      {0.3161024291972920, 0.2057459412081716, 0.0},
      {0.4781516295945364, 0.3161024291972920, 0.0},
      {0.2057459412081715, 0.4781516295945366, 0.0},
      {0.4781516295945364, 0.2057459412081716, 0.0},
      {0.3161024291972920, 0.4781516295945365, 0.0},
      {0.2057459412081715, 0.3161024291972921, 0.0},
      {0.0236802291430403, 0.0236802291430406, 0.0},
      {0.9526395417139191, 0.0236802291430402, 0.0},
      {0.0236802291430406, 0.9526395417139194, 0.0},
      {0.2017143602625017, 0.0286457371054149, 0.0},
      {0.7696399026320835, 0.2017143602625017, 0.0},
      {0.0286457371054150, 0.7696399026320836, 0.0},
      {0.7696399026320834, 0.0286457371054149, 0.0},
      {0.2017143602625018, 0.7696399026320834, 0.0},
      {0.0286457371054148, 0.2017143602625020, 0.0},
      {0.3040953744572968, 0.0287008250916218, 0.0},
      {0.6672038004510813, 0.3040953744572970, 0.0},
      {0.0287008250916218, 0.6672038004510815, 0.0},
      {0.6672038004510813, 0.0287008250916218, 0.0},
      {0.3040953744572970, 0.6672038004510813, 0.0},
      {0.0287008250916217, 0.3040953744572972, 0.0},
      {0.2509406000944529, 0.0290822358487935, 0.0},
      {0.7199771640567535, 0.2509406000944530, 0.0},
      {0.0290822358487937, 0.7199771640567536, 0.0},
      {0.7199771640567536, 0.0290822358487935, 0.0},
      {0.2509406000944530, 0.7199771640567536, 0.0},
      {0.0290822358487933, 0.2509406000944532, 0.0},
      {0.2701136854877469, 0.1217422359406605, 0.0},
      {0.6081440785715926, 0.2701136854877469, 0.0},
      {0.1217422359406605, 0.6081440785715928, 0.0},
      {0.6081440785715926, 0.1217422359406605, 0.0},
      {0.2701136854877470, 0.6081440785715927, 0.0},
      {0.1217422359406603, 0.2701136854877472, 0.0},
      {0.4230078867718429, 0.0279469184369985, 0.0},
      {0.5490451947911587, 0.4230078867718430, 0.0},
      {0.0279469184369985, 0.5490451947911588, 0.0},
      {0.5490451947911587, 0.0279469184369985, 0.0},
      {0.4230078867718429, 0.5490451947911587, 0.0},
      {0.0279469184369984, 0.4230078867718431, 0.0},
      {0.2188381084082620, 0.1221400667583725, 0.0},
      {0.6590218248333656, 0.2188381084082620, 0.0},
      {0.1221400667583725, 0.6590218248333657, 0.0},
      {0.6590218248333655, 0.1221400667583725, 0.0},
      {0.2188381084082620, 0.6590218248333657, 0.0},
      {0.1221400667583722, 0.2188381084082622, 0.0},
      {0.2630564357703722, 0.2630564357703725, 0.0},
      {0.4738871284592553, 0.2630564357703724, 0.0},
      {0.2630564357703724, 0.4738871284592555, 0.0},
      {0.3155582824301563, 0.1604129850910247, 0.0},
      {0.5240287324788191, 0.3155582824301563, 0.0},
      {0.1604129850910246, 0.5240287324788192, 0.0},
      {0.5240287324788190, 0.1604129850910247, 0.0},
      {0.3155582824301562, 0.5240287324788192, 0.0},
      {0.1604129850910245, 0.3155582824301565, 0.0},
      {0.3599942050767867, 0.0520384051030056, 0.0},
      {0.5879673898202077, 0.3599942050767868, 0.0},
      {0.0520384051030056, 0.5879673898202079, 0.0},
      {0.5879673898202077, 0.0520384051030056, 0.0},
      {0.3599942050767868, 0.5879673898202077, 0.0},
      {0.0520384051030055, 0.3599942050767870, 0.0},
      {0.0761492805627097, 0.0280080973264348, 0.0},
      {0.8958426221108554, 0.0761492805627097, 0.0},
      {0.0280080973264349, 0.8958426221108556, 0.0},
      {0.8958426221108555, 0.0280080973264348, 0.0},
      {0.0761492805627098, 0.8958426221108556, 0.0},
      {0.0280080973264346, 0.0761492805627100, 0.0},
      {0.3616871250428565, 0.0283641992932530, 0.0},
      {0.6099486756638904, 0.3616871250428565, 0.0},
      {0.0283641992932530, 0.6099486756638907, 0.0},
      {0.6099486756638904, 0.0283641992932530, 0.0},
      {0.3616871250428564, 0.6099486756638907, 0.0},
      {0.0283641992932528, 0.3616871250428568, 0.0},
      {0.4420693972638305, 0.1158612054723389, 0.0},
      {0.4420693972638306, 0.4420693972638307, 0.0},
      {0.1158612054723388, 0.4420693972638308, 0.0},
      {0.4861601074836898, 0.0276797850326205, 0.0},
      {0.4861601074836898, 0.4861601074836898, 0.0},
      {0.0276797850326204, 0.4861601074836900, 0.0},
      {0.3969570429794791, 0.0818149550216045, 0.0},
      {0.5212280019989164, 0.3969570429794792, 0.0},
      {0.0818149550216045, 0.5212280019989165, 0.0},
      {0.5212280019989164, 0.0818149550216045, 0.0},
      {0.3969570429794791, 0.5212280019989165, 0.0},
      {0.0818149550216044, 0.3969570429794793, 0.0},
      {0.2635821046503857, 0.2121338963165612, 0.0},
      {0.5242839990330530, 0.2635821046503858, 0.0},
      {0.2121338963165611, 0.5242839990330532, 0.0},
      {0.5242839990330530, 0.2121338963165612, 0.0},
      {0.2635821046503858, 0.5242839990330532, 0.0},
      {0.2121338963165610, 0.2635821046503860, 0.0},
      {0.1925350542122712, 0.0524452042229163, 0.0},
      {0.7550197415648126, 0.1925350542122712, 0.0},
      {0.0524452042229163, 0.7550197415648128, 0.0},
      {0.7550197415648127, 0.0524452042229163, 0.0},
      {0.1925350542122711, 0.7550197415648128, 0.0},
      {0.0524452042229161, 0.1925350542122714, 0.0},
      {0.1550537980019329, 0.0291776764327326, 0.0},
      {0.8157685255653345, 0.1550537980019329, 0.0},
      {0.0291776764327327, 0.8157685255653346, 0.0},
      {0.8157685255653344, 0.0291776764327326, 0.0},
      {0.1550537980019330, 0.8157685255653346, 0.0},
      {0.0291776764327324, 0.1550537980019333, 0.0},
      {0.0808139183670610, 0.0116752954894915, 0.0},
      {0.9075107861434476, 0.0808139183670610, 0.0},
      {0.0116752954894916, 0.9075107861434477, 0.0},
      {0.9075107861434475, 0.0116752954894915, 0.0},
      {0.0808139183670610, 0.9075107861434477, 0.0},
      {0.0116752954894912, 0.0808139183670613, 0.0},
      {0.1621461225224503, 0.1621461225224505, 0.0},
      {0.6757077549550992, 0.1621461225224504, 0.0},
      {0.1621461225224506, 0.6757077549550993, 0.0},
      {0.0494903005695828, 0.0113529551848956, 0.0},
      {0.9391567442455215, 0.0494903005695827, 0.0},
      {0.0113529551848958, 0.9391567442455218, 0.0},
      {0.9391567442455215, 0.0113529551848956, 0.0},
      {0.0494903005695828, 0.9391567442455218, 0.0},
      {0.0113529551848954, 0.0494903005695831, 0.0},
      {0.3736911818051820, 0.2526176363896360, 0.0},
      {0.3736911818051820, 0.3736911818051821, 0.0},
      {0.2526176363896359, 0.3736911818051821, 0.0},
      {0.3013661191750608, 0.0527412086548005, 0.0},
      {0.6458926721701388, 0.3013661191750608, 0.0},
      {0.0527412086548005, 0.6458926721701389, 0.0},
      {0.6458926721701387, 0.0527412086548005, 0.0},
      {0.3013661191750608, 0.6458926721701388, 0.0},
      {0.0527412086548003, 0.3013661191750611, 0.0},
      {0.2125512689048137, 0.2125512689048139, 0.0},
      {0.5748974621903724, 0.2125512689048138, 0.0},
      {0.2125512689048139, 0.5748974621903725, 0.0},
      {0.1455590941243064, 0.0539363844971338, 0.0},
      {0.8005045213785598, 0.1455590941243064, 0.0},
      {0.0539363844971339, 0.8005045213785600, 0.0},
      {0.8005045213785598, 0.0539363844971338, 0.0},
      {0.1455590941243065, 0.8005045213785600, 0.0},
      {0.0539363844971336, 0.1455590941243067, 0.0},
      {0.0339239311459584, 0.0020268965838007, 0.0},
      {0.9640491722702409, 0.0339239311459583, 0.0},
      {0.0020268965838009, 0.9640491722702410, 0.0},
      {0.9640491722702410, 0.0020268965838007, 0.0},
      {0.0339239311459585, 0.9640491722702410, 0.0},
      {0.0020268965838004, 0.0339239311459588, 0.0},
      {0.4598321741495011, 0.0803356517009978, 0.0},
      {0.4598321741495011, 0.4598321741495012, 0.0},
      {0.0803356517009977, 0.4598321741495013, 0.0},
      {0.4609203982983521, 0.0113741139787509, 0.0},
      {0.5277054877228970, 0.4609203982983522, 0.0},
      {0.0113741139787508, 0.5277054877228973, 0.0},
      {0.5277054877228969, 0.0113741139787509, 0.0},
      {0.4609203982983522, 0.5277054877228970, 0.0},
      {0.0113741139787508, 0.4609203982983524, 0.0},
      {0.2096506369927426, 0.1647197357980591, 0.0},
      {0.6256296272091983, 0.2096506369927426, 0.0},
      {0.1647197357980590, 0.6256296272091985, 0.0},
      {0.6256296272091983, 0.1647197357980591, 0.0},
      {0.2096506369927426, 0.6256296272091985, 0.0},
      {0.1647197357980589, 0.2096506369927428, 0.0},
      {0.3356188307276363, 0.0824852827778116, 0.0},
      {0.5818958864945520, 0.3356188307276364, 0.0},
      {0.0824852827778115, 0.5818958864945523, 0.0},
      {0.5818958864945520, 0.0824852827778116, 0.0},
      {0.3356188307276363, 0.5818958864945523, 0.0},
      {0.0824852827778114, 0.3356188307276366, 0.0},
      {0.1191983560948026, 0.0118660258094384, 0.0},
      {0.8689356180957589, 0.1191983560948026, 0.0},
      {0.0118660258094385, 0.8689356180957590, 0.0},
      {0.8689356180957590, 0.0118660258094384, 0.0},
      {0.1191983560948028, 0.8689356180957590, 0.0},
      {0.0118660258094381, 0.1191983560948030, 0.0},
      {0.1127517427175889, 0.0289885801232180, 0.0},
      {0.8582596771591932, 0.1127517427175889, 0.0},
      {0.0289885801232180, 0.8582596771591934, 0.0},
      {0.8582596771591932, 0.0289885801232180, 0.0},
      {0.1127517427175889, 0.8582596771591934, 0.0},
      {0.0289885801232177, 0.1127517427175892, 0.0},
      {0.2446044849937105, 0.0536092519706684, 0.0},
      {0.7017862630356211, 0.2446044849937105, 0.0},
      {0.0536092519706684, 0.7017862630356213, 0.0},
      {0.7017862630356212, 0.0536092519706684, 0.0},
      {0.2446044849937105, 0.7017862630356212, 0.0},
      {0.0536092519706682, 0.2446044849937108, 0.0},
      {0.2613278952130154, 0.1649672925863881, 0.0},
      {0.5737048122005965, 0.2613278952130155, 0.0},
      {0.1649672925863881, 0.5737048122005965, 0.0},
      {0.5737048122005964, 0.1649672925863881, 0.0},
      {0.2613278952130155, 0.5737048122005965, 0.0},
      {0.1649672925863880, 0.2613278952130156, 0.0},
      {0.2767102223588470, 0.0841334586353880, 0.0},
      {0.6391563190057649, 0.2767102223588471, 0.0},
      {0.0841334586353880, 0.6391563190057651, 0.0},
      {0.6391563190057650, 0.0841334586353880, 0.0},
      {0.2767102223588471, 0.6391563190057651, 0.0},
      {0.0841334586353878, 0.2767102223588473, 0.0},
      {0.3310576983055140, 0.0117181407852650, 0.0},
      {0.6572241609092210, 0.3310576983055140, 0.0},
      {0.0117181407852651, 0.6572241609092212, 0.0},
      {0.6572241609092210, 0.0117181407852650, 0.0},
      {0.3310576983055140, 0.6572241609092211, 0.0},
      {0.0117181407852649, 0.3310576983055142, 0.0},
      {0.1711659603535230, 0.1198251592753853, 0.0},
      {0.7090088803710917, 0.1711659603535231, 0.0},
      {0.1198251592753853, 0.7090088803710918, 0.0},
      {0.7090088803710918, 0.1198251592753853, 0.0},
      {0.1711659603535231, 0.7090088803710918, 0.0},
      {0.1198251592753851, 0.1711659603535233, 0.0},
      {0.2148380935404156, 0.0118084740509012, 0.0},
      {0.7733534324086832, 0.2148380935404156, 0.0},
      {0.0118084740509012, 0.7733534324086834, 0.0},
      {0.7733534324086832, 0.0118084740509012, 0.0},
      {0.2148380935404157, 0.7733534324086833, 0.0},
      {0.0118084740509010, 0.2148380935404159, 0.0},
      {0.1640325957882373, 0.0118982357159540, 0.0},
      {0.8240691684958087, 0.1640325957882373, 0.0},
      {0.0118982357159542, 0.8240691684958088, 0.0},
      {0.8240691684958087, 0.0118982357159540, 0.0},
      {0.1640325957882374, 0.8240691684958088, 0.0},
      {0.0118982357159538, 0.1640325957882376, 0.0},
      {0.2705995205192744, 0.0118476002079960, 0.0},
      {0.7175528792727295, 0.2705995205192745, 0.0},
      {0.0118476002079961, 0.7175528792727297, 0.0},
      {0.7175528792727296, 0.0118476002079960, 0.0},
      {0.2705995205192745, 0.7175528792727297, 0.0},
      {0.0118476002079958, 0.2705995205192748, 0.0},
      {0.3948815299054212, 0.0115294467270000, 0.0},
      {0.5935890233675787, 0.3948815299054213, 0.0},
      {0.0115294467270000, 0.5935890233675789, 0.0},
      {0.5935890233675788, 0.0115294467270000, 0.0},
      {0.3948815299054213, 0.5935890233675788, 0.0},
      {0.0115294467269999, 0.3948815299054215, 0.0},
      {0.4989185291486734, 0.0021629417026532, 0.0},
      {0.4989185291486734, 0.4989185291486735, 0.0},
      {0.0021629417026531, 0.4989185291486736, 0.0},
      {0.0976801440609630, 0.0022497058285755, 0.0},
      {0.9000701501104615, 0.0976801440609630, 0.0},
      {0.0022497058285756, 0.9000701501104617, 0.0},
      {0.9000701501104615, 0.0022497058285755, 0.0},
      {0.0976801440609631, 0.9000701501104617, 0.0},
      {0.0022497058285752, 0.0976801440609633, 0.0},
      {0.4316631884764961, 0.0021780831191386, 0.0},
      {0.5661587284043652, 0.4316631884764962, 0.0},
      {0.0021780831191386, 0.5661587284043654, 0.0},
      {0.5661587284043652, 0.0021780831191386, 0.0},
      {0.4316631884764962, 0.5661587284043653, 0.0},
      {0.0021780831191386, 0.4316631884764964, 0.0},
      {0.1403238605725803, 0.0022594310974726, 0.0},
      {0.8574167083299471, 0.1403238605725803, 0.0},
      {0.0022594310974726, 0.8574167083299473, 0.0},
      {0.8574167083299471, 0.0022594310974726, 0.0},
      {0.1403238605725803, 0.8574167083299473, 0.0},
      {0.0022594310974724, 0.1403238605725806, 0.0},
      {0.1249745067489113, 0.1249745067489116, 0.0},
      {0.7500509865021772, 0.1249745067489113, 0.0},
      {0.1249745067489116, 0.7500509865021772, 0.0},
      {0.2437969001075017, 0.0022489510449106, 0.0},
      {0.7539541488475877, 0.2437969001075017, 0.0},
      {0.0022489510449106, 0.7539541488475879, 0.0},
      {0.7539541488475877, 0.0022489510449106, 0.0},
      {0.2437969001075018, 0.7539541488475878, 0.0},
      {0.0022489510449104, 0.2437969001075020, 0.0},
      {0.0619939934379152, 0.0022107829440680, 0.0},
      {0.9357952236180167, 0.0619939934379152, 0.0},
      {0.0022107829440681, 0.9357952236180170, 0.0},
      {0.9357952236180167, 0.0022107829440680, 0.0},
      {0.0619939934379153, 0.9357952236180170, 0.0},
      {0.0022107829440677, 0.0619939934379155, 0.0},
      {0.3029504928419904, 0.0022499314295439, 0.0},
      {0.6947995757284657, 0.3029504928419904, 0.0},
      {0.0022499314295440, 0.6947995757284657, 0.0},
      {0.6947995757284656, 0.0022499314295439, 0.0},
      {0.3029504928419905, 0.6947995757284657, 0.0},
      {0.0022499314295438, 0.3029504928419907, 0.0},
      {0.1892661005029103, 0.0022640392620408, 0.0},
      {0.8084698602350489, 0.1892661005029103, 0.0},
      {0.0022640392620409, 0.8084698602350490, 0.0},
      {0.8084698602350490, 0.0022640392620408, 0.0},
      {0.1892661005029104, 0.8084698602350490, 0.0},
      {0.0022640392620406, 0.1892661005029106, 0.0},
      {0.3659190279368091, 0.0022135119210076, 0.0},
      {0.6318674601421832, 0.3659190279368091, 0.0},
      {0.0022135119210076, 0.6318674601421835, 0.0},
      {0.6318674601421833, 0.0022135119210076, 0.0},
      {0.3659190279368091, 0.6318674601421833, 0.0},
      {0.0022135119210075, 0.3659190279368094, 0.0},
      {0.0026050966947360, 0.0026050966947363, 0.0},
      {0.9947898066105276, 0.0026050966947360, 0.0},
      {0.0026050966947365, 0.9947898066105276, 0.0}},
      {0.0021573284791668,
      0.0021573284791668,
      0.0021573284791668,
      0.0001063910032624,
      0.0001063910032624,
      0.0001063910032624,
      0.0001063910032624,
      0.0001063910032624,
      0.0001063910032624,
      0.0011257413724643,
      0.0011257413724643,
      0.0011257413724643,
      0.0011257413724643,
      0.0011257413724643,
      0.0011257413724643,
      0.0011325350577440,
      0.0011325350577440,
      0.0011325350577440,
      0.0011325350577440,
      0.0011325350577440,
      0.0011325350577440,
      0.0023495856156259,
      0.0023495856156259,
      0.0023495856156259,
      0.0007013647873401,
      0.0007013647873401,
      0.0007013647873401,
      0.0007013647873401,
      0.0007013647873401,
      0.0007013647873401,
      0.0000489049584064,
      0.0000489049584064,
      0.0000489049584064,
      0.0000489049584064,
      0.0000489049584064,
      0.0000489049584064,
      0.0002019314934017,
      0.0002019314934017,
      0.0002019314934017,
      0.0002019314934017,
      0.0002019314934017,
      0.0002019314934017,
      0.0006182462048526,
      0.0006182462048526,
      0.0006182462048526,
      0.0025768276603066,
      0.0025768276603066,
      0.0025768276603066,
      0.0025768276603066,
      0.0025768276603066,
      0.0025768276603066,
      0.0022547355281920,
      0.0022547355281920,
      0.0022547355281920,
      0.0022547355281920,
      0.0022547355281920,
      0.0022547355281920,
      0.0022579986890965,
      0.0022579986890965,
      0.0022579986890965,
      0.0009572944836030,
      0.0009572944836030,
      0.0009572944836030,
      0.0009572944836030,
      0.0009572944836030,
      0.0009572944836030,
      0.0013031325914997,
      0.0013031325914997,
      0.0013031325914997,
      0.0013031325914997,
      0.0013031325914997,
      0.0013031325914997,
      0.0014722405331371,
      0.0014722405331371,
      0.0014722405331371,
      0.0015380786371433,
      0.0015380786371433,
      0.0015380786371433,
      0.0015380786371433,
      0.0015380786371433,
      0.0015380786371433,
      0.0010415012431974,
      0.0010415012431974,
      0.0010415012431974,
      0.0005164061562559,
      0.0005164061562559,
      0.0005164061562559,
      0.0005164061562559,
      0.0005164061562559,
      0.0005164061562559,
      0.0021771806152064,
      0.0021771806152064,
      0.0021771806152064,
      0.0021771806152064,
      0.0021771806152064,
      0.0021771806152064,
      0.0016960439589115,
      0.0016960439589115,
      0.0016960439589115,
      0.0016960439589115,
      0.0016960439589115,
      0.0016960439589115,
      0.0022665783523017,
      0.0022665783523017,
      0.0022665783523017,
      0.0022665783523017,
      0.0022665783523017,
      0.0022665783523017,
      0.0026848596537823,
      0.0026848596537823,
      0.0026848596537823,
      0.0026848596537823,
      0.0026848596537823,
      0.0026848596537823,
      0.0025056259451339,
      0.0025056259451339,
      0.0025056259451339,
      0.0025056259451339,
      0.0025056259451339,
      0.0025056259451339,
      0.0003315039316334,
      0.0003315039316334,
      0.0003315039316334,
      0.0009766971373945,
      0.0009766971373945,
      0.0009766971373945,
      0.0009766971373945,
      0.0009766971373945,
      0.0009766971373945,
      0.0011350766457997,
      0.0011350766457997,
      0.0011350766457997,
      0.0011350766457997,
      0.0011350766457997,
      0.0011350766457997,
      0.0010693364915243,
      0.0010693364915243,
      0.0010693364915243,
      0.0010693364915243,
      0.0010693364915243,
      0.0010693364915243,
      0.0021390670916362,
      0.0021390670916362,
      0.0021390670916362,
      0.0021390670916362,
      0.0021390670916362,
      0.0021390670916362,
      0.0012535096722214,
      0.0012535096722214,
      0.0012535096722214,
      0.0012535096722214,
      0.0012535096722214,
      0.0012535096722214,
      0.0019868655501992,
      0.0019868655501992,
      0.0019868655501992,
      0.0019868655501992,
      0.0019868655501992,
      0.0019868655501992,
      0.0027991272163015,
      0.0027991272163015,
      0.0027991272163015,
      0.0023702068431689,
      0.0023702068431689,
      0.0023702068431689,
      0.0023702068431689,
      0.0023702068431689,
      0.0023702068431689,
      0.0015804510788923,
      0.0015804510788923,
      0.0015804510788923,
      0.0015804510788923,
      0.0015804510788923,
      0.0015804510788923,
      0.0006532465158918,
      0.0006532465158918,
      0.0006532465158918,
      0.0006532465158918,
      0.0006532465158918,
      0.0006532465158918,
      0.0012112082852970,
      0.0012112082852970,
      0.0012112082852970,
      0.0012112082852970,
      0.0012112082852970,
      0.0012112082852970,
      0.0022759072567849,
      0.0022759072567849,
      0.0022759072567849,
      0.0012536406752904,
      0.0012536406752904,
      0.0012536406752904,
      0.0020668792951942,
      0.0020668792951942,
      0.0020668792951942,
      0.0020668792951942,
      0.0020668792951942,
      0.0020668792951942,
      0.0025525665906056,
      0.0025525665906056,
      0.0025525665906056,
      0.0025525665906056,
      0.0025525665906056,
      0.0025525665906056,
      0.0013488116669917,
      0.0013488116669917,
      0.0013488116669917,
      0.0013488116669917,
      0.0013488116669917,
      0.0013488116669917,
      0.0009404821989536,
      0.0009404821989536,
      0.0009404821989536,
      0.0009404821989536,
      0.0009404821989536,
      0.0009404821989536,
      0.0004571095051372,
      0.0004571095051372,
      0.0004571095051372,
      0.0004571095051372,
      0.0004571095051372,
      0.0004571095051372,
      0.0020322149629336,
      0.0020322149629336,
      0.0020322149629336,
      0.0003505328014345,
      0.0003505328014345,
      0.0003505328014345,
      0.0003505328014345,
      0.0003505328014345,
      0.0003505328014345,
      0.0027525567706757,
      0.0027525567706757,
      0.0027525567706757,
      0.0015950528243575,
      0.0015950528243575,
      0.0015950528243575,
      0.0015950528243575,
      0.0015950528243575,
      0.0015950528243575,
      0.0024723307999349,
      0.0024723307999349,
      0.0024723307999349,
      0.0012169129205528,
      0.0012169129205528,
      0.0012169129205528,
      0.0012169129205528,
      0.0012169129205528,
      0.0012169129205528,
      0.0001254237257288,
      0.0001254237257288,
      0.0001254237257288,
      0.0001254237257288,
      0.0001254237257288,
      0.0001254237257288,
      0.0020328949037971,
      0.0020328949037971,
      0.0020328949037971,
      0.0008554065638802,
      0.0008554065638802,
      0.0008554065638802,
      0.0008554065638802,
      0.0008554065638802,
      0.0008554065638802,
      0.0022640060667147,
      0.0022640060667147,
      0.0022640060667147,
      0.0022640060667147,
      0.0022640060667147,
      0.0022640060667147,
      0.0020040696150745,
      0.0020040696150745,
      0.0020040696150745,
      0.0020040696150745,
      0.0020040696150745,
      0.0020040696150745,
      0.0005597701999537,
      0.0005597701999537,
      0.0005597701999537,
      0.0005597701999537,
      0.0005597701999537,
      0.0005597701999537,
      0.0008278245406139,
      0.0008278245406139,
      0.0008278245406139,
      0.0008278245406139,
      0.0008278245406139,
      0.0008278245406139,
      0.0015372786056368,
      0.0015372786056368,
      0.0015372786056368,
      0.0015372786056368,
      0.0015372786056368,
      0.0015372786056368,
      0.0024150352120985,
      0.0024150352120985,
      0.0024150352120985,
      0.0024150352120985,
      0.0024150352120985,
      0.0024150352120985,
      0.0019493487019107,
      0.0019493487019107,
      0.0019493487019107,
      0.0019493487019107,
      0.0019493487019107,
      0.0019493487019107,
      0.0008229259955774,
      0.0008229259955774,
      0.0008229259955774,
      0.0008229259955774,
      0.0008229259955774,
      0.0008229259955774,
      0.0018559524354638,
      0.0018559524354638,
      0.0018559524354638,
      0.0018559524354638,
      0.0018559524354638,
      0.0018559524354638,
      0.0007092878039308,
      0.0007092878039308,
      0.0007092878039308,
      0.0007092878039308,
      0.0007092878039308,
      0.0007092878039308,
      0.0006445290977855,
      0.0006445290977855,
      0.0006445290977855,
      0.0006445290977855,
      0.0006445290977855,
      0.0006445290977855,
      0.0007807450274529,
      0.0007807450274529,
      0.0007807450274529,
      0.0007807450274529,
      0.0007807450274529,
      0.0007807450274529,
      0.0008476359454730,
      0.0008476359454730,
      0.0008476359454730,
      0.0008476359454730,
      0.0008476359454730,
      0.0008476359454730,
      0.0003744154306020,
      0.0003744154306020,
      0.0003744154306020,
      0.0002264011362839,
      0.0002264011362839,
      0.0002264011362839,
      0.0002264011362839,
      0.0002264011362839,
      0.0002264011362839,
      0.0003726699155332,
      0.0003726699155332,
      0.0003726699155332,
      0.0003726699155332,
      0.0003726699155332,
      0.0003726699155332,
      0.0002661740736294,
      0.0002661740736294,
      0.0002661740736294,
      0.0002661740736294,
      0.0002661740736294,
      0.0002661740736294,
      0.0018160699278319,
      0.0018160699278319,
      0.0018160699278319,
      0.0003288098418132,
      0.0003288098418132,
      0.0003288098418132,
      0.0003288098418132,
      0.0003288098418132,
      0.0003288098418132,
      0.0001810345255654,
      0.0001810345255654,
      0.0001810345255654,
      0.0001810345255654,
      0.0001810345255654,
      0.0001810345255654,
      0.0003530059728353,
      0.0003530059728353,
      0.0003530059728353,
      0.0003530059728353,
      0.0003530059728353,
      0.0003530059728353,
      0.0003009385511100,
      0.0003009385511100,
      0.0003009385511100,
      0.0003009385511100,
      0.0003009385511100,
      0.0003009385511100,
      0.0003663196732748,
      0.0003663196732748,
      0.0003663196732748,
      0.0003663196732748,
      0.0003663196732748,
      0.0003663196732748,
      0.0000446268307090,
      0.0000446268307090,
      0.0000446268307090}
    },
    {
      423,
      {{0.3477244587108094, 0.3045510825783811, 0.0},
      {0.3477244587108095, 0.3477244587108095, 0.0},
      {0.3045510825783810, 0.3477244587108096, 0.0},
      {0.3695672915901662, 0.2157006441696169, 0.0},
      {0.4147320642402169, 0.3695672915901663, 0.0},
      {0.2157006441696168, 0.4147320642402171, 0.0},
      {0.4147320642402170, 0.2157006441696169, 0.0},
      {0.3695672915901662, 0.4147320642402170, 0.0},
      {0.2157006441696168, 0.3695672915901664, 0.0},
      {0.3084818979933622, 0.1083014999220040, 0.0},
      {0.5832166020846338, 0.3084818979933623, 0.0},
      {0.1083014999220039, 0.5832166020846340, 0.0},
      {0.5832166020846338, 0.1083014999220040, 0.0},
      {0.3084818979933622, 0.5832166020846339, 0.0},
      {0.1083014999220039, 0.3084818979933624, 0.0},
      {0.3196427292245513, 0.0346959527711364, 0.0},
      {0.6456613180043123, 0.3196427292245514, 0.0},
      {0.0346959527711365, 0.6456613180043125, 0.0},
      {0.6456613180043123, 0.0346959527711364, 0.0},
      {0.3196427292245514, 0.6456613180043123, 0.0},
      {0.0346959527711363, 0.3196427292245516, 0.0},
      {0.3734312712553221, 0.2531374574893559, 0.0},
      {0.3734312712553221, 0.3734312712553222, 0.0},
      {0.2531374574893558, 0.3734312712553222, 0.0},
      {0.9962559654771519, 0.0013460639309495, 0.0},
      {0.0023979705918987, 0.9962559654771520, 0.0},
      {0.0013460639309493, 0.0023979705918990, 0.0},
      {0.0023979705918986, 0.0013460639309495, 0.0},
      {0.9962559654771519, 0.0023979705918986, 0.0},
      {0.0013460639309497, 0.9962559654771520, 0.0},
      {0.3280848178088313, 0.0191707840803216, 0.0},
      {0.6527443981108471, 0.3280848178088313, 0.0},
      {0.0191707840803216, 0.6527443981108473, 0.0},
      {0.6527443981108471, 0.0191707840803216, 0.0},
      {0.3280848178088314, 0.6527443981108472, 0.0},
      {0.0191707840803215, 0.3280848178088315, 0.0},
      {0.3198129082727195, 0.2763615653108097, 0.0},
      {0.4038255264164708, 0.3198129082727196, 0.0},
      {0.2763615653108096, 0.4038255264164710, 0.0},
      {0.4038255264164707, 0.2763615653108097, 0.0},
      {0.3198129082727195, 0.4038255264164710, 0.0},
      {0.2763615653108096, 0.3198129082727196, 0.0},
      {0.1589266013618200, 0.0016411683438175, 0.0},
      {0.8394322302943624, 0.1589266013618200, 0.0},
      {0.0016411683438175, 0.8394322302943628, 0.0},
      {0.8394322302943625, 0.0016411683438175, 0.0},
      {0.1589266013618200, 0.8394322302943628, 0.0},
      {0.0016411683438173, 0.1589266013618204, 0.0},
      {0.1962759771972356, 0.0533231934213235, 0.0},
      {0.7504008293814409, 0.1962759771972356, 0.0},
      {0.0533231934213236, 0.7504008293814409, 0.0},
      {0.7504008293814409, 0.0533231934213235, 0.0},
      {0.1962759771972357, 0.7504008293814409, 0.0},
      {0.0533231934213233, 0.1962759771972359, 0.0},
      {0.4893390535628532, 0.0213218928742936, 0.0},
      {0.4893390535628532, 0.4893390535628533, 0.0},
      {0.0213218928742936, 0.4893390535628534, 0.0},
      {0.2933421845901817, 0.0829288364011384, 0.0},
      {0.6237289790086799, 0.2933421845901818, 0.0},
      {0.0829288364011385, 0.6237289790086800, 0.0},
      {0.6237289790086799, 0.0829288364011384, 0.0},
      {0.2933421845901818, 0.6237289790086800, 0.0},
      {0.0829288364011382, 0.2933421845901819, 0.0},
      {0.0484109248251399, 0.0484109248251402, 0.0},
      {0.9031781503497199, 0.0484109248251399, 0.0},
      {0.0484109248251403, 0.9031781503497200, 0.0},
      {0.0801595487479756, 0.0801595487479759, 0.0},
      {0.8396809025040485, 0.0801595487479756, 0.0},
      {0.0801595487479760, 0.8396809025040486, 0.0},
      {0.1247273258813976, 0.0233190171491815, 0.0},
      {0.8519536569694208, 0.1247273258813976, 0.0},
      {0.0233190171491816, 0.8519536569694212, 0.0},
      {0.8519536569694209, 0.0233190171491815, 0.0},
      {0.1247273258813976, 0.8519536569694212, 0.0},
      {0.0233190171491812, 0.1247273258813979, 0.0},
      {0.3115932459432796, 0.1387130939593856, 0.0},
      {0.5496936600973348, 0.3115932459432796, 0.0},
      {0.1387130939593856, 0.5496936600973350, 0.0},
      {0.5496936600973348, 0.1387130939593856, 0.0},
      {0.3115932459432795, 0.5496936600973350, 0.0},
      {0.1387130939593854, 0.3115932459432798, 0.0},
      {0.4790902659592135, 0.0418194680815730, 0.0},
      {0.4790902659592136, 0.4790902659592136, 0.0},
      {0.0418194680815729, 0.4790902659592137, 0.0},
      {0.4666424153484346, 0.0667151693031307, 0.0},
      {0.4666424153484346, 0.4666424153484347, 0.0},
      {0.0667151693031307, 0.4666424153484348, 0.0},
      {0.1105414410318808, 0.0105406394081225, 0.0},
      {0.8789179195599967, 0.1105414410318807, 0.0},
      {0.0105406394081227, 0.8789179195599968, 0.0},
      {0.8789179195599968, 0.0105406394081225, 0.0},
      {0.1105414410318809, 0.8789179195599968, 0.0},
      {0.0105406394081222, 0.1105414410318811, 0.0},
      {0.2974171962759172, 0.0103472505675079, 0.0},
      {0.6922355531565748, 0.2974171962759173, 0.0},
      {0.0103472505675080, 0.6922355531565749, 0.0},
      {0.6922355531565748, 0.0103472505675079, 0.0},
      {0.2974171962759173, 0.6922355531565749, 0.0},
      {0.0103472505675078, 0.2974171962759176, 0.0},
      {0.2940680586522785, 0.0537818816587767, 0.0},
      {0.6521500596889448, 0.2940680586522786, 0.0},
      {0.0537818816587767, 0.6521500596889449, 0.0},
      {0.6521500596889448, 0.0537818816587767, 0.0},
      {0.2940680586522786, 0.6521500596889449, 0.0},
      {0.0537818816587766, 0.2940680586522788, 0.0},
      {0.1114012176063108, 0.0697603580204464, 0.0},
      {0.8188384243732427, 0.1114012176063108, 0.0},
      {0.0697603580204465, 0.8188384243732429, 0.0},
      {0.8188384243732427, 0.0697603580204464, 0.0},
      {0.1114012176063108, 0.8188384243732429, 0.0},
      {0.0697603580204462, 0.1114012176063111, 0.0},
      {0.4437434538636293, 0.0155023027792511, 0.0},
      {0.5407542433571197, 0.4437434538636293, 0.0},
      {0.0155023027792509, 0.5407542433571200, 0.0},
      {0.5407542433571196, 0.0155023027792511, 0.0},
      {0.4437434538636293, 0.5407542433571197, 0.0},
      {0.0155023027792509, 0.4437434538636296, 0.0},
      {0.3676357119294025, 0.0396225072588169, 0.0},
      {0.5927417808117805, 0.3676357119294025, 0.0},
      {0.0396225072588170, 0.5927417808117806, 0.0},
      {0.5927417808117805, 0.0396225072588169, 0.0},
      {0.3676357119294025, 0.5927417808117806, 0.0},
      {0.0396225072588168, 0.3676357119294028, 0.0},
      {0.0483834188936243, 0.0118107009236641, 0.0},
      {0.9398058801827116, 0.0483834188936242, 0.0},
      {0.0118107009236643, 0.9398058801827117, 0.0},
      {0.9398058801827116, 0.0118107009236641, 0.0},
      {0.0483834188936244, 0.9398058801827117, 0.0},
      {0.0118107009236638, 0.0483834188936246, 0.0},
      {0.4261985460348483, 0.0352878923463722, 0.0},
      {0.5385135616187795, 0.4261985460348483, 0.0},
      {0.0352878923463722, 0.5385135616187797, 0.0},
      {0.5385135616187795, 0.0352878923463722, 0.0},
      {0.4261985460348482, 0.5385135616187796, 0.0},
      {0.0352878923463721, 0.4261985460348485, 0.0},
      {0.3464037075083426, 0.0647150930019751, 0.0},
      {0.5888811994896823, 0.3464037075083426, 0.0},
      {0.0647150930019751, 0.5888811994896825, 0.0},
      {0.5888811994896823, 0.0647150930019751, 0.0},
      {0.3464037075083425, 0.5888811994896824, 0.0},
      {0.0647150930019750, 0.3464037075083428, 0.0},
      {0.0761686440792015, 0.0104225715676421, 0.0},
      {0.9134087843531564, 0.0761686440792014, 0.0},
      {0.0104225715676421, 0.9134087843531568, 0.0},
      {0.9134087843531565, 0.0104225715676421, 0.0},
      {0.0761686440792015, 0.9134087843531568, 0.0},
      {0.0104225715676419, 0.0761686440792017, 0.0},
      {0.3198558059892070, 0.2271049175306690, 0.0},
      {0.4530392764801240, 0.3198558059892071, 0.0},
      {0.2271049175306689, 0.4530392764801242, 0.0},
      {0.4530392764801241, 0.2271049175306690, 0.0},
      {0.3198558059892070, 0.4530392764801242, 0.0},
      {0.2271049175306689, 0.3198558059892072, 0.0},
      {0.1149767499763278, 0.0431548880805706, 0.0},
      {0.8418683619431016, 0.1149767499763278, 0.0},
      {0.0431548880805707, 0.8418683619431018, 0.0},
      {0.8418683619431017, 0.0431548880805706, 0.0},
      {0.1149767499763279, 0.8418683619431018, 0.0},
      {0.0431548880805703, 0.1149767499763281, 0.0},
      {0.1553411874863350, 0.0489749482183904, 0.0},
      {0.7956838642952746, 0.1553411874863350, 0.0},
      {0.0489749482183905, 0.7956838642952748, 0.0},
      {0.7956838642952746, 0.0489749482183904, 0.0},
      {0.1553411874863350, 0.7956838642952748, 0.0},
      {0.0489749482183902, 0.1553411874863353, 0.0},
      {0.1536641394780652, 0.0094545488420267, 0.0},
      {0.8368813116799081, 0.1536641394780652, 0.0},
      {0.0094545488420268, 0.8368813116799082, 0.0},
      {0.8368813116799082, 0.0094545488420267, 0.0},
      {0.1536641394780653, 0.8368813116799082, 0.0},
      {0.0094545488420265, 0.1536641394780656, 0.0},
      {0.0798538207718234, 0.0018943058791126, 0.0},
      {0.9182518733490639, 0.0798538207718234, 0.0},
      {0.0018943058791128, 0.9182518733490641, 0.0},
      {0.9182518733490640, 0.0018943058791126, 0.0},
      {0.0798538207718235, 0.9182518733490641, 0.0},
      {0.0018943058791124, 0.0798538207718237, 0.0},
      {0.0117045744671133, 0.0117045744671136, 0.0},
      {0.9765908510657730, 0.0117045744671133, 0.0},
      {0.0117045744671137, 0.9765908510657733, 0.0},
      {0.1978601121285484, 0.0120283224036929, 0.0},
      {0.7901115654677587, 0.1978601121285484, 0.0},
      {0.0120283224036928, 0.7901115654677591, 0.0},
      {0.7901115654677586, 0.0120283224036929, 0.0},
      {0.1978601121285484, 0.7901115654677587, 0.0},
      {0.0120283224036928, 0.1978601121285486, 0.0},
      {0.3169153803807818, 0.0019044677779363, 0.0},
      {0.6811801518412819, 0.3169153803807819, 0.0},
      {0.0019044677779363, 0.6811801518412820, 0.0},
      {0.6811801518412819, 0.0019044677779363, 0.0},
      {0.3169153803807819, 0.6811801518412820, 0.0},
      {0.0019044677779362, 0.3169153803807821, 0.0},
      {0.1645639771634135, 0.0262052237923565, 0.0},
      {0.8092307990442300, 0.1645639771634134, 0.0},
      {0.0262052237923566, 0.8092307990442302, 0.0},
      {0.8092307990442301, 0.0262052237923565, 0.0},
      {0.1645639771634135, 0.8092307990442302, 0.0},
      {0.0262052237923562, 0.1645639771634138, 0.0},
      {0.3611298359799379, 0.0944709215240264, 0.0},
      {0.5443992424960357, 0.3611298359799380, 0.0},
      {0.0944709215240264, 0.5443992424960359, 0.0},
      {0.5443992424960357, 0.0944709215240264, 0.0},
      {0.3611298359799380, 0.5443992424960358, 0.0},
      {0.0944709215240263, 0.3611298359799381, 0.0},
      {0.4198259774542294, 0.0963233490337837, 0.0},
      {0.4838506735119869, 0.4198259774542294, 0.0},
      {0.0963233490337836, 0.4838506735119871, 0.0},
      {0.4838506735119869, 0.0963233490337837, 0.0},
      {0.4198259774542294, 0.4838506735119870, 0.0},
      {0.0963233490337836, 0.4198259774542296, 0.0},
      {0.2431752245746631, 0.0534903172634274, 0.0},
      {0.7033344581619095, 0.2431752245746631, 0.0},
      {0.0534903172634275, 0.7033344581619096, 0.0},
      {0.7033344581619095, 0.0534903172634274, 0.0},
      {0.2431752245746631, 0.7033344581619096, 0.0},
      {0.0534903172634272, 0.2431752245746634, 0.0},
      {0.1459832231136435, 0.0795244786008372, 0.0},
      {0.7744922982855192, 0.1459832231136435, 0.0},
      {0.0795244786008372, 0.7744922982855196, 0.0},
      {0.7744922982855195, 0.0795244786008372, 0.0},
      {0.1459832231136434, 0.7744922982855196, 0.0},
      {0.0795244786008370, 0.1459832231136438, 0.0},
      {0.0272392832679803, 0.0110307547814719, 0.0},
      {0.9617299619505478, 0.0272392832679802, 0.0},
      {0.0110307547814721, 0.9617299619505479, 0.0},
      {0.9617299619505478, 0.0110307547814719, 0.0},
      {0.0272392832679803, 0.9617299619505479, 0.0},
      {0.0110307547814717, 0.0272392832679806, 0.0},
      {0.2453137089647784, 0.0121862329639789, 0.0},
      {0.7425000580712428, 0.2453137089647784, 0.0},
      {0.0121862329639789, 0.7425000580712430, 0.0},
      {0.7425000580712428, 0.0121862329639789, 0.0},
      {0.2453137089647784, 0.7425000580712430, 0.0},
      {0.0121862329639786, 0.2453137089647787, 0.0},
      {0.2678465137111688, 0.2678465137111690, 0.0},
      {0.4643069725776622, 0.2678465137111689, 0.0},
      {0.2678465137111689, 0.4643069725776624, 0.0},
      {0.1165542190734529, 0.0021132182624776, 0.0},
      {0.8813325626640695, 0.1165542190734529, 0.0},
      {0.0021132182624777, 0.8813325626640697, 0.0},
      {0.8813325626640695, 0.0021132182624776, 0.0},
      {0.1165542190734529, 0.8813325626640697, 0.0},
      {0.0021132182624774, 0.1165542190734533, 0.0},
      {0.0535856828700446, 0.0272189906580712, 0.0},
      {0.9191953264718842, 0.0535856828700446, 0.0},
      {0.0272189906580713, 0.9191953264718844, 0.0},
      {0.9191953264718841, 0.0272189906580712, 0.0},
      {0.0535856828700446, 0.9191953264718844, 0.0},
      {0.0272189906580709, 0.0535856828700449, 0.0},
      {0.0851355181500169, 0.0262562590124721, 0.0},
      {0.8886082228375111, 0.0851355181500168, 0.0},
      {0.0262562590124721, 0.8886082228375113, 0.0},
      {0.8886082228375112, 0.0262562590124721, 0.0},
      {0.0851355181500170, 0.8886082228375112, 0.0},
      {0.0262562590124718, 0.0851355181500172, 0.0},
      {0.4084926257856331, 0.0624750974503704, 0.0},
      {0.5290322767639966, 0.4084926257856331, 0.0},
      {0.0624750974503704, 0.5290322767639968, 0.0},
      {0.5290322767639964, 0.0624750974503704, 0.0},
      {0.4084926257856331, 0.5290322767639967, 0.0},
      {0.0624750974503703, 0.4084926257856333, 0.0},
      {0.3822510450948138, 0.1737762098761935, 0.0},
      {0.4439727450289927, 0.3822510450948139, 0.0},
      {0.1737762098761934, 0.4439727450289929, 0.0},
      {0.4439727450289928, 0.1737762098761935, 0.0},
      {0.3822510450948139, 0.4439727450289928, 0.0},
      {0.1737762098761934, 0.3822510450948140, 0.0},
      {0.0259740322065890, 0.0020491260430696, 0.0},
      {0.9719768417503413, 0.0259740322065890, 0.0},
      {0.0020491260430698, 0.9719768417503415, 0.0},
      {0.9719768417503415, 0.0020491260430696, 0.0},
      {0.0259740322065892, 0.9719768417503415, 0.0},
      {0.0020491260430693, 0.0259740322065894, 0.0},
      {0.3856891887600643, 0.0188900416615420, 0.0},
      {0.5954207695783936, 0.3856891887600644, 0.0},
      {0.0188900416615421, 0.5954207695783937, 0.0},
      {0.5954207695783936, 0.0188900416615420, 0.0},
      {0.3856891887600644, 0.5954207695783937, 0.0},
      {0.0188900416615419, 0.3856891887600646, 0.0},
      {0.1907464794314031, 0.0826189529456984, 0.0},
      {0.7266345676228985, 0.1907464794314031, 0.0},
      {0.0826189529456984, 0.7266345676228987, 0.0},
      {0.7266345676228985, 0.0826189529456984, 0.0},
      {0.1907464794314032, 0.7266345676228986, 0.0},
      {0.0826189529456982, 0.1907464794314034, 0.0},
      {0.0493146516048518, 0.0022799567867531, 0.0},
      {0.9484053916083951, 0.0493146516048518, 0.0},
      {0.0022799567867532, 0.9484053916083952, 0.0},
      {0.9484053916083951, 0.0022799567867531, 0.0},
      {0.0493146516048519, 0.9484053916083952, 0.0},
      {0.0022799567867528, 0.0493146516048521, 0.0},
      {0.4264825454980466, 0.0047722521203026, 0.0},
      {0.5687452023816508, 0.4264825454980467, 0.0},
      {0.0047722521203026, 0.5687452023816510, 0.0},
      {0.5687452023816508, 0.0047722521203026, 0.0},
      {0.4264825454980467, 0.5687452023816508, 0.0},
      {0.0047722521203026, 0.4264825454980468, 0.0},
      {0.1513647526705651, 0.1162023564115881, 0.0},
      {0.7324328909178469, 0.1513647526705651, 0.0},
      {0.1162023564115880, 0.7324328909178470, 0.0},
      {0.7324328909178468, 0.1162023564115881, 0.0},
      {0.1513647526705652, 0.7324328909178470, 0.0},
      {0.1162023564115878, 0.1513647526705654, 0.0},
      {0.2410855603909864, 0.0855252642423671, 0.0},
      {0.6733891753666466, 0.2410855603909865, 0.0},
      {0.0855252642423671, 0.6733891753666467, 0.0},
      {0.6733891753666466, 0.0855252642423671, 0.0},
      {0.2410855603909864, 0.6733891753666467, 0.0},
      {0.0855252642423669, 0.2410855603909867, 0.0},
      {0.2120362112054797, 0.0297767883228067, 0.0},
      {0.7581870004717137, 0.2120362112054797, 0.0},
      {0.0297767883228067, 0.7581870004717138, 0.0},
      {0.7581870004717137, 0.0297767883228067, 0.0},
      {0.2120362112054797, 0.7581870004717138, 0.0},
      {0.0297767883228064, 0.2120362112054800, 0.0},
      {0.0773420280842538, 0.0502535370358714, 0.0},
      {0.8724044348798747, 0.0773420280842539, 0.0},
      {0.0502535370358714, 0.8724044348798750, 0.0},
      {0.8724044348798747, 0.0502535370358714, 0.0},
      {0.0773420280842540, 0.8724044348798748, 0.0},
      {0.0502535370358712, 0.0773420280842542, 0.0},
      {0.2659270236545313, 0.2163002071023544, 0.0},
      {0.5177727692431142, 0.2659270236545314, 0.0},
      {0.2163002071023543, 0.5177727692431144, 0.0},
      {0.5177727692431143, 0.2163002071023544, 0.0},
      {0.2659270236545314, 0.5177727692431144, 0.0},
      {0.2163002071023542, 0.2659270236545315, 0.0},
      {0.2131382724801908, 0.2131382724801910, 0.0},
      {0.5737234550396183, 0.2131382724801908, 0.0},
      {0.2131382724801909, 0.5737234550396184, 0.0},
      {0.0102781124040352, 0.0023182076844771, 0.0},
      {0.9874036799114877, 0.0102781124040352, 0.0},
      {0.0023182076844773, 0.9874036799114877, 0.0},
      {0.9874036799114878, 0.0023182076844771, 0.0},
      {0.0102781124040354, 0.9874036799114877, 0.0},
      {0.0023182076844769, 0.0102781124040356, 0.0},
      {0.4336266613666109, 0.1327466772667781, 0.0},
      {0.4336266613666109, 0.4336266613666110, 0.0},
      {0.1327466772667780, 0.4336266613666112, 0.0},
      {0.3637486804898151, 0.0064301877897617, 0.0},
      {0.6298211317204232, 0.3637486804898151, 0.0},
      {0.0064301877897617, 0.6298211317204234, 0.0},
      {0.6298211317204232, 0.0064301877897617, 0.0},
      {0.3637486804898151, 0.6298211317204232, 0.0},
      {0.0064301877897616, 0.3637486804898154, 0.0},
      {0.3698202051067137, 0.1331111150654715, 0.0},
      {0.4970686798278148, 0.3698202051067138, 0.0},
      {0.1331111150654714, 0.4970686798278151, 0.0},
      {0.4970686798278148, 0.1331111150654715, 0.0},
      {0.3698202051067137, 0.4970686798278149, 0.0},
      {0.1331111150654714, 0.3698202051067139, 0.0},
      {0.1599009144405388, 0.1599009144405390, 0.0},
      {0.6801981711189222, 0.1599009144405389, 0.0},
      {0.1599009144405390, 0.6801981711189224, 0.0},
      {0.0280636634926782, 0.0280636634926785, 0.0},
      {0.9438726730146434, 0.0280636634926781, 0.0},
      {0.0280636634926786, 0.9438726730146435, 0.0},
      {0.2622368556255414, 0.1682524853715630, 0.0},
      {0.5695106590028955, 0.2622368556255416, 0.0},
      {0.1682524853715630, 0.5695106590028957, 0.0},
      {0.5695106590028955, 0.1682524853715630, 0.0},
      {0.2622368556255415, 0.5695106590028957, 0.0},
      {0.1682524853715628, 0.2622368556255417, 0.0},
      {0.2058652741989988, 0.0023776437008402, 0.0},
      {0.7917570821001609, 0.2058652741989988, 0.0},
      {0.0023776437008403, 0.7917570821001612, 0.0},
      {0.7917570821001609, 0.0023776437008402, 0.0},
      {0.2058652741989988, 0.7917570821001612, 0.0},
      {0.0023776437008400, 0.2058652741989991, 0.0},
      {0.3210612992391711, 0.1792751769147349, 0.0},
      {0.4996635238460939, 0.3210612992391713, 0.0},
      {0.1792751769147349, 0.4996635238460941, 0.0},
      {0.4996635238460939, 0.1792751769147349, 0.0},
      {0.3210612992391711, 0.4996635238460940, 0.0},
      {0.1792751769147348, 0.3210612992391713, 0.0},
      {0.2670089253528111, 0.0284633516120191, 0.0},
      {0.7045277230351698, 0.2670089253528111, 0.0},
      {0.0284633516120191, 0.7045277230351701, 0.0},
      {0.7045277230351698, 0.0284633516120191, 0.0},
      {0.2670089253528112, 0.7045277230351698, 0.0},
      {0.0284633516120189, 0.2670089253528115, 0.0},
      {0.2586045410103510, 0.0022264670918959, 0.0},
      {0.7391689918977531, 0.2586045410103511, 0.0},
      {0.0022264670918959, 0.7391689918977533, 0.0},
      {0.7391689918977531, 0.0022264670918959, 0.0},
      {0.2586045410103511, 0.7391689918977531, 0.0},
      {0.0022264670918957, 0.2586045410103514, 0.0},
      {0.1996247626773604, 0.1203643695722496, 0.0},
      {0.6800108677503900, 0.1996247626773605, 0.0},
      {0.1203643695722495, 0.6800108677503902, 0.0},
      {0.6800108677503900, 0.1203643695722496, 0.0},
      {0.1996247626773605, 0.6800108677503902, 0.0},
      {0.1203643695722494, 0.1996247626773607, 0.0},
      {0.2088908643844065, 0.1643344616867156, 0.0},
      {0.6267746739288779, 0.2088908643844065, 0.0},
      {0.1643344616867157, 0.6267746739288781, 0.0},
      {0.6267746739288779, 0.1643344616867156, 0.0},
      {0.2088908643844065, 0.6267746739288781, 0.0},
      {0.1643344616867155, 0.2088908643844067, 0.0},
      {0.1080471292423059, 0.1080471292423062, 0.0},
      {0.7839057415153878, 0.1080471292423059, 0.0},
      {0.1080471292423062, 0.7839057415153881, 0.0},
      {0.2533336006018089, 0.1238874588376767, 0.0},
      {0.6227789405605143, 0.2533336006018089, 0.0},
      {0.1238874588376767, 0.6227789405605146, 0.0},
      {0.6227789405605143, 0.1238874588376767, 0.0},
      {0.2533336006018089, 0.6227789405605145, 0.0},
      {0.1238874588376766, 0.2533336006018092, 0.0},
      {0.4971756372384661, 0.0056487255230679, 0.0},
      {0.4971756372384660, 0.4971756372384661, 0.0},
      {0.0056487255230678, 0.4971756372384663, 0.0},
      {0.4637961227465663, 0.0001600103893843, 0.0},
      {0.5360438668640494, 0.4637961227465663, 0.0},
      {0.0001600103893842, 0.5360438668640496, 0.0},
      {0.5360438668640494, 0.0001600103893843, 0.0},
      {0.4637961227465663, 0.5360438668640495, 0.0},
      {0.0001600103893842, 0.4637961227465666, 0.0},
      {0.3837177990505464, 0.0001085092017437, 0.0},
      {0.6161736917477099, 0.3837177990505464, 0.0},
      {0.0001085092017437, 0.6161736917477101, 0.0},
      {0.6161736917477099, 0.0001085092017437, 0.0},
      {0.3837177990505465, 0.6161736917477100, 0.0},
      {0.0001085092017435, 0.3837177990505467, 0.0}},
      {0.0021825693768913,
      0.0021825693768913,
      0.0021825693768913,
      0.0018599657061483,
      0.0018599657061483,
      0.0018599657061483,
      0.0018599657061483,
      0.0018599657061483,
      0.0018599657061483,
      0.0015163054722851,
      0.0015163054722851,
      0.0015163054722851,
      0.0015163054722851,
      0.0015163054722851,
      0.0015163054722851,
      0.0009235054324168,
      0.0009235054324168,
      0.0009235054324168,
      0.0009235054324168,
      0.0009235054324168,
      0.0009235054324168,
      0.0019845875226513,
      0.0019845875226513,
      0.0019845875226513,
      0.0000118864540424,
      0.0000118864540424,
      0.0000118864540424,
      0.0000118864540424,
      0.0000118864540424,
      0.0000118864540424,
      0.0007724837263205,
      0.0007724837263205,
      0.0007724837263205,
      0.0007724837263205,
      0.0007724837263205,
      0.0007724837263205,
      0.0025159434913497,
      0.0025159434913497,
      0.0025159434913497,
      0.0025159434913497,
      0.0025159434913497,
      0.0025159434913497,
      0.0001943678834005,
      0.0001943678834005,
      0.0001943678834005,
      0.0001943678834005,
      0.0001943678834005,
      0.0001943678834005,
      0.0011464846676598,
      0.0011464846676598,
      0.0011464846676598,
      0.0011464846676598,
      0.0011464846676598,
      0.0011464846676598,
      0.0009110572683922,
      0.0009110572683922,
      0.0009110572683922,
      0.0016295040978854,
      0.0016295040978854,
      0.0016295040978854,
      0.0016295040978854,
      0.0016295040978854,
      0.0016295040978854,
      0.0006057236250325,
      0.0006057236250325,
      0.0006057236250325,
      0.0009654772448136,
      0.0009654772448136,
      0.0009654772448136,
      0.0006497171967008,
      0.0006497171967008,
      0.0006497171967008,
      0.0006497171967008,
      0.0006497171967008,
      0.0006497171967008,
      0.0020221158995708,
      0.0020221158995708,
      0.0020221158995708,
      0.0020221158995708,
      0.0020221158995708,
      0.0020221158995708,
      0.0013200121878206,
      0.0013200121878206,
      0.0013200121878206,
      0.0016420960432031,
      0.0016420960432031,
      0.0016420960432031,
      0.0004322395535472,
      0.0004322395535472,
      0.0004322395535472,
      0.0004322395535472,
      0.0004322395535472,
      0.0004322395535472,
      0.0006329857182379,
      0.0006329857182379,
      0.0006329857182379,
      0.0006329857182379,
      0.0006329857182379,
      0.0006329857182379,
      0.0013712942976207,
      0.0013712942976207,
      0.0013712942976207,
      0.0013712942976207,
      0.0013712942976207,
      0.0013712942976207,
      0.0010577037768328,
      0.0010577037768328,
      0.0010577037768328,
      0.0010577037768328,
      0.0010577037768328,
      0.0010577037768328,
      0.0008115772636429,
      0.0008115772636429,
      0.0008115772636429,
      0.0008115772636429,
      0.0008115772636429,
      0.0008115772636429,
      0.0012706368667027,
      0.0012706368667027,
      0.0012706368667027,
      0.0012706368667027,
      0.0012706368667027,
      0.0012706368667027,
      0.0003113177360561,
      0.0003113177360561,
      0.0003113177360561,
      0.0003113177360561,
      0.0003113177360561,
      0.0003113177360561,
      0.0013209201899881,
      0.0013209201899881,
      0.0013209201899881,
      0.0013209201899881,
      0.0013209201899881,
      0.0013209201899881,
      0.0016326585513787,
      0.0016326585513787,
      0.0016326585513787,
      0.0016326585513787,
      0.0016326585513787,
      0.0016326585513787,
      0.0003827572929612,
      0.0003827572929612,
      0.0003827572929612,
      0.0003827572929612,
      0.0003827572929612,
      0.0003827572929612,
      0.0025820680472357,
      0.0025820680472357,
      0.0025820680472357,
      0.0025820680472357,
      0.0025820680472357,
      0.0025820680472357,
      0.0009277194906379,
      0.0009277194906379,
      0.0009277194906379,
      0.0009277194906379,
      0.0009277194906379,
      0.0009277194906379,
      0.0011399050056812,
      0.0011399050056812,
      0.0011399050056812,
      0.0011399050056812,
      0.0011399050056812,
      0.0011399050056812,
      0.0005133973085980,
      0.0005133973085980,
      0.0005133973085980,
      0.0005133973085980,
      0.0005133973085980,
      0.0005133973085980,
      0.0001664215605633,
      0.0001664215605633,
      0.0001664215605633,
      0.0001664215605633,
      0.0001664215605633,
      0.0001664215605633,
      0.0001614724792174,
      0.0001614724792174,
      0.0001614724792174,
      0.0006078746557259,
      0.0006078746557259,
      0.0006078746557259,
      0.0006078746557259,
      0.0006078746557259,
      0.0006078746557259,
      0.0002946557864478,
      0.0002946557864478,
      0.0002946557864478,
      0.0002946557864478,
      0.0002946557864478,
      0.0002946557864478,
      0.0008671329022474,
      0.0008671329022474,
      0.0008671329022474,
      0.0008671329022474,
      0.0008671329022474,
      0.0008671329022474,
      0.0020249375378802,
      0.0020249375378802,
      0.0020249375378802,
      0.0020249375378802,
      0.0020249375378802,
      0.0020249375378802,
      0.0021371161488564,
      0.0021371161488564,
      0.0021371161488564,
      0.0021371161488564,
      0.0021371161488564,
      0.0021371161488564,
      0.0014167542737901,
      0.0014167542737901,
      0.0014167542737901,
      0.0014167542737901,
      0.0014167542737901,
      0.0014167542737901,
      0.0013719062729419,
      0.0013719062729419,
      0.0013719062729419,
      0.0013719062729419,
      0.0013719062729419,
      0.0013719062729419,
      0.0002365107281775,
      0.0002365107281775,
      0.0002365107281775,
      0.0002365107281775,
      0.0002365107281775,
      0.0002365107281775,
      0.0006866588720805,
      0.0006866588720805,
      0.0006866588720805,
      0.0006866588720805,
      0.0006866588720805,
      0.0006866588720805,
      0.0027155231519976,
      0.0027155231519976,
      0.0027155231519976,
      0.0002116504957303,
      0.0002116504957303,
      0.0002116504957303,
      0.0002116504957303,
      0.0002116504957303,
      0.0002116504957303,
      0.0005332305935819,
      0.0005332305935819,
      0.0005332305935819,
      0.0005332305935819,
      0.0005332305935819,
      0.0005332305935819,
      0.0006912005366558,
      0.0006912005366558,
      0.0006912005366558,
      0.0006912005366558,
      0.0006912005366558,
      0.0006912005366558,
      0.0018292558965135,
      0.0018292558965135,
      0.0018292558965135,
      0.0018292558965135,
      0.0018292558965135,
      0.0018292558965135,
      0.0026050024274333,
      0.0026050024274333,
      0.0026050024274333,
      0.0026050024274333,
      0.0026050024274333,
      0.0026050024274333,
      0.0001036552802954,
      0.0001036552802954,
      0.0001036552802954,
      0.0001036552802954,
      0.0001036552802954,
      0.0001036552802954,
      0.0009791581434316,
      0.0009791581434316,
      0.0009791581434316,
      0.0009791581434316,
      0.0009791581434316,
      0.0009791581434316,
      0.0016044612649594,
      0.0016044612649594,
      0.0016044612649594,
      0.0016044612649594,
      0.0016044612649594,
      0.0016044612649594,
      0.0001572075773066,
      0.0001572075773066,
      0.0001572075773066,
      0.0001572075773066,
      0.0001572075773066,
      0.0001572075773066,
      0.0005392572775602,
      0.0005392572775602,
      0.0005392572775602,
      0.0005392572775602,
      0.0005392572775602,
      0.0005392572775602,
      0.0018002134883076,
      0.0018002134883076,
      0.0018002134883076,
      0.0018002134883076,
      0.0018002134883076,
      0.0018002134883076,
      0.0018305957089213,
      0.0018305957089213,
      0.0018305957089213,
      0.0018305957089213,
      0.0018305957089213,
      0.0018305957089213,
      0.0010808561569376,
      0.0010808561569376,
      0.0010808561569376,
      0.0010808561569376,
      0.0010808561569376,
      0.0010808561569376,
      0.0009005281381931,
      0.0009005281381931,
      0.0009005281381931,
      0.0009005281381931,
      0.0009005281381931,
      0.0009005281381931,
      0.0027237796471379,
      0.0027237796471379,
      0.0027237796471379,
      0.0027237796471379,
      0.0027237796471379,
      0.0027237796471379,
      0.0025762802722419,
      0.0025762802722419,
      0.0025762802722419,
      0.0000698926007622,
      0.0000698926007622,
      0.0000698926007622,
      0.0000698926007622,
      0.0000698926007622,
      0.0000698926007622,
      0.0024965025749709,
      0.0024965025749709,
      0.0024965025749709,
      0.0005981005925977,
      0.0005981005925977,
      0.0005981005925977,
      0.0005981005925977,
      0.0005981005925977,
      0.0005981005925977,
      0.0024762829117918,
      0.0024762829117918,
      0.0024762829117918,
      0.0024762829117918,
      0.0024762829117918,
      0.0024762829117918,
      0.0021217492459271,
      0.0021217492459271,
      0.0021217492459271,
      0.0004438002805123,
      0.0004438002805123,
      0.0004438002805123,
      0.0025276029369482,
      0.0025276029369482,
      0.0025276029369482,
      0.0025276029369482,
      0.0025276029369482,
      0.0025276029369482,
      0.0003027955309496,
      0.0003027955309496,
      0.0003027955309496,
      0.0003027955309496,
      0.0003027955309496,
      0.0003027955309496,
      0.0026788951054618,
      0.0026788951054618,
      0.0026788951054618,
      0.0026788951054618,
      0.0026788951054618,
      0.0026788951054618,
      0.0011827676122502,
      0.0011827676122502,
      0.0011827676122502,
      0.0011827676122502,
      0.0011827676122502,
      0.0011827676122502,
      0.0003180880278225,
      0.0003180880278225,
      0.0003180880278225,
      0.0003180880278225,
      0.0003180880278225,
      0.0003180880278225,
      0.0020583897158792,
      0.0020583897158792,
      0.0020583897158792,
      0.0020583897158792,
      0.0020583897158792,
      0.0020583897158792,
      0.0023717830850862,
      0.0023717830850862,
      0.0023717830850862,
      0.0023717830850862,
      0.0023717830850862,
      0.0023717830850862,
      0.0015869641744859,
      0.0015869641744859,
      0.0015869641744859,
      0.0023011337125741,
      0.0023011337125741,
      0.0023011337125741,
      0.0023011337125741,
      0.0023011337125741,
      0.0023011337125741,
      0.0006475399556948,
      0.0006475399556948,
      0.0006475399556948,
      0.0001175044612403,
      0.0001175044612403,
      0.0001175044612403,
      0.0001175044612403,
      0.0001175044612403,
      0.0001175044612403,
      0.0001199368394655,
      0.0001199368394655,
      0.0001199368394655,
      0.0001199368394655,
      0.0001199368394655,
      0.0001199368394655}
    },
    {
      435,
      {{0.1747098543071731, 0.0400081550569363, 0.0},
      {0.7852819906358907, 0.1747098543071731, 0.0},
      {0.0400081550569363, 0.7852819906358908, 0.0},
      {0.7852819906358907, 0.0400081550569363, 0.0},
      {0.1747098543071732, 0.7852819906358907, 0.0},
      {0.0400081550569361, 0.1747098543071734, 0.0},
      {0.3869650571638262, 0.0086624172314518, 0.0},
      {0.6043725256047221, 0.3869650571638262, 0.0},
      {0.0086624172314518, 0.6043725256047222, 0.0},
      {0.6043725256047220, 0.0086624172314518, 0.0},
      {0.3869650571638262, 0.6043725256047221, 0.0},
      {0.0086624172314517, 0.3869650571638265, 0.0},
      {0.1260609334354054, 0.0013011091483539, 0.0},
      {0.8726379574162407, 0.1260609334354054, 0.0},
      {0.0013011091483540, 0.8726379574162408, 0.0},
      {0.8726379574162407, 0.0013011091483539, 0.0},
      {0.1260609334354055, 0.8726379574162408, 0.0},
      {0.0013011091483536, 0.1260609334354057, 0.0},
      {0.1783730162296315, 0.1783730162296318, 0.0},
      {0.6432539675407367, 0.1783730162296316, 0.0},
      {0.1783730162296319, 0.6432539675407367, 0.0},
      {0.2839249520005005, 0.2456285053783459, 0.0},
      {0.4704465426211537, 0.2839249520005006, 0.0},
      {0.2456285053783458, 0.4704465426211538, 0.0},
      {0.4704465426211537, 0.2456285053783459, 0.0},
      {0.2839249520005004, 0.4704465426211538, 0.0},
      {0.2456285053783457, 0.2839249520005007, 0.0},
      {0.3302482693344623, 0.2507416361736093, 0.0},
      {0.4190100944919284, 0.3302482693344624, 0.0},
      {0.2507416361736092, 0.4190100944919286, 0.0},
      {0.4190100944919285, 0.2507416361736093, 0.0},
      {0.3302482693344623, 0.4190100944919286, 0.0},
      {0.2507416361736091, 0.3302482693344625, 0.0},
      {0.0728406536060113, 0.0728406536060116, 0.0},
      {0.8543186927879770, 0.0728406536060113, 0.0},
      {0.0728406536060117, 0.8543186927879771, 0.0},
      {0.2560459410681578, 0.0015097374938804, 0.0},
      {0.7424443214379617, 0.2560459410681578, 0.0},
      {0.0015097374938804, 0.7424443214379620, 0.0},
      {0.7424443214379618, 0.0015097374938804, 0.0},
      {0.2560459410681578, 0.7424443214379620, 0.0},
      {0.0015097374938802, 0.2560459410681581, 0.0},
      {0.2603953161344762, 0.1089833802129252, 0.0},
      {0.6306213036525985, 0.2603953161344763, 0.0},
      {0.1089833802129252, 0.6306213036525987, 0.0},
      {0.6306213036525986, 0.1089833802129252, 0.0},
      {0.2603953161344763, 0.6306213036525986, 0.0},
      {0.1089833802129251, 0.2603953161344765, 0.0},
      {0.4289883556394070, 0.1420232887211860, 0.0},
      {0.4289883556394070, 0.4289883556394071, 0.0},
      {0.1420232887211859, 0.4289883556394072, 0.0},
      {0.1707126963793420, 0.1019687169087761, 0.0},
      {0.7273185867118819, 0.1707126963793421, 0.0},
      {0.1019687169087761, 0.7273185867118821, 0.0},
      {0.7273185867118819, 0.1019687169087761, 0.0},
      {0.1707126963793419, 0.7273185867118821, 0.0},
      {0.1019687169087760, 0.1707126963793423, 0.0},
      {0.3746440580951249, 0.0197722885748599, 0.0},
      {0.6055836533300153, 0.3746440580951250, 0.0},
      {0.0197722885748599, 0.6055836533300155, 0.0},
      {0.6055836533300152, 0.0197722885748599, 0.0},
      {0.3746440580951250, 0.6055836533300152, 0.0},
      {0.0197722885748598, 0.3746440580951251, 0.0},
      {0.4351684572939831, 0.0013541852880937, 0.0},
      {0.5634773574179233, 0.4351684572939832, 0.0},
      {0.0013541852880936, 0.5634773574179235, 0.0},
      {0.5634773574179233, 0.0013541852880937, 0.0},
      {0.4351684572939831, 0.5634773574179233, 0.0},
      {0.0013541852880936, 0.4351684572939833, 0.0},
      {0.1390923912291039, 0.0876428525159975, 0.0},
      {0.7732647562548985, 0.1390923912291039, 0.0},
      {0.0876428525159975, 0.7732647562548988, 0.0},
      {0.7732647562548985, 0.0876428525159975, 0.0},
      {0.1390923912291041, 0.7732647562548988, 0.0},
      {0.0876428525159973, 0.1390923912291042, 0.0},
      {0.2801506543012279, 0.0784155789651995, 0.0},
      {0.6414337667335724, 0.2801506543012280, 0.0},
      {0.0784155789651995, 0.6414337667335728, 0.0},
      {0.6414337667335726, 0.0784155789651995, 0.0},
      {0.2801506543012280, 0.6414337667335727, 0.0},
      {0.0784155789651994, 0.2801506543012282, 0.0},
      {0.3463311874275595, 0.0627477726380917, 0.0},
      {0.5909210399343486, 0.3463311874275596, 0.0},
      {0.0627477726380918, 0.5909210399343489, 0.0},
      {0.5909210399343486, 0.0627477726380917, 0.0},
      {0.3463311874275596, 0.5909210399343489, 0.0},
      {0.0627477726380916, 0.3463311874275598, 0.0},
      {0.3811688378455703, 0.2376623243088594, 0.0},
      {0.3811688378455702, 0.3811688378455704, 0.0},
      {0.2376623243088594, 0.3811688378455704, 0.0},
      {0.1142447933587818, 0.0071057304538391, 0.0},
      {0.8786494761873791, 0.1142447933587818, 0.0},
      {0.0071057304538392, 0.8786494761873792, 0.0},
      {0.8786494761873791, 0.0071057304538391, 0.0},
      {0.1142447933587818, 0.8786494761873792, 0.0},
      {0.0071057304538389, 0.1142447933587821, 0.0},
      {0.2601477827144039, 0.0408904742282701, 0.0},
      {0.6989617430573259, 0.2601477827144039, 0.0},
      {0.0408904742282701, 0.6989617430573262, 0.0},
      {0.6989617430573259, 0.0408904742282701, 0.0},
      {0.2601477827144039, 0.6989617430573262, 0.0},
      {0.0408904742282700, 0.2601477827144042, 0.0},
      {0.3385954503697221, 0.0324948830275604, 0.0},
      {0.6289096666027175, 0.3385954503697222, 0.0},
      {0.0324948830275603, 0.6289096666027179, 0.0},
      {0.6289096666027175, 0.0324948830275604, 0.0},
      {0.3385954503697222, 0.6289096666027175, 0.0},
      {0.0324948830275602, 0.3385954503697224, 0.0},
      {0.4405249471606921, 0.0087290442899362, 0.0},
      {0.5507460085493718, 0.4405249471606921, 0.0},
      {0.0087290442899361, 0.5507460085493719, 0.0},
      {0.5507460085493717, 0.0087290442899362, 0.0},
      {0.4405249471606921, 0.5507460085493718, 0.0},
      {0.0087290442899361, 0.4405249471606923, 0.0},
      {0.0857234837407636, 0.0015747586640192, 0.0},
      {0.9127017575952171, 0.0857234837407636, 0.0},
      {0.0015747586640192, 0.9127017575952174, 0.0},
      {0.9127017575952172, 0.0015747586640192, 0.0},
      {0.0857234837407637, 0.9127017575952174, 0.0},
      {0.0015747586640189, 0.0857234837407640, 0.0},
      {0.2007383675971185, 0.2007383675971187, 0.0},
      {0.5985232648057627, 0.2007383675971186, 0.0},
      {0.2007383675971187, 0.5985232648057629, 0.0},
      {0.2859840866397785, 0.2015322055357084, 0.0},
      {0.5124837078245130, 0.2859840866397786, 0.0},
      {0.2015322055357083, 0.5124837078245132, 0.0},
      {0.5124837078245130, 0.2015322055357084, 0.0},
      {0.2859840866397785, 0.5124837078245132, 0.0},
      {0.2015322055357083, 0.2859840866397787, 0.0},
      {0.3022430648527080, 0.0514181635143590, 0.0},
      {0.6463387716329330, 0.3022430648527080, 0.0},
      {0.0514181635143590, 0.6463387716329332, 0.0},
      {0.6463387716329331, 0.0514181635143590, 0.0},
      {0.3022430648527081, 0.6463387716329331, 0.0},
      {0.0514181635143589, 0.3022430648527083, 0.0},
      {0.1145574257288858, 0.0190287488235584, 0.0},
      {0.8664138254475557, 0.1145574257288858, 0.0},
      {0.0190287488235586, 0.8664138254475557, 0.0},
      {0.8664138254475557, 0.0190287488235584, 0.0},
      {0.1145574257288858, 0.8664138254475560, 0.0},
      {0.0190287488235582, 0.1145574257288862, 0.0},
      {0.2122964300195307, 0.0407880592078402, 0.0},
      {0.7469155107726292, 0.2122964300195307, 0.0},
      {0.0407880592078402, 0.7469155107726293, 0.0},
      {0.7469155107726291, 0.0407880592078402, 0.0},
      {0.2122964300195309, 0.7469155107726292, 0.0},
      {0.0407880592078400, 0.2122964300195310, 0.0},
      {0.0462782231758691, 0.0462782231758694, 0.0},
      {0.9074435536482615, 0.0462782231758691, 0.0},
      {0.0462782231758695, 0.9074435536482617, 0.0},
      {0.2971430888632479, 0.2971430888632481, 0.0},
      {0.4057138222735041, 0.2971430888632480, 0.0},
      {0.2971430888632479, 0.4057138222735042, 0.0},
      {0.1465284297099020, 0.0285146118965766, 0.0},
      {0.8249569583935215, 0.1465284297099020, 0.0},
      {0.0285146118965767, 0.8249569583935216, 0.0},
      {0.8249569583935213, 0.0285146118965766, 0.0},
      {0.1465284297099020, 0.8249569583935216, 0.0},
      {0.0285146118965764, 0.1465284297099023, 0.0},
      {0.3740816461088270, 0.1537086586116403, 0.0},
      {0.4722096952795328, 0.3740816461088270, 0.0},
      {0.1537086586116402, 0.4722096952795329, 0.0},
      {0.4722096952795327, 0.1537086586116403, 0.0},
      {0.3740816461088270, 0.4722096952795328, 0.0},
      {0.1537086586116402, 0.3740816461088272, 0.0},
      {0.0817068303997551, 0.0246053203840889, 0.0},
      {0.8936878492161561, 0.0817068303997551, 0.0},
      {0.0246053203840890, 0.8936878492161562, 0.0},
      {0.8936878492161561, 0.0246053203840889, 0.0},
      {0.0817068303997551, 0.8936878492161562, 0.0},
      {0.0246053203840887, 0.0817068303997554, 0.0},
      {0.0108421217690070, 0.0108421217690073, 0.0},
      {0.9783157564619855, 0.0108421217690070, 0.0},
      {0.0108421217690075, 0.9783157564619858, 0.0},
      {0.2941474788808447, 0.0235355019299826, 0.0},
      {0.6823170191891728, 0.2941474788808447, 0.0},
      {0.0235355019299825, 0.6823170191891730, 0.0},
      {0.6823170191891728, 0.0235355019299826, 0.0},
      {0.2941474788808447, 0.6823170191891729, 0.0},
      {0.0235355019299824, 0.2941474788808450, 0.0},
      {0.1750703694826515, 0.1377167212601208, 0.0},
      {0.6872129092572276, 0.1750703694826516, 0.0},
      {0.1377167212601208, 0.6872129092572279, 0.0},
      {0.6872129092572277, 0.1377167212601208, 0.0},
      {0.1750703694826515, 0.6872129092572279, 0.0},
      {0.1377167212601206, 0.1750703694826518, 0.0},
      {0.1028533816631290, 0.1028533816631293, 0.0},
      {0.7942932366737417, 0.1028533816631290, 0.0},
      {0.1028533816631293, 0.7942932366737419, 0.0},
      {0.0476205378057730, 0.0109753100892318, 0.0},
      {0.9414041521049951, 0.0476205378057730, 0.0},
      {0.0109753100892320, 0.9414041521049953, 0.0},
      {0.9414041521049952, 0.0109753100892318, 0.0},
      {0.0476205378057731, 0.9414041521049953, 0.0},
      {0.0109753100892315, 0.0476205378057733, 0.0},
      {0.3175724815133269, 0.1604656572027248, 0.0},
      {0.5219618612839483, 0.3175724815133270, 0.0},
      {0.1604656572027248, 0.5219618612839485, 0.0},
      {0.5219618612839483, 0.1604656572027248, 0.0},
      {0.3175724815133269, 0.5219618612839484, 0.0},
      {0.1604656572027247, 0.3175724815133271, 0.0},
      {0.2881380539950774, 0.1329492677376998, 0.0},
      {0.5789126782672228, 0.2881380539950775, 0.0},
      {0.1329492677376997, 0.5789126782672230, 0.0},
      {0.5789126782672228, 0.1329492677376998, 0.0},
      {0.2881380539950775, 0.5789126782672228, 0.0},
      {0.1329492677376996, 0.2881380539950777, 0.0},
      {0.2679708893032962, 0.0087549512313114, 0.0},
      {0.7232741594653924, 0.2679708893032962, 0.0},
      {0.0087549512313114, 0.7232741594653926, 0.0},
      {0.7232741594653924, 0.0087549512313114, 0.0},
      {0.2679708893032962, 0.7232741594653926, 0.0},
      {0.0087549512313113, 0.2679708893032964, 0.0},
      {0.4841735634166949, 0.0316528731666103, 0.0},
      {0.4841735634166949, 0.4841735634166949, 0.0},
      {0.0316528731666102, 0.4841735634166951, 0.0},
      {0.2393790592030269, 0.0207437619833566, 0.0},
      {0.7398771788136165, 0.2393790592030269, 0.0},
      {0.0207437619833566, 0.7398771788136167, 0.0},
      {0.7398771788136166, 0.0207437619833566, 0.0},
      {0.2393790592030269, 0.7398771788136167, 0.0},
      {0.0207437619833564, 0.2393790592030272, 0.0},
      {0.1918532715697031, 0.0197602896840750, 0.0},
      {0.7883864387462218, 0.1918532715697031, 0.0},
      {0.0197602896840751, 0.7883864387462221, 0.0},
      {0.7883864387462219, 0.0197602896840750, 0.0},
      {0.1918532715697031, 0.7883864387462221, 0.0},
      {0.0197602896840748, 0.1918532715697034, 0.0},
      {0.3873242425341850, 0.0429392857768671, 0.0},
      {0.5697364716889479, 0.3873242425341851, 0.0},
      {0.0429392857768670, 0.5697364716889481, 0.0},
      {0.5697364716889479, 0.0429392857768671, 0.0},
      {0.3873242425341850, 0.5697364716889480, 0.0},
      {0.0429392857768670, 0.3873242425341852, 0.0},
      {0.3535324158284108, 0.1178924181582904, 0.0},
      {0.5285751660132987, 0.3535324158284109, 0.0},
      {0.1178924181582904, 0.5285751660132989, 0.0},
      {0.5285751660132987, 0.1178924181582904, 0.0},
      {0.3535324158284108, 0.5285751660132989, 0.0},
      {0.1178924181582903, 0.3535324158284111, 0.0},
      {0.0521774890148561, 0.0253747294625837, 0.0},
      {0.9224477815225601, 0.0521774890148561, 0.0},
      {0.0253747294625839, 0.9224477815225602, 0.0},
      {0.9224477815225601, 0.0253747294625837, 0.0},
      {0.0521774890148562, 0.9224477815225602, 0.0},
      {0.0253747294625835, 0.0521774890148565, 0.0},
      {0.3237021161513139, 0.0935293820848460, 0.0},
      {0.5827685017638400, 0.3237021161513140, 0.0},
      {0.0935293820848460, 0.5827685017638402, 0.0},
      {0.5827685017638401, 0.0935293820848460, 0.0},
      {0.3237021161513140, 0.5827685017638401, 0.0},
      {0.0935293820848459, 0.3237021161513142, 0.0},
      {0.3720593606748486, 0.0019871034880731, 0.0},
      {0.6259535358370784, 0.3720593606748486, 0.0},
      {0.0019871034880730, 0.6259535358370786, 0.0},
      {0.6259535358370782, 0.0019871034880731, 0.0},
      {0.3720593606748486, 0.6259535358370784, 0.0},
      {0.0019871034880730, 0.3720593606748488, 0.0},
      {0.4136925346071049, 0.1086901962935059, 0.0},
      {0.4776172690993892, 0.4136925346071050, 0.0},
      {0.1086901962935059, 0.4776172690993893, 0.0},
      {0.4776172690993891, 0.1086901962935059, 0.0},
      {0.4136925346071050, 0.4776172690993892, 0.0},
      {0.1086901962935058, 0.4136925346071051, 0.0},
      {0.2508654436414371, 0.1749023341509636, 0.0},
      {0.5742322222075992, 0.2508654436414372, 0.0},
      {0.1749023341509636, 0.5742322222075994, 0.0},
      {0.5742322222075992, 0.1749023341509636, 0.0},
      {0.2508654436414372, 0.5742322222075994, 0.0},
      {0.1749023341509635, 0.2508654436414374, 0.0},
      {0.2368477035387836, 0.0669659133716485, 0.0},
      {0.6961863830895678, 0.2368477035387836, 0.0},
      {0.0669659133716486, 0.6961863830895680, 0.0},
      {0.6961863830895679, 0.0669659133716485, 0.0},
      {0.2368477035387836, 0.6961863830895680, 0.0},
      {0.0669659133716483, 0.2368477035387838, 0.0},
      {0.1099576830426482, 0.0409791337707322, 0.0},
      {0.8490631831866196, 0.1099576830426482, 0.0},
      {0.0409791337707323, 0.8490631831866197, 0.0},
      {0.8490631831866197, 0.0409791337707322, 0.0},
      {0.1099576830426484, 0.8490631831866197, 0.0},
      {0.0409791337707320, 0.1099576830426485, 0.0},
      {0.4297319028951815, 0.0244664150758458, 0.0},
      {0.5458016820289727, 0.4297319028951816, 0.0},
      {0.0244664150758457, 0.5458016820289728, 0.0},
      {0.5458016820289727, 0.0244664150758458, 0.0},
      {0.4297319028951816, 0.5458016820289727, 0.0},
      {0.0244664150758457, 0.4297319028951818, 0.0},
      {0.1437680374333725, 0.0566880883410084, 0.0},
      {0.7995438742256191, 0.1437680374333725, 0.0},
      {0.0566880883410085, 0.7995438742256192, 0.0},
      {0.7995438742256191, 0.0566880883410084, 0.0},
      {0.1437680374333726, 0.7995438742256192, 0.0},
      {0.0566880883410082, 0.1437680374333728, 0.0},
      {0.0774106779986908, 0.0096524833615464, 0.0},
      {0.9129368386397628, 0.0774106779986908, 0.0},
      {0.0096524833615466, 0.9129368386397629, 0.0},
      {0.9129368386397629, 0.0096524833615464, 0.0},
      {0.0774106779986908, 0.9129368386397629, 0.0},
      {0.0096524833615462, 0.0774106779986911, 0.0},
      {0.1557060056653469, 0.0110995676273212, 0.0},
      {0.8331944267073318, 0.1557060056653469, 0.0},
      {0.0110995676273212, 0.8331944267073321, 0.0},
      {0.8331944267073320, 0.0110995676273212, 0.0},
      {0.1557060056653469, 0.8331944267073321, 0.0},
      {0.0110995676273209, 0.1557060056653472, 0.0},
      {0.3128533024496512, 0.0019740717879365, 0.0},
      {0.6851726257624122, 0.3128533024496513, 0.0},
      {0.0019740717879365, 0.6851726257624124, 0.0},
      {0.6851726257624123, 0.0019740717879365, 0.0},
      {0.3128533024496513, 0.6851726257624124, 0.0},
      {0.0019740717879363, 0.3128533024496515, 0.0},
      {0.0260902661381798, 0.0110588672805733, 0.0},
      {0.9628508665812467, 0.0260902661381798, 0.0},
      {0.0110588672805734, 0.9628508665812471, 0.0},
      {0.9628508665812469, 0.0110588672805733, 0.0},
      {0.0260902661381799, 0.9628508665812471, 0.0},
      {0.0110588672805730, 0.0260902661381801, 0.0},
      {0.1656728608179916, 0.0024625444223662, 0.0},
      {0.8318645947596423, 0.1656728608179916, 0.0},
      {0.0024625444223663, 0.8318645947596424, 0.0},
      {0.8318645947596421, 0.0024625444223662, 0.0},
      {0.1656728608179916, 0.8318645947596424, 0.0},
      {0.0024625444223660, 0.1656728608179919, 0.0},
      {0.4932611955660527, 0.0134776088678946, 0.0},
      {0.4932611955660527, 0.4932611955660528, 0.0},
      {0.0134776088678945, 0.4932611955660529, 0.0},
      {0.0742257546763114, 0.0463950129826374, 0.0},
      {0.8793792323410512, 0.0742257546763114, 0.0},
      {0.0463950129826375, 0.8793792323410514, 0.0},
      {0.8793792323410513, 0.0463950129826374, 0.0},
      {0.0742257546763114, 0.8793792323410514, 0.0},
      {0.0463950129826371, 0.0742257546763117, 0.0},
      {0.0288339172370140, 0.0020785999248742, 0.0},
      {0.9690874828381117, 0.0288339172370141, 0.0},
      {0.0020785999248743, 0.9690874828381119, 0.0},
      {0.9690874828381117, 0.0020785999248742, 0.0},
      {0.0288339172370142, 0.9690874828381119, 0.0},
      {0.0020785999248739, 0.0288339172370144, 0.0},
      {0.1868226606723151, 0.0683532967983220, 0.0},
      {0.7448240425293629, 0.1868226606723151, 0.0},
      {0.0683532967983221, 0.7448240425293630, 0.0},
      {0.7448240425293629, 0.0683532967983220, 0.0},
      {0.1868226606723152, 0.7448240425293630, 0.0},
      {0.0683532967983218, 0.1868226606723153, 0.0},
      {0.3270839281856873, 0.0106763043029805, 0.0},
      {0.6622397675113322, 0.3270839281856874, 0.0},
      {0.0106763043029805, 0.6622397675113323, 0.0},
      {0.6622397675113322, 0.0106763043029805, 0.0},
      {0.3270839281856875, 0.6622397675113322, 0.0},
      {0.0106763043029803, 0.3270839281856877, 0.0},
      {0.1038911034338268, 0.0691692834636133, 0.0},
      {0.8269396131025599, 0.1038911034338268, 0.0},
      {0.0691692834636134, 0.8269396131025600, 0.0},
      {0.8269396131025599, 0.0691692834636133, 0.0},
      {0.1038911034338268, 0.8269396131025600, 0.0},
      {0.0691692834636131, 0.1038911034338271, 0.0},
      {0.2155342882429546, 0.1006688094346437, 0.0},
      {0.6837969023224015, 0.2155342882429546, 0.0},
      {0.1006688094346437, 0.6837969023224019, 0.0},
      {0.6837969023224016, 0.1006688094346437, 0.0},
      {0.2155342882429546, 0.6837969023224019, 0.0},
      {0.1006688094346435, 0.2155342882429548, 0.0},
      {0.4057324719755224, 0.1885350560489551, 0.0},
      {0.4057324719755224, 0.4057324719755226, 0.0},
      {0.1885350560489550, 0.4057324719755226, 0.0},
      {0.3431516130273899, 0.2012179375284443, 0.0},
      {0.4556304494441659, 0.3431516130273900, 0.0},
      {0.2012179375284442, 0.4556304494441660, 0.0},
      {0.4556304494441658, 0.2012179375284443, 0.0},
      {0.3431516130273899, 0.4556304494441659, 0.0},
      {0.2012179375284441, 0.3431516130273902, 0.0},
      {0.2224087587233254, 0.1434532484926369, 0.0},
      {0.6341379927840377, 0.2224087587233255, 0.0},
      {0.1434532484926369, 0.6341379927840379, 0.0},
      {0.6341379927840377, 0.1434532484926369, 0.0},
      {0.2224087587233254, 0.6341379927840379, 0.0},
      {0.1434532484926367, 0.2224087587233256, 0.0},
      {0.0534809769143942, 0.0021251605376943, 0.0},
      {0.9443938625479114, 0.0534809769143942, 0.0},
      {0.0021251605376946, 0.9443938625479115, 0.0},
      {0.9443938625479115, 0.0021251605376943, 0.0},
      {0.0534809769143944, 0.9443938625479115, 0.0},
      {0.0021251605376941, 0.0534809769143946, 0.0},
      {0.1304216449251314, 0.1304216449251317, 0.0},
      {0.7391567101497368, 0.1304216449251314, 0.0},
      {0.1304216449251318, 0.7391567101497369, 0.0},
      {0.4387889688450269, 0.0514225024494020, 0.0},
      {0.5097885287055710, 0.4387889688450270, 0.0},
      {0.0514225024494020, 0.5097885287055712, 0.0},
      {0.5097885287055710, 0.0514225024494020, 0.0},
      {0.4387889688450270, 0.5097885287055710, 0.0},
      {0.0514225024494020, 0.4387889688450272, 0.0},
      {0.4614348056302963, 0.0771303887394075, 0.0},
      {0.4614348056302963, 0.4614348056302963, 0.0},
      {0.0771303887394074, 0.4614348056302964, 0.0},
      {0.0275739602431112, 0.0275739602431115, 0.0},
      {0.9448520795137773, 0.0275739602431112, 0.0},
      {0.0275739602431116, 0.9448520795137775, 0.0},
      {0.2318424453754739, 0.2318424453754741, 0.0},
      {0.5363151092490520, 0.2318424453754740, 0.0},
      {0.2318424453754740, 0.5363151092490521, 0.0},
      {0.3517217766277714, 0.2965564467444571, 0.0},
      {0.3517217766277714, 0.3517217766277715, 0.0},
      {0.2965564467444570, 0.3517217766277716, 0.0},
      {0.0022124213099818, 0.0022124213099821, 0.0},
      {0.9955751573800362, 0.0022124213099817, 0.0},
      {0.0022124213099822, 0.9955751573800363, 0.0},
      {0.3919321013755558, 0.0781526329255783, 0.0},
      {0.5299152656988659, 0.3919321013755560, 0.0},
      {0.0781526329255782, 0.5299152656988662, 0.0},
      {0.5299152656988659, 0.0781526329255783, 0.0},
      {0.3919321013755558, 0.5299152656988660, 0.0},
      {0.0781526329255781, 0.3919321013755561, 0.0},
      {0.2110821798970346, 0.0064969244459218, 0.0},
      {0.7824208956570436, 0.2110821798970346, 0.0},
      {0.0064969244459219, 0.7824208956570438, 0.0},
      {0.7824208956570435, 0.0064969244459218, 0.0},
      {0.2110821798970346, 0.7824208956570438, 0.0},
      {0.0064969244459216, 0.2110821798970349, 0.0},
      {0.0117098573109518, 0.0020903064191529, 0.0},
      {0.9861998362698954, 0.0117098573109518, 0.0},
      {0.0020903064191530, 0.9861998362698955, 0.0},
      {0.9861998362698954, 0.0020903064191529, 0.0},
      {0.0117098573109518, 0.9861998362698955, 0.0},
      {0.0020903064191526, 0.0117098573109521, 0.0},
      {0.4987895646915128, 0.0024208706169743, 0.0},
      {0.4987895646915129, 0.4987895646915129, 0.0},
      {0.0024208706169742, 0.4987895646915131, 0.0},
      {0.2063833537599898, 0.0001398635181475, 0.0},
      {0.7934767827218625, 0.2063833537599899, 0.0},
      {0.0001398635181475, 0.7934767827218628, 0.0},
      {0.7934767827218626, 0.0001398635181475, 0.0},
      {0.2063833537599898, 0.7934767827218628, 0.0},
      {0.0001398635181473, 0.2063833537599901, 0.0}},
      {0.0007589746913754,
      0.0007589746913754,
      0.0007589746913754,
      0.0007589746913754,
      0.0007589746913754,
      0.0007589746913754,
      0.0004820181544435,
      0.0004820181544435,
      0.0004820181544435,
      0.0004820181544435,
      0.0004820181544435,
      0.0004820181544435,
      0.0001406751832661,
      0.0001406751832661,
      0.0001406751832661,
      0.0001406751832661,
      0.0001406751832661,
      0.0001406751832661,
      0.0016149388372262,
      0.0016149388372262,
      0.0016149388372262,
      0.0023431106687326,
      0.0023431106687326,
      0.0023431106687326,
      0.0023431106687326,
      0.0023431106687326,
      0.0023431106687326,
      0.0022589494991916,
      0.0022589494991916,
      0.0022589494991916,
      0.0022589494991916,
      0.0022589494991916,
      0.0022589494991916,
      0.0007895458930092,
      0.0007895458930092,
      0.0007895458930092,
      0.0002132067406496,
      0.0002132067406496,
      0.0002132067406496,
      0.0002132067406496,
      0.0002132067406496,
      0.0002132067406496,
      0.0016234183150526,
      0.0016234183150526,
      0.0016234183150526,
      0.0016234183150526,
      0.0016234183150526,
      0.0016234183150526,
      0.0019654527394101,
      0.0019654527394101,
      0.0019654527394101,
      0.0013556749468707,
      0.0013556749468707,
      0.0013556749468707,
      0.0013556749468707,
      0.0013556749468707,
      0.0013556749468707,
      0.0008163791134712,
      0.0008163791134712,
      0.0008163791134712,
      0.0008163791134712,
      0.0008163791134712,
      0.0008163791134712,
      0.0002395150880895,
      0.0002395150880895,
      0.0002395150880895,
      0.0002395150880895,
      0.0002395150880895,
      0.0002395150880895,
      0.0013001359147194,
      0.0013001359147194,
      0.0013001359147194,
      0.0013001359147194,
      0.0013001359147194,
      0.0013001359147194,
      0.0015369038310028,
      0.0015369038310028,
      0.0015369038310028,
      0.0015369038310028,
      0.0015369038310028,
      0.0015369038310028,
      0.0014939099251137,
      0.0014939099251137,
      0.0014939099251137,
      0.0014939099251137,
      0.0014939099251137,
      0.0014939099251137,
      0.0024979296124831,
      0.0024979296124831,
      0.0024979296124831,
      0.0003462956748393,
      0.0003462956748393,
      0.0003462956748393,
      0.0003462956748393,
      0.0003462956748393,
      0.0003462956748393,
      0.0011320947479483,
      0.0011320947479483,
      0.0011320947479483,
      0.0011320947479483,
      0.0011320947479483,
      0.0011320947479483,
      0.0010841809686357,
      0.0010841809686357,
      0.0010841809686357,
      0.0010841809686357,
      0.0010841809686357,
      0.0010841809686357,
      0.0006301672412378,
      0.0006301672412378,
      0.0006301672412378,
      0.0006301672412378,
      0.0006301672412378,
      0.0006301672412378,
      0.0001509640027210,
      0.0001509640027210,
      0.0001509640027210,
      0.0001509640027210,
      0.0001509640027210,
      0.0001509640027210,
      0.0020389594514398,
      0.0020389594514398,
      0.0020389594514398,
      0.0022808115137282,
      0.0022808115137282,
      0.0022808115137282,
      0.0022808115137282,
      0.0022808115137282,
      0.0022808115137282,
      0.0013215713819515,
      0.0013215713819515,
      0.0013215713819515,
      0.0013215713819515,
      0.0013215713819515,
      0.0013215713819515,
      0.0005657299646417,
      0.0005657299646417,
      0.0005657299646417,
      0.0005657299646417,
      0.0005657299646417,
      0.0005657299646417,
      0.0010544365941910,
      0.0010544365941910,
      0.0010544365941910,
      0.0010544365941910,
      0.0010544365941910,
      0.0010544365941910,
      0.0005858932741372,
      0.0005858932741372,
      0.0005858932741372,
      0.0025420268789960,
      0.0025420268789960,
      0.0025420268789960,
      0.0007804093505606,
      0.0007804093505606,
      0.0007804093505606,
      0.0007804093505606,
      0.0007804093505606,
      0.0007804093505606,
      0.0023154255562052,
      0.0023154255562052,
      0.0023154255562052,
      0.0023154255562052,
      0.0023154255562052,
      0.0023154255562052,
      0.0005824189122617,
      0.0005824189122617,
      0.0005824189122617,
      0.0005824189122617,
      0.0005824189122617,
      0.0005824189122617,
      0.0001454967198956,
      0.0001454967198956,
      0.0001454967198956,
      0.0009539237935564,
      0.0009539237935564,
      0.0009539237935564,
      0.0009539237935564,
      0.0009539237935564,
      0.0009539237935564,
      0.0018103679665177,
      0.0018103679665177,
      0.0018103679665177,
      0.0018103679665177,
      0.0018103679665177,
      0.0018103679665177,
      0.0012339913193696,
      0.0012339913193696,
      0.0012339913193696,
      0.0003023529583056,
      0.0003023529583056,
      0.0003023529583056,
      0.0003023529583056,
      0.0003023529583056,
      0.0003023529583056,
      0.0022357890144113,
      0.0022357890144113,
      0.0022357890144113,
      0.0022357890144113,
      0.0022357890144113,
      0.0022357890144113,
      0.0021608932779723,
      0.0021608932779723,
      0.0021608932779723,
      0.0021608932779723,
      0.0021608932779723,
      0.0021608932779723,
      0.0005917229840015,
      0.0005917229840015,
      0.0005917229840015,
      0.0005917229840015,
      0.0005917229840015,
      0.0005917229840015,
      0.0012237774459376,
      0.0012237774459376,
      0.0012237774459376,
      0.0008549230596296,
      0.0008549230596296,
      0.0008549230596296,
      0.0008549230596296,
      0.0008549230596296,
      0.0008549230596296,
      0.0007630333887660,
      0.0007630333887660,
      0.0007630333887660,
      0.0007630333887660,
      0.0007630333887660,
      0.0007630333887660,
      0.0013764510347318,
      0.0013764510347318,
      0.0013764510347318,
      0.0013764510347318,
      0.0013764510347318,
      0.0013764510347318,
      0.0021816136183956,
      0.0021816136183956,
      0.0021816136183956,
      0.0021816136183956,
      0.0021816136183956,
      0.0021816136183956,
      0.0004831426781163,
      0.0004831426781163,
      0.0004831426781163,
      0.0004831426781163,
      0.0004831426781163,
      0.0004831426781163,
      0.0018907763340343,
      0.0018907763340343,
      0.0018907763340343,
      0.0018907763340343,
      0.0018907763340343,
      0.0018907763340343,
      0.0003017846563734,
      0.0003017846563734,
      0.0003017846563734,
      0.0003017846563734,
      0.0003017846563734,
      0.0003017846563734,
      0.0021275819804195,
      0.0021275819804195,
      0.0021275819804195,
      0.0021275819804195,
      0.0021275819804195,
      0.0021275819804195,
      0.0024245266612046,
      0.0024245266612046,
      0.0024245266612046,
      0.0024245266612046,
      0.0024245266612046,
      0.0024245266612046,
      0.0015127459076676,
      0.0015127459076676,
      0.0015127459076676,
      0.0015127459076676,
      0.0015127459076676,
      0.0015127459076676,
      0.0009151329554746,
      0.0009151329554746,
      0.0009151329554746,
      0.0009151329554746,
      0.0009151329554746,
      0.0009151329554746,
      0.0011556455726362,
      0.0011556455726362,
      0.0011556455726362,
      0.0011556455726362,
      0.0011556455726362,
      0.0011556455726362,
      0.0011488090641466,
      0.0011488090641466,
      0.0011488090641466,
      0.0011488090641466,
      0.0011488090641466,
      0.0011488090641466,
      0.0003868713007276,
      0.0003868713007276,
      0.0003868713007276,
      0.0003868713007276,
      0.0003868713007276,
      0.0003868713007276,
      0.0005545553953316,
      0.0005545553953316,
      0.0005545553953316,
      0.0005545553953316,
      0.0005545553953316,
      0.0005545553953316,
      0.0002958640035393,
      0.0002958640035393,
      0.0002958640035393,
      0.0002958640035393,
      0.0002958640035393,
      0.0002958640035393,
      0.0002331645391699,
      0.0002331645391699,
      0.0002331645391699,
      0.0002331645391699,
      0.0002331645391699,
      0.0002331645391699,
      0.0002530272571071,
      0.0002530272571071,
      0.0002530272571071,
      0.0002530272571071,
      0.0002530272571071,
      0.0002530272571071,
      0.0008417355883929,
      0.0008417355883929,
      0.0008417355883929,
      0.0007731026217951,
      0.0007731026217951,
      0.0007731026217951,
      0.0007731026217951,
      0.0007731026217951,
      0.0007731026217951,
      0.0001119083862635,
      0.0001119083862635,
      0.0001119083862635,
      0.0001119083862635,
      0.0001119083862635,
      0.0001119083862635,
      0.0014780925919470,
      0.0014780925919470,
      0.0014780925919470,
      0.0014780925919470,
      0.0014780925919470,
      0.0014780925919470,
      0.0007145209245851,
      0.0007145209245851,
      0.0007145209245851,
      0.0007145209245851,
      0.0007145209245851,
      0.0007145209245851,
      0.0010729346742088,
      0.0010729346742088,
      0.0010729346742088,
      0.0010729346742088,
      0.0010729346742088,
      0.0010729346742088,
      0.0017923837097285,
      0.0017923837097285,
      0.0017923837097285,
      0.0017923837097285,
      0.0017923837097285,
      0.0017923837097285,
      0.0026827112703014,
      0.0026827112703014,
      0.0026827112703014,
      0.0027263678891136,
      0.0027263678891136,
      0.0027263678891136,
      0.0027263678891136,
      0.0027263678891136,
      0.0027263678891136,
      0.0021908328878286,
      0.0021908328878286,
      0.0021908328878286,
      0.0021908328878286,
      0.0021908328878286,
      0.0021908328878286,
      0.0001535872679823,
      0.0001535872679823,
      0.0001535872679823,
      0.0001535872679823,
      0.0001535872679823,
      0.0001535872679823,
      0.0017496340158808,
      0.0017496340158808,
      0.0017496340158808,
      0.0016190885082294,
      0.0016190885082294,
      0.0016190885082294,
      0.0016190885082294,
      0.0016190885082294,
      0.0016190885082294,
      0.0019800098957544,
      0.0019800098957544,
      0.0019800098957544,
      0.0004159754444519,
      0.0004159754444519,
      0.0004159754444519,
      0.0026822045354474,
      0.0026822045354474,
      0.0026822045354474,
      0.0030633868037771,
      0.0030633868037771,
      0.0030633868037771,
      0.0000323057149543,
      0.0000323057149543,
      0.0000323057149543,
      0.0020526914294637,
      0.0020526914294637,
      0.0020526914294637,
      0.0020526914294637,
      0.0020526914294637,
      0.0020526914294637,
      0.0005341793661711,
      0.0005341793661711,
      0.0005341793661711,
      0.0005341793661711,
      0.0005341793661711,
      0.0005341793661711,
      0.0000711599458075,
      0.0000711599458075,
      0.0000711599458075,
      0.0000711599458075,
      0.0000711599458075,
      0.0000711599458075,
      0.0003942775827266,
      0.0003942775827266,
      0.0003942775827266,
      0.0000802792352761,
      0.0000802792352761,
      0.0000802792352761,
      0.0000802792352761,
      0.0000802792352761,
      0.0000802792352761}
    },
    {
      453,
      {{0.2563558633847908, 0.2563558633847910, 0.0},
      {0.4872882732304182, 0.2563558633847909, 0.0},
      {0.2563558633847909, 0.4872882732304183, 0.0},
      {0.4433230868583420, 0.0009224561060705, 0.0},
      {0.5557544570355875, 0.4433230868583421, 0.0},
      {0.0009224561060704, 0.5557544570355878, 0.0},
      {0.5557544570355875, 0.0009224561060705, 0.0},
      {0.4433230868583420, 0.5557544570355876, 0.0},
      {0.0009224561060704, 0.4433230868583423, 0.0},
      {0.0259368938757904, 0.0224379912623131, 0.0},
      {0.9516251148618965, 0.0259368938757903, 0.0},
      {0.0224379912623133, 0.9516251148618966, 0.0},
      {0.9516251148618965, 0.0224379912623131, 0.0},
      {0.0259368938757906, 0.9516251148618966, 0.0},
      {0.0224379912623128, 0.0259368938757907, 0.0},
      {0.2401441052632695, 0.2227907282937329, 0.0},
      {0.5370651664429976, 0.2401441052632695, 0.0},
      {0.2227907282937328, 0.5370651664429978, 0.0},
      {0.5370651664429976, 0.2227907282937329, 0.0},
      {0.2401441052632695, 0.5370651664429978, 0.0},
      {0.2227907282937327, 0.2401441052632697, 0.0},
      {0.1345526180266427, 0.0231202139702463, 0.0},
      {0.8423271680031109, 0.1345526180266427, 0.0},
      {0.0231202139702464, 0.8423271680031111, 0.0},
      {0.8423271680031110, 0.0231202139702463, 0.0},
      {0.1345526180266427, 0.8423271680031111, 0.0},
      {0.0231202139702461, 0.1345526180266430, 0.0},
      {0.0450355724673314, 0.0008567797950090, 0.0},
      {0.9541076477376595, 0.0450355724673314, 0.0},
      {0.0008567797950091, 0.9541076477376597, 0.0},
      {0.9541076477376597, 0.0008567797950090, 0.0},
      {0.0450355724673315, 0.9541076477376597, 0.0},
      {0.0008567797950087, 0.0450355724673318, 0.0},
      {0.4087145665677719, 0.0048086376608720, 0.0},
      {0.5864767957713561, 0.4087145665677719, 0.0},
      {0.0048086376608720, 0.5864767957713563, 0.0},
      {0.5864767957713561, 0.0048086376608720, 0.0},
      {0.4087145665677719, 0.5864767957713561, 0.0},
      {0.0048086376608719, 0.4087145665677722, 0.0},
      {0.1714634577399832, 0.0234930123594386, 0.0},
      {0.8050435299005781, 0.1714634577399832, 0.0},
      {0.0234930123594386, 0.8050435299005784, 0.0},
      {0.8050435299005783, 0.0234930123594386, 0.0},
      {0.1714634577399832, 0.8050435299005784, 0.0},
      {0.0234930123594384, 0.1714634577399836, 0.0},
      {0.0016305781421460, 0.0016305781421463, 0.0},
      {0.9967388437157076, 0.0016305781421460, 0.0},
      {0.0016305781421465, 0.9967388437157076, 0.0},
      {0.2101059247050331, 0.0933966471293636, 0.0},
      {0.6964974281656032, 0.2101059247050332, 0.0},
      {0.0933966471293636, 0.6964974281656034, 0.0},
      {0.6964974281656033, 0.0933966471293636, 0.0},
      {0.2101059247050331, 0.6964974281656034, 0.0},
      {0.0933966471293634, 0.2101059247050334, 0.0},
      {0.1392968201185850, 0.0417966366511482, 0.0},
      {0.8189065432302668, 0.1392968201185851, 0.0},
      {0.0417966366511482, 0.8189065432302669, 0.0},
      {0.8189065432302668, 0.0417966366511482, 0.0},
      {0.1392968201185851, 0.8189065432302669, 0.0},
      {0.0417966366511480, 0.1392968201185854, 0.0},
      {0.0703820539127592, 0.0493573118928613, 0.0},
      {0.8802606341943795, 0.0703820539127592, 0.0},
      {0.0493573118928615, 0.8802606341943795, 0.0},
      {0.8802606341943796, 0.0493573118928613, 0.0},
      {0.0703820539127593, 0.8802606341943796, 0.0},
      {0.0493573118928611, 0.0703820539127595, 0.0},
      {0.4903268692484512, 0.0193462615030976, 0.0},
      {0.4903268692484513, 0.4903268692484513, 0.0},
      {0.0193462615030975, 0.4903268692484514, 0.0},
      {0.3726064881564940, 0.1528141070121611, 0.0},
      {0.4745794048313449, 0.3726064881564941, 0.0},
      {0.1528141070121611, 0.4745794048313450, 0.0},
      {0.4745794048313449, 0.1528141070121611, 0.0},
      {0.3726064881564940, 0.4745794048313449, 0.0},
      {0.1528141070121610, 0.3726064881564942, 0.0},
      {0.1479157172776441, 0.0651154299306216, 0.0},
      {0.7869688527917342, 0.1479157172776441, 0.0},
      {0.0651154299306216, 0.7869688527917345, 0.0},
      {0.7869688527917343, 0.0651154299306216, 0.0},
      {0.1479157172776442, 0.7869688527917345, 0.0},
      {0.0651154299306213, 0.1479157172776444, 0.0},
      {0.2140246435816437, 0.0234621431760208, 0.0},
      {0.7625132132423355, 0.2140246435816437, 0.0},
      {0.0234621431760209, 0.7625132132423356, 0.0},
      {0.7625132132423355, 0.0234621431760208, 0.0},
      {0.2140246435816437, 0.7625132132423356, 0.0},
      {0.0234621431760206, 0.2140246435816440, 0.0},
      {0.3302674030581440, 0.0520322782344111, 0.0},
      {0.6177003187074449, 0.3302674030581440, 0.0},
      {0.0520322782344111, 0.6177003187074451, 0.0},
      {0.6177003187074449, 0.0520322782344111, 0.0},
      {0.3302674030581441, 0.6177003187074450, 0.0},
      {0.0520322782344110, 0.3302674030581442, 0.0},
      {0.3699848635444039, 0.0014945191055323, 0.0},
      {0.6285206173500637, 0.3699848635444040, 0.0},
      {0.0014945191055323, 0.6285206173500639, 0.0},
      {0.6285206173500638, 0.0014945191055323, 0.0},
      {0.3699848635444039, 0.6285206173500638, 0.0},
      {0.0014945191055322, 0.3699848635444041, 0.0},
      {0.0675115690433361, 0.0120437469347925, 0.0},
      {0.9204446840218713, 0.0675115690433361, 0.0},
      {0.0120437469347927, 0.9204446840218714, 0.0},
      {0.9204446840218714, 0.0120437469347925, 0.0},
      {0.0675115690433362, 0.9204446840218714, 0.0},
      {0.0120437469347923, 0.0675115690433364, 0.0},
      {0.4370988411800967, 0.0566847855434612, 0.0},
      {0.5062163732764420, 0.4370988411800968, 0.0},
      {0.0566847855434612, 0.5062163732764422, 0.0},
      {0.5062163732764420, 0.0566847855434612, 0.0},
      {0.4370988411800968, 0.5062163732764421, 0.0},
      {0.0566847855434611, 0.4370988411800970, 0.0},
      {0.4225871685075206, 0.1548256629849588, 0.0},
      {0.4225871685075206, 0.4225871685075207, 0.0},
      {0.1548256629849586, 0.4225871685075208, 0.0},
      {0.4145122211577853, 0.1148601420448157, 0.0},
      {0.4706276367973991, 0.4145122211577854, 0.0},
      {0.1148601420448156, 0.4706276367973992, 0.0},
      {0.4706276367973991, 0.1148601420448157, 0.0},
      {0.4145122211577853, 0.4706276367973991, 0.0},
      {0.1148601420448156, 0.4145122211577855, 0.0},
      {0.3623805122689274, 0.0312648361867089, 0.0},
      {0.6063546515443636, 0.3623805122689275, 0.0},
      {0.0312648361867088, 0.6063546515443639, 0.0},
      {0.6063546515443636, 0.0312648361867089, 0.0},
      {0.3623805122689275, 0.6063546515443637, 0.0},
      {0.0312648361867088, 0.3623805122689277, 0.0},
      {0.3122512855112035, 0.1022466401455418, 0.0},
      {0.5855020743432547, 0.3122512855112036, 0.0},
      {0.1022466401455417, 0.5855020743432550, 0.0},
      {0.5855020743432547, 0.1022466401455418, 0.0},
      {0.3122512855112035, 0.5855020743432549, 0.0},
      {0.1022466401455416, 0.3122512855112037, 0.0},
      {0.4805756764340300, 0.0388486471319399, 0.0},
      {0.4805756764340301, 0.4805756764340301, 0.0},
      {0.0388486471319398, 0.4805756764340303, 0.0},
      {0.3926893217036879, 0.0483493018730729, 0.0},
      {0.5589613764232392, 0.3926893217036880, 0.0},
      {0.0483493018730729, 0.5589613764232394, 0.0},
      {0.5589613764232392, 0.0483493018730729, 0.0},
      {0.3926893217036880, 0.5589613764232393, 0.0},
      {0.0483493018730728, 0.3926893217036881, 0.0},
      {0.0260638541622880, 0.0099795769296427, 0.0},
      {0.9639565689080692, 0.0260638541622880, 0.0},
      {0.0099795769296427, 0.9639565689080695, 0.0},
      {0.9639565689080694, 0.0099795769296427, 0.0},
      {0.0260638541622881, 0.9639565689080695, 0.0},
      {0.0099795769296424, 0.0260638541622883, 0.0},
      {0.3983909958449554, 0.0839129407699284, 0.0},
      {0.5176960633851162, 0.3983909958449555, 0.0},
      {0.0839129407699283, 0.5176960633851163, 0.0},
      {0.5176960633851162, 0.0839129407699284, 0.0},
      {0.3983909958449555, 0.5176960633851163, 0.0},
      {0.0839129407699283, 0.3983909958449556, 0.0},
      {0.0458803913209149, 0.0067281317012843, 0.0},
      {0.9473914769778008, 0.0458803913209149, 0.0},
      {0.0067281317012844, 0.9473914769778009, 0.0},
      {0.9473914769778008, 0.0067281317012843, 0.0},
      {0.0458803913209150, 0.9473914769778009, 0.0},
      {0.0067281317012840, 0.0458803913209153, 0.0},
      {0.2510983078651668, 0.1773080248323685, 0.0},
      {0.5715936673024646, 0.2510983078651669, 0.0},
      {0.1773080248323685, 0.5715936673024649, 0.0},
      {0.5715936673024647, 0.1773080248323685, 0.0},
      {0.2510983078651668, 0.5715936673024649, 0.0},
      {0.1773080248323682, 0.2510983078651671, 0.0},
      {0.1831369339346112, 0.0426840877350856, 0.0},
      {0.7741789783303032, 0.1831369339346113, 0.0},
      {0.0426840877350856, 0.7741789783303034, 0.0},
      {0.7741789783303031, 0.0426840877350856, 0.0},
      {0.1831369339346113, 0.7741789783303031, 0.0},
      {0.0426840877350855, 0.1831369339346115, 0.0},
      {0.3540463032783623, 0.0743927680400382, 0.0},
      {0.5715609286815995, 0.3540463032783624, 0.0},
      {0.0743927680400381, 0.5715609286815998, 0.0},
      {0.5715609286815995, 0.0743927680400382, 0.0},
      {0.3540463032783623, 0.5715609286815997, 0.0},
      {0.0743927680400380, 0.3540463032783625, 0.0},
      {0.2992109871101671, 0.2435689477970363, 0.0},
      {0.4572200650927965, 0.2992109871101672, 0.0},
      {0.2435689477970363, 0.4572200650927967, 0.0},
      {0.4572200650927966, 0.2435689477970363, 0.0},
      {0.2992109871101671, 0.4572200650927967, 0.0},
      {0.2435689477970362, 0.2992109871101673, 0.0},
      {0.2908392329854116, 0.0763915697459878, 0.0},
      {0.6327691972686006, 0.2908392329854117, 0.0},
      {0.0763915697459878, 0.6327691972686007, 0.0},
      {0.6327691972686005, 0.0763915697459878, 0.0},
      {0.2908392329854118, 0.6327691972686006, 0.0},
      {0.0763915697459877, 0.2908392329854119, 0.0},
      {0.1861461318533070, 0.1242245896688861, 0.0},
      {0.6896292784778069, 0.1861461318533070, 0.0},
      {0.1242245896688862, 0.6896292784778070, 0.0},
      {0.6896292784778069, 0.1242245896688861, 0.0},
      {0.1861461318533070, 0.6896292784778070, 0.0},
      {0.1242245896688860, 0.1861461318533073, 0.0},
      {0.1336271749712246, 0.0097790083705655, 0.0},
      {0.8565938166582099, 0.1336271749712246, 0.0},
      {0.0097790083705656, 0.8565938166582100, 0.0},
      {0.8565938166582099, 0.0097790083705655, 0.0},
      {0.1336271749712248, 0.8565938166582099, 0.0},
      {0.0097790083705652, 0.1336271749712250, 0.0},
      {0.0689763351482116, 0.0279432020481641, 0.0},
      {0.9030804628036242, 0.0689763351482116, 0.0},
      {0.0279432020481643, 0.9030804628036244, 0.0},
      {0.9030804628036242, 0.0279432020481641, 0.0},
      {0.0689763351482117, 0.9030804628036244, 0.0},
      {0.0279432020481639, 0.0689763351482120, 0.0},
      {0.2614910192698080, 0.0248082685108521, 0.0},
      {0.7137007122193398, 0.2614910192698081, 0.0},
      {0.0248082685108522, 0.7137007122193399, 0.0},
      {0.7137007122193398, 0.0248082685108521, 0.0},
      {0.2614910192698081, 0.7137007122193399, 0.0},
      {0.0248082685108520, 0.2614910192698083, 0.0},
      {0.0439794560591168, 0.0211260004278839, 0.0},
      {0.9348945435129993, 0.0439794560591167, 0.0},
      {0.0211260004278840, 0.9348945435129995, 0.0},
      {0.9348945435129993, 0.0211260004278839, 0.0},
      {0.0439794560591169, 0.9348945435129995, 0.0},
      {0.0211260004278836, 0.0439794560591171, 0.0},
      {0.2469572317073108, 0.0681498702540453, 0.0},
      {0.6848928980386439, 0.2469572317073107, 0.0},
      {0.0681498702540453, 0.6848928980386442, 0.0},
      {0.6848928980386439, 0.0681498702540453, 0.0},
      {0.2469572317073108, 0.6848928980386441, 0.0},
      {0.0681498702540452, 0.2469572317073110, 0.0},
      {0.1916073029798411, 0.0687953069004347, 0.0},
      {0.7395973901197243, 0.1916073029798411, 0.0},
      {0.0687953069004347, 0.7395973901197244, 0.0},
      {0.7395973901197241, 0.0687953069004347, 0.0},
      {0.1916073029798412, 0.7395973901197243, 0.0},
      {0.0687953069004345, 0.1916073029798415, 0.0},
      {0.3173959823085313, 0.1563534771200215, 0.0},
      {0.5262505405714473, 0.3173959823085313, 0.0},
      {0.1563534771200214, 0.5262505405714475, 0.0},
      {0.5262505405714473, 0.1563534771200215, 0.0},
      {0.3173959823085312, 0.5262505405714474, 0.0},
      {0.1563534771200213, 0.3173959823085315, 0.0},
      {0.3938393062998427, 0.0157887711356748, 0.0},
      {0.5903719225644825, 0.3938393062998428, 0.0},
      {0.0157887711356747, 0.5903719225644827, 0.0},
      {0.5903719225644825, 0.0157887711356748, 0.0},
      {0.3938393062998428, 0.5903719225644826, 0.0},
      {0.0157887711356747, 0.3938393062998430, 0.0},
      {0.3181446682127857, 0.0234464079672303, 0.0},
      {0.6584089238199839, 0.3181446682127858, 0.0},
      {0.0234464079672303, 0.6584089238199842, 0.0},
      {0.6584089238199841, 0.0234464079672303, 0.0},
      {0.3181446682127858, 0.6584089238199841, 0.0},
      {0.0234464079672302, 0.3181446682127860, 0.0},
      {0.2803644875579968, 0.1359298134713862, 0.0},
      {0.5837056989706170, 0.2803644875579969, 0.0},
      {0.1359298134713862, 0.5837056989706172, 0.0},
      {0.5837056989706170, 0.1359298134713862, 0.0},
      {0.2803644875579968, 0.5837056989706172, 0.0},
      {0.1359298134713861, 0.2803644875579970, 0.0},
      {0.2299757300841567, 0.0446955970130421, 0.0},
      {0.7253286729028012, 0.2299757300841567, 0.0},
      {0.0446955970130422, 0.7253286729028013, 0.0},
      {0.7253286729028012, 0.0446955970130421, 0.0},
      {0.2299757300841567, 0.7253286729028013, 0.0},
      {0.0446955970130419, 0.2299757300841569, 0.0},
      {0.3494881250371351, 0.2460562506891596, 0.0},
      {0.4044556242737054, 0.3494881250371351, 0.0},
      {0.2460562506891595, 0.4044556242737055, 0.0},
      {0.4044556242737053, 0.2460562506891596, 0.0},
      {0.3494881250371351, 0.4044556242737055, 0.0},
      {0.2460562506891595, 0.3494881250371352, 0.0},
      {0.0964083579062225, 0.0107634727377169, 0.0},
      {0.8928281693560606, 0.0964083579062225, 0.0},
      {0.0107634727377169, 0.8928281693560609, 0.0},
      {0.8928281693560607, 0.0107634727377169, 0.0},
      {0.0964083579062225, 0.8928281693560609, 0.0},
      {0.0107634727377166, 0.0964083579062228, 0.0},
      {0.1003380079665996, 0.0267977697259695, 0.0},
      {0.8728642223074309, 0.1003380079665996, 0.0},
      {0.0267977697259696, 0.8728642223074310, 0.0},
      {0.8728642223074310, 0.0267977697259695, 0.0},
      {0.1003380079665996, 0.8728642223074310, 0.0},
      {0.0267977697259692, 0.1003380079665999, 0.0},
      {0.2894378430839789, 0.1999016566395393, 0.0},
      {0.5106605002764818, 0.2894378430839790, 0.0},
      {0.1999016566395393, 0.5106605002764819, 0.0},
      {0.5106605002764818, 0.1999016566395393, 0.0},
      {0.2894378430839789, 0.5106605002764819, 0.0},
      {0.1999016566395392, 0.2894378430839791, 0.0},
      {0.4595867718244119, 0.0808264563511763, 0.0},
      {0.4595867718244119, 0.4595867718244119, 0.0},
      {0.0808264563511762, 0.4595867718244120, 0.0},
      {0.1786164539550919, 0.0096597928476050, 0.0},
      {0.8117237531973031, 0.1786164539550919, 0.0},
      {0.0096597928476050, 0.8117237531973033, 0.0},
      {0.8117237531973032, 0.0096597928476050, 0.0},
      {0.1786164539550919, 0.8117237531973033, 0.0},
      {0.0096597928476048, 0.1786164539550922, 0.0},
      {0.1133332341350742, 0.1133332341350745, 0.0},
      {0.7733335317298511, 0.1133332341350743, 0.0},
      {0.1133332341350746, 0.7733335317298513, 0.0},
      {0.2861539226312519, 0.0436162890256224, 0.0},
      {0.6702297883431256, 0.2861539226312520, 0.0},
      {0.0436162890256225, 0.6702297883431257, 0.0},
      {0.6702297883431256, 0.0436162890256224, 0.0},
      {0.2861539226312521, 0.6702297883431256, 0.0},
      {0.0436162890256222, 0.2861539226312523, 0.0},
      {0.2293288959608979, 0.0096337955497694, 0.0},
      {0.7610373084893327, 0.2293288959608979, 0.0},
      {0.0096337955497694, 0.7610373084893329, 0.0},
      {0.7610373084893327, 0.0096337955497694, 0.0},
      {0.2293288959608979, 0.7610373084893329, 0.0},
      {0.0096337955497692, 0.2293288959608982, 0.0},
      {0.3542128961664716, 0.1164570021517526, 0.0},
      {0.5293301016817759, 0.3542128961664717, 0.0},
      {0.1164570021517525, 0.5293301016817761, 0.0},
      {0.5293301016817759, 0.1164570021517526, 0.0},
      {0.3542128961664716, 0.5293301016817760, 0.0},
      {0.1164570021517524, 0.3542128961664718, 0.0},
      {0.3449444915449854, 0.0094768771491045, 0.0},
      {0.6455786313059102, 0.3449444915449854, 0.0},
      {0.0094768771491044, 0.6455786313059104, 0.0},
      {0.6455786313059101, 0.0094768771491045, 0.0},
      {0.3449444915449854, 0.6455786313059102, 0.0},
      {0.0094768771491043, 0.3449444915449857, 0.0},
      {0.4316732308235539, 0.0279666162959973, 0.0},
      {0.5403601528804488, 0.4316732308235540, 0.0},
      {0.0279666162959972, 0.5403601528804491, 0.0},
      {0.5403601528804488, 0.0279666162959973, 0.0},
      {0.4316732308235539, 0.5403601528804489, 0.0},
      {0.0279666162959972, 0.4316732308235541, 0.0},
      {0.2207972237518396, 0.1472658441967969, 0.0},
      {0.6319369320513635, 0.2207972237518397, 0.0},
      {0.1472658441967969, 0.6319369320513636, 0.0},
      {0.6319369320513635, 0.1472658441967969, 0.0},
      {0.2207972237518397, 0.6319369320513636, 0.0},
      {0.1472658441967967, 0.2207972237518398, 0.0},
      {0.0109390130646629, 0.0109390130646632, 0.0},
      {0.9781219738706738, 0.0109390130646629, 0.0},
      {0.0109390130646634, 0.9781219738706741, 0.0},
      {0.1531728350976843, 0.0018705669110342, 0.0},
      {0.8449565979912814, 0.1531728350976843, 0.0},
      {0.0018705669110343, 0.8449565979912816, 0.0},
      {0.8449565979912815, 0.0018705669110342, 0.0},
      {0.1531728350976843, 0.8449565979912816, 0.0},
      {0.0018705669110340, 0.1531728350976846, 0.0},
      {0.0786463619513997, 0.0786463619514000, 0.0},
      {0.8427072760972003, 0.0786463619513997, 0.0},
      {0.0786463619514000, 0.8427072760972005, 0.0},
      {0.0091986167658109, 0.0020671401785267, 0.0},
      {0.9887342430556623, 0.0091986167658108, 0.0},
      {0.0020671401785269, 0.9887342430556626, 0.0},
      {0.9887342430556624, 0.0020671401785267, 0.0},
      {0.0091986167658110, 0.9887342430556626, 0.0},
      {0.0020671401785264, 0.0091986167658113, 0.0},
      {0.0235452342574103, 0.0020063808420238, 0.0},
      {0.9744483849005658, 0.0235452342574103, 0.0},
      {0.0020063808420239, 0.9744483849005661, 0.0},
      {0.9744483849005658, 0.0020063808420238, 0.0},
      {0.0235452342574104, 0.9744483849005661, 0.0},
      {0.0020063808420235, 0.0235452342574107, 0.0},
      {0.3457284797184662, 0.1965919922795324, 0.0},
      {0.4576795280020015, 0.3457284797184662, 0.0},
      {0.1965919922795323, 0.4576795280020016, 0.0},
      {0.4576795280020015, 0.1965919922795324, 0.0},
      {0.3457284797184662, 0.4576795280020016, 0.0},
      {0.1965919922795323, 0.3457284797184664, 0.0},
      {0.2501693088121573, 0.1074788661022307, 0.0},
      {0.6423518250856121, 0.2501693088121573, 0.0},
      {0.1074788661022307, 0.6423518250856122, 0.0},
      {0.6423518250856121, 0.1074788661022307, 0.0},
      {0.2501693088121574, 0.6423518250856121, 0.0},
      {0.1074788661022305, 0.2501693088121575, 0.0},
      {0.2555498626016329, 0.0018618093532657, 0.0},
      {0.7425883280451013, 0.2555498626016329, 0.0},
      {0.0018618093532657, 0.7425883280451016, 0.0},
      {0.7425883280451014, 0.0018618093532657, 0.0},
      {0.2555498626016329, 0.7425883280451016, 0.0},
      {0.0018618093532655, 0.2555498626016332, 0.0},
      {0.1032934352626250, 0.0502072925915597, 0.0},
      {0.8464992721458153, 0.1032934352626250, 0.0},
      {0.0502072925915598, 0.8464992721458154, 0.0},
      {0.8464992721458153, 0.0502072925915597, 0.0},
      {0.1032934352626251, 0.8464992721458154, 0.0},
      {0.0502072925915595, 0.1032934352626254, 0.0},
      {0.1702588823641313, 0.1702588823641316, 0.0},
      {0.6594822352717371, 0.1702588823641314, 0.0},
      {0.1702588823641316, 0.6594822352717372, 0.0},
      {0.1134827791905402, 0.0790987667568746, 0.0},
      {0.8074184540525852, 0.1134827791905402, 0.0},
      {0.0790987667568747, 0.8074184540525853, 0.0},
      {0.8074184540525853, 0.0790987667568746, 0.0},
      {0.1134827791905403, 0.8074184540525853, 0.0},
      {0.0790987667568744, 0.1134827791905405, 0.0},
      {0.2020015191395728, 0.0018288696800151, 0.0},
      {0.7961696111804121, 0.2020015191395728, 0.0},
      {0.0018288696800152, 0.7961696111804121, 0.0},
      {0.7961696111804121, 0.0018288696800151, 0.0},
      {0.2020015191395729, 0.7961696111804121, 0.0},
      {0.0018288696800149, 0.2020015191395731, 0.0},
      {0.1996334593368171, 0.1996334593368174, 0.0},
      {0.6007330813263655, 0.1996334593368172, 0.0},
      {0.1996334593368173, 0.6007330813263656, 0.0},
      {0.1554072933416661, 0.0974984606749395, 0.0},
      {0.7470942459833944, 0.1554072933416662, 0.0},
      {0.0974984606749396, 0.7470942459833945, 0.0},
      {0.7470942459833944, 0.0974984606749395, 0.0},
      {0.1554072933416663, 0.7470942459833945, 0.0},
      {0.0974984606749393, 0.1554072933416664, 0.0},
      {0.2837331258499088, 0.0102710427379366, 0.0},
      {0.7059958314121547, 0.2837331258499088, 0.0},
      {0.0102710427379366, 0.7059958314121549, 0.0},
      {0.7059958314121546, 0.0102710427379366, 0.0},
      {0.2837331258499087, 0.7059958314121548, 0.0},
      {0.0102710427379364, 0.2837331258499091, 0.0},
      {0.4006689325488300, 0.1986621349023400, 0.0},
      {0.4006689325488300, 0.4006689325488301, 0.0},
      {0.1986621349023399, 0.4006689325488302, 0.0},
      {0.1406138740563841, 0.1406138740563844, 0.0},
      {0.7187722518872315, 0.1406138740563842, 0.0},
      {0.1406138740563844, 0.7187722518872316, 0.0},
      {0.1101376107119611, 0.0019450667745237, 0.0},
      {0.8879173225135152, 0.1101376107119611, 0.0},
      {0.0019450667745238, 0.8879173225135153, 0.0},
      {0.8879173225135152, 0.0019450667745237, 0.0},
      {0.1101376107119612, 0.8879173225135153, 0.0},
      {0.0019450667745234, 0.1101376107119614, 0.0},
      {0.3511903644454155, 0.2976192711091690, 0.0},
      {0.3511903644454155, 0.3511903644454157, 0.0},
      {0.2976192711091689, 0.3511903644454157, 0.0},
      {0.0422503625151380, 0.0422503625151383, 0.0},
      {0.9154992749697237, 0.0422503625151379, 0.0},
      {0.0422503625151384, 0.9154992749697239, 0.0},
      {0.3116269863962924, 0.0020127149292621, 0.0},
      {0.6863602986744455, 0.3116269863962924, 0.0},
      {0.0020127149292622, 0.6863602986744456, 0.0},
      {0.6863602986744455, 0.0020127149292621, 0.0},
      {0.3116269863962924, 0.6863602986744456, 0.0},
      {0.0020127149292620, 0.3116269863962926, 0.0},
      {0.4986838983120579, 0.0026322033758842, 0.0},
      {0.4986838983120579, 0.4986838983120580, 0.0},
      {0.0026322033758840, 0.4986838983120582, 0.0},
      {0.4560656275499566, 0.0098798242146206, 0.0},
      {0.5340545482354229, 0.4560656275499567, 0.0},
      {0.0098798242146205, 0.5340545482354231, 0.0},
      {0.5340545482354229, 0.0098798242146206, 0.0},
      {0.4560656275499566, 0.5340545482354230, 0.0},
      {0.0098798242146206, 0.4560656275499568, 0.0},
      {0.2964798675129709, 0.2964798675129711, 0.0},
      {0.4070402649740581, 0.2964798675129710, 0.0},
      {0.2964798675129710, 0.4070402649740582, 0.0},
      {0.0740590746798509, 0.0022261525470101, 0.0},
      {0.9237147727731391, 0.0740590746798509, 0.0},
      {0.0022261525470102, 0.9237147727731391, 0.0},
      {0.9237147727731391, 0.0022261525470101, 0.0},
      {0.0740590746798510, 0.9237147727731391, 0.0},
      {0.0022261525470098, 0.0740590746798513, 0.0}},
      {0.0016643475873566,
      0.0016643475873566,
      0.0016643475873566,
      0.0001736640797462,
      0.0001736640797462,
      0.0001736640797462,
      0.0001736640797462,
      0.0001736640797462,
      0.0001736640797462,
      0.0001500250509550,
      0.0001500250509550,
      0.0001500250509550,
      0.0001500250509550,
      0.0001500250509550,
      0.0001500250509550,
      0.0013652010296149,
      0.0013652010296149,
      0.0013652010296149,
      0.0013652010296149,
      0.0013652010296149,
      0.0013652010296149,
      0.0005664857258923,
      0.0005664857258923,
      0.0005664857258923,
      0.0005664857258923,
      0.0005664857258923,
      0.0005664857258923,
      0.0000682609204984,
      0.0000682609204984,
      0.0000682609204984,
      0.0000682609204984,
      0.0000682609204984,
      0.0000682609204984,
      0.0003846880333921,
      0.0003846880333921,
      0.0003846880333921,
      0.0003846880333921,
      0.0003846880333921,
      0.0003846880333921,
      0.0006484865973060,
      0.0006484865973060,
      0.0006484865973060,
      0.0006484865973060,
      0.0006484865973060,
      0.0006484865973060,
      0.0000182966121232,
      0.0000182966121232,
      0.0000182966121232,
      0.0014029275810552,
      0.0014029275810552,
      0.0014029275810552,
      0.0014029275810552,
      0.0014029275810552,
      0.0014029275810552,
      0.0008544121114967,
      0.0008544121114967,
      0.0008544121114967,
      0.0008544121114967,
      0.0008544121114967,
      0.0008544121114967,
      0.0007417634716093,
      0.0007417634716093,
      0.0007417634716093,
      0.0007417634716093,
      0.0007417634716093,
      0.0007417634716093,
      0.0008256981786396,
      0.0008256981786396,
      0.0008256981786396,
      0.0021270972166203,
      0.0021270972166203,
      0.0021270972166203,
      0.0021270972166203,
      0.0021270972166203,
      0.0021270972166203,
      0.0010988997933592,
      0.0010988997933592,
      0.0010988997933592,
      0.0010988997933592,
      0.0010988997933592,
      0.0010988997933592,
      0.0007742909739458,
      0.0007742909739458,
      0.0007742909739458,
      0.0007742909739458,
      0.0007742909739458,
      0.0007742909739458,
      0.0013016402650768,
      0.0013016402650768,
      0.0013016402650768,
      0.0013016402650768,
      0.0013016402650768,
      0.0013016402650768,
      0.0002330851320517,
      0.0002330851320517,
      0.0002330851320517,
      0.0002330851320517,
      0.0002330851320517,
      0.0002330851320517,
      0.0003287885781867,
      0.0003287885781867,
      0.0003287885781867,
      0.0003287885781867,
      0.0003287885781867,
      0.0003287885781867,
      0.0013932327047497,
      0.0013932327047497,
      0.0013932327047497,
      0.0013932327047497,
      0.0013932327047497,
      0.0013932327047497,
      0.0021295483390957,
      0.0021295483390957,
      0.0021295483390957,
      0.0020648284765708,
      0.0020648284765708,
      0.0020648284765708,
      0.0020648284765708,
      0.0020648284765708,
      0.0020648284765708,
      0.0010771414586637,
      0.0010771414586637,
      0.0010771414586637,
      0.0010771414586637,
      0.0010771414586637,
      0.0010771414586637,
      0.0017379761008355,
      0.0017379761008355,
      0.0017379761008355,
      0.0017379761008355,
      0.0017379761008355,
      0.0017379761008355,
      0.0012121399787050,
      0.0012121399787050,
      0.0012121399787050,
      0.0013533071271986,
      0.0013533071271986,
      0.0013533071271986,
      0.0013533071271986,
      0.0013533071271986,
      0.0013533071271986,
      0.0001928896434911,
      0.0001928896434911,
      0.0001928896434911,
      0.0001928896434911,
      0.0001928896434911,
      0.0001928896434911,
      0.0017198332146128,
      0.0017198332146128,
      0.0017198332146128,
      0.0017198332146128,
      0.0017198332146128,
      0.0017198332146128,
      0.0002190134685452,
      0.0002190134685452,
      0.0002190134685452,
      0.0002190134685452,
      0.0002190134685452,
      0.0002190134685452,
      0.0022030072639051,
      0.0022030072639051,
      0.0022030072639051,
      0.0022030072639051,
      0.0022030072639051,
      0.0022030072639051,
      0.0010087504042277,
      0.0010087504042277,
      0.0010087504042277,
      0.0010087504042277,
      0.0010087504042277,
      0.0010087504042277,
      0.0016191393022749,
      0.0016191393022749,
      0.0016191393022749,
      0.0016191393022749,
      0.0016191393022749,
      0.0016191393022749,
      0.0023331053995970,
      0.0023331053995970,
      0.0023331053995970,
      0.0023331053995970,
      0.0023331053995970,
      0.0023331053995970,
      0.0015796129088775,
      0.0015796129088775,
      0.0015796129088775,
      0.0015796129088775,
      0.0015796129088775,
      0.0015796129088775,
      0.0017367922592231,
      0.0017367922592231,
      0.0017367922592231,
      0.0017367922592231,
      0.0017367922592231,
      0.0017367922592231,
      0.0004476964402049,
      0.0004476964402049,
      0.0004476964402049,
      0.0004476964402049,
      0.0004476964402049,
      0.0004476964402049,
      0.0005402392418127,
      0.0005402392418127,
      0.0005402392418127,
      0.0005402392418127,
      0.0005402392418127,
      0.0005402392418127,
      0.0009064872245550,
      0.0009064872245550,
      0.0009064872245550,
      0.0009064872245550,
      0.0009064872245550,
      0.0009064872245550,
      0.0003825785652431,
      0.0003825785652431,
      0.0003825785652431,
      0.0003825785652431,
      0.0003825785652431,
      0.0003825785652431,
      0.0014288408521395,
      0.0014288408521395,
      0.0014288408521395,
      0.0014288408521395,
      0.0014288408521395,
      0.0014288408521395,
      0.0013279060452758,
      0.0013279060452758,
      0.0013279060452758,
      0.0013279060452758,
      0.0013279060452758,
      0.0013279060452758,
      0.0022387188105533,
      0.0022387188105533,
      0.0022387188105533,
      0.0022387188105533,
      0.0022387188105533,
      0.0022387188105533,
      0.0008267586306367,
      0.0008267586306367,
      0.0008267586306367,
      0.0008267586306367,
      0.0008267586306367,
      0.0008267586306367,
      0.0009724948178572,
      0.0009724948178572,
      0.0009724948178572,
      0.0009724948178572,
      0.0009724948178572,
      0.0009724948178572,
      0.0020272587903587,
      0.0020272587903587,
      0.0020272587903587,
      0.0020272587903587,
      0.0020272587903587,
      0.0020272587903587,
      0.0011735985049339,
      0.0011735985049339,
      0.0011735985049339,
      0.0011735985049339,
      0.0011735985049339,
      0.0011735985049339,
      0.0027098983409458,
      0.0027098983409458,
      0.0027098983409458,
      0.0027098983409458,
      0.0027098983409458,
      0.0027098983409458,
      0.0004107095577781,
      0.0004107095577781,
      0.0004107095577781,
      0.0004107095577781,
      0.0004107095577781,
      0.0004107095577781,
      0.0006665615373910,
      0.0006665615373910,
      0.0006665615373910,
      0.0006665615373910,
      0.0006665615373910,
      0.0006665615373910,
      0.0023363011519203,
      0.0023363011519203,
      0.0023363011519203,
      0.0023363011519203,
      0.0023363011519203,
      0.0023363011519203,
      0.0017913556055310,
      0.0017913556055310,
      0.0017913556055310,
      0.0005236746221755,
      0.0005236746221755,
      0.0005236746221755,
      0.0005236746221755,
      0.0005236746221755,
      0.0005236746221755,
      0.0013897994002223,
      0.0013897994002223,
      0.0013897994002223,
      0.0012785064104471,
      0.0012785064104471,
      0.0012785064104471,
      0.0012785064104471,
      0.0012785064104471,
      0.0012785064104471,
      0.0005768829640657,
      0.0005768829640657,
      0.0005768829640657,
      0.0005768829640657,
      0.0005768829640657,
      0.0005768829640657,
      0.0021118787915446,
      0.0021118787915446,
      0.0021118787915446,
      0.0021118787915446,
      0.0021118787915446,
      0.0021118787915446,
      0.0006687087572516,
      0.0006687087572516,
      0.0006687087572516,
      0.0006687087572516,
      0.0006687087572516,
      0.0006687087572516,
      0.0011907116522871,
      0.0011907116522871,
      0.0011907116522871,
      0.0011907116522871,
      0.0011907116522871,
      0.0011907116522871,
      0.0021414262260095,
      0.0021414262260095,
      0.0021414262260095,
      0.0021414262260095,
      0.0021414262260095,
      0.0021414262260095,
      0.0001499760934574,
      0.0001499760934574,
      0.0001499760934574,
      0.0002203469792688,
      0.0002203469792688,
      0.0002203469792688,
      0.0002203469792688,
      0.0002203469792688,
      0.0002203469792688,
      0.0010339358335399,
      0.0010339358335399,
      0.0010339358335399,
      0.0000578226288754,
      0.0000578226288754,
      0.0000578226288754,
      0.0000578226288754,
      0.0000578226288754,
      0.0000578226288754,
      0.0000912125509020,
      0.0000912125509020,
      0.0000912125509020,
      0.0000912125509020,
      0.0000912125509020,
      0.0000912125509020,
      0.0025675579236344,
      0.0025675579236344,
      0.0025675579236344,
      0.0025675579236344,
      0.0025675579236344,
      0.0025675579236344,
      0.0019525278550773,
      0.0019525278550773,
      0.0019525278550773,
      0.0019525278550773,
      0.0019525278550773,
      0.0019525278550773,
      0.0002641087799170,
      0.0002641087799170,
      0.0002641087799170,
      0.0002641087799170,
      0.0002641087799170,
      0.0002641087799170,
      0.0009495317379236,
      0.0009495317379236,
      0.0009495317379236,
      0.0009495317379236,
      0.0009495317379236,
      0.0009495317379236,
      0.0020138554942591,
      0.0020138554942591,
      0.0020138554942591,
      0.0011670296560819,
      0.0011670296560819,
      0.0011670296560819,
      0.0011670296560819,
      0.0011670296560819,
      0.0011670296560819,
      0.0002413353389898,
      0.0002413353389898,
      0.0002413353389898,
      0.0002413353389898,
      0.0002413353389898,
      0.0002413353389898,
      0.0022853369445631,
      0.0022853369445631,
      0.0022853369445631,
      0.0016245097246607,
      0.0016245097246607,
      0.0016245097246607,
      0.0016245097246607,
      0.0016245097246607,
      0.0016245097246607,
      0.0006598834583838,
      0.0006598834583838,
      0.0006598834583838,
      0.0006598834583838,
      0.0006598834583838,
      0.0006598834583838,
      0.0025826779637270,
      0.0025826779637270,
      0.0025826779637270,
      0.0017426316767263,
      0.0017426316767263,
      0.0017426316767263,
      0.0001983646305222,
      0.0001983646305222,
      0.0001983646305222,
      0.0001983646305222,
      0.0001983646305222,
      0.0001983646305222,
      0.0029284072269592,
      0.0029284072269592,
      0.0029284072269592,
      0.0005994743919192,
      0.0005994743919192,
      0.0005994743919192,
      0.0002927967205105,
      0.0002927967205105,
      0.0002927967205105,
      0.0002927967205105,
      0.0002927967205105,
      0.0002927967205105,
      0.0003746610270526,
      0.0003746610270526,
      0.0003746610270526,
      0.0007003874423756,
      0.0007003874423756,
      0.0007003874423756,
      0.0007003874423756,
      0.0007003874423756,
      0.0007003874423756,
      0.0028868194517962,
      0.0028868194517962,
      0.0028868194517962,
      0.0001852527744071,
      0.0001852527744071,
      0.0001852527744071,
      0.0001852527744071,
      0.0001852527744071,
      0.0001852527744071}
    }
  };
    
} // end namespace Intrepid2
