/* SPDX-License-Identifier: GPL-2.0 */
#include <linux/bitops.h>
#include <linux/device.h>
#include <linux/regmap.h>

/* BMP380 specific registers */
#define BMP380_REG_CMD			0x7E
#define BMP380_REG_CONFIG		0x1F
#define BMP380_REG_ODR			0x1D
#define BMP380_REG_OSR			0x1C
#define BMP380_REG_POWER_CONTROL	0x1B
#define BMP380_REG_IF_CONFIG		0x1A
#define BMP380_REG_INT_CONTROL		0x19
#define BMP380_REG_INT_STATUS		0x11
#define BMP380_REG_EVENT		0x10
#define BMP380_REG_STATUS		0x03
#define BMP380_REG_ERROR		0x02
#define BMP380_REG_ID			0x00

#define BMP380_REG_FIFO_CONFIG_1	0x18
#define BMP380_REG_FIFO_CONFIG_2	0x17
#define BMP380_REG_FIFO_WATERMARK_MSB	0x16
#define BMP380_REG_FIFO_WATERMARK_LSB	0x15
#define BMP380_REG_FIFO_DATA		0x14
#define BMP380_REG_FIFO_LENGTH_MSB	0x13
#define BMP380_REG_FIFO_LENGTH_LSB	0x12

#define BMP380_REG_SENSOR_TIME_MSB	0x0E
#define BMP380_REG_SENSOR_TIME_LSB	0x0D
#define BMP380_REG_SENSOR_TIME_XLSB	0x0C

#define BMP380_REG_TEMP_MSB		0x09
#define BMP380_REG_TEMP_LSB		0x08
#define BMP380_REG_TEMP_XLSB		0x07

#define BMP380_REG_PRESS_MSB		0x06
#define BMP380_REG_PRESS_LSB		0x05
#define BMP380_REG_PRESS_XLSB		0x04

#define BMP380_REG_CALIB_TEMP_START	0x31
#define BMP380_CALIB_REG_COUNT		21

#define BMP380_FILTER_MASK		GENMASK(3, 1)
#define BMP380_FILTER_OFF		0
#define BMP380_FILTER_1X		1
#define BMP380_FILTER_3X		2
#define BMP380_FILTER_7X		3
#define BMP380_FILTER_15X		4
#define BMP380_FILTER_31X		5
#define BMP380_FILTER_63X		6
#define BMP380_FILTER_127X		7

#define BMP380_OSRS_TEMP_MASK		GENMASK(5, 3)
#define BMP380_OSRS_PRESS_MASK		GENMASK(2, 0)

#define BMP380_ODRS_MASK		GENMASK(4, 0)

#define BMP380_CTRL_SENSORS_MASK	GENMASK(1, 0)
#define BMP380_CTRL_SENSORS_PRESS_EN	BIT(0)
#define BMP380_CTRL_SENSORS_TEMP_EN	BIT(1)
#define BMP380_MODE_MASK		GENMASK(5, 4)
#define BMP380_MODE_SLEEP		0
#define BMP380_MODE_FORCED		1
#define BMP380_MODE_NORMAL		3

#define BMP380_MIN_TEMP			-4000
#define BMP380_MAX_TEMP			8500
#define BMP380_MIN_PRES			3000000
#define BMP380_MAX_PRES			12500000

#define BMP380_CMD_NOOP			0x00
#define BMP380_CMD_EXTMODE_EN_MID	0x34
#define BMP380_CMD_FIFO_FLUSH		0xB0
#define BMP380_CMD_SOFT_RESET		0xB6

#define BMP380_STATUS_CMD_RDY_MASK	BIT(4)
#define BMP380_STATUS_DRDY_PRESS_MASK	BIT(5)
#define BMP380_STATUS_DRDY_TEMP_MASK	BIT(6)

#define BMP380_ERR_FATAL_MASK		BIT(0)
#define BMP380_ERR_CMD_MASK		BIT(1)
#define BMP380_ERR_CONF_MASK		BIT(2)

#define BMP380_TEMP_SKIPPED		0x800000
#define BMP380_PRESS_SKIPPED		0x800000

/* BMP280 specific registers */
#define BMP280_REG_HUMIDITY_LSB		0xFE
#define BMP280_REG_HUMIDITY_MSB		0xFD
#define BMP280_REG_TEMP_XLSB		0xFC
#define BMP280_REG_TEMP_LSB		0xFB
#define BMP280_REG_TEMP_MSB		0xFA
#define BMP280_REG_PRESS_XLSB		0xF9
#define BMP280_REG_PRESS_LSB		0xF8
#define BMP280_REG_PRESS_MSB		0xF7

/* Helper mask to truncate excess 4 bits on pressure and temp readings */
#define BMP280_MEAS_TRIM_MASK		GENMASK(24, 4)

#define BMP280_REG_CONFIG		0xF5
#define BMP280_REG_CTRL_MEAS		0xF4
#define BMP280_REG_STATUS		0xF3
#define BMP280_REG_CTRL_HUMIDITY	0xF2

/* Due to non linear mapping, and data sizes we can't do a bulk read */
#define BMP280_REG_COMP_H1		0xA1
#define BMP280_REG_COMP_H2		0xE1
#define BMP280_REG_COMP_H3		0xE3
#define BMP280_REG_COMP_H4		0xE4
#define BMP280_REG_COMP_H5		0xE5
#define BMP280_REG_COMP_H6		0xE7

#define BMP280_REG_COMP_TEMP_START	0x88
#define BMP280_COMP_TEMP_REG_COUNT	6

#define BMP280_REG_COMP_PRESS_START	0x8E
#define BMP280_COMP_PRESS_REG_COUNT	18

#define BMP280_COMP_H5_MASK		GENMASK(15, 4)

#define BMP280_CONTIGUOUS_CALIB_REGS	(BMP280_COMP_TEMP_REG_COUNT + \
					 BMP280_COMP_PRESS_REG_COUNT)

#define BMP280_FILTER_MASK		GENMASK(4, 2)
#define BMP280_FILTER_OFF		0
#define BMP280_FILTER_2X		1
#define BMP280_FILTER_4X		2
#define BMP280_FILTER_8X		3
#define BMP280_FILTER_16X		4

#define BMP280_OSRS_HUMIDITY_MASK	GENMASK(2, 0)
#define BMP280_OSRS_HUMIDITY_SKIP	0
#define BMP280_OSRS_HUMIDITY_1X		1
#define BMP280_OSRS_HUMIDITY_2X		2
#define BMP280_OSRS_HUMIDITY_4X		3
#define BMP280_OSRS_HUMIDITY_8X		4
#define BMP280_OSRS_HUMIDITY_16X	5

#define BMP280_OSRS_TEMP_MASK		GENMASK(7, 5)
#define BMP280_OSRS_TEMP_SKIP		0
#define BMP280_OSRS_TEMP_1X		1
#define BMP280_OSRS_TEMP_2X		2
#define BMP280_OSRS_TEMP_4X		3
#define BMP280_OSRS_TEMP_8X		4
#define BMP280_OSRS_TEMP_16X		5

#define BMP280_OSRS_PRESS_MASK		GENMASK(4, 2)
#define BMP280_OSRS_PRESS_SKIP		0
#define BMP280_OSRS_PRESS_1X		1
#define BMP280_OSRS_PRESS_2X		2
#define BMP280_OSRS_PRESS_4X		3
#define BMP280_OSRS_PRESS_8X		4
#define BMP280_OSRS_PRESS_16X		5

#define BMP280_MODE_MASK		GENMASK(1, 0)
#define BMP280_MODE_SLEEP		0
#define BMP280_MODE_FORCED		1
#define BMP280_MODE_NORMAL		3

/* BMP180 specific registers */
#define BMP180_REG_OUT_XLSB		0xF8
#define BMP180_REG_OUT_LSB		0xF7
#define BMP180_REG_OUT_MSB		0xF6

#define BMP180_REG_CALIB_START		0xAA
#define BMP180_REG_CALIB_COUNT		22

#define BMP180_MEAS_CTRL_MASK		GENMASK(4, 0)
#define BMP180_MEAS_TEMP		0x0E
#define BMP180_MEAS_PRESS		0x14
#define BMP180_MEAS_SCO			BIT(5)
#define BMP180_OSRS_PRESS_MASK		GENMASK(7, 6)
#define BMP180_MEAS_PRESS_1X		0
#define BMP180_MEAS_PRESS_2X		1
#define BMP180_MEAS_PRESS_4X		2
#define BMP180_MEAS_PRESS_8X		3

/* BMP180 and BMP280 common registers */
#define BMP280_REG_CTRL_MEAS		0xF4
#define BMP280_REG_RESET		0xE0
#define BMP280_REG_ID			0xD0

#define BMP380_CHIP_ID			0x50
#define BMP180_CHIP_ID			0x55
#define BMP280_CHIP_ID			0x58
#define BME280_CHIP_ID			0x60
#define BMP280_SOFT_RESET_VAL		0xB6

/* BMP280 register skipped special values */
#define BMP280_TEMP_SKIPPED		0x80000
#define BMP280_PRESS_SKIPPED		0x80000
#define BMP280_HUMIDITY_SKIPPED		0x8000

/* Regmap configurations */
extern const struct regmap_config bmp180_regmap_config;
extern const struct regmap_config bmp280_regmap_config;
extern const struct regmap_config bmp380_regmap_config;

/* Probe called from different transports */
int bmp280_common_probe(struct device *dev,
			struct regmap *regmap,
			unsigned int chip,
			const char *name,
			int irq);

/* PM ops */
extern const struct dev_pm_ops bmp280_dev_pm_ops;
